// Reports page - generic reporting UI powered by api/reports.php
(function(){
  window.pages = window.pages || {};

  const esc = (s)=>String(s ?? '').replace(/[&<>"']/g,c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c]));
  const getRole = ()=> (window.auth?.currentUser?.role_slug) || (window.auth?.currentUser?.role) || '';
  const isReportsAllowed = ()=> ['super_admin','admin','hr_head','hr_officer'].includes(getRole());

  async function render(container){
    if (!isReportsAllowed()){
      container.innerHTML = '<div class="alert alert-warning">Access denied. Reports are available to Admin and HR roles only.</div>';
      return;
    }
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Reports</h5>
          <div class="d-flex flex-wrap gap-2">
            <div class="dropdown">
              <button class="btn btn-sm btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown"><i class="fas fa-folder-open"></i> Saved</button>
              <div class="dropdown-menu dropdown-menu-end p-2" style="min-width:280px; max-height:260px; overflow:auto;">
                <div id="repSavedList">Loading...</div>
                <div class="dropdown-divider"></div>
                <button class="btn btn-sm btn-outline-primary w-100" id="repSaveBtn"><i class="fas fa-save"></i> Save current</button>
              </div>
            </div>
            <button class="btn btn-sm btn-outline-secondary" id="repReset"><i class="fas fa-rotate"></i> Reset</button>
            <button class="btn btn-sm btn-primary" id="repRun"><i class="fas fa-play"></i> Run</button>
            <button class="btn btn-sm btn-outline-primary" id="repEmail"><i class="fas fa-paper-plane"></i> Email</button>
            <button class="btn btn-sm btn-outline-primary" id="repSchedule"><i class="fas fa-calendar"></i> Schedule</button>
          </div>
        </div>
        <div class="card-body">
          <div class="row g-3 align-items-end">
            <div class="col-md-3">
              <label class="form-label">Entity</label>
              <select id="repEntity" class="form-select"></select>
            </div>
            <div class="col-md-3">
              <label class="form-label">Date From</label>
              <input type="date" id="repDateFrom" class="form-control" />
            </div>
            <div class="col-md-3">
              <label class="form-label">Date To</label>
              <input type="date" id="repDateTo" class="form-control" />
            </div>
            <div class="col-md-3">
              <label class="form-label">Search</label>
              <input type="text" id="repSearch" class="form-control" placeholder="Search..." />
            </div>
            <div class="col-12">
              <a class="small" href="#" id="repToggleConfig"><i class="fas fa-sliders"></i> Configure columns & filters</a>
            </div>
          </div>

          <div id="repConfig" class="border rounded p-3 mt-3" style="display:none;">
            <div class="row g-3">
              <div class="col-md-6">
                <label class="form-label">Columns</label>
                <div id="repCols" class="vstack gap-1" style="max-height:220px; overflow:auto; border:1px dashed #ddd; padding:8px; border-radius:6px;"></div>
              </div>
              <div class="col-md-6">
                <label class="form-label">Filters</label>
                <div id="repFilters" class="vstack gap-2" style="max-height:220px; overflow:auto;"></div>
                <button class="btn btn-sm btn-outline-primary mt-2" id="repAddFilter"><i class="fas fa-plus"></i> Add filter</button>
              </div>
            </div>
          </div>

          <div class="d-flex justify-content-between align-items-center mt-3">
            <div class="d-flex align-items-center gap-2">
              <button class="btn btn-sm btn-outline-secondary" id="repCSV"><i class="fas fa-file-csv"></i> Export CSV</button>
              <button class="btn btn-sm btn-outline-secondary" id="repXLSX"><i class="fas fa-file-excel"></i> Export Excel</button>
            </div>
            <div class="d-flex align-items-center gap-2">
              <select id="repChartType" class="form-select form-select-sm" style="width:130px;">
                <option value="bar">Bar</option>
                <option value="line">Line</option>
                <option value="pie">Pie</option>
              </select>
              <select id="repChartX" class="form-select form-select-sm" style="width:160px;"></select>
              <select id="repChartY" class="form-select form-select-sm" style="width:160px;"></select>
              <select id="repChartAgg" class="form-select form-select-sm" style="width:140px;">
                <option value="count">Count</option>
                <option value="sum">Sum</option>
                <option value="avg">Average</option>
              </select>
              <button class="btn btn-sm btn-outline-primary" id="repRenderChart"><i class="fas fa-chart-column"></i> Chart</button>
            </div>
          </div>

          <div class="table-responsive mt-3">
            <table class="table table-sm align-middle" id="repTable">
              <thead><tr id="repHead"></tr></thead>
              <tbody id="repBody"><tr><td>Loading...</td></tr></tbody>
            </table>
          </div>

          <div class="d-flex justify-content-between align-items-center">
            <div class="text-muted small" id="repInfo"></div>
            <div class="btn-group btn-group-sm" role="group">
              <button class="btn btn-outline-secondary" id="repPrev">Prev</button>
              <button class="btn btn-outline-secondary" id="repNext">Next</button>
            </div>
          </div>

          <div class="mt-3">
            <canvas id="repChart" height="110"></canvas>
          </div>
        </div>
      </div>
    `;

    // State
    const state = { schema:{}, entity:'', columns:[], selectedCols:[], filters:[], sort:'', page:1, pageSize:50, rows:[], savedReports:[], selectedSavedId:null };

    // UI refs
    const entitySel = container.querySelector('#repEntity');
    const dateFrom = container.querySelector('#repDateFrom');
    const dateTo = container.querySelector('#repDateTo');
    const searchInput = container.querySelector('#repSearch');
    const colsWrap = container.querySelector('#repCols');
    const filtersWrap = container.querySelector('#repFilters');
    const configBox = container.querySelector('#repConfig');
    const headRow = container.querySelector('#repHead');
    const body = container.querySelector('#repBody');
    const info = container.querySelector('#repInfo');
    const savedListWrap = container.querySelector('#repSavedList');

    // Load schema
    const schemaAll = await fetch('api/reports.php?action=schema', { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
    if (schemaAll?.success) state.schema = schemaAll.data || {}; else state.schema = {};

    // Populate entities
    entitySel.innerHTML = Object.keys(state.schema).map(k=>`<option value="${k}">${k.replace('_',' ').replace(/\b\w/g,c=>c.toUpperCase())}</option>`).join('');
    state.entity = entitySel.value || Object.keys(state.schema)[0] || '';

    await refreshSaved();

    function buildColumnsUI(){
      const cols = (state.schema[state.entity]?.columns)||[];
      const defaults = (state.schema[state.entity]?.default_columns)||[];
      state.columns = cols;
      if (state.selectedCols.length===0) state.selectedCols = defaults.slice();
      colsWrap.innerHTML = cols.map(c=>{
        const checked = state.selectedCols.includes(c) ? 'checked' : '';
        return `<label class="form-check"><input type="checkbox" class="form-check-input" data-col="${c}" ${checked}/> <span class="form-check-label">${esc(c)}</span></label>`;
      }).join('');
      colsWrap.querySelectorAll('[data-col]').forEach(cb=>{
        cb.addEventListener('change', ()=>{
          const name = cb.getAttribute('data-col');
          if (cb.checked){ if (!state.selectedCols.includes(name)) state.selectedCols.push(name); }
          else { state.selectedCols = state.selectedCols.filter(x=>x!==name); }
        });
      });
      // Chart selectors
      const xSel = container.querySelector('#repChartX');
      const ySel = container.querySelector('#repChartY');
      xSel.innerHTML = cols.map(c=>`<option value="${c}">${esc(c)}</option>`).join('');
      ySel.innerHTML = cols.map(c=>`<option value="${c}">${esc(c)}</option>`).join('');
    }

    function addFilterRow(init){
      const cols = (state.schema[state.entity]?.columns)||[];
      const id = Math.random().toString(36).slice(2);
      const row = document.createElement('div');
      row.className = 'row g-2 align-items-end';
      row.dataset.id = id;
      row.innerHTML = `
        <div class="col-md-4">
          <select class="form-select" data-role="field">${cols.map(c=>`<option value="${c}">${esc(c)}</option>`).join('')}</select>
        </div>
        <div class="col-md-3">
          <select class="form-select" data-role="op">
            <option value="eq">=</option>
            <option value="ne">!=</option>
            <option value="like">contains</option>
            <option value="gt">></option>
            <option value="lt"><</option>
            <option value="gte">>=</option>
            <option value="lte"><=</option>
            <option value="between">between</option>
            <option value="in">in (comma)</option>
          </select>
        </div>
        <div class="col-md-4" data-role="valwrap">
          <input class="form-control" data-role="val" placeholder="Value" />
        </div>
        <div class="col-md-1 text-end">
          <button class="btn btn-sm btn-outline-danger" data-role="del"><i class="fas fa-trash"></i></button>
        </div>
      `;
      filtersWrap.appendChild(row);
      const opSel = row.querySelector('[data-role="op"]');
      const valWrap = row.querySelector('[data-role="valwrap"]');
      opSel.addEventListener('change', ()=>{
        const op = opSel.value;
        if (op==='between'){
          valWrap.innerHTML = `<div class="input-group"><input class="form-control" data-role="val_a" placeholder="From" /><span class="input-group-text">to</span><input class="form-control" data-role="val_b" placeholder="To" /></div>`;
        } else {
          valWrap.innerHTML = `<input class="form-control" data-role="val" placeholder="Value" />`;
        }
      });
      row.querySelector('[data-role="del"]').addEventListener('click',()=>{ row.remove(); });
      if (init){
        row.querySelector('[data-role="field"]').value = init.field || cols[0];
        row.querySelector('[data-role="op"]').value = init.op || 'eq';
        if (init.op==='between'){
          opSel.dispatchEvent(new Event('change'));
          row.querySelector('[data-role="val_a"]').value = init.value?.[0] || '';
          row.querySelector('[data-role="val_b"]').value = init.value?.[1] || '';
        } else {
          row.querySelector('[data-role="val"]').value = Array.isArray(init.value) ? init.value.join(',') : (init.value ?? '');
        }
      }
    }

    function readFilters(){
      const out = [];
      filtersWrap.querySelectorAll('[data-id]').forEach(row=>{
        const field = row.querySelector('[data-role="field"]').value;
        const op = row.querySelector('[data-role="op"]').value;
        if (op==='between'){
          const a = row.querySelector('[data-role="val_a"]').value;
          const b = row.querySelector('[data-role="val_b"]').value;
          if (a!=='' && b!=='') out.push({ field, op, value:[a,b] });
        } else {
          let v = row.querySelector('[data-role="val"]').value;
          if (op==='in') v = v.split(',').map(s=>s.trim()).filter(Boolean);
          if (v!=='' && !(Array.isArray(v)&&v.length===0)) out.push({ field, op, value:v });
        }
      });
      return out;
    }

    function buildQuery(){
      const qs = new URLSearchParams();
      qs.set('entity', state.entity);
      if (state.selectedCols.length) qs.set('columns', state.selectedCols.join(','));
      if (dateFrom.value) qs.set('date_from', dateFrom.value);
      if (dateTo.value) qs.set('date_to', dateTo.value);
      if (searchInput.value.trim()) qs.set('search', searchInput.value.trim());
      const filters = readFilters();
      if (filters.length) qs.set('filters', JSON.stringify(filters));
      if (state.sort) qs.set('sort', state.sort);
      qs.set('page', String(state.page));
      qs.set('page_size', String(state.pageSize));
      return qs.toString();
    }

    async function run(){
      body.innerHTML = `<tr><td>Loading...</td></tr>`;
      const qs = buildQuery();
      const res = await fetch('api/reports.php?' + qs, { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
      if (!res?.success){ body.innerHTML = `<tr><td class="text-danger">Failed to load report</td></tr>`; return; }
      const rows = res.data?.rows || [];
      const cols = res.data?.columns || [];
      state.rows = rows; state.selectedCols = cols.length ? cols : state.selectedCols;
      // head
      headRow.innerHTML = cols.map(c=>`<th data-col="${c}" class="sortable">${esc(c)} ${state.sort.startsWith(c+':')? (state.sort.endsWith(':desc')? '▼':'▲'):''}</th>`).join('');
      // body
      if (!rows.length){ body.innerHTML = `<tr><td class="text-muted">No data</td></tr>`; }
      else {
        body.innerHTML = rows.map(r=>`<tr>` + cols.map(c=>`<td>${esc(r[c])}</td>`).join('') + `</tr>`).join('');
      }
      // info
      info.textContent = `Page ${res.data.page} • ${rows.length} rows`;
      // header sorting
      headRow.querySelectorAll('.sortable').forEach(th=>{
        th.addEventListener('click',()=>{
          const c = th.getAttribute('data-col');
          const cur = state.sort;
          const next = !cur || !cur.startsWith(c+':') ? `${c}:asc` : (cur.endsWith(':asc')? `${c}:desc` : '');
          state.sort = next; state.page=1; run();
        });
      });
    }

    function toggleConfig(){ configBox.style.display = configBox.style.display==='none'? 'block':'none'; }

    async function refreshSaved(){
      const res = await fetch('api/reports.php?action=saved', { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
      if (!res?.success){ savedListWrap.innerHTML = '<div class="text-danger small">Failed to load</div>'; return; }
      state.savedReports = res.data || [];
      if (!state.savedReports.length){ savedListWrap.innerHTML = '<div class="text-muted small">No saved reports yet</div>'; return; }
      savedListWrap.innerHTML = state.savedReports.map(sr=>`
        <div class="d-flex justify-content-between align-items-center py-1">
          <a href="#" data-role="open" data-id="${sr.id}">${esc(sr.name)}</a>
          <button class="btn btn-xs btn-link text-danger" data-role="del" data-id="${sr.id}"><i class="fas fa-trash"></i></button>
        </div>
      `).join('');
      savedListWrap.querySelectorAll('[data-role="open"]').forEach(a=> a.addEventListener('click', async (e)=>{
        e.preventDefault();
        const id = a.getAttribute('data-id'); state.selectedSavedId = id;
        const res = await fetch('api/reports.php?action=saved_show&id=' + encodeURIComponent(id), { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
        if (!res?.success){ window.auth?.showNotification?.(res?.message||'Failed to load', 'error'); return; }
        const sr = res.data;
        // Apply to UI state
        if (sr.entity && state.schema[sr.entity]){ state.entity = sr.entity; entitySel.value = sr.entity; }
        state.selectedCols = Array.isArray(sr.columns) && sr.columns.length ? sr.columns : (state.schema[state.entity]?.default_columns || []);
        filtersWrap.innerHTML = '';
        (Array.isArray(sr.filters)? sr.filters : []).forEach(f=> addFilterRow(f));
        dateFrom.value = sr.date_from || '';
        dateTo.value = sr.date_to || '';
        state.sort = sr.sort || '';
        buildColumnsUI();
        state.page = 1;
        await run();
      }));
      savedListWrap.querySelectorAll('[data-role="del"]').forEach(btn=> btn.addEventListener('click', async ()=>{
        if (!confirm('Delete this saved report?')) return;
        const res = await fetch('api/reports.php?action=delete', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ id: btn.getAttribute('data-id') }) }).then(r=>r.json()).catch(()=>null);
        if (res?.success){ await refreshSaved(); } else { window.auth?.showNotification?.(res?.message||'Failed', 'error'); }
      }));
    }

    // Wire
    container.querySelector('#repToggleConfig').addEventListener('click',(e)=>{ e.preventDefault(); toggleConfig(); });
    container.querySelector('#repAddFilter').addEventListener('click',()=> addFilterRow());
    container.querySelector('#repRun').addEventListener('click',()=>{ state.page=1; run(); });
    container.querySelector('#repReset').addEventListener('click',()=>{ state.sort=''; state.page=1; searchInput.value=''; dateFrom.value=''; dateTo.value=''; filtersWrap.innerHTML=''; state.selectedCols=[]; buildColumnsUI(); run(); });
    container.querySelector('#repPrev').addEventListener('click',()=>{ if (state.page>1){ state.page--; run(); } });
    container.querySelector('#repNext').addEventListener('click',()=>{ state.page++; run(); });
    container.querySelector('#repSaveBtn').addEventListener('click', async ()=>{
      const name = prompt('Name this report'); if (!name) return;
      const payload = { name, entity: state.entity, columns: state.selectedCols, filters: readFilters(), date_from: dateFrom.value||null, date_to: dateTo.value||null, sort: state.sort||null };
      const res = await fetch('api/reports.php?action=save', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (res?.success){ window.auth?.showNotification?.('Saved', 'success'); refreshSaved(); } else { window.auth?.showNotification?.(res?.message||'Save failed', 'error'); }
    });
    container.querySelector('#repEmail').addEventListener('click', async ()=>{
      const rcpt = prompt('Recipients (comma separated emails)'); if (!rcpt) return;
      const payload = { entity: state.entity, columns: state.selectedCols, filters: readFilters(), date_from: dateFrom.value||null, date_to: dateTo.value||null, sort: state.sort||null, recipients: rcpt, subject: `Report: ${state.entity}` };
      const res = await fetch('api/reports.php?action=email', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (res?.success){ window.auth?.showNotification?.('Email sent', 'success'); } else { window.auth?.showNotification?.(res?.message||'Email failed', 'error'); }
    });
    container.querySelector('#repSchedule').addEventListener('click', async ()=>{
      const name = prompt('Save this report first. Enter name'); if (!name) return;
      const savePayload = { name, entity: state.entity, columns: state.selectedCols, filters: readFilters(), date_from: dateFrom.value||null, date_to: dateTo.value||null, sort: state.sort||null };
      const saveRes = await fetch('api/reports.php?action=save', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(savePayload) }).then(r=>r.json()).catch(()=>null);
      if (!saveRes?.success){ window.auth?.showNotification?.(saveRes?.message||'Save failed', 'error'); return; }
      const rid = saveRes.data?.id; const rcpt = prompt('Recipients (comma separated emails)'); if (!rcpt) return;
      const freq = prompt('Frequency (daily/weekly/monthly/quarterly)', 'monthly') || 'monthly';
      const schedulePayload = { saved_report_id: Number(rid), frequency: freq, recipients: rcpt };
      const schRes = await fetch('api/reports.php?action=schedule', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(schedulePayload) }).then(r=>r.json()).catch(()=>null);
      if (schRes?.success){ window.auth?.showNotification?.('Scheduled', 'success'); } else { window.auth?.showNotification?.(schRes?.message||'Schedule failed', 'error'); }
    });

    entitySel.addEventListener('change',()=>{ state.entity = entitySel.value; state.selectedCols=[]; filtersWrap.innerHTML=''; buildColumnsUI(); run(); });

    // Export
    container.querySelector('#repCSV').addEventListener('click', ()=> exportCSV(state.selectedCols, state.rows));
    container.querySelector('#repXLSX').addEventListener('click', ()=> exportXLSX(state.selectedCols, state.rows));

    // Chart
    container.querySelector('#repRenderChart').addEventListener('click', ()=> renderChart(container, state));

    // Init
    buildColumnsUI();
    run();
  }

  function exportCSV(cols, rows){
    if (!cols || !rows) return;
    const csv = [cols.join(',')].concat(rows.map(r=> cols.map(c=> csvCell(r[c])).join(','))).join('\n');
    const blob = new Blob([csv], { type:'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob); const a = document.createElement('a'); a.href=url; a.download='report.csv'; a.click(); setTimeout(()=>URL.revokeObjectURL(url), 1000);
  }
  function csvCell(v){
    const s = v==null? '' : String(v);
    if (s.includes(',') || s.includes('"') || s.includes('\n')) return '"' + s.replace(/"/g,'""') + '"';
    return s;
  }
  function exportXLSX(cols, rows){
    if (!window.XLSX){ alert('Excel export not available'); return; }
    const data = [cols].concat(rows.map(r=> cols.map(c=> r[c])));
    const ws = XLSX.utils.aoa_to_sheet(data);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Report');
    XLSX.writeFile(wb, 'report.xlsx');
  }

  function renderChart(container, state){
    const type = container.querySelector('#repChartType').value;
    const xField = container.querySelector('#repChartX').value;
    const yField = container.querySelector('#repChartY').value;
    const agg = container.querySelector('#repChartAgg').value;
    const rows = state.rows || [];
    if (!rows.length){ window.auth?.showNotification?.('No data to chart', 'warning'); return; }

    // Aggregate
    const map = new Map();
    for (const r of rows){
      const k = r[xField];
      const y = Number(r[yField] ?? 0);
      if (!map.has(k)) map.set(k, { count:0, sum:0 });
      const obj = map.get(k); obj.count += 1; obj.sum += (isNaN(y)?0:y);
    }
    const labels = Array.from(map.keys());
    const values = labels.map(k=> agg==='count'? map.get(k).count : agg==='avg'? (map.get(k).sum / (map.get(k).count||1)) : map.get(k).sum);

    const ctx = container.querySelector('#repChart').getContext('2d');
    if (window._repChart) { window._repChart.destroy(); }
    window._repChart = new Chart(ctx, { type, data:{ labels, datasets:[{ label: `${agg.toUpperCase()} of ${yField} by ${xField}`, data: values, backgroundColor: 'rgba(13,110,253,0.5)', borderColor: 'rgba(13,110,253,1)', borderWidth:1 }] }, options:{ responsive:true, plugins:{ legend:{ display: type!=='bar' && type!=='line' } } } });
  }

  window.pages.reports = { render };
})();
