// Recruitment module page
(function(){
  window.pages = window.pages || {};

  const esc = s=>{ try{ return String(s).replace(/[&<>"']/g,c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); }catch(_){ return String(s||''); } };
  const notify = (m,t)=>{ try{ window.auth?.showNotification?.(m,t||'info'); }catch(_){} };

  const allowedRoles = ['super_admin','admin','hr_head','hr_officer'];
  function role(){ return (window.auth?.currentUser?.role_slug) || (window.auth?.currentUser?.role) || 'employee'; }
  function isAllowed(){ return allowedRoles.includes(role()); }

  const state = { lists: null, jobs: [], candidates: [], applications: [] };

  async function loadLists(){
    try{ const r = await fetch('api/recruitment.php?action=lists', { credentials:'same-origin' }); const j = await r.json(); state.lists = j?.success ? (j.data||{}) : {}; }catch(_){ state.lists = {}; }
  }

  function render(container){
    if (!isAllowed()) { container.innerHTML = '<div class="alert alert-warning">Access denied. Recruitment is available to Admin and HR roles.</div>'; return; }

    container.innerHTML = `
      <ul class="nav nav-tabs rec-tabs" role="tablist">
        <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#rec-jobs" type="button" role="tab">Jobs</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#rec-cands" type="button" role="tab">Candidates</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#rec-apps" type="button" role="tab">Applications</button></li>
      </ul>
      <div class="tab-content mt-3">
        <div class="tab-pane fade show active" id="rec-jobs" role="tabpanel">
          <div class="row g-3">
            <div class="col-lg-4">
              <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Create Job</h6><button class="btn btn-sm btn-outline-secondary" id="recLoadJobs"><i class="fas fa-rotate"></i></button></div>
                <div class="card-body">
                  <div class="mb-2"><label class="form-label">Title</label><input id="jobTitle" class="form-control" placeholder="e.g., Software Engineer" /></div>
                  <div class="mb-2"><label class="form-label">Department</label><select id="jobDept" class="form-select"><option value="">--</option></select></div>
                  <div class="mb-2"><label class="form-label">Position</label><select id="jobPos" class="form-select"><option value="">--</option></select></div>
                  <div class="mb-2"><label class="form-label">Employment Type</label><select id="jobType" class="form-select"></select></div>
                  <div class="mb-2"><label class="form-label">Deadline</label><input id="jobDeadline" type="date" class="form-control" /></div>
                  <div class="row g-2 mb-2"><div class="col"><label class="form-label">Min Salary</label><input id="jobMin" type="number" class="form-control" /></div><div class="col"><label class="form-label">Max Salary</label><input id="jobMax" type="number" class="form-control" /></div></div>
                  <div class="form-check mb-2"><input class="form-check-input" type="checkbox" id="jobRemote"><label class="form-check-label" for="jobRemote">Remote OK</label></div>
                  <div class="mb-2"><label class="form-label">Description</label><textarea id="jobDesc" class="form-control" rows="2"></textarea></div>
                  <div class="mb-2"><label class="form-label">Requirements</label><textarea id="jobReq" class="form-control" rows="2"></textarea></div>
                  <div class="mb-3"><label class="form-label">Responsibilities</label><textarea id="jobResp" class="form-control" rows="2"></textarea></div>
                  <div class="mb-3"><label class="form-label">Status</label><select id="jobStatus" class="form-select"></select></div>
                  <div class="text-end"><button class="btn btn-primary" id="createJobBtn"><i class="fas fa-plus"></i> Create Job</button></div>
                </div>
              </div>
            </div>
            <div class="col-lg-8">
              <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Job Postings</h6></div>
                <div class="card-body">
                  <div class="d-flex flex-wrap gap-2 align-items-end mb-2">
                    <div>
                      <label class="form-label small">Status</label>
                      <select id="jobFilterStatus" class="form-select form-select-sm" style="min-width:140px"></select>
                    </div>
                    <div>
                      <label class="form-label small">Department</label>
                      <select id="jobFilterDept" class="form-select form-select-sm" style="min-width:200px"><option value=""></option></select>
                    </div>
                    <button class="btn btn-sm btn-outline-primary" id="jobFilterApply">Apply</button>
                  </div>
                  <div class="table-responsive">
                    <table class="table table-sm align-middle">
                      <thead><tr><th>Title</th><th>Department</th><th>Position</th><th>Type</th><th>Salary</th><th>Deadline</th><th>Status</th><th></th></tr></thead>
                      <tbody id="jobsTBody"><tr><td colspan="8" class="text-muted">Loading...</td></tr></tbody>
                    </table>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <div class="tab-pane fade" id="rec-cands" role="tabpanel">
          <div class="row g-3">
            <div class="col-lg-4">
              <div class="card h-100">
                <div class="card-header"><h6 class="mb-0">Add Candidate</h6></div>
                <div class="card-body">
                  <div class="row g-2 mb-2"><div class="col"><label class="form-label">First Name</label><input id="cFirst" class="form-control" /></div><div class="col"><label class="form-label">Last Name</label><input id="cLast" class="form-control" /></div></div>
                  <div class="mb-2"><label class="form-label">Email</label><input id="cEmail" type="email" class="form-control" /></div>
                  <div class="mb-2"><label class="form-label">Phone</label><input id="cPhone" class="form-control" /></div>
                  <div class="mb-2"><label class="form-label">LinkedIn</label><input id="cLinkedIn" class="form-control" placeholder="https://linkedin.com/in/..." /></div>
                  <div class="mb-2"><label class="form-label">Portfolio</label><input id="cPortfolio" class="form-control" placeholder="https://..." /></div>
                  <div class="mb-2"><label class="form-label">Source</label><input id="cSource" class="form-control" placeholder="direct/referral/job board" /></div>
                  <div class="mb-3"><label class="form-label">Status</label><select id="cStatus" class="form-select"></select></div>
                  <div class="text-end"><button class="btn btn-primary" id="addCandBtn"><i class="fas fa-user-plus"></i> Add</button></div>
                </div>
              </div>
            </div>
            <div class="col-lg-8">
              <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Candidates</h6><button class="btn btn-sm btn-outline-secondary" id="reloadCands"><i class="fas fa-rotate"></i></button></div>
                <div class="card-body">
                  <div class="table-responsive">
                    <table class="table table-sm align-middle">
                      <thead><tr><th>Name</th><th>Email</th><th>Phone</th><th>Source</th><th>Status</th><th>Resume</th><th></th></tr></thead>
                      <tbody id="candTBody"><tr><td colspan="7" class="text-muted">Loading...</td></tr></tbody>
                    </table>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <div class="tab-pane fade" id="rec-apps" role="tabpanel">
          <div class="row g-3">
            <div class="col-lg-4">
              <div class="card h-100">
                <div class="card-header"><h6 class="mb-0">Create Application</h6></div>
                <div class="card-body">
                  <div class="mb-2"><label class="form-label">Job</label><select id="aJob" class="form-select"></select></div>
                  <div class="mb-2"><label class="form-label">Candidate</label><select id="aCand" class="form-select"></select></div>
                  <div class="mb-2"><label class="form-label">Status</label><select id="aStatus" class="form-select"></select></div>
                  <div class="mb-3"><label class="form-label">Cover Letter</label><textarea id="aCover" class="form-control" rows="3"></textarea></div>
                  <div class="text-end"><button class="btn btn-primary" id="addAppBtn"><i class="fas fa-file-circle-plus"></i> Create</button></div>
                </div>
              </div>
            </div>
            <div class="col-lg-8">
              <div class="card h-100">
                <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Applications</h6><button class="btn btn-sm btn-outline-secondary" id="reloadApps"><i class="fas fa-rotate"></i></button></div>
                <div class="card-body">
                  <div class="d-flex flex-wrap gap-2 align-items-end mb-3">
                    <div>
                      <label class="form-label small">Status</label>
                      <select id="appsFilterStatus" class="form-select form-select-sm" style="min-width:160px"></select>
                    </div>
                    <div>
                      <label class="form-label small">Department</label>
                      <select id="appsFilterDept" class="form-select form-select-sm" style="min-width:200px"><option value=""></option></select>
                    </div>
                    <div>
                      <label class="form-label small">Job</label>
                      <select id="appsFilterJob" class="form-select form-select-sm" style="min-width:200px"><option value=""></option></select>
                    </div>
                    <button class="btn btn-sm btn-outline-primary" id="appsFilterApply">Apply</button>
                    <div class="ms-auto">
                      <div class="btn-group btn-group-sm" role="group">
                        <button class="btn btn-outline-secondary active" id="appsViewList">List</button>
                        <button class="btn btn-outline-secondary" id="appsViewBoard">Pipeline</button>
                      </div>
                    </div>
                  </div>
                  <div id="appsListWrap">
                    <div class="table-responsive">
                      <table class="table table-sm align-middle">
                        <thead><tr><th>Candidate</th><th>Job</th><th>Date</th><th>Status</th><th>Files</th><th></th></tr></thead>
                        <tbody id="appsTBody"><tr><td colspan="6" class="text-muted">Loading...</td></tr></tbody>
                      </table>
                    </div>
                  </div>
                  <div id="appsBoardWrap" style="display:none; overflow:auto;">
                    <div id="appsBoard" class="d-flex gap-3" style="min-height:300px"></div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <!-- Edit Job Modal -->
      <div class="modal fade" id="editJobModal" tabindex="-1">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header"><h5 class="modal-title">Edit Job</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
            <div class="modal-body">
              <input type="hidden" id="ejId" />
              <div class="mb-2"><label class="form-label">Title</label><input id="ejTitle" class="form-control" /></div>
              <div class="mb-2"><label class="form-label">Employment Type</label><select id="ejType" class="form-select"></select></div>
              <div class="row g-2 mb-2"><div class="col"><label class="form-label">Min Salary</label><input id="ejMin" type="number" class="form-control" /></div><div class="col"><label class="form-label">Max Salary</label><input id="ejMax" type="number" class="form-control" /></div></div>
              <div class="mb-2"><label class="form-label">Deadline</label><input id="ejDeadline" type="date" class="form-control" /></div>
              <div class="mb-2"><label class="form-label">Status</label><select id="ejStatus" class="form-select"></select></div>
              <div class="form-check"><input class="form-check-input" type="checkbox" id="ejRemote"><label class="form-check-label" for="ejRemote">Remote OK</label></div>
            </div>
            <div class="modal-footer"><button class="btn btn-primary" id="saveJobBtn">Save</button></div>
          </div>
        </div>
      </div>
      
      <!-- Job Details Modal -->
      <div class="modal fade" id="jobDetailModal" tabindex="-1">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
          <div class="modal-content">
            <div class="modal-header"><h5 class="modal-title" id="jdTitle">Job Details</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
            <div class="modal-body">
              <div class="small text-muted mb-2" id="jdMeta"></div>
              <div class="mb-3"><h6>Description</h6><div id="jdDesc" class="text-prewrap"></div></div>
              <div class="mb-3"><h6>Requirements</h6><div id="jdReq" class="text-prewrap"></div></div>
              <div class="mb-3"><h6>Responsibilities</h6><div id="jdResp" class="text-prewrap"></div></div>
            </div>
          </div>
        </div>
      </div>

      <!-- Interview Modal -->
      <div class="modal fade" id="interviewModal" tabindex="-1">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header"><h5 class="modal-title">Schedule Interview</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
            <div class="modal-body">
              <input type="hidden" id="ivAppId" />
              <div class="mb-2"><label class="form-label">When</label><input id="ivWhen" type="datetime-local" class="form-control" /></div>
              <div class="mb-2"><label class="form-label">Interviewer</label><input id="ivInterviewer" class="form-control" /></div>
              <div class="mb-2"><label class="form-label">Notes</label><textarea id="ivNotes" rows="3" class="form-control"></textarea></div>
              <div class="mb-2"><label class="form-label">Rating</label><select id="ivRating" class="form-select"><option value="">-</option>${[1,2,3,4,5].map(n=>`<option>${n}</option>`).join('')}</select></div>
              <div><h6 class="mt-3">Previous Interviews</h6><ul class="list-group" id="ivList"><li class="list-group-item text-muted">None</li></ul></div>
            </div>
            <div class="modal-footer"><button class="btn btn-primary" id="ivSaveBtn">Save</button></div>
          </div>
        </div>
      </div>
    `;

    // Load supporting lists then data
    loadLists().then(()=>{
      fillLists(container);
      Promise.all([loadJobs(container), loadCandidates(container), loadApplications(container)]).catch(()=>{});
    });

    // Wire actions
    container.querySelector('#createJobBtn')?.addEventListener('click', ()=>createJob(container));
    container.querySelector('#recLoadJobs')?.addEventListener('click', ()=>loadJobs(container));
    container.querySelector('#jobFilterApply')?.addEventListener('click', ()=>loadJobs(container));
    container.querySelector('#addCandBtn')?.addEventListener('click', ()=>createCandidate(container));
    container.querySelector('#reloadCands')?.addEventListener('click', ()=>loadCandidates(container));
    container.querySelector('#addAppBtn')?.addEventListener('click', ()=>createApplication(container));
    container.querySelector('#reloadApps')?.addEventListener('click', ()=>{ loadApplications(container); buildPipeline(container); });
    container.querySelector('#appsFilterApply')?.addEventListener('click', ()=>{ loadApplications(container); buildPipeline(container); });
    container.querySelector('#appsViewList')?.addEventListener('click', ()=>{ document.getElementById('appsListWrap').style.display='block'; document.getElementById('appsBoardWrap').style.display='none'; document.getElementById('appsViewList').classList.add('active'); document.getElementById('appsViewBoard').classList.remove('active'); });
    container.querySelector('#appsViewBoard')?.addEventListener('click', ()=>{ document.getElementById('appsListWrap').style.display='none'; document.getElementById('appsBoardWrap').style.display='block'; document.getElementById('appsViewBoard').classList.add('active'); document.getElementById('appsViewList').classList.remove('active'); buildPipeline(container); });
    container.querySelector('#saveJobBtn')?.addEventListener('click', ()=>saveJob(container));
    // Delegated: resume upload
    container.querySelector('#candTBody')?.addEventListener('change', (e)=>{
      const t = e.target; if (!(t && t.matches('.cand-resume-input'))) return; const id = Number(t.getAttribute('data-candidate')); const file = t.files && t.files[0]; if (!id || !file) return; uploadCandidateResume(id, file).then(()=>loadCandidates(container));
    });
    // Delegated: cover upload, schedule interview, convert
    container.querySelector('#appsTBody')?.addEventListener('change', (e)=>{
      const t = e.target; if (t && t.matches('.app-cover-input')){ const id = Number(t.getAttribute('data-app')); const file = t.files && t.files[0]; if (id && file) uploadApplicationCover(id, file).then(()=>loadApplications(container)); }
    });
    container.querySelector('#appsTBody')?.addEventListener('click', (e)=>{
      const btn = e.target.closest('[data-app-action]'); if (!btn) return; const id = Number(btn.getAttribute('data-app-id')); const act = btn.getAttribute('data-app-action');
      if (act==='interview') openInterviewModal(id);
      if (act==='convert') convertToEmployee(id);
    });
    document.getElementById('ivSaveBtn')?.addEventListener('click', ()=>saveInterview(container));
  }

  function fillLists(container){
    const L = state.lists || {};
    const depSel = container.querySelector('#jobDept'); if (depSel) depSel.innerHTML = '<option value=""></option>' + (Array.isArray(L.departments)?L.departments:[]).map(d=>`<option value="${d.id}">${esc(d.name)}</option>`).join('');
    const posSel = container.querySelector('#jobPos'); if (posSel) posSel.innerHTML = '<option value=""></option>' + (Array.isArray(L.positions)?L.positions:[]).map(p=>`<option value="${p.id}">${esc(p.title)}</option>`).join('');
    const typeSel = container.querySelector('#jobType'); if (typeSel) typeSel.innerHTML = (Array.isArray(L.employment_types)?L.employment_types:[]).map(t=>`<option>${t}</option>`).join('');
    const stSel = container.querySelector('#jobStatus'); if (stSel) stSel.innerHTML = (Array.isArray(L.job_statuses)?L.job_statuses:[]).map(s=>`<option>${s}</option>`).join('');

    const cst = container.querySelector('#cStatus'); if (cst) cst.innerHTML = (Array.isArray(L.candidate_statuses)?L.candidate_statuses:[]).map(s=>`<option>${s}</option>`).join('');

    const aStatus = container.querySelector('#aStatus'); if (aStatus) aStatus.innerHTML = (Array.isArray(L.application_statuses)?L.application_statuses:[]).map(s=>`<option>${s}</option>`).join('');
    // Filter dropdowns
    const jf = container.querySelector('#jobFilterStatus'); if (jf) jf.innerHTML = ['','draft','open','closed'].map(s=>`<option value="${s}">${s||'All'}</option>`).join('');
    const jfd = container.querySelector('#jobFilterDept'); if (jfd) jfd.innerHTML = '<option value=""></option>' + (Array.isArray(L.departments)?L.departments:[]).map(d=>`<option value="${d.id}">${esc(d.name)}</option>`).join('');
    const afs = container.querySelector('#appsFilterStatus'); if (afs) afs.innerHTML = ['','applied','shortlisted','interview','offered','hired','rejected'].map(s=>`<option value="${s}">${s||'All'}</option>`).join('');
    const afd = container.querySelector('#appsFilterDept'); if (afd) afd.innerHTML = '<option value=""></option>' + (Array.isArray(L.departments)?L.departments:[]).map(d=>`<option value="${d.id}">${esc(d.name)}</option>`).join('');
  }

  async function loadJobs(container){
    const tb = container.querySelector('#jobsTBody'); if (!tb) return;
    tb.innerHTML = '<tr><td colspan="8" class="text-muted">Loading...</td></tr>';
    try{
      const stF = document.getElementById('jobFilterStatus')?.value || '';
      const depF = document.getElementById('jobFilterDept')?.value || '';
      const qs = new URLSearchParams(); if (stF) qs.set('status', stF); if (depF) qs.set('department_id', depF);
      const r = await fetch('api/recruitment.php?action=jobs' + (qs.toString()?('&'+qs.toString()):''), { credentials:'same-origin' });
      const j = await r.json();
      state.jobs = j?.success ? (j.data||[]) : [];
      tb.innerHTML = state.jobs.length ? state.jobs.map(job=>{
        const sal = (job.min_salary||job.max_salary) ? `${Number(job.min_salary||0)} - ${Number(job.max_salary||0)}` : '-';
        const acts = `
          <div class="btn-group btn-group-sm">
            <button class="btn btn-outline-info" data-view-job="${job.id}"><i class="fas fa-eye"></i></button>
            <button class="btn btn-outline-primary" data-edit-job="${job.id}"><i class="fas fa-pen"></i></button>
            <div class="btn-group">
              <button class="btn btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">${esc(job.status||'open')}</button>
              <div class="dropdown-menu dropdown-menu-end">
                ${['draft','open','closed'].map(s=>`<a class="dropdown-item" href="#" data-job-status="${job.id}" data-status="${s}">${s}</a>`).join('')}
              </div>
            </div>
          </div>`;
        return `<tr>
          <td>${esc(job.title||'')}</td>
          <td>${esc(job.department_name||'-')}</td>
          <td>${esc(job.position_title||'-')}</td>
          <td>${esc(job.employment_type||'-')}</td>
          <td>${esc(sal)}</td>
          <td>${esc(job.deadline||'')}</td>
          <td>${esc(job.status||'')}</td>
          <td class="text-end">${acts}</td>
        </tr>`;
      }).join('') : '<tr><td colspan="8" class="text-muted">No jobs found.</td></tr>';
      // bind actions
      tb.querySelectorAll('[data-job-status]')?.forEach(a=>{
        if (a._b) return; a._b = true; a.addEventListener('click', (ev)=>{ ev.preventDefault(); updateJobStatus(container, Number(a.getAttribute('data-job-status')), a.getAttribute('data-status')); });
      });
      tb.querySelectorAll('[data-view-job]')?.forEach(btn=>{ if (btn._b) return; btn._b = true; btn.addEventListener('click', ()=>{ const id = Number(btn.getAttribute('data-view-job')); openJobDetails(id); }); });
      tb.querySelectorAll('[data-edit-job]')?.forEach(btn=>{
        if (btn._b) return; btn._b = true;
        btn.addEventListener('click', ()=>{
          const id = Number(btn.getAttribute('data-edit-job'));
          const job = state.jobs.find(j=>Number(j.id)===id); if (!job) return;
          openEditJob(container, job);
        });
      });
      // refresh job select for applications
      const aj = container.querySelector('#aJob'); if (aj) aj.innerHTML = (state.jobs||[]).map(j=>`<option value="${j.id}">${esc(j.title)} (${esc(j.status)})</option>`).join('');
      const afJob = container.querySelector('#appsFilterJob'); if (afJob) afJob.innerHTML = '<option value=""></option>' + (state.jobs||[]).map(j=>`<option value="${j.id}">${esc(j.title)}</option>`).join('');
    }catch(_){ tb.innerHTML = '<tr><td colspan="8" class="text-danger">Failed to load jobs.</td></tr>'; }
  }

  async function createJob(container){
    const title = container.querySelector('#jobTitle').value.trim();
    if (!title){ notify('Title is required','warning'); return; }
    const payload = {
      title,
      department_id: numVal(container.querySelector('#jobDept').value),
      position_id: numVal(container.querySelector('#jobPos').value),
      employment_type: container.querySelector('#jobType').value,
      deadline: container.querySelector('#jobDeadline').value || null,
      min_salary: numVal(container.querySelector('#jobMin').value),
      max_salary: numVal(container.querySelector('#jobMax').value),
      remote_work: container.querySelector('#jobRemote').checked ? 1 : 0,
      description: container.querySelector('#jobDesc').value,
      requirements: container.querySelector('#jobReq').value,
      responsibilities: container.querySelector('#jobResp').value,
      status: container.querySelector('#jobStatus').value || 'open'
    };
    try{
      const r = await fetch('api/recruitment.php?action=create_job', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
      const j = await r.json();
      if (j?.success){ notify('Job created','success'); loadJobs(container); container.querySelector('#jobTitle').value=''; }
      else notify(j?.message||'Create failed','error');
    }catch(_){ notify('Create failed','error'); }
  }

  async function updateJobStatus(container, id, status){
    try{
      const r = await fetch('api/recruitment.php?action=toggle_job', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ id, status }) });
      const j = await r.json();
      if (j?.success){ notify('Status updated','success'); loadJobs(container); }
      else notify(j?.message||'Update failed','error');
    }catch(_){ notify('Update failed','error'); }
  }

  function openEditJob(container, job){
    const modal = document.getElementById('editJobModal'); if (!modal) return;
    document.getElementById('ejId').value = job.id;
    document.getElementById('ejTitle').value = job.title||'';
    const L = state.lists || {};
    const ejt = document.getElementById('ejType'); if (ejt) ejt.innerHTML = (Array.isArray(L.employment_types)?L.employment_types:[]).map(t=>`<option ${job.employment_type===t?'selected':''}>${t}</option>`).join('');
    document.getElementById('ejMin').value = job.min_salary||'';
    document.getElementById('ejMax').value = job.max_salary||'';
    document.getElementById('ejDeadline').value = (job.deadline||'').slice(0,10);
    const ejs = document.getElementById('ejStatus'); if (ejs) ejs.innerHTML = (Array.isArray(L.job_statuses)?L.job_statuses:[]).map(s=>`<option ${job.status===s?'selected':''}>${s}</option>`).join('');
    document.getElementById('ejRemote').checked = !!job.remote_work;
    bootstrap.Modal.getOrCreateInstance(modal).show();
  }

  async function saveJob(container){
    const id = numVal(document.getElementById('ejId').value);
    if (!id) return;
    const payload = {
      id,
      title: document.getElementById('ejTitle').value,
      employment_type: document.getElementById('ejType').value,
      min_salary: numVal(document.getElementById('ejMin').value),
      max_salary: numVal(document.getElementById('ejMax').value),
      deadline: document.getElementById('ejDeadline').value || null,
      status: document.getElementById('ejStatus').value,
      remote_work: document.getElementById('ejRemote').checked ? 1 : 0
    };
    try{
      const r = await fetch('api/recruitment.php?action=update_job', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
      const j = await r.json();
      if (j?.success){ notify('Job updated','success'); loadJobs(container); bootstrap.Modal.getOrCreateInstance(document.getElementById('editJobModal')).hide(); }
      else notify(j?.message||'Update failed','error');
    }catch(_){ notify('Update failed','error'); }
  }

  async function loadCandidates(container){
    const tb = container.querySelector('#candTBody'); if (!tb) return;
    tb.innerHTML = '<tr><td colspan="7" class="text-muted">Loading...</td></tr>';
    try{
      const r = await fetch('api/recruitment.php?action=candidates', { credentials:'same-origin' });
      const j = await r.json();
      state.candidates = j?.success ? (j.data||[]) : [];
      tb.innerHTML = state.candidates.length ? state.candidates.map(c=>{
        const nm = esc(((c.first_name||'')+' '+(c.last_name||'')).trim());
        const resume = c.resume_path ? `<a href="${esc(c.resume_path)}" target="_blank">View</a>` : `<label class=\"btn btn-sm btn-outline-secondary mb-0\">Upload<input type=\"file\" accept=\".pdf,.doc,.docx,.png,.jpg,.jpeg,.txt\" class=\"d-none cand-resume-input\" data-candidate=\"${c.id}\"/></label>`;
        return `<tr>
          <td>${nm}</td>
          <td>${esc(c.email||'')}</td>
          <td>${esc(c.phone||'')}</td>
          <td>${esc(c.source||'')}</td>
          <td>${esc(c.status||'')}</td>
          <td>${resume}</td>
          <td></td>
        </tr>`;
      }).join('') : '<tr><td colspan="7" class="text-muted">No candidates found.</td></tr>';
      // refresh candidate select for apps
      const ac = container.querySelector('#aCand'); if (ac) ac.innerHTML = (state.candidates||[]).map(c=>`<option value="${c.id}">${esc(((c.first_name||'')+' '+(c.last_name||'')).trim() || c.email || ('#'+c.id))}</option>`).join('');
    }catch(_){ tb.innerHTML = '<tr><td colspan="7" class="text-danger">Failed to load candidates.</td></tr>'; }
  }

  async function createCandidate(container){
    const first = container.querySelector('#cFirst').value.trim();
    const last = container.querySelector('#cLast').value.trim();
    if (!first && !last){ notify('Name is required','warning'); return; }
    const payload = {
      first_name: first, last_name: last,
      email: container.querySelector('#cEmail').value.trim() || null,
      phone: container.querySelector('#cPhone').value.trim() || null,
      linkedin: container.querySelector('#cLinkedIn').value.trim() || null,
      portfolio: container.querySelector('#cPortfolio').value.trim() || null,
      source: container.querySelector('#cSource').value.trim() || 'direct',
      status: container.querySelector('#cStatus').value || 'new'
    };
    try{
      const r = await fetch('api/recruitment.php?action=create_candidate', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
      const j = await r.json();
      if (j?.success){ notify('Candidate added','success'); loadCandidates(container); container.querySelector('#cFirst').value=''; container.querySelector('#cLast').value=''; }
      else notify(j?.message||'Add failed','error');
    }catch(_){ notify('Add failed','error'); }
  }

  async function loadApplications(container){
    const tb = container.querySelector('#appsTBody'); if (!tb) return;
    tb.innerHTML = '<tr><td colspan="6" class="text-muted">Loading...</td></tr>';
    try{
      const st = document.getElementById('appsFilterStatus')?.value || '';
      const dep = document.getElementById('appsFilterDept')?.value || '';
      const job = document.getElementById('appsFilterJob')?.value || '';
      const qs = new URLSearchParams(); if (st) qs.set('status', st); if (dep) qs.set('department_id', dep); if (job) qs.set('job_id', job);
      const r = await fetch('api/recruitment.php?action=applications' + (qs.toString()?('&'+qs.toString()):''), { credentials:'same-origin' });
      const j = await r.json();
      state.applications = j?.success ? (j.data||[]) : [];
      tb.innerHTML = state.applications.length ? state.applications.map(a=>{
        const nm = esc(((a.first_name||'')+' '+(a.last_name||'')).trim() || a.email || ('#'+a.candidate_id));
        const when = esc((a.application_date||'').replace('T',' '));
        const statusSel = `<select class="form-select form-select-sm" data-app-status="${a.id}">${(state.lists?.application_statuses||[]).map(s=>`<option ${a.status===s?'selected':''}>${s}</option>`).join('')}</select>`;
        // parse cover_file from notes if json
        let cover = ''; try{ const n = a.notes && JSON.parse(a.notes); if (n && n.cover_file) cover = `<a href="${esc(n.cover_file)}" target="_blank">Cover</a>`; }catch(_){ }
        const filesCell = `${a.resume_path?`<a href="${esc(a.resume_path)}" target="_blank">Resume</a>`:''} ${cover} <label class="btn btn-sm btn-outline-secondary mb-0">Upload Cover<input type="file" class="d-none app-cover-input" data-app="${a.id}" accept=".pdf,.doc,.docx,.png,.jpg,.jpeg,.txt"/></label>`;
        const actions = `<div class="btn-group btn-group-sm"><button class="btn btn-outline-secondary" data-app-action="interview" data-app-id="${a.id}"><i class="fas fa-calendar"></i> Interview</button>${a.status==='hired'?`<button class="btn btn-outline-success" data-app-action="convert" data-app-id="${a.id}"><i class="fas fa-user-check"></i> Convert</button>`:''}</div>`;
        return `<tr>
          <td>${nm}</td>
          <td>${esc(a.job_title||'')}</td>
          <td>${esc(when)}</td>
          <td style="max-width:180px;">${statusSel}</td>
          <td>${filesCell}</td>
          <td class="text-end">${actions}</td>
        </tr>`;
      }).join('') : '<tr><td colspan="6" class="text-muted">No applications found.</td></tr>';
      // bind status update
      tb.querySelectorAll('[data-app-status]')?.forEach(sel=>{
        if (sel._b) return; sel._b = true; sel.addEventListener('change', ()=>{
          const id = Number(sel.getAttribute('data-app-status'));
          const status = sel.value;
          updateApplicationStatus(container, id, status);
        });
      });
      // refresh selects for create
      const aj = container.querySelector('#aJob'); if (aj && (!aj.options || aj.options.length===0)) aj.innerHTML = (state.jobs||[]).map(j=>`<option value="${j.id}">${esc(j.title)} (${esc(j.status)})</option>`).join('');
      const ac = container.querySelector('#aCand'); if (ac && (!ac.options || ac.options.length===0)) ac.innerHTML = (state.candidates||[]).map(c=>`<option value="${c.id}">${esc(((c.first_name||'')+' '+(c.last_name||'')).trim() || c.email || ('#'+c.id))}</option>`).join('');
    }catch(_){ tb.innerHTML = '<tr><td colspan="6" class="text-danger">Failed to load applications.</td></tr>'; }
  }

  async function createApplication(container){
    const jobId = numVal(container.querySelector('#aJob').value);
    const candId = numVal(container.querySelector('#aCand').value);
    if (!jobId || !candId){ notify('Select job and candidate','warning'); return; }
    const payload = { job_posting_id: jobId, candidate_id: candId, status: container.querySelector('#aStatus').value, cover_letter: container.querySelector('#aCover').value };
    try{
      const r = await fetch('api/recruitment.php?action=create_application', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
      const j = await r.json();
      if (j?.success){ notify('Application created','success'); loadApplications(container); container.querySelector('#aCover').value=''; }
      else notify(j?.message||'Create failed','error');
    }catch(_){ notify('Create failed','error'); }
  }

  async function updateApplicationStatus(container, id, status){
    try{
      const r = await fetch('api/recruitment.php?action=update_application', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ id, status }) });
      const j = await r.json();
      if (j?.success){ notify('Application updated','success'); buildPipeline(container); }
      else notify(j?.message||'Update failed','error');
    }catch(_){ notify('Update failed','error'); }
  }

  function numVal(v){ const n = Number(v); return Number.isFinite(n) && n>0 ? n : null; }

  // ======= Extra helpers =======
  async function openJobDetails(id){
    try{
      const r = await fetch('api/recruitment.php?action=job&id='+encodeURIComponent(id), { credentials:'same-origin' });
      const j = await r.json(); if (!j?.success) { notify(j?.message||'Failed to load job','error'); return; }
      const d = j.data||{};
      document.getElementById('jdTitle').textContent = d.title||'Job Details';
      document.getElementById('jdMeta').textContent = `${d.department_name||''} • ${d.position_title||''} • ${d.employment_type||''}`;
      document.getElementById('jdDesc').innerHTML = `<div style="white-space:pre-wrap">${esc(d.description||'')}</div>`;
      document.getElementById('jdReq').innerHTML = `<div style="white-space:pre-wrap">${esc(d.requirements||'')}</div>`;
      document.getElementById('jdResp').innerHTML = `<div style="white-space:pre-wrap">${esc(d.responsibilities||'')}</div>`;
      bootstrap.Modal.getOrCreateInstance(document.getElementById('jobDetailModal')).show();
    }catch(_){ notify('Failed to load job','error'); }
  }

  async function uploadCandidateResume(candidateId, file){
    const fd = new FormData(); fd.append('candidate_id', candidateId); fd.append('file', file);
    const r = await fetch('api/recruitment.php?action=upload_resume', { method:'POST', body: fd, credentials:'same-origin' });
    const j = await r.json(); if (!j?.success) notify(j?.message||'Upload failed','error'); else notify('Resume uploaded','success');
  }

  async function uploadApplicationCover(appId, file){
    const fd = new FormData(); fd.append('application_id', appId); fd.append('file', file);
    const r = await fetch('api/recruitment.php?action=upload_cover', { method:'POST', body: fd, credentials:'same-origin' });
    const j = await r.json(); if (!j?.success) notify(j?.message||'Upload failed','error'); else notify('Cover uploaded','success');
  }

  async function openInterviewModal(appId){
    document.getElementById('ivAppId').value = appId;
    // Load previous
    const ul = document.getElementById('ivList'); ul.innerHTML = '<li class="list-group-item text-muted">Loading...</li>';
    try{
      const r = await fetch('api/recruitment.php?action=list_interviews&application_id='+encodeURIComponent(appId), { credentials:'same-origin' });
      const j = await r.json(); const rows = j?.success ? (j.data||[]) : [];
      ul.innerHTML = rows.length ? rows.map(it=>{
        const md = it.metadata||{}; const t = esc((md.scheduled_at||'').replace('T',' ')); const iv = esc(md.interviewer||''); const rt = md.rating?` • Rating: ${md.rating}`:''; const nt = md.notes?`<div class="small text-muted">${esc(md.notes)}</div>`:'';
        return `<li class="list-group-item"><div class="fw-semibold">${t} • ${iv}${rt}</div>${nt}</li>`; }).join('') : '<li class="list-group-item text-muted">None</li>';
    }catch(_){ ul.innerHTML = '<li class="list-group-item text-danger">Failed to load</li>'; }
    bootstrap.Modal.getOrCreateInstance(document.getElementById('interviewModal')).show();
  }

  async function saveInterview(container){
    const id = Number(document.getElementById('ivAppId').value||0);
    const payload = { application_id: id, scheduled_at: document.getElementById('ivWhen').value, interviewer: document.getElementById('ivInterviewer').value, notes: document.getElementById('ivNotes').value, rating: Number(document.getElementById('ivRating').value||0)||null };
    try{
      const r = await fetch('api/recruitment.php?action=schedule_interview', { method:'POST', body: JSON.stringify(payload), headers:{'Content-Type':'application/json'}, credentials:'same-origin' });
      const j = await r.json(); if (j?.success){ notify('Interview saved','success'); openInterviewModal(id); }
      else notify(j?.message||'Failed','error');
    }catch(_){ notify('Failed','error'); }
  }

  async function convertToEmployee(appId){
    try{
      const r = await fetch('api/recruitment.php?action=convert_hired', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ application_id: appId }) });
      const j = await r.json(); if (j?.success){ notify('Converted to employee','success'); }
      else notify(j?.message||'Conversion failed','error');
    }catch(_){ notify('Conversion failed','error'); }
  }

  function buildPipeline(container){
    const wrap = document.getElementById('appsBoard'); if (!wrap) return; wrap.innerHTML = '';
    const statuses = ['applied','shortlisted','interview','offered','hired','rejected'];
    const groups = {}; statuses.forEach(s=>groups[s]=[]);
    (state.applications||[]).forEach(a=>{ if (!groups[a.status]) groups[a.status]=[]; groups[a.status].push(a); });
    statuses.forEach(stat=>{
      const col = document.createElement('div'); col.className = 'flex-fill'; col.style.minWidth = '260px';
      col.innerHTML = `<div class="card h-100"><div class="card-header py-2"><strong class="text-capitalize">${esc(stat)}</strong> <span class="badge bg-secondary">${(groups[stat]||[]).length}</span></div><div class="card-body" data-drop-status="${stat}" style="min-height:220px;"></div></div>`;
      wrap.appendChild(col);
      const body = col.querySelector('[data-drop-status]');
      ;(groups[stat]||[]).forEach(a=>{
        const nm = esc(((a.first_name||'')+' '+(a.last_name||'')).trim() || a.email || ('#'+a.candidate_id));
        const card = document.createElement('div'); card.className = 'card mb-2'; card.draggable = true; card.setAttribute('data-app-id', a.id);
        card.innerHTML = `<div class="card-body p-2"><div class="fw-semibold">${nm}</div><div class="small text-muted">${esc(a.job_title||'')}</div><div class="d-flex gap-2 mt-1"><button class="btn btn-xs btn-outline-secondary" data-app-action="interview" data-app-id="${a.id}"><i class="fas fa-calendar"></i></button>${a.status==='hired'?`<button class="btn btn-xs btn-outline-success" data-app-action="convert" data-app-id="${a.id}"><i class=\"fas fa-user-check\"></i></button>`:''}</div></div>`;
        body.appendChild(card);
        card.addEventListener('dragstart', (ev)=>{ ev.dataTransfer.setData('text/plain', String(a.id)); });
      });
      body.addEventListener('dragover', (ev)=>{ ev.preventDefault(); });
      body.addEventListener('drop', (ev)=>{ ev.preventDefault(); const id = Number(ev.dataTransfer.getData('text/plain')); const st = body.getAttribute('data-drop-status'); if (!id || !st) return; updateApplicationStatus(container, id, st); });
    });
  }

  window.pages.recruitment = { render };
})();
