// Payroll page module (Setup, Management/Execution, Approvals)
(function () {
  window.pages = window.pages || {};

  function canApprovePayroll() {
    try { return window.auth?.canApprove('payroll'); } catch(_) { return false; }
  }

  async function fetchPayslip(id) {
    try {
      window.logEvent?.('info', 'payslip_fetch_start', { id });
      const res = await fetch(`api/payroll.php?action=payslip&id=${encodeURIComponent(id)}`, { credentials: 'same-origin' });
      const text = await res.text();
      let json;
      try { json = JSON.parse(text); }
      catch(_) {
        window.logEvent?.('error', 'payslip_fetch_parse_error', { id, snippet: text && text.slice ? text.slice(0,200) : null });
        return null;
      }
      if (!json?.success) {
        window.logEvent?.('error', 'payslip_fetch_failed', { id, message: json?.message });
        return null;
      }
      window.logEvent?.('info', 'payslip_fetch_ok', { id });
      return json.data;
    } catch (e) {
      window.logEvent?.('error', 'payslip_fetch_exception', { id, error: String(e?.message || e) });
      return null;
    }
  }

  function currency(v) {
    const n = Number(v||0);
    if (Number.isNaN(n)) return String(v||'');
    return n.toFixed(2);
  }

  // Local HTML escaper to avoid relying on other modules
  function escapeHtml(s){
    return String(s ?? '').replace(/[&<>"']/g, c => ({
      '&':'&amp;', '<':'&lt;', '>':'&gt;', '"':'&quot;', "'":'&#39;'
    })[c]);
  }

  function buildPayslipHtml(p) {
    const allowances = Array.isArray(p.allowances) ? p.allowances : [];
    const deductions = Array.isArray(p.deductions) ? p.deductions : [];
    const fmt = (v)=>currency(v);
    const monthTxt = (()=>{
      const s = String(p.pay_period_start||'');
      const d = s && s.length>=7 ? new Date(s) : null;
      try { return d ? d.toLocaleString('en', { month: 'long', year: 'numeric' }) : `${s}`; } catch(_) { return `${s}`; }
    })();

    // Build side-by-side earnings vs deductions rows
    const earnRows = [{ name:'Basic Pay', amount:+p.basic_salary||0 }].concat(allowances.map(a=>({ name: a.name||a.code||'Allowance', amount: +a.amount||0 })));
    const dedRows = [
      { name:'Employee SSF (5.5%)', amount:+p.ssnit_employee||0 },
      { name:'PAYE', amount:+p.total_tax||0 },
      ...deductions.map(d=>({ name: d.name||d.code||'Deduction', amount: +d.amount||0 }))
    ];
    const maxLen = Math.max(earnRows.length, dedRows.length);
    while (earnRows.length < maxLen) earnRows.push({ name:'', amount:'' });
    while (dedRows.length < maxLen) dedRows.push({ name:'', amount:'' });
    const totalEarnings = (+p.basic_salary||0) + (+(p.total_allowances||0));
    const totalDeductions = (+p.ssnit_employee||0) + (+p.total_tax||0) + (+(p.total_deductions||0));
    const net = +p.net_salary || (totalEarnings - totalDeductions);

    const baseHref = (()=>{ try { return String(window.location.href.split('#')[0] || '/'); } catch(_) { return '/'; } })();
    const logoUrl = (()=>{ 
      try { 
        const path = String(p.company_logo || 'assets/img/smartquantumhrlogo.png'); 
        return new URL(path, baseHref).href; 
      } catch(_) { 
        return String(p.company_logo || 'assets/img/smartquantumhrlogo.png'); 
      } 
    })();
    const logsUrl = (()=>{ try { return new URL('api/logs.php', baseHref).href; } catch(_) { return 'api/logs.php'; } })();

    const css = `
      <style>
        body { font-family: 'Times New Roman', Times, serif; color:#000; }
        .sheet { width: 980px; margin: 20px auto; border: 2px solid #000; padding: 12px; }
        .title { text-align:center; }
        .title h1 { margin:0; font-size:28px; font-weight:bold; }
        .title h2 { margin:2px 0 0 0; font-size:16px; font-weight:normal; }
        .subtitle { text-align:center; margin: 10px 0; font-size:16px; font-weight:bold; }
        .info { width:100%; font-size:14px; margin-bottom:8px; }
        .info td { padding:2px 6px; }
        .hr { border-top:2px solid #000; margin:8px 0; }
        table { width:100%; border-collapse:collapse; }
        th, td { border:1px solid #000; padding:6px 8px; font-size:14px; }
        th { background:#f5f5f5; }
        .no-border td { border: none; }
        .right { text-align:right; }
        .center { text-align:center; }
        .footnote { text-align:center; margin:12px 0; font-style:italic; }
        .signatures { width:100%; margin-top:24px; }
        .signatures td { padding-top:30px; }
      </style>`;

    const rows = earnRows.map((er, i)=>{
      const dr = dedRows[i];
      return `<tr>
        <td>${escapeHtml(er.name)}</td>
        <td class="right">${er.amount===''?'' : fmt(er.amount)}</td>
        <td>${escapeHtml(dr.name)}</td>
        <td class="right">${dr.amount===''?'' : fmt(dr.amount)}</td>
      </tr>`;
    }).join('');

    const html = `<!doctype html><html><head><meta charset="utf-8" />${css}<base href="${escapeHtml(baseHref)}"><title>Payslip</title>
    <script>(function(){
      function send(payload){
        var url='${logsUrl}';
        try{ navigator.sendBeacon(url, new Blob([JSON.stringify(payload)], {type:'application/json'})); }
        catch(_){ try{ fetch(url, { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) }); }catch(e){} }
      }
      window.addEventListener('error', function(e){ try { send({ level:'error', message:'payslip_window_error', context:{ error: String(e.message||''), stack: String((e.error&&e.error.stack)||'') } }); } catch(_){} });
      document.addEventListener('DOMContentLoaded', function(){ try { var pid = document.body && document.body.getAttribute('data-payroll-id'); send({ level:'info', message:'payslip_window_loaded', context:{ payroll_id: pid||null } }); } catch(_){} });
    })();</script></head>
    <body data-payroll-id="${escapeHtml(String(p.id || ''))}">
      <div class="sheet">
        <div class="title">
          <img src="${logoUrl}" alt="Company Logo" style="height:50px;margin-bottom:6px;"/>
          <h1>${escapeHtml(p.company_name || 'Ghana Demo Company')}</h1>
          <h2>${escapeHtml(p.company_address || 'Demo')}</h2>
        </div>
        <div class="subtitle">Payslip for the period of ${escapeHtml(monthTxt)}</div>
        <table class="info">
          <tr>
            <td>Employee Id</td><td>: ${escapeHtml(p.employee_number || '')}</td>
            <td>Name</td><td>: ${escapeHtml(p.employee_name || '')}</td>
          </tr>
          <tr>
            <td>Department</td><td>: ${escapeHtml(p.department_name || '')}</td>
            <td>Designation</td><td>: ${escapeHtml(p.position_title || '')}</td>
          </tr>
          <tr>
            <td>Date Of Birth</td><td>: ${escapeHtml(p.date_of_birth || '')}</td>
            <td>Hire Date</td><td>: ${escapeHtml(p.hire_date || '')}</td>
          </tr>
          <tr>
            <td>Pay Date</td><td>: ${escapeHtml(p.pay_period_end || '')}</td>
            <td>Employer SSF</td><td>: ${fmt(p.ssnit_employer)}</td>
          </tr>
        </table>
        <div class="hr"></div>
        <table>
          <thead>
            <tr>
              <th>Earnings</th><th class="right">Amount</th>
              <th>Deductions</th><th class="right">Amount</th>
            </tr>
          </thead>
          <tbody>
            ${rows}
            <tr>
              <th>Total Earnings</th><th class="right">${fmt(totalEarnings)}</th>
              <th>Total Deductions</th><th class="right">${fmt(totalDeductions)}</th>
            </tr>
            <tr>
              <th colspan="2" class="center">Net Pay (Rounded)</th>
              <th colspan="2" class="center">${fmt(net)}</th>
            </tr>
          </tbody>
        </table>
        <div class="footnote">(All figures in Ghana Cedi)</div>
        <table class="signatures no-border">
          <tr>
            <td>_______________________________<br/>Employer's Signature</td>
            <td class="right">_______________________________<br/>Employee's Signature</td>
          </tr>
        </table>
        <div class="no-print" style="text-align:right; margin-top:10px;"><button onclick="window.print()" class="btn btn-primary">Print / Save PDF</button></div>
      </div>
    </body></html>`;
    return html;
  }

  // Write payslip HTML into an existing window (opened synchronously by caller)
  function renderPayslipIntoWindow(w, payslipData, { autoPrint = false } = {}) {
    if (!w) return;
    const html = buildPayslipHtml(payslipData);
    try {
      w.document.open();
      w.document.write(html);
      w.document.close();
    } catch (_) {
      // Fallback: Blob URL navigation if document.write is restricted
      const blob = new Blob([html], { type: 'text/html' });
      const url = URL.createObjectURL(blob);
      w.location.href = url;
      setTimeout(() => { try { URL.revokeObjectURL(url); } catch(_){} }, 30000);
    }
    if (autoPrint) {
      try { w.focus(); } catch(_) {}
      setTimeout(() => { try { w.print(); } catch(_) {} }, 500);
    }
  }

  function openPayslipWindow(payslipData, { autoPrint = false } = {}) {
    const w = window.open('', '_blank', 'noopener,noreferrer');
    if (!w) return;
    try { renderPayslipIntoWindow(w, payslipData, { autoPrint }); }
    catch (e) {
      try { w.close(); } catch(_) {}
      window.auth?.showNotification?.('Failed to render payslip','error');
      console && console.error && console.error('Payslip render error:', e);
    }
  }
  function canEditPayroll() {
    try { return window.auth?.canEdit('payroll'); } catch(_) { return false; }
  }

  async function loadSetupLists() {
    const res = await fetch('api/payroll.php?action=setup_lists', { credentials: 'same-origin' });
    const data = await res.json();
    if (!data?.success) throw new Error(data?.message || 'Failed');
    return data.data || { allowance_types: [], deduction_types: [] };
  }

  async function saveAllowanceType(payload) {
    const res = await fetch('api/payroll.php?action=save_allowance_type', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }
  async function saveDeductionType(payload) {
    const res = await fetch('api/payroll.php?action=save_deduction_type', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }
  async function deleteAllowanceType(id) {
    const res = await fetch(`api/payroll.php?action=delete_allowance_type&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }
  async function deleteDeductionType(id) {
    const res = await fetch(`api/payroll.php?action=delete_deduction_type&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }

  async function loadEmployees() {
    const res = await fetch('api/employees.php', { credentials: 'same-origin' });
    const data = await res.json();
    if (!data?.success) return [];
    return Array.isArray(data.data) ? data.data : [];
  }

  // Grades APIs
  async function loadGrades() {
    const res = await fetch('api/grades.php', { credentials: 'same-origin' });
    const data = await res.json();
    if (!data?.success) return [];
    return Array.isArray(data.data) ? data.data : [];
  }
  async function saveGrade(payload) {
    const res = await fetch('api/grades.php?action=save', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }
  async function deleteGrade(id) {
    const res = await fetch(`api/grades.php?action=delete&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }

  // Grade components APIs
  async function loadGradeDetail(id) {
    const res = await fetch(`api/grades.php?action=detail&id=${encodeURIComponent(id)}`, { credentials: 'same-origin' });
    const data = await res.json();
    if (!data?.success) return null;
    return data.data;
  }
  async function saveGradeAllowance(payload) {
    const res = await fetch('api/grades.php?action=save_allowance', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }
  async function deleteGradeAllowance(id) {
    const res = await fetch(`api/grades.php?action=delete_allowance&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }
  async function saveGradeDeduction(payload) {
    const res = await fetch('api/grades.php?action=save_deduction', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }
  async function deleteGradeDeduction(id) {
    const res = await fetch(`api/grades.php?action=delete_deduction&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }

  async function processPayroll(body) {
    const res = await fetch('api/payroll.php?action=process', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(body) });
    return res.json();
  }

  async function approvePayroll(id) {
    try {
      const res = await fetch(`api/payroll.php?action=approve&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
      const text = await res.text();
      try { return JSON.parse(text); } catch (_) { return { success: false, message: `Server error (${res.status})` }; }
    } catch (e) {
      return { success: false, message: e?.message || 'Network error' };
    }
  }

  // Trigger payout: bank_manual, integration, paystack_bank, paystack_momo
  async function payoutPayrollAPI(body) {
    try {
      const res = await fetch('api/payroll.php?action=payout', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'same-origin',
        body: JSON.stringify(body)
      });
      const text = await res.text();
      try { return JSON.parse(text); } catch (_) { return { success:false, message: `Server error (${res.status})` }; }
    } catch (e) {
      return { success:false, message: e?.message || 'Network error' };
    }
  }

  async function approvePayrollBulk({ ids = [], period_month = null, period_year = null } = {}) {
    try {
      const body = { };
      if (Array.isArray(ids) && ids.length) body.ids = ids.map(x=>String(x));
      if (period_month) body.period_month = period_month;
      if (period_year) body.period_year = period_year;
      const res = await fetch('api/payroll.php?action=approve_all', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(body) });
      const text = await res.text();
      try { return JSON.parse(text); } catch (_) { return { success: false, message: `Server error (${res.status})` }; }
    } catch (e) {
      return { success: false, message: e?.message || 'Network error' };
    }
  }

  async function fetchPayrollPreview(id) {
    try {
      const res = await fetch(`api/payroll.php?action=preview&id=${encodeURIComponent(id)}`, { credentials:'same-origin' });
      const data = await res.json();
      return data?.success ? data.data : null;
    } catch (_) { return null; }
  }

  async function loadPayrolls(params = {}) {
    const qs = new URLSearchParams(params);
    const res = await fetch(`api/payroll.php${qs.toString()? '?' + qs.toString(): ''}`, { credentials:'same-origin' });
    const data = await res.json();
    if (!data?.success) return [];
    return Array.isArray(data.data) ? data.data : [];
  }

  function setupTabHtml() {
    return `
      <div class="row g-3">
        <div class="col-md-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Allowance Types</h6>
              <button type="button" class="btn btn-sm btn-outline-primary" id="addAllowanceType">Add</button>
            </div>
            <div class="card-body">
              <div id="allowanceTypes">Loading...</div>
            </div>
          </div>
        </div>
        <div class="col-md-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Deduction Types</h6>
              <button type="button" class="btn btn-sm btn-outline-primary" id="addDeductionType">Add</button>
            </div>
            <div class="card-body">
              <div id="deductionTypes">Loading...</div>
            </div>
          </div>
        </div>
        <div class="col-12">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Grades</h6>
              <button type="button" class="btn btn-sm btn-outline-primary" id="addGrade">Add Grade</button>
            </div>
            <div class="card-body">
              <div id="gradesList">Loading...</div>
            </div>
          </div>
        </div>
      </div>`;
  }

  function renderTypeList(list, { isAllowance = true }) {
    if (!Array.isArray(list) || list.length === 0) return '<div class="text-muted">No records</div>';
    return `
      <div class="table-responsive">
        <table class="table table-sm">
          <thead><tr><th>Code</th><th>Name</th>${isAllowance?'<th>Taxable</th><th>Taxable %</th>':''}<th style="width:160px">Actions</th></tr></thead>
          <tbody>
            ${list.map(x=>`
              <tr data-id="${x.id}">
                <td><input class="form-control form-control-sm" value="${escapeHtml(x.code||'')}" data-field="code" /></td>
                <td><input class="form-control form-control-sm" value="${escapeHtml(x.name||'')}" data-field="name" /></td>
                ${isAllowance?`<td><input class="form-check-input" type="checkbox" data-field="taxable" ${x.taxable? 'checked':''} /></td>
                <td><input type="number" min="0" max="100" class="form-control form-control-sm" data-field="taxable_rate" value="${(x.taxable_rate??'') === null ? '' : String(x.taxable_rate)}" /></td>`:''}
                <td>
                  <button class="btn btn-sm btn-success me-1" data-action="save-type" data-kind="${isAllowance?'allowance':'deduction'}">Save</button>
                  <button class="btn btn-sm btn-outline-danger" data-action="delete-type" data-kind="${isAllowance?'allowance':'deduction'}">Delete</button>
                </td>
              </tr>`).join('')}
          </tbody>
        </table>
      </div>`;
  }

  function renderGradesTable(list) {
    const items = Array.isArray(list) ? list : [];
    const rows = items.map(g=>`
      <tr data-id="${g.id}">
        <td><input class="form-control form-control-sm" data-field="name" value="${escapeHtml(g.name||'')}" /></td>
        <td><input class="form-control form-control-sm" data-field="code" value="${escapeHtml(g.code||'')}" /></td>
        <td><input type="number" step="0.01" class="form-control form-control-sm" data-field="base_salary" value="${g.base_salary ?? ''}" /></td>
        <td>
          <button class="btn btn-sm btn-outline-secondary me-1" data-action="open-components">Components</button>
          <button class="btn btn-sm btn-success me-1" data-action="save-grade">Save</button>
          <button class="btn btn-sm btn-outline-danger" data-action="delete-grade">Delete</button>
        </td>
      </tr>`).join('');
    return `
      <div class="table-responsive">
        <table class="table table-sm">
          <thead><tr><th>Name</th><th>Code</th><th>Base Salary</th><th style="width:160px">Actions</th></tr></thead>
          <tbody>
            ${rows || '<tr data-empty="1"><td colspan="4" class="text-muted">No grades defined</td></tr>'}
          </tbody>
        </table>
      </div>`;
  }

  function ensureGradeComponentsModal(container, lists) {
    let modal = document.getElementById('gradeComponentsModal');
    if (!modal) {
      const wrap = document.createElement('div');
      wrap.innerHTML = `
        <div class="modal fade" id="gradeComponentsModal" tabindex="-1" aria-hidden="true">
          <div class="modal-dialog modal-lg">
            <div class="modal-content">
              <div class="modal-header">
                <h5 class="modal-title">Grade Components</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
              </div>
              <div class="modal-body">
                <div class="row g-3">
                  <div class="col-md-6">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                      <h6 class="mb-0">Allowances</h6>
                      <button type="button" class="btn btn-sm btn-outline-primary" id="addGradeAllowance">Add</button>
                    </div>
                    <div class="table-responsive">
                      <table class="table table-sm" id="gradeAllowancesTbl">
                        <thead><tr><th>Type</th><th>Mode</th><th>Amount</th><th>Rate %</th><th style="width:160px">Actions</th></tr></thead>
                        <tbody></tbody>
                      </table>
                    </div>
                  </div>
                  <div class="col-md-6">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                      <h6 class="mb-0">Deductions</h6>
                      <button type="button" class="btn btn-sm btn-outline-primary" id="addGradeDeduction">Add</button>
                    </div>
                    <div class="table-responsive">
                      <table class="table table-sm" id="gradeDeductionsTbl">
                        <thead><tr><th>Type</th><th>Mode</th><th>Amount</th><th>Rate %</th><th style="width:160px">Actions</th></tr></thead>
                        <tbody></tbody>
                      </table>
                    </div>
                  </div>
                </div>
              </div>
              <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
              </div>
            </div>
          </div>
        </div>`;
      document.body.appendChild(wrap.firstElementChild);
      modal = document.getElementById('gradeComponentsModal');

      const fillSelect = (opts, val) => {
        const o = ['<option value="">-- Select --</option>'].concat((opts||[]).map(x=>`<option value="${x.id}">${escapeHtml(x.code||'')}${x.code?' - ':''}${escapeHtml(x.name||'')}</option>`)).join('');
        const sel = document.createElement('select');
        sel.className = 'form-select form-select-sm';
        sel.innerHTML = o;
        if (val) sel.value = String(val);
        return sel;
      };

      // Add handlers
      modal.addEventListener('click', async (e) => {
        const btn = e.target.closest('button[data-action]');
        if (!btn) return;
        const action = btn.getAttribute('data-action');
        const tr = btn.closest('tr');
        const gid = parseInt(modal.getAttribute('data-grade-id') || '0', 10);
        if (!gid) return;

        if (action === 'save-allowance') {
          const id = parseInt(tr?.getAttribute('data-id') || '0', 10) || undefined;
          const typeId = parseInt(tr?.querySelector('select[data-field="type"]').value || '0', 10);
          const mode = tr?.querySelector('select[data-field="mode"]').value || 'fixed';
          const amountStr = tr?.querySelector('input[data-field="amount"]').value || '';
          const rateStr = tr?.querySelector('input[data-field="rate"]').value || '';
          const amount = mode==='fixed' ? (parseFloat(amountStr||'0')||0) : 0;
          const rate = mode==='fixed' ? null : (rateStr===''? null : (parseFloat(rateStr||'0')||0));
          if (!typeId) { window.auth?.showNotification?.('Select allowance type', 'error'); return; }
          const res = await saveGradeAllowance({ id, grade_id: gid, allowance_type_id: typeId, amount, calc_mode: mode, rate });
          if (res?.success) {
            window.auth?.showNotification?.('Saved', 'success');
            if (res.data?.id && !id) tr.setAttribute('data-id', String(res.data.id));
          } else {
            window.auth?.showNotification?.(res?.message || 'Failed to save', 'error');
          }
        } else if (action === 'delete-allowance') {
          const id = tr?.getAttribute('data-id');
          if (!id) { tr.remove(); return; }
          if (!confirm('Delete this allowance?')) return;
          const res = await deleteGradeAllowance(id);
          if (res?.success) { tr.remove(); } else { window.auth?.showNotification?.(res?.message || 'Failed', 'error'); }
        } else if (action === 'save-deduction') {
          const id = parseInt(tr?.getAttribute('data-id') || '0', 10) || undefined;
          const typeId = parseInt(tr?.querySelector('select[data-field="type"]').value || '0', 10);
          const mode = tr?.querySelector('select[data-field="mode"]').value || 'fixed';
          const amountStr = tr?.querySelector('input[data-field="amount"]').value || '';
          const rateStr = tr?.querySelector('input[data-field="rate"]').value || '';
          const amount = mode==='fixed' ? (parseFloat(amountStr||'0')||0) : 0;
          const rate = mode==='fixed' ? null : (rateStr===''? null : (parseFloat(rateStr||'0')||0));
          if (!typeId) { window.auth?.showNotification?.('Select deduction type', 'error'); return; }
          const res = await saveGradeDeduction({ id, grade_id: gid, deduction_type_id: typeId, amount, calc_mode: mode, rate });
          if (res?.success) {
            window.auth?.showNotification?.('Saved', 'success');
            if (res.data?.id && !id) tr.setAttribute('data-id', String(res.data.id));
          } else {
            window.auth?.showNotification?.(res?.message || 'Failed to save', 'error');
          }
        } else if (action === 'delete-deduction') {
          const id = tr?.getAttribute('data-id');
          if (!id) { tr.remove(); return; }
          if (!confirm('Delete this deduction?')) return;
          const res = await deleteGradeDeduction(id);
          if (res?.success) { tr.remove(); } else { window.auth?.showNotification?.(res?.message || 'Failed', 'error'); }
        }
      });

      // Store helper to rebuild rows when opening
      modal._fillComponents = (detail) => {
        const aBody = modal.querySelector('#gradeAllowancesTbl tbody');
        const dBody = modal.querySelector('#gradeDeductionsTbl tbody');
        aBody.innerHTML = '';
        dBody.innerHTML = '';
        const allowanceTypes = lists?.allowance_types || [];
        const deductionTypes = lists?.deduction_types || [];
        const addRow = (tbody, opts, row, isAllowance) => {
          const tr = document.createElement('tr');
          tr.setAttribute('data-id', row?.id ? String(row.id) : '');
          const tdType = document.createElement('td');
          const sel = fillSelect(opts, row && (isAllowance ? row.allowance_type_id : row.deduction_type_id));
          sel.setAttribute('data-field', 'type');
          tdType.appendChild(sel);
          const tdMode = document.createElement('td');
          tdMode.innerHTML = `<select class="form-select form-select-sm" data-field="mode">
              <option value="fixed">Fixed Amount</option>
              <option value="percent_basic">% of Basic</option>
              <option value="percent_gross">% of Gross</option>
            </select>`;
          const tdAmt = document.createElement('td');
          tdAmt.innerHTML = `<input type="number" step="0.01" class="form-control form-control-sm" data-field="amount" value="${row?.amount ?? ''}" />`;
          const tdRate = document.createElement('td');
          tdRate.innerHTML = `<input type="number" step="0.01" class="form-control form-control-sm" data-field="rate" value="${(row?.rate ?? '')}" />`;
          const tdAct = document.createElement('td');
          tdAct.innerHTML = `<button class="btn btn-sm btn-success me-1" data-action="${isAllowance?'save-allowance':'save-deduction'}">Save</button>
                             <button class="btn btn-sm btn-outline-danger" data-action="${isAllowance?'delete-allowance':'delete-deduction'}">Delete</button>`;
          tr.appendChild(tdType); tr.appendChild(tdMode); tr.appendChild(tdAmt); tr.appendChild(tdRate); tr.appendChild(tdAct);
          // Initialize mode and toggle visibility
          const modeSel = tdMode.querySelector('select[data-field="mode"]');
          if (row?.calc_mode) modeSel.value = String(row.calc_mode);
          const toggle = ()=>{
            const m = modeSel.value;
            tdAmt.style.display = (m==='fixed')? '' : 'none';
            tdRate.style.display = (m==='fixed')? 'none' : '';
          };
          modeSel.addEventListener('change', toggle); toggle();
          tbody.appendChild(tr);
        };
        (detail.allowances || []).forEach(row => addRow(aBody, allowanceTypes, row, true));
        (detail.deductions || []).forEach(row => addRow(dBody, deductionTypes, row, false));

        // Add buttons
        modal.querySelector('#addGradeAllowance').onclick = () => addRow(aBody, allowanceTypes, null, true);
        modal.querySelector('#addGradeDeduction').onclick = () => addRow(dBody, deductionTypes, null, false);
      };
    }
    return modal;
  }

  function ensureGradeModal(container) {
    let modal = document.getElementById('gradeModal');
    if (!modal) {
      const wrap = document.createElement('div');
      wrap.innerHTML = `
        <div class="modal fade" id="gradeModal" tabindex="-1" aria-hidden="true">
          <div class="modal-dialog">
            <div class="modal-content">
              <div class="modal-header">
                <h5 class="modal-title" id="gradeModalTitle">Add Grade</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
              </div>
              <div class="modal-body">
                <div class="mb-3">
                  <label class="form-label">Name</label>
                  <input type="text" class="form-control" id="gradeName" />
                </div>
                <div class="mb-3">
                  <label class="form-label">Code</label>
                  <input type="text" class="form-control" id="gradeCode" />
                </div>
                <div class="mb-3">
                  <label class="form-label">Base Salary</label>
                  <input type="number" step="0.01" class="form-control" id="gradeBase" />
                </div>
              </div>
              <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveGradeBtn">Save</button>
              </div>
            </div>
          </div>
        </div>`;
      document.body.appendChild(wrap.firstElementChild);
      modal = document.getElementById('gradeModal');
      const saveBtn = modal.querySelector('#saveGradeBtn');
      saveBtn.addEventListener('click', async () => {
        const name = modal.querySelector('#gradeName').value.trim();
        const code = modal.querySelector('#gradeCode').value.trim();
        const baseStr = modal.querySelector('#gradeBase').value;
        const base_salary = baseStr === '' ? null : parseFloat(baseStr);
        if (!name) { window.auth?.showNotification?.('Name is required', 'error'); return; }
        const idAttr = modal.getAttribute('data-id');
        const id = idAttr ? parseInt(idAttr, 10) : undefined;
        const res = await saveGrade({ id, name, code, base_salary });
        if (res?.success) {
          window.auth?.showNotification?.('Saved', 'success');
          const grades2 = await loadGrades();
          const gWrap = container.querySelector('#gradesList');
          if (gWrap) gWrap.innerHTML = renderGradesTable(grades2);
          if (window.bootstrap && modal) {
            const bs = bootstrap.Modal.getOrCreateInstance(modal);
            bs.hide();
          }
        } else {
          window.auth?.showNotification?.(res?.message || 'Failed to save grade', 'error');
        }
      });
    }
    return modal;
  }

  function runTabHtml() {
    return `
      <div class="card">
        <div class="card-body">
          <div class="row g-3 align-items-end">
            <div class="col-md-3">
              <label class="form-label">Pay Period Start</label>
              <input type="date" class="form-control" id="ppStart" />
            </div>
            <div class="col-md-3">
              <label class="form-label">Pay Period End</label>
              <input type="date" class="form-control" id="ppEnd" />
            </div>
            <div class="col-md-6">
              <label class="form-label">Employees</label>
              <select multiple class="form-select" id="payEmployees" size="8"></select>
              <div class="form-text"><button type="button" class="btn btn-link p-0" id="selectAllEmp">Select All</button></div>
            </div>
            <div class="col-12">
              <button type="button" class="btn btn-primary" id="btnProcessPayroll">Process Payroll</button>
            </div>
          </div>
          <hr/>
          <div id="runResult"></div>
        </div>
      </div>`;
  }

  function approvalsTabHtml() {
    return `
      <div class="card">
        <div class="card-body">
          <div id="approvalsList">Loading...</div>
        </div>
      </div>`;
  }

  function historyTabHtml() {
    return `
      <div class="card">
        <div class="card-body">
          <div id="historyList">Loading...</div>
        </div>
      </div>`;
  }

  function reportsTabHtml() {
    // Month/year filters + 4 report sections with table and CSV buttons
    const months = ['','01','02','03','04','05','06','07','08','09','10','11','12'];
    const monthOpts = months.map((m,i)=>`<option value="${m}">${i===0?'-- Month --':m}</option>`).join('');
    const year = new Date().getFullYear();
    const years = [year-1, year, year+1];
    const yearOpts = ['<option value="">-- Year --</option>'].concat(years.map(y=>`<option value="${y}">${y}</option>`)).join('');
    return `
      <div class="card">
        <div class="card-body">
          <div class="row g-3 align-items-end mb-3">
            <div class="col-md-2">
              <label class="form-label">Month</label>
              <select class="form-select" id="repMonth">${monthOpts}</select>
            </div>
            <div class="col-md-2">
              <label class="form-label">Year</label>
              <select class="form-select" id="repYear">${yearOpts}</select>
            </div>
            <div class="col-md-8 d-flex gap-2">
              <button type="button" class="btn btn-primary" id="btnLoadReports">Load Reports</button>
              <button type="button" class="btn btn-outline-secondary" data-csv="paye">Download PAYE CSV</button>
              <button type="button" class="btn btn-outline-secondary" data-csv="ssnit">Download SSNIT CSV</button>
              <button type="button" class="btn btn-outline-secondary" data-csv="summary">Download Summary CSV</button>
              <button type="button" class="btn btn-outline-secondary" data-csv="payslips">Download Payslips CSV</button>
            </div>
          </div>

          <div class="row g-3">
            <div class="col-12">
              <div class="card">
                <div class="card-header"><strong>PAYE Schedule</strong></div>
                <div class="card-body" id="payeReport">Use filters then click Load Reports.</div>
              </div>
            </div>
            <div class="col-12">
              <div class="card">
                <div class="card-header"><strong>SSNIT Contribution Report</strong></div>
                <div class="card-body" id="ssnitReport"></div>
              </div>
            </div>
            <div class="col-12">
              <div class="card">
                <div class="card-header"><strong>Payroll Summary</strong></div>
                <div class="card-body" id="summaryReport"></div>
              </div>
            </div>
            <div class="col-12">
              <div class="card">
                <div class="card-header"><strong>Payslips</strong></div>
                <div class="card-body" id="payslipsReport"></div>
              </div>
            </div>
          </div>
        </div>
      </div>`;
  }

  async function render(container) {
    const showApprovals = !!canApprovePayroll();
    const canEdit = !!canEditPayroll();
    const showReports = canEdit || showApprovals; // HR roles can view reports
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Payroll</h5>
        </div>
        <div class="card-body">
          <ul class="nav nav-tabs" role="tablist">
            ${canEdit ? `<li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#tab-setup" type="button" role="tab">Setup</button></li>` : ''}
            ${canEdit ? `<li class="nav-item" role="presentation"><button class="nav-link ${!canEdit ? 'active' : ''}" data-bs-toggle="tab" data-bs-target="#tab-run" type="button" role="tab">Run</button></li>` : ''}
            ${showApprovals ? `<li class="nav-item" role="presentation"><button class="nav-link ${!canEdit ? 'active' : ''}" data-bs-toggle="tab" data-bs-target="#tab-approvals" type="button" role="tab">Approvals</button></li>` : ''}
            ${showReports ? `<li class="nav-item" role="presentation"><button class="nav-link ${!canEdit && !showApprovals ? 'active' : ''}" data-bs-toggle="tab" data-bs-target="#tab-reports" type="button" role="tab">Reports</button></li>` : ''}
            <li class="nav-item" role="presentation"><button class="nav-link ${!canEdit && !showApprovals && !showReports ? 'active' : ''}" data-bs-toggle="tab" data-bs-target="#tab-history" type="button" role="tab">History</button></li>
          </ul>
          <div class="tab-content mt-3">
            ${canEdit ? `<div class="tab-pane fade show active" id="tab-setup" role="tabpanel">${setupTabHtml()}</div>` : ''}
            ${canEdit ? `<div class="tab-pane fade ${!canEdit ? 'show active' : ''}" id="tab-run" role="tabpanel">${runTabHtml()}</div>` : ''}
            ${showApprovals ? `<div class="tab-pane fade ${!canEdit ? 'show active' : ''}" id="tab-approvals" role="tabpanel">${approvalsTabHtml()}</div>` : ''}
            ${showReports ? `<div class="tab-pane fade ${!canEdit && !showApprovals ? 'show active' : ''}" id="tab-reports" role="tabpanel">${reportsTabHtml()}</div>` : ''}
            <div class="tab-pane fade ${!canEdit && !showApprovals && !showReports ? 'show active' : ''}" id="tab-history" role="tabpanel">${historyTabHtml()}</div>
          </div>
        </div>
      </div>`;

    // Setup tab
    if (canEdit) {
      try {
        const lists = await loadSetupLists();
        const aWrap = container.querySelector('#allowanceTypes');
        const dWrap = container.querySelector('#deductionTypes');
        if (aWrap) aWrap.innerHTML = renderTypeList(lists.allowance_types, { isAllowance: true });
        if (dWrap) dWrap.innerHTML = renderTypeList(lists.deduction_types, { isAllowance: false });
        // Grades
        const grades = await loadGrades();
        const gWrap = container.querySelector('#gradesList');
        if (gWrap) gWrap.innerHTML = renderGradesTable(grades);

        // Add new type rows
        container.querySelector('#addAllowanceType')?.addEventListener('click', () => {
          const tbody = container.querySelector('#allowanceTypes tbody');
          if (!tbody) return;
          const tr = document.createElement('tr');
          tr.innerHTML = `<td><input class="form-control form-control-sm" data-field="code" /></td>
                          <td><input class="form-control form-control-sm" data-field="name" /></td>
                          <td><input class="form-check-input" type="checkbox" data-field="taxable" checked /></td>
                          <td><input type="number" min="0" max="100" class="form-control form-control-sm" data-field="taxable_rate" value="100" /></td>
                          <td>
                            <button class="btn btn-sm btn-success me-1" data-action="save-type" data-kind="allowance">Save</button>
                            <button class="btn btn-sm btn-outline-danger" data-action="delete-row">Remove</button>
                          </td>`;
          tbody.prepend(tr);
        });
        container.querySelector('#addDeductionType')?.addEventListener('click', () => {
          const tbody = container.querySelector('#deductionTypes tbody');
          if (!tbody) return;
          const tr = document.createElement('tr');
          tr.innerHTML = `<td><input class="form-control form-control-sm" data-field="code" /></td>
                          <td><input class="form-control form-control-sm" data-field="name" /></td>
                          <td>
                            <button class="btn btn-sm btn-success me-1" data-action="save-type" data-kind="deduction">Save</button>
                            <button class="btn btn-sm btn-outline-danger" data-action="delete-row">Remove</button>
                          </td>`;
          tbody.prepend(tr);
        });
        container.querySelector('#addGrade')?.addEventListener('click', () => {
          const modal = ensureGradeModal(container);
          if (!modal) return;
          modal.setAttribute('data-id', '');
          modal.querySelector('#gradeModalTitle').textContent = 'Add Grade';
          modal.querySelector('#gradeName').value = '';
          modal.querySelector('#gradeCode').value = '';
          modal.querySelector('#gradeBase').value = '';
          if (window.bootstrap) {
            const bs = bootstrap.Modal.getOrCreateInstance(modal);
            bs.show();
          } else {
            // Fallback: simple display (non-animated)
            modal.style.display = 'block';
            modal.classList.add('show');
          }
        });

        container.addEventListener('click', async (e) => {
          const btn = e.target.closest('button[data-action]');
          if (!btn) return;
          const action = btn.getAttribute('data-action');
          const tr = btn.closest('tr');
          if (action === 'open-components') {
            const idStr = tr?.getAttribute('data-id');
            const gid = idStr ? parseInt(idStr, 10) : 0;
            if (!gid) { window.auth?.showNotification?.('Please save the grade first', 'error'); return; }
            const detail = await loadGradeDetail(gid);
            if (!detail) { window.auth?.showNotification?.('Failed to load grade', 'error'); return; }
            const modal = ensureGradeComponentsModal(container, lists);
            if (!modal) return;
            modal.setAttribute('data-grade-id', String(gid));
            modal.querySelector('.modal-title').textContent = `Grade Components — ${detail.name || ('#'+gid)}`;
            modal._fillComponents(detail);
            if (window.bootstrap) {
              const bs = bootstrap.Modal.getOrCreateInstance(modal);
              bs.show();
            } else {
              modal.style.display = 'block';
              modal.classList.add('show');
            }
            return;
          }
          if (action === 'save-type') {
            const kind = btn.getAttribute('data-kind');
            const id = parseInt(tr?.getAttribute('data-id') || '0', 10) || undefined;
            const name = tr?.querySelector('[data-field="name"]').value.trim();
            const code = tr?.querySelector('[data-field="code"]').value.trim();
            if (!name || !code) { window.auth?.showNotification?.('Code and Name are required', 'error'); return; }
            const api = kind === 'allowance' ? saveAllowanceType : saveDeductionType;
            const payload = { id, name, code };
            if (kind === 'allowance') {
              const taxable = !!tr?.querySelector('[data-field="taxable"]').checked;
              const rateEl = tr?.querySelector('[data-field="taxable_rate"]');
              const taxable_rate = rateEl && rateEl.value !== '' ? parseFloat(rateEl.value) : null;
              payload.taxable = taxable ? 1 : 0;
              if (taxable_rate !== null && !Number.isNaN(taxable_rate)) payload.taxable_rate = taxable_rate;
            }
            const result = await api(payload);
            if (result?.success) {
              window.auth?.showNotification?.('Saved', 'success');
              const lists2 = await loadSetupLists();
              const aW = container.querySelector('#allowanceTypes');
              const dW = container.querySelector('#deductionTypes');
              if (aW) aW.innerHTML = renderTypeList(lists2.allowance_types, { isAllowance: true });
              if (dW) dW.innerHTML = renderTypeList(lists2.deduction_types, { isAllowance: false });
            } else {
              window.auth?.showNotification?.(result?.message || 'Failed to save', 'error');
            }
          } else if (action === 'delete-type') {
            const kind = btn.getAttribute('data-kind');
            const id = tr?.getAttribute('data-id');
            if (!id) { tr?.remove(); return; }
            if (!confirm('Delete this item?')) return;
            const api = kind === 'allowance' ? deleteAllowanceType : deleteDeductionType;
            const result = await api(id);
            if (result?.success) {
              window.auth?.showNotification?.('Deleted', 'success');
              tr.remove();
            } else {
              window.auth?.showNotification?.(result?.message || 'Failed to delete', 'error');
            }
          } else if (action === 'delete-row') {
            tr?.remove();
          }
        });
      } catch (e) {
        container.querySelector('#allowanceTypes')?.replaceChildren();
        container.querySelector('#deductionTypes')?.replaceChildren();
      }
    }

    // Run tab
    if (canEdit) {
      try {
        const select = container.querySelector('#payEmployees');
        const list = await loadEmployees();
        if (select) select.innerHTML = list.map(e => `<option value="${e.id}">${escapeHtml(e.employee_number || '')} — ${escapeHtml(e.full_name || (e.first_name + ' ' + e.last_name))}</option>`).join('');
        container.querySelector('#selectAllEmp')?.addEventListener('click', () => {
          Array.from(select.options).forEach(o => o.selected = true);
        });
        const btnProc = container.querySelector('#btnProcessPayroll');
        btnProc?.addEventListener('click', async (e) => {
          const start = container.querySelector('#ppStart').value;
          const end = container.querySelector('#ppEnd').value;
          const ids = Array.from(select.selectedOptions).map(o => parseInt(o.value, 10)).filter(Boolean);
          if (!start || !end || !ids.length) { window.auth?.showNotification?.('Select dates and at least one employee', 'error'); return; }
          const btnEl = e.currentTarget;
          const originalText = btnEl?.innerHTML;
          if (btnEl) { btnEl.disabled = true; btnEl.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span> Processing...'; }
          window.auth?.showNotification?.('Processing payroll...', 'info');
          let res;
          try {
            res = await processPayroll({ employees: ids, pay_period_start: start, pay_period_end: end });
          } finally {
            if (btnEl) { btnEl.disabled = false; btnEl.innerHTML = originalText; }
          }
          const box = container.querySelector('#runResult');
          if (res?.success) {
            const out = res.data || {};
            const processed = out.processed_count || 0;
            const total = out.total_employees || ids.length;
            box.innerHTML = `<div class="alert alert-success">Processed ${processed} of ${total} employees.</div>` + (Array.isArray(out.errors) && out.errors.length ? `<div class="alert alert-warning"><div>${out.errors.map(e=>`<div>${escapeHtml(e)}</div>`).join('')}</div></div>` : '');
            window.auth?.showNotification?.(`Processed ${processed} of ${total}`, 'success');
            try { box.scrollIntoView({ behavior: 'smooth', block: 'start' }); } catch(_) {}
          } else {
            const msg = res?.message || 'Failed';
            box.innerHTML = `<div class="alert alert-danger">${escapeHtml(msg)}</div>`;
            window.auth?.showNotification?.(msg, 'error');
          }
        });
      } catch (_) {}
    }

    // Approvals tab
    if (showApprovals) {
      try {
        const wrap = container.querySelector('#approvalsList');
        if (!wrap) return;

        const ensurePayrollPreviewModal = () => {
          let modal = document.getElementById('payrollPreviewModal');
          if (!modal) {
            const div = document.createElement('div');
            div.innerHTML = `
              <div class="modal fade" id="payrollPreviewModal" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-lg">
                  <div class="modal-content">
                    <div class="modal-header">
                      <h5 class="modal-title">Payroll Preview</h5>
                      <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body"><div id="previewContent">Loading...</div></div>
                    <div class="modal-footer">
                      <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                      <button type="button" class="btn btn-primary" id="btnApproveSingle" data-id="">Approve</button>
                    </div>
                  </div>
                </div>
              </div>`;
            document.body.appendChild(div.firstElementChild);
            modal = document.getElementById('payrollPreviewModal');
            // attach approve handler
            modal.querySelector('#btnApproveSingle').addEventListener('click', async (e) => {
              const id = modal.querySelector('#btnApproveSingle').getAttribute('data-id');
              if (!id) return;
              const btn = modal.querySelector('#btnApproveSingle');
              btn.disabled = true;
              const res = await approvePayroll(id);
              btn.disabled = false;
              if (res?.success) {
                window.auth?.showNotification?.('Payroll approved', 'success');
                // remove row from approvals list if present
                const tr = wrap.querySelector(`tr[data-id="${CSS.escape(String(id))}"]`);
                tr?.remove();
                if (window.bootstrap) bootstrap.Modal.getOrCreateInstance(modal).hide();
              } else {
                window.auth?.showNotification?.(res?.message || 'Failed to approve', 'error');
              }
            });
          }
          return modal;
        };

        const ensureApproveAllModal = () => {
          let modal = document.getElementById('approveAllModal');
          if (!modal) {
            const div = document.createElement('div');
            div.innerHTML = `
              <div class="modal fade" id="approveAllModal" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-lg">
                  <div class="modal-content">
                    <div class="modal-header">
                      <h5 class="modal-title">Approve All</h5>
                      <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                      <div id="approveAllBody">Loading...</div>
                    </div>
                    <div class="modal-footer">
                      <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                      <button type="button" class="btn btn-success" id="btnApproveAllConfirm">Approve All</button>
                    </div>
                  </div>
                </div>
              </div>`;
            document.body.appendChild(div.firstElementChild);
            modal = document.getElementById('approveAllModal');
            modal.querySelector('#btnApproveAllConfirm').addEventListener('click', async ()=>{
              let ids = [];
              try {
                const idsAttr = modal.getAttribute('data-ids') || '[]';
                ids = JSON.parse(idsAttr);
              } catch(_) { ids = []; }
              if (!ids.length) {
                // Fallback: read from approvals list cache attribute if present
                try {
                  const wrap = document.getElementById('approvalsList');
                  const wrapIds = wrap?.getAttribute('data-ids');
                  if (wrapIds) { ids = JSON.parse(wrapIds); }
                } catch(_) {}
              }
              if (!Array.isArray(ids) || !ids.length) { window.auth?.showNotification?.('Nothing to approve', 'warning'); return; }
              const btn = modal.querySelector('#btnApproveAllConfirm');
              btn.disabled = true;
              window.auth?.showNotification?.('Approving selected payrolls...', 'info');
              const res = await approvePayrollBulk({ ids });
              btn.disabled = false;
              if (res?.success) {
                const cnt = (res.data && typeof res.data.approved_count!=='undefined') ? res.data.approved_count : ids.length;
                window.auth?.showNotification?.(`Approved ${cnt}`, 'success');
                await refreshApprovals();
                if (window.bootstrap) bootstrap.Modal.getOrCreateInstance(modal).hide();
              } else {
                window.auth?.showNotification?.(res?.message || 'Failed to approve all', 'error');
              }
            });
          }
          return modal;
        };

        let approvalsCache = [];

        const renderApprovalsTable = (items) => {
          const rows = items.map(p=>`<tr data-id="${p.id}">
              <td>${escapeHtml(p.employee_name || '')}</td>
              <td>${escapeHtml(p.pay_period_start || '')} - ${escapeHtml(p.pay_period_end || '')}</td>
              <td>${escapeHtml(p.net_salary || '')}</td>
              <td>${escapeHtml(p.status || '')}</td>
              <td>
                <button class="btn btn-sm btn-outline-secondary me-1" data-action="preview-payroll">Preview</button>
                <button class="btn btn-sm btn-success" data-action="approve-payroll">Approve</button>
              </td>
            </tr>`).join('');
          const header = `
            <div class="d-flex justify-content-between align-items-center mb-2">
              <h6 class="mb-0">Awaiting Approval</h6>
              <div>
                <button type="button" class="btn btn-sm btn-outline-primary" id="btnPreviewAll">Preview All</button>
              </div>
            </div>`;
          return header + (items.length ? `<div class="table-responsive"><table class="table table-sm">
              <thead><tr><th>Employee</th><th>Period</th><th>Net</th><th>Status</th><th style="width:200px">Actions</th></tr></thead>
              <tbody>${rows}</tbody>
            </table></div>` : '<div class="text-muted">No calculated payrolls awaiting approval</div>');
        };

        const refreshApprovals = async () => {
          const list = await loadPayrolls({ status: 'calculated' });
          approvalsCache = Array.isArray(list) ? list : [];
          wrap.innerHTML = renderApprovalsTable(approvalsCache);
          // Store ids for Preview All
          wrap.setAttribute('data-ids', JSON.stringify(approvalsCache.map(p=>p.id)));
        };

        await refreshApprovals();

        // Delegated actions: approve, preview single, preview all
        wrap.addEventListener('click', async (e) => {
          const btn = e.target.closest('button');
          if (!btn) return;
          const action = btn.getAttribute('data-action');
          if (action === 'approve-payroll') {
            const tr = btn.closest('tr');
            const id = tr?.getAttribute('data-id');
            if (!id) return;
            btn.disabled = true;
            const res = await approvePayroll(id);
            btn.disabled = false;
            if (res?.success) {
              window.auth?.showNotification?.('Payroll approved', 'success');
              tr.remove();
            } else {
              window.auth?.showNotification?.(res?.message || 'Failed to approve', 'error');
            }
          } else if (action === 'preview-payroll') {
            const tr = btn.closest('tr');
            const id = tr?.getAttribute('data-id');
            if (!id) return;
            const data = await fetchPayrollPreview(id);
            if (!data) { window.auth?.showNotification?.('Failed to load preview', 'error'); return; }
            const modal = ensurePayrollPreviewModal();
            modal.querySelector('.modal-title').textContent = `Payroll Preview — ${data.employee_name || ''}`;
            modal.querySelector('#btnApproveSingle').setAttribute('data-id', String(id));
            // build compact preview content
            const allowances = Array.isArray(data.allowances) ? data.allowances : [];
            const deductions = Array.isArray(data.deductions) ? data.deductions : [];
            const rowsAllow = allowances.map(a=>`<tr><td>${escapeHtml(a.name||a.code||'')}</td><td class="text-end">${currency(a.amount)}</td><td class="text-muted">${escapeHtml(a.source||'')}</td></tr>`).join('') || '<tr><td colspan="3" class="text-muted">None</td></tr>';
            const rowsDed = deductions.map(d=>`<tr><td>${escapeHtml(d.name||d.code||'')}</td><td class="text-end">${currency(d.amount)}</td><td class="text-muted">${escapeHtml(d.source||'')}</td></tr>`).join('') || '<tr><td colspan="3" class="text-muted">None</td></tr>';
            const html = `
              <div>
                <div class="mb-2"><strong>${escapeHtml(data.employee_name||'')}</strong> — ${escapeHtml(data.position_title||'')}</div>
                <div class="mb-2">Period: ${escapeHtml(data.pay_period_start||'')} - ${escapeHtml(data.pay_period_end||'')}</div>
                <div class="row g-3">
                  <div class="col-md-6">
                    <div class="border rounded">
                      <div class="p-2 bg-light fw-bold">Earnings</div>
                      <div class="p-2">
                        <table class="table table-sm">
                          <thead><tr><th>Description</th><th class="text-end">Amount</th><th></th></tr></thead>
                          <tbody>
                            <tr><td>Basic Salary</td><td class="text-end">${currency(data.basic_salary)}</td><td></td></tr>
                            ${rowsAllow}
                          </tbody>
                          <tfoot><tr><th>Total Earnings (Gross)</th><th class="text-end">${currency(data.gross_salary)}</th><th></th></tr></tfoot>
                        </table>
                      </div>
                    </div>
                  </div>
                  <div class="col-md-6">
                    <div class="border rounded">
                      <div class="p-2 bg-light fw-bold">Deductions</div>
                      <div class="p-2">
                        <table class="table table-sm">
                          <thead><tr><th>Description</th><th class="text-end">Amount</th><th></th></tr></thead>
                          <tbody>
                            <tr><td>SSNIT (Employee 5.5%)</td><td class="text-end">${currency(data.ssnit_employee)}</td><td></td></tr>
                            <tr><td>PAYE</td><td class="text-end">${currency(data.total_tax)}</td><td></td></tr>
                            ${rowsDed}
                          </tbody>
                          <tfoot><tr><th>Total Deductions</th><th class="text-end">${currency((+data.ssnit_employee||0)+(+data.total_tax||0)+(+data.total_deductions||0))}</th><th></th></tr></tfoot>
                        </table>
                      </div>
                    </div>
                  </div>
                </div>
                <div class="mt-2"><strong>Net Pay:</strong> ${currency(data.net_salary)} — <span class="text-muted">Employer SSNIT (13%): ${currency(data.ssnit_employer)}</span></div>
              </div>`;
            modal.querySelector('#previewContent').innerHTML = html;
            if (window.bootstrap) bootstrap.Modal.getOrCreateInstance(modal).show(); else { modal.style.display='block'; modal.classList.add('show'); }
          } else if (btn.id === 'btnPreviewAll') {
            const idsAttr = wrap.getAttribute('data-ids') || '[]';
            let ids = [];
            try { ids = JSON.parse(idsAttr); } catch(_) { ids = []; }
            const modal = ensureApproveAllModal();
            modal.setAttribute('data-ids', JSON.stringify(ids));
            const body = modal.querySelector('#approveAllBody');
            if (approvalsCache.length) {
              // Build table with requested columns and totals
              let tBasic = 0, tAllow = 0, tGross = 0, tSSNITE = 0, tSSNITR = 0, tPAYE = 0, tOtherDed = 0, tNet = 0;
              const rows = approvalsCache.map(p=>{
                const basic = +p.basic_salary || 0;
                const allow = +p.total_allowances || 0; // benefits
                const gross = +p.gross_salary || (basic + allow);
                const ssnE = +p.ssnit_employee || 0; // 5.5%
                const ssnR = +p.ssnit_employer || 0; // 13%
                const paye = +p.total_tax || 0;
                const other = +p.total_deductions || 0;
                const net = +p.net_salary || (gross - ssnE - paye - other);
                tBasic += basic; tAllow += allow; tGross += gross; tSSNITE += ssnE; tSSNITR += ssnR; tPAYE += paye; tOtherDed += other; tNet += net;
                return `<tr>
                  <td>${escapeHtml(p.employee_name || '')}</td>
                  <td>${escapeHtml(p.pay_period_start || '')} - ${escapeHtml(p.pay_period_end || '')}</td>
                  <td class="text-end">${currency(basic)}</td>
                  <td class="text-end">${currency(allow)}</td>
                  <td class="text-end">${currency(gross)}</td>
                  <td class="text-end">${currency(ssnE)}</td>
                  <td class="text-end">${currency(ssnR)}</td>
                  <td class="text-end">${currency(paye)}</td>
                  <td class="text-end">${currency(other)}</td>
                  <td class="text-end">${currency(net)}</td>
                  <td>${escapeHtml(p.status || '')}</td>
                </tr>`;
              }).join('');
              const totals = `<tr class="fw-bold">
                  <td colspan="2" class="text-end">Totals:</td>
                  <td class="text-end">${currency(tBasic)}</td>
                  <td class="text-end">${currency(tAllow)}</td>
                  <td class="text-end">${currency(tGross)}</td>
                  <td class="text-end">${currency(tSSNITE)}</td>
                  <td class="text-end">${currency(tSSNITR)}</td>
                  <td class="text-end">${currency(tPAYE)}</td>
                  <td class="text-end">${currency(tOtherDed)}</td>
                  <td class="text-end">${currency(tNet)}</td>
                  <td></td>
                </tr>`;
              const html = `<div class="table-responsive"><table class="table table-sm">
                <thead>
                  <tr>
                    <th>Employee</th>
                    <th>Period</th>
                    <th class="text-end">Basic</th>
                    <th class="text-end">Benefits</th>
                    <th class="text-end">Gross</th>
                    <th class="text-end">SSNIT 5.5%</th>
                    <th class="text-end">SSNIT 13%</th>
                    <th class="text-end">PAYE</th>
                    <th class="text-end">Other Deductions</th>
                    <th class="text-end">Net</th>
                    <th>Status</th>
                  </tr>
                </thead>
                <tbody>${rows}</tbody>
                <tfoot>${totals}</tfoot>
              </table></div>`;
              body.innerHTML = html;
            } else {
              body.innerHTML = '<div class="text-muted">No calculated payrolls</div>';
            }
            if (window.bootstrap) bootstrap.Modal.getOrCreateInstance(modal).show(); else { modal.style.display='block'; modal.classList.add('show'); }
          }
        });
      } catch (_) {}
    }

    // Payout modal (for HR approvers)
    const ensurePayoutModal = () => {
      let modal = document.getElementById('payoutModal');
      if (!modal) {
        const div = document.createElement('div');
        div.innerHTML = `
          <div class="modal fade" id="payoutModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog">
              <div class="modal-content">
                <div class="modal-header">
                  <h5 class="modal-title">Payout Salary</h5>
                  <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                  <div class="mb-3">
                    <label class="form-label">Channel</label>
                    <select class="form-select" id="payChannel">
                      <option value="bank_manual">Mark as Paid (Manual/Bank)</option>
                      <option value="integration">Mark as Paid (External Integration)</option>
                      <option value="paystack_bank">Paystack: Bank Transfer</option>
                      <option value="paystack_momo">Paystack: Mobile Money</option>
                    </select>
                  </div>
                  <div class="mb-3">
                    <label class="form-label">Recipient Name</label>
                    <input type="text" class="form-control" id="payoutRecipient" placeholder="Employee name" />
                  </div>
                  <div class="row g-3" data-bank>
                    <div class="col-md-6">
                      <label class="form-label">Bank Code</label>
                      <input type="text" class="form-control" id="bankCode" placeholder="e.g. 058" />
                    </div>
                    <div class="col-md-6">
                      <label class="form-label">Account Number</label>
                      <input type="text" class="form-control" id="bankAccount" placeholder="Account number" />
                    </div>
                  </div>
                  <div class="row g-3" data-momo style="display:none">
                    <div class="col-md-6">
                      <label class="form-label">MoMo Provider (bank_code)</label>
                      <input type="text" class="form-control" id="momoProvider" placeholder="e.g. MPS (MTN)" />
                    </div>
                    <div class="col-md-6">
                      <label class="form-label">MoMo Number</label>
                      <input type="text" class="form-control" id="momoAccount" placeholder="Phone number" />
                    </div>
                  </div>
                  <div class="row g-3 mt-1">
                    <div class="col-md-6">
                      <label class="form-label">Amount (override)</label>
                      <input type="number" step="0.01" class="form-control" id="payoutAmount" placeholder="Leave blank to use Net" />
                    </div>
                    <div class="col-md-6">
                      <label class="form-label">Reference (optional)</label>
                      <input type="text" class="form-control" id="payoutRef" placeholder="Reference" />
                    </div>
                  </div>
                </div>
                <div class="modal-footer">
                  <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                  <button type="button" class="btn btn-primary" id="btnPayoutSubmit" data-id="">Submit Payout</button>
                </div>
              </div>
            </div>
          </div>`;
        document.body.appendChild(div.firstElementChild);
        modal = document.getElementById('payoutModal');
        const channelSel = modal.querySelector('#payChannel');
        const bankRow = modal.querySelector('[data-bank]');
        const momoRow = modal.querySelector('[data-momo]');
        const toggle = () => {
          const ch = channelSel.value;
          const isBank = ch === 'paystack_bank';
          const isMomo = ch === 'paystack_momo';
          bankRow.style.display = isBank ? '' : 'none';
          momoRow.style.display = isMomo ? '' : 'none';
        };
        channelSel.addEventListener('change', toggle); toggle();
        modal.querySelector('#btnPayoutSubmit').addEventListener('click', async ()=>{
          const id = parseInt(modal.querySelector('#btnPayoutSubmit').getAttribute('data-id')||'0',10);
          if (!id) return;
          const channel = channelSel.value;
          const recipient_name = modal.querySelector('#payoutRecipient').value.trim();
          const amountStr = modal.querySelector('#payoutAmount').value;
          const reference = modal.querySelector('#payoutRef').value.trim();
          const details = { recipient_name };
          if (channel === 'paystack_bank') {
            details.bank_code = modal.querySelector('#bankCode').value.trim();
            details.account_number = modal.querySelector('#bankAccount').value.trim();
          } else if (channel === 'paystack_momo') {
            details.momo_provider = modal.querySelector('#momoProvider').value.trim();
            details.account_number = modal.querySelector('#momoAccount').value.trim();
          }
          const body = { id, channel, details };
          if (amountStr !== '') { const amt = parseFloat(amountStr); if (!Number.isNaN(amt)) body.amount = amt; }
          if (reference) body.reference = reference;
          const btn = modal.querySelector('#btnPayoutSubmit');
          btn.disabled = true;
          const res = await payoutPayrollAPI(body);
          btn.disabled = false;
          if (res?.success) {
            window.auth?.showNotification?.('Payout submitted', 'success');
            if (window.bootstrap) bootstrap.Modal.getOrCreateInstance(modal).hide(); else { modal.style.display='none'; modal.classList.remove('show'); }
            // Try to refresh history list if present
            try {
              const hist2 = await loadPayrolls({});
              const w = document.querySelector('#historyList');
              if (w) {
                w.innerHTML = hist2.length ? `
                  <div class="table-responsive">
                    <table class="table table-sm">
                      <thead><tr><th>Employee</th><th>Period</th><th>Net</th><th>Status</th><th style="width:260px">Actions</th></tr></thead>
                      <tbody>${hist2.map(p=>{
                        const canSlip = ['approved','paid'].includes(String(p.status||'').toLowerCase());
                        const canPayout = ['approved'].includes(String(p.status||'').toLowerCase()) && !!canApprovePayroll();
                        const btns = canSlip ? `
                          <button class="btn btn-sm btn-outline-primary me-1" data-action="view-payslip" data-id="${p.id}">View Payslip</button>
                          <button class="btn btn-sm btn-primary me-1" data-action="pdf-payslip" data-id="${p.id}">Download PDF</button>
                          ${canPayout ? `<button class=\"btn btn-sm btn-warning\" data-action=\"payout\" data-id=\"${p.id}\">Payout</button>`:''}
                        ` : '<span class="text-muted">—</span>';
                        return `<tr data-id="${p.id}"><td>${escapeHtml(p.employee_name || '')}</td><td>${escapeHtml(p.pay_period_start || '')} - ${escapeHtml(p.pay_period_end || '')}</td><td>${escapeHtml(p.net_salary || '')}</td><td>${escapeHtml(p.status || '')}</td><td>${btns}</td></tr>`;
                      }).join('')}</tbody>
                    </table>
                  </div>` : '<div class="text-muted">No payroll records yet</div>';
              }
            } catch(_) {}
          } else {
            window.auth?.showNotification?.(res?.message || 'Payout failed', 'error');
          }
        });
      }
      return modal;
    };

    // History tab
    try {
      const hist = await loadPayrolls({});
      const w = container.querySelector('#historyList');
      if (w) {
        w.innerHTML = hist.length ? `
          <div class="table-responsive">
            <table class="table table-sm">
              <thead><tr><th>Employee</th><th>Period</th><th>Net</th><th>Status</th><th style="width:260px">Actions</th></tr></thead>
              <tbody>${hist.map(p=>{
                const canSlip = ['approved','paid'].includes(String(p.status||'').toLowerCase());
                const canPayout = ['approved'].includes(String(p.status||'').toLowerCase()) && !!canApprovePayroll();
                const btns = canSlip ? `
                  <button class=\"btn btn-sm btn-outline-primary me-1\" data-action=\"view-payslip\" data-id=\"${p.id}\">View Payslip</button>
                  <button class=\"btn btn-sm btn-primary me-1\" data-action=\"pdf-payslip\" data-id=\"${p.id}\">Download PDF</button>
                  ${canPayout ? `<button class=\"btn btn-sm btn-warning\" data-action=\"payout\" data-id=\"${p.id}\">Payout</button>`:''}
                ` : '<span class=\"text-muted\">—</span>';
                return `<tr data-id=\"${p.id}\"><td>${escapeHtml(p.employee_name || '')}</td><td>${escapeHtml(p.pay_period_start || '')} - ${escapeHtml(p.pay_period_end || '')}</td><td>${escapeHtml(p.net_salary || '')}</td><td>${escapeHtml(p.status || '')}</td><td>${btns}</td></tr>`;
              }).join('')}</tbody>
            </table>
          </div>` : '<div class="text-muted">No payroll records yet</div>';
        // Attach delegated actions for payslip view/pdf within historyList
        w.addEventListener('click', async (e) => {
          const btn = e.target.closest('button[data-action]');
          if (!btn) return;
          const id = btn.getAttribute('data-id');
          const action = btn.getAttribute('data-action');
          if (!id) return;
          try {
            // Open window synchronously to avoid popup blockers, then load data
            if (action === 'view-payslip' || action === 'pdf-payslip') {
              const viewWin = window.open('', '_blank', 'noopener,noreferrer');
              if (viewWin) {
                try { viewWin.document.write('<!doctype html><title>Loading…</title><body style="font-family:sans-serif;padding:20px">Loading payslip…</body>'); viewWin.document.close(); } catch(_) {}
              }
              const data = await fetchPayslip(id);
              if (!data) {
                window.auth?.showNotification?.('Payslip not found', 'error');
                try { viewWin && viewWin.close(); } catch(_) {}
                return;
              }
              const auto = (action === 'pdf-payslip');
              if (viewWin) renderPayslipIntoWindow(viewWin, data, { autoPrint: auto });
              else openPayslipWindow(data, { autoPrint: auto });
              return;
            }
            if (action === 'payout') {
              if (!canApprovePayroll()) { window.auth?.showNotification?.('Insufficient permissions', 'error'); return; }
              const modal = ensurePayoutModal();
              if (!modal) return;
              modal.querySelector('#btnPayoutSubmit').setAttribute('data-id', String(id));
              // Try to prefill recipient name via preview
              try {
                const data = await fetchPayrollPreview(id);
                if (data) {
                  modal.querySelector('#payoutRecipient').value = String(data.employee_name || '');
                  modal.querySelector('#payoutAmount').value = String(data.net_salary ?? '');
                }
              } catch(_) {}
              if (window.bootstrap) bootstrap.Modal.getOrCreateInstance(modal).show(); else { modal.style.display='block'; modal.classList.add('show'); }
              return;
            }
          } catch (_) {
            window.auth?.showNotification?.('Failed to load payslip', 'error');
          }
        });
      }
    } catch (_) {}

    // Reports tab
    if (showReports) {
      const doLoad = async () => {
        const m = (container.querySelector('#repMonth')?.value || '').trim();
        const y = (container.querySelector('#repYear')?.value || '').trim();
        const params = new URLSearchParams();
        if (m) params.set('period_month', m);
        if (y) params.set('period_year', y);
        const qs = params.toString() ? '&' + params.toString() : '';
        try {
          const [paye, ssnit, summary, payslips] = await Promise.all([
            fetch('api/payroll.php?action=report_paye' + qs, { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null),
            fetch('api/payroll.php?action=report_ssnit' + qs, { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null),
            fetch('api/payroll.php?action=report_summary' + qs, { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null),
            fetch('api/payroll.php?action=report_payslips' + qs, { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null),
          ]);

          const payeWrap = container.querySelector('#payeReport');
          const ssnitWrap = container.querySelector('#ssnitReport');
          const sumWrap = container.querySelector('#summaryReport');
          const slipWrap = container.querySelector('#payslipsReport');

          if (payeWrap) {
            const rows = paye?.success ? (paye.data || []) : [];
            payeWrap.innerHTML = rows.length ? `<div class="table-responsive"><table class="table table-sm"><thead><tr><th>Emp #</th><th>Name</th><th>Start</th><th>End</th><th>Basic</th><th>Taxable Allow</th><th>Exempt Allow</th><th>Chargeable</th><th>PAYE</th><th>Status</th></tr></thead><tbody>${rows.map(r=>`<tr><td>${escapeHtml(r.employee_number||'')}</td><td>${escapeHtml(r.employee_name||'')}</td><td>${escapeHtml(r.pay_period_start||'')}</td><td>${escapeHtml(r.pay_period_end||'')}</td><td>${escapeHtml(r.basic_salary||'')}</td><td>${escapeHtml(r.taxable_allowances||'')}</td><td>${escapeHtml(r.exempt_allowances||'')}</td><td>${escapeHtml(r.chargeable_income||'')}</td><td>${escapeHtml(r.paye||'')}</td><td>${escapeHtml(r.status||'')}</td></tr>`).join('')}</tbody></table></div>` : '<div class="text-muted">No data</div>';
          }
          if (ssnitWrap) {
            const rows = ssnit?.success ? (ssnit.data || []) : [];
            ssnitWrap.innerHTML = rows.length ? `<div class="table-responsive"><table class="table table-sm"><thead><tr><th>Emp #</th><th>Name</th><th>Start</th><th>End</th><th>Gross</th><th>SSNIT Emp 5.5%</th><th>SSNIT Empr 13%</th><th>Status</th></tr></thead><tbody>${rows.map(r=>`<tr><td>${escapeHtml(r.employee_number||'')}</td><td>${escapeHtml(r.employee_name||'')}</td><td>${escapeHtml(r.pay_period_start||'')}</td><td>${escapeHtml(r.pay_period_end||'')}</td><td>${escapeHtml(r.gross_salary||'')}</td><td>${escapeHtml(r.ssnit_employee||'')}</td><td>${escapeHtml(r.ssnit_employer||'')}</td><td>${escapeHtml(r.status||'')}</td></tr>`).join('')}</tbody></table></div>` : '<div class="text-muted">No data</div>';
          }
          if (sumWrap) {
            const ok = summary?.success && summary.data;
            const rows = ok ? (summary.data.rows || []) : [];
            const totals = ok ? (summary.data.totals || {}) : {};
            const tbl = rows.length ? `<div class="table-responsive"><table class="table table-sm"><thead><tr><th>Emp #</th><th>Name</th><th>Start</th><th>End</th><th>Gross</th><th>SSNIT Emp</th><th>SSNIT Empr</th><th>PAYE</th><th>Other Deductions</th><th>Net</th><th>Employer Cost</th></tr></thead><tbody>${rows.map(r=>`<tr><td>${escapeHtml(r.employee_number||'')}</td><td>${escapeHtml(r.employee_name||'')}</td><td>${escapeHtml(r.pay_period_start||'')}</td><td>${escapeHtml(r.pay_period_end||'')}</td><td>${escapeHtml(r.gross_salary||'')}</td><td>${escapeHtml(r.ssnit_employee||'')}</td><td>${escapeHtml(r.ssnit_employer||'')}</td><td>${escapeHtml(r.total_tax||'')}</td><td>${escapeHtml(r.total_deductions||'')}</td><td>${escapeHtml(r.net_salary||'')}</td><td>${escapeHtml(((+r.gross_salary||0)+(+r.ssnit_employer||0)).toFixed ? ((+r.gross_salary||0)+(+r.ssnit_employer||0)).toFixed(2) : '')}</td></tr>`).join('')}</tbody></table></div>` : '<div class="text-muted">No data</div>';
            const totHtml = `<div class="mt-2"><strong>Totals:</strong> Gross=${escapeHtml((totals.total_gross??'').toString())}, SSNIT Emp=${escapeHtml((totals.ssnit_employee??'').toString())}, SSNIT Empr=${escapeHtml((totals.ssnit_employer??'').toString())}, PAYE=${escapeHtml((totals.paye??'').toString())}, Other Ded=${escapeHtml((totals.other_deductions??'').toString())}, Net=${escapeHtml((totals.total_net??'').toString())}, Employer Cost=${escapeHtml((totals.employer_cost??'').toString())}</div>`;
            sumWrap.innerHTML = tbl + totHtml;
          }
          if (slipWrap) {
            const rows = payslips?.success ? (payslips.data || []) : [];
            slipWrap.innerHTML = rows.length ? `<div class="table-responsive"><table class="table table-sm"><thead><tr><th>Emp #</th><th>Name</th><th>Start</th><th>End</th><th>Basic</th><th>Allowances</th><th>Other Deductions</th><th>PAYE</th><th>Net</th><th>Status</th></tr></thead><tbody>${rows.map(r=>`<tr><td>${escapeHtml(r.employee_number||'')}</td><td>${escapeHtml(r.employee_name||'')}</td><td>${escapeHtml(r.pay_period_start||'')}</td><td>${escapeHtml(r.pay_period_end||'')}</td><td>${escapeHtml(r.basic_salary||'')}</td><td>${escapeHtml(r.total_allowances||'')}</td><td>${escapeHtml(r.total_deductions||'')}</td><td>${escapeHtml(r.total_tax||'')}</td><td>${escapeHtml(r.net_salary||'')}</td><td>${escapeHtml(r.status||'')}</td></tr>`).join('')}</tbody></table></div>` : '<div class="text-muted">No data</div>';
          }
        } catch (_) {
          ['#payeReport','#ssnitReport','#summaryReport','#payslipsReport'].forEach(sel=>{ const el = container.querySelector(sel); if (el) el.textContent = 'Failed to load.'; });
        }
      };
      container.querySelector('#btnLoadReports')?.addEventListener('click', doLoad);
      container.addEventListener('click', (e)=>{
        const btn = e.target.closest('button[data-csv]');
        if (!btn) return;
        const kind = btn.getAttribute('data-csv');
        const m = (container.querySelector('#repMonth')?.value || '').trim();
        const y = (container.querySelector('#repYear')?.value || '').trim();
        const params = new URLSearchParams({ format: 'csv' });
        if (m) params.set('period_month', m);
        if (y) params.set('period_year', y);
        const url = `api/payroll.php?action=report_${kind}&${params.toString()}`;
        window.location.href = url;
      });
    }
  }

  function escapeHtml(s){ return String(s ?? '').replace(/[&<>"']/g, c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c])); }

  // Expose for My Payslips page reuse
  window.buildPayslipHtml = buildPayslipHtml;
  window.fetchPayslip = fetchPayslip;
  window.renderPayslipIntoWindow = renderPayslipIntoWindow;
  window.openPayslipWindow = openPayslipWindow;

  window.pages.payroll = { render };
})();
