// Onboarding Checklists UI
(function(){
  window.pages = window.pages || {};

  function canManage(){
    const r = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role;
    return ['super_admin','admin','hr_head','hr_officer'].includes(r);
  }

  async function render(container){
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Onboarding</h5>
          <div class="d-flex gap-2">
            ${canManage()? '<button class="btn btn-sm btn-primary" id="obNew"><i class="fas fa-plus"></i> New Onboarding</button>':''}
            <button class="btn btn-sm btn-outline-secondary" id="obReload"><i class="fas fa-rotate"></i> Reload</button>
            <div class="form-check form-check-inline ms-2">
              <input class="form-check-input" type="checkbox" id="obMineOnly" />
              <label class="form-check-label" for="obMineOnly">Show mine</label>
            </div>
          </div>
        </div>
        <div class="card-body">
          <div id="obCreateBox" class="mb-3" style="display:none;">
            <div class="border rounded p-3">
              <div class="row g-3">
                <div class="col-md-4">
                  <label class="form-label">Employee</label>
                  <select id="obEmp" class="form-select"></select>
                </div>
                <div class="col-md-4">
                  <label class="form-label">Title</label>
                  <input id="obTitle" class="form-control" placeholder="Onboarding title" />
                </div>
                <div class="col-12">
                  <label class="form-label">Tasks</label>
                  <div id="obTasks" class="vstack gap-2"></div>
                  <button class="btn btn-sm btn-outline-primary mt-2" id="obAddTask"><i class="fas fa-plus"></i> Add Task</button>
                </div>
                <div class="col-12 text-end">
                  <button class="btn btn-light" id="obCancel">Cancel</button>
                  <button class="btn btn-primary" id="obCreate">Create</button>
                </div>
              </div>
            </div>
          </div>

          <div class="table-responsive">
            <table class="table table-sm align-middle">
              <thead>
                <tr>
                  <th>Title</th>
                  <th>Employee</th>
                  <th>Progress</th>
                  <th>Status</th>
                  <th>Started</th>
                  <th style="width:120px;">Actions</th>
                </tr>
              </thead>
              <tbody id="obTable"><tr><td colspan="6">Loading...</td></tr></tbody>
            </table>
          </div>
        </div>
      </div>

      <div class="modal fade" id="obViewModal" tabindex="-1">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
          <div class="modal-content">
            <div class="modal-header"><h5 class="modal-title">Onboarding</h5><button class="btn-close" data-bs-dismiss="modal"></button></div>
            <div class="modal-body">
              <div id="obViewBody">Loading...</div>
            </div>
            <div class="modal-footer">
              <button class="btn btn-light" data-bs-dismiss="modal">Close</button>
            </div>
          </div>
        </div>
      </div>
    `;

    const tableBody = container.querySelector('#obTable');
    const mineChk = container.querySelector('#obMineOnly');

    // Controls
    container.querySelector('#obReload').addEventListener('click', ()=> loadList());
    const obNew = container.querySelector('#obNew');
    const obBox = container.querySelector('#obCreateBox');
    const obCancel = container.querySelector('#obCancel');
    const obEmp = container.querySelector('#obEmp');
    const obTitle = container.querySelector('#obTitle');
    const obTasksWrap = container.querySelector('#obTasks');
    const obAddTask = container.querySelector('#obAddTask');

    if (obNew){ obNew.addEventListener('click', async ()=>{ obBox.style.display = obBox.style.display==='none'? 'block':'none'; if (obBox.style.display==='block'){ await loadEmployees(obEmp); if (obTasksWrap.childElementCount===0) addTaskRow(); } }); }
    obCancel?.addEventListener('click', ()=>{ obBox.style.display='none'; obTasksWrap.innerHTML=''; obTitle.value=''; });
    obAddTask?.addEventListener('click', addTaskRow);

    container.querySelector('#obCreate')?.addEventListener('click', async ()=>{
      const employee_id = Number(obEmp.value||0); const title = obTitle.value.trim()||'Onboarding';
      const tasks = readTasks();
      if (!employee_id){ note('Select employee','warning'); return; }
      const res = await fetch('api/onboarding.php?action=create', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ employee_id, title, tasks }) }).then(r=>r.json()).catch(()=>null);
      if (res?.success){ note('Onboarding created','success'); obBox.style.display='none'; obTasksWrap.innerHTML=''; obTitle.value=''; await loadList(); }
      else { note(res?.message||'Create failed','error'); }
    });

    mineChk?.addEventListener('change', ()=> loadList());

    // Delegated actions
    container.addEventListener('click', async (e)=>{
      const a = e.target.closest('[data-action]'); if (!a) return;
      const act = a.getAttribute('data-action'); const id = a.getAttribute('data-id');
      if (act==='view'){ openView(id); }
    });

    function addTaskRow(){
      const row = document.createElement('div');
      row.className = 'row g-2 align-items-end';
      row.innerHTML = `
        <div class="col-md-6"><input class="form-control" data-role="title" placeholder="Task title (e.g., Submit ID documents)" /></div>
        <div class="col-md-3">
          <select class="form-select" data-role="role">
            <option value="employee">Employee</option>
            <option value="manager">Manager</option>
            <option value="hr">HR</option>
            <option value="it">IT</option>
          </select>
        </div>
        <div class="col-md-2 form-check">
          <input class="form-check-input" type="checkbox" data-role="esign" id="ch${Math.random().toString(36).slice(2)}" />
          <label class="form-check-label">e‑sign</label>
        </div>
        <div class="col-md-1 text-end"><button class="btn btn-sm btn-outline-danger" data-role="del"><i class="fas fa-trash"></i></button></div>
      `;
      row.querySelector('[data-role="del"]').addEventListener('click', ()=> row.remove());
      obTasksWrap.appendChild(row);
    }

    function readTasks(){
      const out = [];
      obTasksWrap.querySelectorAll('.row').forEach(r=>{
        const title = r.querySelector('[data-role="title"]').value.trim();
        const role = r.querySelector('[data-role="role"]').value;
        const es = r.querySelector('[data-role="esign"]').checked;
        if (title) out.push({ title, assignee_role: role, requires_esign: es?1:0 });
      });
      return out;
    }

    async function loadEmployees(sel){
      try{
        const res = await fetch('api/employees.php', { credentials:'same-origin' }); const js = await res.json();
        const opts = (js?.data||[]).map(e=> `<option value="${e.id}">${esc(e.full_name || (e.first_name||'')+' '+(e.last_name||''))} (${esc(e.employee_number||'')})</option>`);
        sel.innerHTML = opts.join('');
      }catch(_){ sel.innerHTML = ''; }
    }

    async function loadList(){
      tableBody.innerHTML = '<tr><td colspan="6">Loading...</td></tr>';
      const mine = mineChk?.checked ? 1 : 0;
      try{
        const res = await fetch('api/onboarding.php' + (mine? '?mine=1':''), { credentials:'same-origin' });
        const js = await res.json();
        if (!js?.success){ tableBody.innerHTML = '<tr><td colspan="6" class="text-danger">Failed to load</td></tr>'; return; }
        const list = js.data || [];
        tableBody.innerHTML = list.map(rowToHtml).join('') || '<tr><td colspan="6" class="text-muted">No onboarding</td></tr>';
      }catch(_){ tableBody.innerHTML = '<tr><td colspan="6" class="text-danger">Failed to load</td></tr>'; }
    }

    function rowToHtml(r){
      const prog = `${Number(r.tasks_done||0)}/${Number(r.tasks_total||0)}`;
      const badge = r.status==='completed'? 'bg-success' : (r.status==='open'? 'bg-primary':'bg-secondary');
      return `<tr>
        <td>${esc(r.title||'')}</td>
        <td>${esc(r.employee_name||'')} <span class="text-muted">${esc(r.employee_number||'')}</span></td>
        <td>${prog}</td>
        <td><span class="badge ${badge}">${esc(r.status||'')}</span></td>
        <td>${esc(r.started_at||'')}</td>
        <td><button class="btn btn-sm btn-outline-primary" data-action="view" data-id="${r.id}"><i class="fas fa-eye"></i></button></td>
      </tr>`;
    }

    async function openView(id){
      const wrap = document.getElementById('obViewBody');
      wrap.innerHTML = 'Loading...';
      try{
        const res = await fetch('api/onboarding.php?action=instance&id='+encodeURIComponent(id), { credentials:'same-origin' });
        const js = await res.json();
        if (!js?.success){ wrap.innerHTML = '<div class="text-danger">Failed to load</div>'; return; }
        const inst = js.data.instance; const tasks = js.data.tasks||[];
        wrap.innerHTML = `
          <div class="mb-2"><strong>${esc(inst.title||'')}</strong></div>
          <div class="text-muted mb-2">${esc(inst.employee_name||'')} • ${esc(inst.employee_number||'')}</div>
          <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead><tr><th>Task</th><th>Assignee</th><th>Status</th><th>e‑sign</th><th style="width:160px;">Actions</th></tr></thead>
            <tbody>
              ${tasks.map(t=> taskRow(inst.id, t)).join('') || '<tr><td colspan="5" class="text-muted">No tasks</td></tr>'}
            </tbody>
          </table>
          </div>
        `;
        // wire
        wrap.querySelectorAll('[data-action]').forEach(btn=>{
          btn.addEventListener('click', async ()=>{
            const act = btn.getAttribute('data-action'); const tid = btn.getAttribute('data-id');
            if (act==='done'){ await updateTask(inst.id, tid, { status:'done' }); await openView(id); }
            if (act==='esign'){
              const nm = prompt('Type your full name to sign'); if (!nm) return;
              await updateTask(inst.id, tid, { esign_name: nm, status:'done' }); await openView(id);
            }
          });
        });
        const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById('obViewModal')); modal.show();
      }catch(_){ wrap.innerHTML = '<div class="text-danger">Failed to load</div>'; }
    }

    function taskRow(iid, t){
      const ass = t.assignee_role?.toUpperCase?.() || '-';
      const status = t.status==='done'? '<span class="badge bg-success">done</span>':'<span class="badge bg-secondary">pending</span>';
      const es = t.requires_esign? (t.esign_name? `${esc(t.esign_name)} <small class="text-muted">(${esc(t.esign_at||'')})</small>`:'Required') : '-';
      const buttons = t.status==='done'? '' : `
        <div class="btn-group btn-group-sm">
          <button class="btn btn-outline-success" data-action="done" data-id="${t.id}"><i class="fas fa-check"></i></button>
          ${t.requires_esign? `<button class="btn btn-outline-primary" data-action="esign" data-id="${t.id}"><i class="fas fa-signature"></i></button>`:''}
        </div>`;
      return `<tr>
        <td>${esc(t.title||'')}</td>
        <td>${ass}</td>
        <td>${status}</td>
        <td>${es}</td>
        <td>${buttons}</td>
      </tr>`;
    }

    async function updateTask(instance_id, task_id, patch){
      const payload = Object.assign({ instance_id:Number(instance_id), task_id:Number(task_id) }, patch);
      const res = await fetch('api/onboarding.php?action=task_update', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (!res?.success) note(res?.message||'Update failed','error'); else note('Updated','success');
    }

    function note(msg, type){ window.auth?.showNotification?.(msg, type||'info'); }
    function esc(s){ return String(s ?? '').replace(/[&<>"']/g, c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c])); }

    // init
    await loadList();
  }

  window.pages.onboarding = { render };
})();
