// Loans UI Module
(function(){
  window.pages = window.pages || {};

  function role(){ const u = window.auth?.currentUser; return (u?.role_slug) || (u?.role); }
  function isHR(){ return ['super_admin','admin','hr_head','hr_officer'].includes(role()); }
  function isManagerOrEmp(){ return ['manager','employee'].includes(role()); }
  function fmt(n){ const v = parseFloat(n || 0); return isFinite(v) ? v.toFixed(2) : '0.00'; }
  const escapeHtml = window.escapeHtml || function(s){
    return String(s==null? '': s)
      .replace(/&/g,'&amp;')
      .replace(/</g,'&lt;')
      .replace(/>/g,'&gt;')
      .replace(/"/g,'&quot;')
      .replace(/'/g,'&#39;');
  };
  function roleLevel(r){ const m={employee:1,manager:2,hr_officer:3,hr_head:4,admin:5,super_admin:6}; return m[r]||0; }
  function requiredApproverRole(applicantRole){
    switch(applicantRole){
      case 'hr_officer': return 'hr_head';
      case 'hr_head': return 'admin';
      case 'admin': return 'super_admin';
      default: return 'hr_officer';
    }
  }

  async function render(container){
    const hr = isHR();
    const me = isManagerOrEmp();
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="mb-0">Loans</h5>
          <small class="text-muted">Staff loan applications and repayments</small>
        </div>
        <div class="card-body">
          ${(me || hr) ? `
          <div class="mb-4">
            <h6>Apply for Loan</h6>
            <div class="row g-3">
              <div class="col-md-3">
                <label class="form-label">Loan Type</label>
                <select id="loanTypeId" class="form-select"></select>
              </div>
              <div class="col-md-3">
                <label class="form-label">Principal</label>
                <input type="number" step="0.01" id="loanPrincipal" class="form-control" placeholder="5000" />
              </div>
              <div class="col-md-2">
                <label class="form-label">Term (months)</label>
                <input type="number" id="loanMonths" class="form-control" placeholder="12" />
              </div>
              <div class="col-md-2">
                <label class="form-label">Interest %</label>
                <input type="number" step="0.01" id="loanRate" class="form-control" placeholder="10" />
              </div>
              <div class="col-md-2">
                <label class="form-label">Interest Type</label>
                <select id="loanInterestType" class="form-select">
                  <option value="flat" selected>Flat</option>
                  <option value="reducing">Reducing (coming soon)</option>
                </select>
              </div>
              <div class="col-md-3">
                <label class="form-label">Start Date</label>
                <input type="date" id="loanStart" class="form-control" />
              </div>
            </div>
            <div class="mt-3">
              <button class="btn btn-primary" id="btnApplyLoan">Submit Application</button>
            </div>
          </div>` : ''}

          <div class="d-flex align-items-center mb-2">
            <h6 class="mb-0">${hr ? 'All Company Loans' : 'My Loans'}</h6>
            <div class="ms-auto d-flex gap-2 align-items-center">
              ${hr ? '<button class="btn btn-sm btn-primary" id="btnApplyForEmployee"><i class="fa fa-user-plus"></i> Apply For Employee</button>' : ''}
              <button class="btn btn-sm btn-outline-secondary" id="btnRefreshLoans"><i class="fa fa-rotate"></i> Refresh</button>
            </div>
          </div>
          <div class="table-responsive">
            <table class="table table-sm align-middle" id="tblLoans">
              <thead>
                <tr>
                  ${hr ? '<th>Employee</th>' : ''}
                  <th>Type</th>
                  <th>Principal</th>
                  <th>Monthly</th>
                  <th>Balance</th>
                  <th>Start</th>
                  <th>Status</th>
                  <th>Actions</th>
                </tr>
              </thead>
              <tbody></tbody>
            </table>
          </div>
          ${hr ? `
          <hr/>
          <div class="mt-4">
            <div class="d-flex align-items-center mb-2">
              <h6 class="mb-0">Loan Types</h6>
              <div class="ms-auto d-flex gap-2 align-items-center">
                <button class="btn btn-sm btn-primary" id="btnAddLoanType"><i class="fa fa-plus"></i> Add Type</button>
                <button class="btn btn-sm btn-outline-secondary" id="btnRefreshTypes"><i class="fa fa-rotate"></i> Refresh</button>
              </div>
            </div>
            <div class="table-responsive">
              <table class="table table-sm align-middle" id="tblLoanTypes">
                <thead>
                  <tr>
                    <th>Name</th>
                    <th>Code</th>
                    <th>Term (months)</th>
                    <th>Rate %</th>
                    <th>Interest Type</th>
                    <th>Status</th>
                    <th>Actions</th>
                  </tr>
                </thead>
                <tbody></tbody>
              </table>
            </div>
          </div>
          ` : ''}

          ${hr ? `
          <div id="hrApplyPanel" class="mt-4 card" style="display:none;">
            <div class="card-header d-flex align-items-center">
              <h6 class="mb-0">Apply for Employee</h6>
              <div class="ms-auto">
                <button class="btn btn-sm btn-outline-secondary" id="btnCloseHrApply">Close</button>
              </div>
            </div>
            <div class="card-body">
              <div class="row g-3 align-items-end">
                <div class="col-md-5">
                  <label class="form-label">Employee</label>
                  <input type="text" id="hrEmpSearch" class="form-control" placeholder="Search name, number, or email" />
                  <select id="hrEmpSelect" class="form-select mt-2"></select>
                </div>
                <div class="col-md-3">
                  <label class="form-label">Loan Type</label>
                  <select id="hrLoanTypeId" class="form-select"></select>
                </div>
                <div class="col-md-2">
                  <label class="form-label">Principal</label>
                  <input type="number" step="0.01" id="hrLoanPrincipal" class="form-control" placeholder="5000" />
                </div>
                <div class="col-md-2">
                  <label class="form-label">Term (months)</label>
                  <input type="number" id="hrLoanMonths" class="form-control" placeholder="12" />
                </div>
                <div class="col-md-2">
                  <label class="form-label">Interest %</label>
                  <input type="number" step="0.01" id="hrLoanRate" class="form-control" placeholder="10" />
                </div>
                <div class="col-md-2">
                  <label class="form-label">Interest Type</label>
                  <select id="hrLoanInterestType" class="form-select">
                    <option value="flat" selected>Flat</option>
                    <option value="reducing">Reducing (coming soon)</option>
                  </select>
                </div>
                <div class="col-md-3">
                  <label class="form-label">Start Date</label>
                  <input type="date" id="hrLoanStart" class="form-control" />
                </div>
                <div class="col-md-12 mt-2">
                  <button class="btn btn-success" id="btnSubmitHrApply">Submit Application</button>
                </div>
              </div>
            </div>
          </div>
          ` : ''}

          <div id="loanDetail" class="mt-4" style="display:none">
            <h6 class="mb-2">Loan Details</h6>
            <div id="loanDetailBody"></div>
          </div>
        </div>
      </div>
    `;

    // Handlers
    const refreshBtn = container.querySelector('#btnRefreshLoans'); if (refreshBtn) refreshBtn.addEventListener('click', loadList);
    const applyBtn = container.querySelector('#btnApplyLoan'); if (applyBtn) applyBtn.addEventListener('click', applyLoan);
    const typeSelect = container.querySelector('#loanTypeId');
    const itypeSelect = container.querySelector('#loanInterestType');
    if (typeSelect){
      typeSelect.addEventListener('change', ()=>{
        const t = (window.__loanTypes||[]).find(x=> String(x.id)===String(typeSelect.value));
        if (t){
          const m = container.querySelector('#loanMonths'); if (m) m.value = t.default_term_months||'';
          const r = container.querySelector('#loanRate'); if (r) r.value = t.default_interest_rate||'';
          if (itypeSelect) itypeSelect.value = t.interest_type||'flat';
        }
      });
    }
    // Type management handlers will be wired after functions are defined below

    // ===== HR Apply for Employee helpers =====
    function debounce(fn, ms){ let to; return (...args)=>{ clearTimeout(to); to=setTimeout(()=>fn.apply(null,args), ms); }; }
    function openHrApplyPanel(){ const p = container.querySelector('#hrApplyPanel'); if (p){ p.style.display=''; p.scrollIntoView({behavior:'smooth'}); } loadEmployeeOptions(''); }
    function closeHrApplyPanel(){ const p = container.querySelector('#hrApplyPanel'); if (p){ p.style.display='none'; } }
    async function populateHrTypes(){
      const sel = container.querySelector('#hrLoanTypeId'); if (!sel) return;
      try {
        const res = await fetch('api/loans.php?action=list_types', { credentials:'same-origin' });
        const json = await res.json();
        const rows = (json?.success && Array.isArray(json.data)) ? json.data.filter(t=>t.status==='active') : [];
        window.__loanTypes = rows; // reuse
        sel.innerHTML = '<option value="">-- Select Type --</option>' + rows.map(t=>`<option value="${t.id}">${escapeHtml(t.name)}</option>`).join('');
      } catch(e){ sel.innerHTML = '<option value="">-- Select Type --</option>'; }
    }
    async function loadEmployeeOptions(search){
      const sel = container.querySelector('#hrEmpSelect'); if (!sel) return;
      try {
        const url = 'api/employees.php'+(search? ('?search='+encodeURIComponent(search)) : '');
        const res = await fetch(url, { credentials:'same-origin' });
        const json = await res.json();
        const rows = (json?.success && Array.isArray(json.data)) ? json.data : [];
        sel.innerHTML = '<option value="">-- Select Employee --</option>' + rows.map(e=>`<option value="${e.id}">${escapeHtml((e.full_name||'').trim()|| (e.first_name+' '+e.last_name))} - ${escapeHtml(e.employee_number||'')}</option>`).join('');
      } catch(e){ sel.innerHTML = '<option value="">-- Select Employee --</option>'; }
    }
    async function applyLoanForEmployee(){
      const employee_id = parseInt(container.querySelector('#hrEmpSelect')?.value || '0', 10);
      const principal = parseFloat(container.querySelector('#hrLoanPrincipal')?.value || '0');
      const term_months = parseInt(container.querySelector('#hrLoanMonths')?.value || '0', 10);
      const interest_rate = parseFloat(container.querySelector('#hrLoanRate')?.value || '0');
      const interest_type = container.querySelector('#hrLoanInterestType')?.value || 'flat';
      const loan_type_id = parseInt(container.querySelector('#hrLoanTypeId')?.value || '0', 10) || null;
      const start_date = container.querySelector('#hrLoanStart')?.value || '';
      if (!(employee_id>0)) { window.auth?.showNotification?.('Select an employee','warning'); return; }
      if (!(principal>0) || !(term_months>0)) { window.auth?.showNotification?.('Enter principal and term','warning'); return; }
      try {
        const res = await fetch('api/loans.php?action=apply', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ employee_id, principal, term_months, interest_rate, interest_type, loan_type_id, start_date }) });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Loan application submitted','success'); closeHrApplyPanel(); loadList(); }
        else window.auth?.showNotification?.(json?.message || 'Failed to apply','error');
      } catch(e){ window.auth?.showNotification?.('Failed to apply','error'); }
    }

    // Wire HR Apply handlers
    const btnApplyForEmp = container.querySelector('#btnApplyForEmployee'); if (btnApplyForEmp) btnApplyForEmp.addEventListener('click', openHrApplyPanel);
    const btnCloseHrApply = container.querySelector('#btnCloseHrApply'); if (btnCloseHrApply) btnCloseHrApply.addEventListener('click', closeHrApplyPanel);
    const btnSubmitHrApply = container.querySelector('#btnSubmitHrApply'); if (btnSubmitHrApply) btnSubmitHrApply.addEventListener('click', applyLoanForEmployee);
    const hrTypeSelect = container.querySelector('#hrLoanTypeId');
    const hrItypeSelect = container.querySelector('#hrLoanInterestType');
    if (hrTypeSelect){
      hrTypeSelect.addEventListener('change', ()=>{
        const t = (window.__loanTypes||[]).find(x=> String(x.id)===String(hrTypeSelect.value));
        if (t){
          const m = container.querySelector('#hrLoanMonths'); if (m) m.value = t.default_term_months||'';
          const r = container.querySelector('#hrLoanRate'); if (r) r.value = t.default_interest_rate||'';
          if (hrItypeSelect) hrItypeSelect.value = t.interest_type||'flat';
        }
      });
    }
    const hrEmpSearch = container.querySelector('#hrEmpSearch'); if (hrEmpSearch) hrEmpSearch.addEventListener('input', debounce(async ()=>{ await loadEmployeeOptions(hrEmpSearch.value||''); }, 400));

    // ===== Loan Types helpers (inside render to access container/hr) =====
    async function loadTypes(){
      if (!hr) return;
      try {
        const res = await fetch('api/loans.php?action=list_types', { credentials:'same-origin' });
        const json = await res.json();
        if (!json?.success) {
          window.auth?.showNotification?.(json?.message || 'Failed to load loan types','warning');
        }
        const rows = (json?.success && Array.isArray(json.data)) ? json.data : [];
        window.__loanTypes = rows;
        const tb = container.querySelector('#tblLoanTypes tbody'); if (!tb) return;
        tb.innerHTML = '';
        for (const t of rows){
          const tr = document.createElement('tr');
          tr.innerHTML = `
            <td>${escapeHtml(t.name||'')}</td>
            <td>${escapeHtml(t.code||'')}</td>
            <td>${t.default_term_months||''}</td>
            <td>${fmt(t.default_interest_rate||0)}</td>
            <td>${escapeHtml(t.interest_type||'')}</td>
            <td><span class="badge ${t.status==='active'?'bg-success':'bg-secondary'}">${String(t.status||'').toUpperCase()}</span></td>
            <td class="d-flex gap-2">
              <button class="btn btn-sm btn-outline-primary" data-act="edit" data-id="${t.id}">Edit</button>
              <button class="btn btn-sm btn-outline-danger" data-act="deactivate" data-id="${t.id}">Deactivate</button>
            </td>`;
          tb.appendChild(tr);
        }
        tb.querySelectorAll('button[data-act]').forEach(btn=>{
          btn.addEventListener('click', async ()=>{
            const id = parseInt(btn.getAttribute('data-id'), 10);
            const act = btn.getAttribute('data-act');
            if (act==='edit') editTypePrompt(id);
            else if (act==='deactivate') deactivateType(id);
          });
        });
      } catch(e){ window.auth?.showNotification?.('Failed to load loan types','error'); }
    }

    async function populateApplyTypes(){
      const sel = container.querySelector('#loanTypeId'); if (!sel) return;
      try {
        const res = await fetch('api/loans.php?action=list_types', { credentials:'same-origin' });
        const json = await res.json();
        const rows = (json?.success && Array.isArray(json.data)) ? json.data.filter(t=>t.status==='active') : [];
        window.__loanTypes = rows;
        sel.innerHTML = '<option value="">-- Select Type --</option>' + rows.map(t=>`<option value="${t.id}">${escapeHtml(t.name)}</option>`).join('');
      } catch(e){ sel.innerHTML = '<option value="">-- Select Type --</option>'; }
    }

    async function addTypePrompt(){
      const name = prompt('Type name'); if (!name) return;
      const code = prompt('Code (optional)') || '';
      const term = parseInt(prompt('Default term (months)')||'0',10);
      const rate = parseFloat(prompt('Default interest rate %')||'0');
      const itype = (prompt('Interest type (flat/reducing)')||'flat').toLowerCase()==='reducing'?'reducing':'flat';
      const body = { name, code, default_term_months: term, default_interest_rate: rate, interest_type: itype, status: 'active' };
      try {
        const res = await fetch('api/loans.php?action=create_type', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(body) });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Type created','success'); await loadTypes(); await populateApplyTypes(); }
        else window.auth?.showNotification?.(json?.message||'Failed','error');
      } catch(e){ window.auth?.showNotification?.('Failed','error'); }
    }

    async function editTypePrompt(id){
      try {
        const res = await fetch('api/loans.php?action=get_type&id='+encodeURIComponent(id), { credentials:'same-origin' });
        const json = await res.json(); if (!(json?.success && json.data)) return;
        const t = json.data;
        const name = prompt('Type name', t.name||''); if (!name) return;
        const code = prompt('Code (optional)', t.code||'') || '';
        const term = parseInt(prompt('Default term (months)', String(t.default_term_months||''))||'0',10);
        const rate = parseFloat(prompt('Default interest rate %', String(t.default_interest_rate||''))||'0');
        const itype = (prompt('Interest type (flat/reducing)', t.interest_type||'flat')||'flat').toLowerCase()==='reducing'?'reducing':'flat';
        const status = (prompt('Status (active/inactive)', t.status||'active')||'active').toLowerCase()==='inactive'?'inactive':'active';
        const body = { name, code, default_term_months: term, default_interest_rate: rate, interest_type: itype, status };
        const up = await fetch('api/loans.php?action=update_type&id='+encodeURIComponent(id), { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(body) });
        const jr = await up.json();
        if (jr?.success){ window.auth?.showNotification?.('Type updated','success'); await loadTypes(); await populateApplyTypes(); }
        else window.auth?.showNotification?.(jr?.message||'Failed','error');
      } catch(e){ window.auth?.showNotification?.('Failed','error'); }
    }

    async function deactivateType(id){
      if (!confirm('Deactivate this type?')) return;
      try {
        const res = await fetch('api/loans.php?action=delete_type&id='+encodeURIComponent(id), { method:'POST', credentials:'same-origin' });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Type deactivated','info'); await loadTypes(); await populateApplyTypes(); }
        else window.auth?.showNotification?.(json?.message||'Failed','error');
      } catch(e){ window.auth?.showNotification?.('Failed','error'); }
    }

    // Now wire handlers for type management
    const btnAddType = container.querySelector('#btnAddLoanType'); if (btnAddType) btnAddType.addEventListener('click', addTypePrompt);
    const btnRefreshTypes = container.querySelector('#btnRefreshTypes'); if (btnRefreshTypes) btnRefreshTypes.addEventListener('click', loadTypes);

    await Promise.all([loadList(), loadTypes(), populateApplyTypes()]);
    if (hr) await populateHrTypes();

    async function loadList(){
      try {
        const res = await fetch('api/loans.php?action=list', { credentials:'same-origin' });
        const rid = res.headers?.get?.('X-Request-ID');
        const text = await res.text();
        let json;
        try {
          json = text ? JSON.parse(text) : null;
        } catch (parseErr) {
          console.error('Loans list not JSON', { rid, status: res.status, body: text?.slice(0,1000) });
          window.auth?.showNotification?.('Failed to load loans','error');
          return;
        }
        if (!json?.success) {
          if (json?.message) window.auth?.showNotification?.(json.message, 'warning');
        }
        const rows = (json?.success && Array.isArray(json.data)) ? json.data : [];
        renderList(rows);
      } catch (e) {
        window.auth?.showNotification?.('Failed to load loans','error');
      }
    }

    function renderList(rows){
      const tb = container.querySelector('#tblLoans tbody');
      tb.innerHTML = '';
      for (const r of rows){
        const cur = window.auth?.currentUser || {};
        const curRole = role();
        const needRole = requiredApproverRole(String(r.applicant_role||'employee'));
        const canApprove = isHR() && roleLevel(curRole) >= roleLevel(needRole) && cur.id && r.applicant_user_id !== cur.id && String(r.status||'')==='pending';
        const tr = document.createElement('tr');
        tr.innerHTML = `
          ${hr ? `<td>${r.employee_name || ''}</td>` : ''}
          <td>${r.loan_type_name || ''}</td>
          <td>${fmt(r.principal)}</td>
          <td>${fmt(r.monthly_installment)}</td>
          <td>${fmt(r.balance_outstanding)}</td>
          <td>${(r.start_date||'').slice(0,10)}</td>
          <td><span class="badge ${badgeClass(r.status)}">${String(r.status||'').toUpperCase()}</span></td>
          <td class="d-flex gap-2">
            <button class="btn btn-sm btn-outline-primary" data-act="view" data-id="${r.id}">View</button>
            ${canApprove ? `<button class="btn btn-sm btn-success" data-act="approve" data-id="${r.id}">Approve</button>` : ''}
            ${canApprove ? `<button class="btn btn-sm btn-danger" data-act="reject" data-id="${r.id}">Reject</button>` : ''}
            ${hr && (r.status==='approved' || r.status==='active') ? `<button class="btn btn-sm btn-outline-secondary" data-act="service" data-id="${r.id}">Service Month</button>` : ''}
          </td>
        `;
        tb.appendChild(tr);
      }
      // Bind row actions
      tb.querySelectorAll('button[data-act]').forEach(btn=>{
        btn.addEventListener('click', async (ev)=>{
          const act = btn.getAttribute('data-act');
          const id = parseInt(btn.getAttribute('data-id'),10);
          if (act==='view') viewLoan(id);
          else if (act==='approve') approveLoan(id);
          else if (act==='reject') rejectLoan(id);
          else if (act==='service') serviceLoanMonth(id);
        });
      });
    }

    function badgeClass(st){
      switch(String(st||'').toLowerCase()){
        case 'pending': return 'bg-warning text-dark';
        case 'approved':
        case 'active': return 'bg-info text-dark';
        case 'completed': return 'bg-success';
        case 'rejected':
        case 'cancelled': return 'bg-danger';
        default: return 'bg-secondary';
      }
    }

    async function applyLoan(){
      const principal = parseFloat(container.querySelector('#loanPrincipal')?.value || '0');
      const term_months = parseInt(container.querySelector('#loanMonths')?.value || '0', 10);
      const interest_rate = parseFloat(container.querySelector('#loanRate')?.value || '0');
      const interest_type = container.querySelector('#loanInterestType')?.value || 'flat';
      const loan_type_id = parseInt(container.querySelector('#loanTypeId')?.value || '0', 10) || null;
      const start_date = container.querySelector('#loanStart')?.value || '';
      if (!(principal>0) || !(term_months>0)) { window.auth?.showNotification?.('Enter principal and term','warning'); return; }
      try {
        const res = await fetch('api/loans.php?action=apply', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ principal, term_months, interest_rate, interest_type, loan_type_id, start_date }) });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Loan application submitted','success'); loadList(); }
        else window.auth?.showNotification?.(json?.message || 'Failed to apply','error');
      } catch(e){ window.auth?.showNotification?.('Failed to apply','error'); }
    }

    async function approveLoan(id){
      if (!confirm('Approve this loan?')) return;
      try {
        const res = await fetch('api/loans.php?action=approve&id='+encodeURIComponent(id), { method:'POST', credentials:'same-origin' });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Loan approved','success'); loadList(); }
        else window.auth?.showNotification?.(json?.message || 'Approval failed','error');
      } catch(e){ window.auth?.showNotification?.('Approval failed','error'); }
    }

    async function rejectLoan(id){
      if (!confirm('Reject this loan?')) return;
      try {
        const res = await fetch('api/loans.php?action=reject&id='+encodeURIComponent(id), { method:'POST', credentials:'same-origin' });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Loan rejected','info'); loadList(); }
        else window.auth?.showNotification?.(json?.message || 'Rejection failed','error');
      } catch(e){ window.auth?.showNotification?.('Rejection failed','error'); }
    }

    async function serviceLoanMonth(id){
      const month = prompt('Enter month to service (YYYY-MM)');
      if (!month || !/^\d{4}-\d{2}$/.test(month)) { window.auth?.showNotification?.('Invalid month','warning'); return; }
      try {
        const res = await fetch('api/loans.php?action=service_month', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ loan_id:id, month }) });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Month serviced','success'); loadList(); }
        else window.auth?.showNotification?.(json?.message || 'Service failed','error');
      } catch(e){ window.auth?.showNotification?.('Service failed','error'); }
    }

    async function viewLoan(id){
      try {
        const res = await fetch('api/loans.php?action=show&id='+encodeURIComponent(id), { credentials:'same-origin' });
        const json = await res.json();
        if (!(json?.success && json.data)) { window.auth?.showNotification?.('Failed to load loan','error'); return; }
        const l = json.data;
        const detail = container.querySelector('#loanDetail');
        const body = container.querySelector('#loanDetailBody');
        const sched = Array.isArray(l.schedule) ? l.schedule : [];
        body.innerHTML = `
          <div class="row g-3">
            <div class="col-md-3"><strong>Principal:</strong> ${fmt(l.principal)}</div>
            <div class="col-md-3"><strong>Total Payable:</strong> ${fmt(l.total_payable)}</div>
            <div class="col-md-3"><strong>Monthly:</strong> ${fmt(l.monthly_installment)}</div>
            <div class="col-md-3"><strong>Balance:</strong> ${fmt(l.balance_outstanding)}</div>
            <div class="col-md-3"><strong>Start Date:</strong> ${(l.start_date||'').slice(0,10)}</div>
            <div class="col-md-3"><strong>Status:</strong> ${String(l.status||'').toUpperCase()}</div>
          </div>
          <div class="mt-3 table-responsive">
            <table class="table table-sm">
              <thead><tr><th>Due Date</th><th>Amount Due</th><th>Amount Paid</th><th>Status</th></tr></thead>
              <tbody>
                ${sched.map(s=>`<tr>
                  <td>${(s.due_date||'').slice(0,10)}</td>
                  <td>${fmt(s.amount_due)}</td>
                  <td>${s.amount_paid!=null?fmt(s.amount_paid):''}</td>
                  <td>${String(s.status||'').toUpperCase()}</td>
                </tr>`).join('')}
              </tbody>
            </table>
          </div>
        `;
        detail.style.display = '';
        detail.scrollIntoView({ behavior:'smooth', block:'start' });
      } catch (e){
        window.auth?.showNotification?.('Failed to load loan','error');
      }
    }
  }

  window.pages.loans = { render };
})();
