// Leave page module
(function () {
  window.pages = window.pages || {};

  function role() {
    const r = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role || 'employee';
    return r;
  }

  function canApproveLeave() {
    try { return window.auth?.canApprove('leave'); } catch(_) { return false; }
  }

  function canManageAllocations() {
    const r = role();
    return ['super_admin','admin','hr_head','hr_officer'].includes(r);
  }

  async function render(container) {
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex align-items-center justify-content-between">
          <h5 class="card-title mb-0">Leave Management</h5>
        </div>
        <div class="card-body">
          <ul class="nav nav-tabs" role="tablist">
            <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#lv-balance" type="button" role="tab">My Balance</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#lv-request" type="button" role="tab">Request Leave</button></li>
            ${canApproveLeave() ? '<li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#lv-approvals" type="button" role="tab">Pending Approvals</button></li>' : ''}
            ${canManageAllocations() ? '<li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#lv-alloc" type="button" role="tab">Allocations</button></li>' : ''}
          </ul>
          <div class="tab-content mt-3">
            <div class="tab-pane fade show active" id="lv-balance" role="tabpanel">
              <div id="balanceWrap" class="row g-3"></div>
            </div>
            <div class="tab-pane fade" id="lv-request" role="tabpanel">
              <form id="leaveRequestForm" class="row g-3">
                <div class="col-md-4">
                  <label class="form-label">Leave Type</label>
                  <select class="form-select" name="leave_type_id" required></select>
                </div>
                <div class="col-md-3">
                  <label class="form-label">Start Date</label>
                  <input type="date" class="form-control" name="start_date" required />
                </div>
                <div class="col-md-3">
                  <label class="form-label">End Date</label>
                  <input type="date" class="form-control" name="end_date" required />
                </div>
                <div class="col-md-4">
                  <label class="form-label">Reliever (optional)</label>
                  <select class="form-select" name="reliever_employee_id">
                    <option value="">-- Select Reliever --</option>
                  </select>
                </div>
                <div class="col-12">
                  <label class="form-label">Reason</label>
                  <textarea class="form-control" name="reason" rows="2" required></textarea>
                </div>
                <div class="col-12 text-end">
                  <button type="submit" class="btn btn-primary">Submit Request</button>
                </div>
              </form>
              <hr class="my-4" />
              <h6>My Requests</h6>
              <div id="myRequests"></div>
            </div>
            ${canApproveLeave() ? `
            <div class="tab-pane fade" id="lv-approvals" role="tabpanel">
              <div id="approvalFilters" class="row g-2 align-items-end mb-3">
                <div class="col-md-3">
                  <label class="form-label">Search (Employee)</label>
                  <input type="text" class="form-control" id="apprFilterQ" placeholder="Name contains..." />
                </div>
                <div class="col-md-2">
                  <label class="form-label">From</label>
                  <input type="date" class="form-control" id="apprFilterFrom" />
                </div>
                <div class="col-md-2">
                  <label class="form-label">To</label>
                  <input type="date" class="form-control" id="apprFilterTo" />
                </div>
                <div class="col-md-3">
                  <label class="form-label">Leave Type</label>
                  <select class="form-select" id="apprFilterType"><option value="">All</option></select>
                </div>
                <div class="col-md-2 text-end">
                  <button class="btn btn-primary w-100" id="apprFilterApply">Apply</button>
                </div>
                <div class="col-md-2 text-end">
                  <button class="btn btn-outline-secondary w-100" id="apprExportCsv">Export CSV</button>
                </div>
              </div>
              <div id="approvalList"></div>
            </div>` : ''}
            ${canManageAllocations() ? `
            <div class="tab-pane fade" id="lv-alloc" role="tabpanel">
              <div class="alert alert-info mb-3">Manage grade-based annual leave allocations per type.</div>
              <div id="allocPanel"></div>
              <div class="text-end mt-3">
                <button id="saveAllocBtn" class="btn btn-success btn-sm"><i class="fas fa-save"></i> Save Allocations</button>
              </div>
            </div>` : ''}
          </div>
        </div>
      </div>`;

    // Load initial data
    await Promise.all([
      loadBalance(container),
      setupRequestForm(container),
      loadMyRequests(container),
      canApproveLeave() ? loadPendingApprovals(container) : Promise.resolve(),
      canManageAllocations() ? loadAllocations(container) : Promise.resolve()
    ]);
  }

  async function loadBalance(container) {
    const wrap = container.querySelector('#balanceWrap'); if (!wrap) return;
    wrap.innerHTML = '<div class="text-muted">Loading...</div>';
    try {
      const res = await fetch('api/leave.php?action=balance', { credentials: 'same-origin' });
      const data = await res.json();
      if (!data?.success) throw new Error();
      const arr = Array.isArray(data.data) ? data.data : [];
      wrap.innerHTML = arr.map(b => {
        const annual = Number(b.days_per_year);
        const used = Number(b.used_days || 0);
        const pending = Number(b.pending_days || 0);
        const remaining = (b.remaining_days != null) ? Number(b.remaining_days) : (annual - used);
        const available = (b.available_days != null) ? Number(b.available_days) : (annual - used - pending);
        return `
        <div class="col-md-3">
          <div class="border rounded p-3 h-100">
            <div class="fw-bold">${escapeHtml(b.name)} (${escapeHtml(b.code)})</div>
            <div class="small text-muted">Annual: ${annual}</div>
            <div>Used: <span class="text-danger">${used}</span></div>
            <div>Pending: <span class="text-warning">${pending}</span></div>
            <div>Available: <span class="text-primary fw-bold">${Math.max(0, available)}</span></div>
            <div>Remaining: <span class="text-success fw-bold">${Math.max(0, remaining)}</span></div>
          </div>
        </div>`;
      }).join('') || '<div class="text-muted">No leave types defined.</div>';
    } catch(_) {
      wrap.innerHTML = '<div class="text-danger">Failed to load balance.</div>';
    }
  }

  async function setupRequestForm(container) {
    const form = container.querySelector('#leaveRequestForm'); if (!form) return;
    // populate leave types
    try {
      const res = await fetch('api/leave.php?action=types', { credentials: 'same-origin' });
      const data = await res.json();
      const sel = form.querySelector('select[name="leave_type_id"]');
      if (sel) sel.innerHTML = (data?.success ? data.data : []).map(t => `<option value="${t.id}">${escapeHtml(t.name)}</option>`).join('');
    } catch(_) {}

    // populate reliever employees (company-wide)
    try {
      const relSel = form.querySelector('select[name="reliever_employee_id"]');
      if (relSel) {
        const resE = await fetch('api/employees.php', { credentials: 'same-origin' });
        const js = await resE.json();
        const rows = (js?.success && Array.isArray(js.data)) ? js.data : [];
        relSel.innerHTML = '<option value="">-- Select Reliever --</option>' + rows.map(e => {
          const name = (e.full_name||'').trim() || `${e.first_name||''} ${e.last_name||''}`.trim();
          const code = e.employee_number || '';
          return `<option value="${e.id}">${escapeHtml(name)}${code? ' - '+escapeHtml(code):''}</option>`;
        }).join('');
      }
    } catch(_) {}

    // Add working days preview hint
    const startInp = form.querySelector('input[name="start_date"]');
    const endInp = form.querySelector('input[name="end_date"]');
    const endCol = endInp?.closest('.col-md-3') || endInp?.parentElement || form;
    if (endCol && !endCol.querySelector('#lvDaysHint')) {
      const hint = document.createElement('div');
      hint.id = 'lvDaysHint';
      hint.className = 'form-text';
      hint.textContent = 'Working days will be calculated excluding weekends and company holidays.';
      endCol.appendChild(hint);
    }

    async function updateDaysHint(){
      const hint = form.querySelector('#lvDaysHint'); if (!hint || !startInp || !endInp) return;
      const s = startInp.value; const e = endInp.value;
      if (!s || !e) { hint.textContent = 'Working days will be calculated excluding weekends and company holidays.'; return; }
      try {
        const from = s; const to = e;
        const res = await fetch(`api/holidays.php?action=range&from=${encodeURIComponent(from)}&to=${encodeURIComponent(to)}`, { credentials:'same-origin' });
        const js = await res.json();
        const hols = (js?.success && Array.isArray(js.data)) ? new Set(js.data.map(r=>r.holiday_date)) : new Set();
        const startTs = Date.parse(s); const endTs = Date.parse(e);
        if (isNaN(startTs) || isNaN(endTs)) { hint.textContent = ''; return; }
        const a = Math.min(startTs, endTs); const b = Math.max(startTs, endTs);
        let count = 0;
        for (let ts=a; ts<=b; ts+=86400000){
          const d = new Date(ts);
          const dow = d.getDay(); // 0 Sun .. 6 Sat
          if (dow === 0 || dow === 6) continue;
          const key = `${d.getFullYear()}-${String(d.getMonth()+1).padStart(2,'0')}-${String(d.getDate()).padStart(2,'0')}`;
          if (hols.has(key)) continue;
          count++;
        }
        hint.textContent = `Working days: ${count}`;
      } catch(_) { /* ignore */ }
    }

    startInp?.addEventListener('change', updateDaysHint);
    endInp?.addEventListener('change', updateDaysHint);
    form.querySelector('select[name="leave_type_id"]').addEventListener('change', updateDaysHint);

    form.addEventListener('submit', async (e) => {
      e.preventDefault();
      const btn = form.querySelector('button[type="submit"]');
      if (btn) { btn.disabled = true; btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Submitting...'; }
      try {
        const payload = Object.fromEntries(new FormData(form).entries());
        const res = await fetch('api/leave.php?action=request', {
          method: 'POST', headers: { 'Content-Type': 'application/json' }, credentials: 'same-origin',
          body: JSON.stringify(payload)
        });
        const data = await res.json();
        if (data?.success) {
          window.auth?.showNotification?.('Leave request submitted', 'success');
          form.reset();
          await Promise.all([loadMyRequests(container), loadBalance(container)]);
        } else {
          window.auth?.showNotification?.(data?.message || 'Submission failed', 'error');
        }
      } catch(err) {
        window.auth?.showNotification?.('Submission failed', 'error');
      } finally {
        if (btn) { btn.disabled = false; btn.textContent = 'Submit Request'; }
      }
    });
  }

  async function loadMyRequests(container) {
    const wrap = container.querySelector('#myRequests'); if (!wrap) return;
    wrap.innerHTML = '<div class="text-muted">Loading...</div>';
    try {
      const res = await fetch('api/leave.php', { credentials: 'same-origin' });
      const data = await res.json();
      if (!data?.success) throw new Error();
      const arr = Array.isArray(data.data) ? data.data : [];
      wrap.innerHTML = arr.length ? (`
        <div class="table-responsive">
        <table class="table table-sm align-middle">
          <thead><tr><th>Type</th><th>Dates</th><th>Days</th><th>Status</th><th></th></tr></thead>
          <tbody>
            ${arr.map(r => `
              <tr>
                <td>${escapeHtml(r.leave_type_name || '')}</td>
                <td>${escapeHtml(r.start_date)} → ${escapeHtml(r.end_date)}</td>
                <td>${Number(r.days_requested)}</td>
                <td><span class="badge bg-${r.status==='approved'?'success':(r.status==='rejected'?'danger':(r.status==='cancelled'?'secondary':'warning'))}">${escapeHtml(r.status)}</span></td>
                <td class="text-end">${r.status==='pending' ? `<button class="btn btn-outline-danger btn-sm" data-cancel-id="${r.id}">Cancel</button>` : ''}</td>
              </tr>`).join('')}
          </tbody>
        </table></div>`) : '<div class="text-muted">No requests yet.</div>';

      wrap.querySelectorAll('[data-cancel-id]').forEach(btn => {
        btn.addEventListener('click', async () => {
          const id = btn.getAttribute('data-cancel-id');
          if (!id) return;
          btn.disabled = true;
          try {
            const res2 = await fetch(`api/leave.php?id=${encodeURIComponent(id)}`, { method: 'DELETE', credentials: 'same-origin' });
            const d2 = await res2.json();
            if (d2?.success) {
              window.auth?.showNotification?.('Request cancelled', 'info');
              await Promise.all([loadMyRequests(container), loadBalance(container)]);
            } else {
              window.auth?.showNotification?.(d2?.message || 'Failed to cancel', 'error');
            }
          } catch(_) { window.auth?.showNotification?.('Failed to cancel', 'error'); }
        });
      });
    } catch(_) { wrap.innerHTML = '<div class="text-danger">Failed to load.</div>'; }
  }

  async function loadPendingApprovals(container) {
    const wrap = container.querySelector('#approvalList'); if (!wrap) return;
    const typeSel = container.querySelector('#apprFilterType');
    const isHr = ['super_admin','admin','hr_head'].includes(role());
    // Populate leave types once
    if (typeSel && typeSel.children.length <= 1) {
      try {
        const resTypes = await fetch('api/leave.php?action=types', { credentials: 'same-origin' });
        const dt = await resTypes.json();
        if (dt?.success) {
          typeSel.insertAdjacentHTML('beforeend', dt.data.map(t => `<option value="${t.id}">${escapeHtml(t.name)}</option>`).join(''));
        }
      } catch(_) {}
    }

    const q = container.querySelector('#apprFilterQ')?.value?.trim() || '';
    const from = container.querySelector('#apprFilterFrom')?.value || '';
    const to = container.querySelector('#apprFilterTo')?.value || '';
    const leave_type_id = typeSel?.value || '';

    const params = new URLSearchParams({ status: 'pending' });
    if (q) params.set('q', q);
    if (from) params.set('from', from);
    if (to) params.set('to', to);
    if (leave_type_id) params.set('leave_type_id', leave_type_id);

    wrap.innerHTML = '<div class="text-muted">Loading...</div>';
    try {
      const res = await fetch(`api/leave.php?${params.toString()}`, { credentials: 'same-origin' });
      const data = await res.json();
      if (!data?.success) throw new Error();
      const arr = Array.isArray(data.data) ? data.data : [];
      window.leaveApprovalsCache = arr;
      // cache/load employees for reliever selection (HR only)
      let empRows = [];
      if (isHr) {
        try {
          if (Array.isArray(window.__employeesCache)) { empRows = window.__employeesCache; }
          else {
            const re = await fetch('api/employees.php', { credentials:'same-origin' });
            const je = await re.json();
            empRows = (je?.success && Array.isArray(je.data)) ? je.data : [];
            window.__employeesCache = empRows;
          }
        } catch(_) { empRows = []; }
      }

      function relieverOptions(selectedId){
        if (!isHr) return '';
        const opts = ['<option value="">Reliever</option>'].concat(empRows.map(e=>{
          const name = (e.full_name||'').trim() || `${e.first_name||''} ${e.last_name||''}`.trim();
          const code = e.employee_number || '';
          const sel = String(selectedId||'') === String(e.id) ? ' selected' : '';
          return `<option value="${e.id}"${sel}>${escapeHtml(name)}${code? ' - '+escapeHtml(code):''}</option>`;
        }));
        return `<select class="form-select form-select-sm" data-reliever-for="REPLACE_ID">${opts.join('')}</select>`;
      }

      wrap.innerHTML = arr.length ? (`
        <div class="table-responsive">
        <table class="table table-sm align-middle" id="approvalsTable">
          <thead><tr><th>Employee</th><th>Type</th><th>Start</th><th>End</th><th>Days</th><th>Status</th><th class="text-end">Actions</th></tr></thead>
          <tbody>
            ${arr.map(r => `
              <tr>
                <td>${escapeHtml(r.employee_name || '')}</td>
                <td>${escapeHtml(r.leave_type_name || '')}</td>
                <td>${escapeHtml(r.start_date)}</td>
                <td>${escapeHtml(r.end_date)}</td>
                <td>${Number(r.days_requested)}</td>
                <td><span class="badge bg-warning">pending</span></td>
                <td class="text-end">
                  ${isHr ? relieverOptions(r.reliever_employee_id).replace('REPLACE_ID', String(r.id)) : ''}
                  <div class="d-inline-flex gap-1 align-items-center mt-1 mt-md-0">
                    <button class="btn btn-success btn-sm" data-approve-id="${r.id}"><i class="fas fa-check"></i></button>
                    <button class="btn btn-danger btn-sm" data-reject-id="${r.id}"><i class="fas fa-times"></i></button>
                  </div>
                </td>
              </tr>`).join('')}
          </tbody>
        </table></div>`) : '<div class="text-muted">No pending approvals.</div>';

      wrap.querySelectorAll('[data-approve-id]').forEach(btn => {
        btn.addEventListener('click', () => {
          const id = btn.getAttribute('data-approve-id');
          const extra = {};
          if (isHr) {
            const sel = wrap.querySelector(`select[data-reliever-for="${id}"]`);
            const v = sel ? parseInt(sel.value||'0',10) : 0;
            if (v>0) extra.reliever_employee_id = v;
          }
          doApproveReject('approve', id, extra, container);
        });
      });
      wrap.querySelectorAll('[data-reject-id]').forEach(btn => {
        btn.addEventListener('click', async () => {
          const id = btn.getAttribute('data-reject-id');
          const reason = prompt('Enter rejection reason');
          if (!id || reason === null) return;
          await doApproveReject('reject', id, { rejection_reason: reason }, container);
        });
      });

      // Hook up filter apply and export
      const applyBtn = container.querySelector('#apprFilterApply');
      if (applyBtn && !applyBtn._bound) {
        applyBtn._bound = true;
        applyBtn.addEventListener('click', async () => { await loadPendingApprovals(container); });
      }
      const exportBtn = container.querySelector('#apprExportCsv');
      if (exportBtn && !exportBtn._bound) {
        exportBtn._bound = true;
        exportBtn.addEventListener('click', () => exportApprovalsCsv(window.leaveApprovalsCache || []));
      }
    } catch(_) {
      wrap.innerHTML = '<div class="text-danger">Failed to load approvals.</div>';
    }
  }

  async function doApproveReject(action, id, extra = {}, containerOverride) {
    if (!id) return;
    try {
      const res = await fetch(`api/leave.php?action=${action}&id=${encodeURIComponent(id)}`, {
        method: 'POST', headers: { 'Content-Type': 'application/json' }, credentials: 'same-origin',
        body: JSON.stringify(extra)
      });
      const data = await res.json();
      if (data?.success) {
        window.auth?.showNotification?.(`Request ${action}d`, 'success');
        // reload approvals and balances
        const container = containerOverride || document.getElementById('pageContent');
        await Promise.all([
          loadPendingApprovals(container),
          loadBalance(container)
        ]);
      } else {
        window.auth?.showNotification?.(data?.message || 'Operation failed', 'error');
      }
    } catch(_) { window.auth?.showNotification?.('Operation failed', 'error'); }
  }

  function exportApprovalsCsv(rows) {
    if (!rows || !rows.length) {
      window.auth?.showNotification?.('No data to export', 'info');
      return;
    }
    const headers = ['Employee','Type','Start','End','Days','Status'];
    const csv = [headers.join(',')].concat(rows.map(r => [
      safeCsv(r.employee_name), safeCsv(r.leave_type_name), safeCsv(r.start_date), safeCsv(r.end_date), String(r.days_requested||0), String(r.status||'pending')
    ].join(','))).join('\n');
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `leave_approvals_${new Date().toISOString().slice(0,10)}.csv`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }

  async function loadAllocations(container) {
    const panel = container.querySelector('#allocPanel'); if (!panel) return;
    panel.innerHTML = '<div class="text-muted">Loading...</div>';
    try {
      const res = await fetch('api/leave.php?action=allocations', { credentials: 'same-origin' });
      const data = await res.json();
      if (!data?.success) throw new Error();
      const types = Array.isArray(data.data?.leave_types) ? data.data.leave_types : [];
      const grades = Array.isArray(data.data?.grades) ? data.data.grades : [];
      const allocRows = Array.isArray(data.data?.allocations) ? data.data.allocations : [];
      const allocMap = new Map(allocRows.map(a => [`${a.grade}|${a.leave_type_id}`, a.days_per_year]));

      panel.innerHTML = `
        <div class="table-responsive">
        <table class="table table-sm align-middle">
          <thead><tr><th>Grade</th>${types.map(t=>`<th>${escapeHtml(t.name)}</th>`).join('')}</tr></thead>
          <tbody>
            ${grades.map(g => `
              <tr>
                <td class="fw-bold">${escapeHtml(g)}</td>
                ${types.map(t => {
                  const key = `${g}|${t.id}`;
                  const val = allocMap.has(key) ? Number(allocMap.get(key)) : '';
                  return `<td><input type="number" class="form-control form-control-sm" min="0" data-alloc="${g}|${t.id}" value="${val}" /></td>`;
                }).join('')}
              </tr>`).join('')}
          </tbody>
        </table></div>`;

      const saveBtn = container.querySelector('#saveAllocBtn');
      if (saveBtn) {
        saveBtn.onclick = async () => {
          const inputs = panel.querySelectorAll('[data-alloc]');
          const payload = [];
          inputs.forEach(inp => {
            const key = inp.getAttribute('data-alloc');
            const [grade, typeId] = key.split('|');
            const days = inp.value === '' ? null : Number(inp.value);
            if (days !== null && !Number.isNaN(days)) {
              payload.push({ grade, leave_type_id: Number(typeId), days_per_year: days });
            }
          });
          try {
            const res2 = await fetch('api/leave.php?action=allocations_save', {
              method: 'POST', headers: { 'Content-Type': 'application/json' }, credentials: 'same-origin',
              body: JSON.stringify({ allocations: payload })
            });
            const d2 = await res2.json();
            if (d2?.success) {
              window.auth?.showNotification?.('Allocations saved', 'success');
              await loadBalance(container);
            } else {
              window.auth?.showNotification?.(d2?.message || 'Save failed', 'error');
            }
          } catch(_) { window.auth?.showNotification?.('Save failed', 'error'); }
        };
      }
    } catch(_) { panel.innerHTML = '<div class="text-danger">Failed to load allocations.</div>'; }
  }

  // helpers
  function escapeHtml(s){
    try { return String(s).replace(/[&<>"']/g, c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); } catch(_) { return String(s||''); }
  }
  function safeCsv(v){
    const s = (v==null?'':String(v));
    if (/[",\n]/.test(s)) return '"' + s.replace(/"/g,'""') + '"';
    return s;
  }

  window.pages.leave = { render };
})();
