// Learning module page
(function(){
  window.pages = window.pages || {};

  function role(){ return (window.auth?.currentUser?.role_slug) || (window.auth?.currentUser?.role) || 'employee'; }
  function isHR(){ return ['super_admin','admin','hr_head','hr_officer'].includes(role()); }
  function canGrade(){ return ['super_admin','admin','hr_head'].includes(role()); }
  function escapeHtml(s){ try { return String(s).replace(/[&<>"']/g, c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); } catch(_) { return String(s||''); } }

  async function render(container){
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex align-items-center justify-content-between">
          <h5 class="card-title mb-0">Learning</h5>
          <small class="text-muted">Curate lessons and plans. Track progress and assessments.</small>
        </div>
        <div class="card-body">
          <ul class="nav nav-tabs" role="tablist">
            <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#lr-courses" type="button" role="tab">Courses</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#lr-myplans" type="button" role="tab">My Plans</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#lr-assess" type="button" role="tab">Assessments</button></li>
            ${isHR()? '<li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#lr-reports" type="button" role="tab">Reports</button></li>': ''}
            ${isHR()? '<li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#lr-manage" type="button" role="tab">Manage</button></li>': ''}
          </ul>
          <div class="tab-content mt-3">
            <div class="tab-pane fade show active" id="lr-courses" role="tabpanel">
              <div class="d-flex align-items-center justify-content-between mb-2">
                <h6 class="mb-0">Courses & Lessons</h6>
                ${isHR()? '<button class="btn btn-sm btn-primary" id="lrAddCourseBtn"><i class="fas fa-plus"></i> Add Course</button>' : ''}
              </div>
              <div id="lrCourses"></div>
              <div id="lrCourseView" class="mt-3"></div>
            </div>
            <div class="tab-pane fade" id="lr-myplans" role="tabpanel">
              <div class="d-flex align-items-center justify-content-between mb-2">
                <h6 class="mb-0">My Learning Plans</h6>
                <button class="btn btn-sm btn-primary" id="lrNewPlanBtn"><i class="fas fa-plus"></i> New Plan</button>
              </div>
              <div id="lrPlans"></div>
              <div id="lrPlanView" class="mt-3"></div>
            </div>
            <div class="tab-pane fade" id="lr-assess" role="tabpanel">
              <div class="d-flex align-items-center justify-content-between mb-2">
                <h6 class="mb-0">Assessments</h6>
              </div>
              <div id="lrAssessList"></div>
              <div id="lrAssessView" class="mt-3"></div>
            </div>
            <div class="tab-pane fade" id="lr-reports" role="tabpanel">
              <div class="d-flex align-items-center justify-content-between mb-2">
                <h6 class="mb-0">Organization Reports</h6>
              </div>
              <div id="lrReports" class="small"></div>
            </div>
            ${isHR()? `
            <div class="tab-pane fade" id="lr-manage" role="tabpanel">
              <div class="row g-3">
                <div class="col-12"><h6 class="mb-2">Create Lesson</h6></div>
                <div class="col-md-6">
                  <label class="form-label">Title</label>
                  <input type="text" class="form-control" id="lrTitle" placeholder="e.g. Security Compliance 101" />
                </div>
                <div class="col-md-3">
                  <label class="form-label">Content Type</label>
                  <select class="form-select" id="lrType">
                    <option value="text">Text</option>
                    <option value="video">Video</option>
                    <option value="link">Link</option>
                  </select>
                </div>
                <div class="col-12" id="lrTextWrap">
                  <label class="form-label">Content (Text)</label>
                  <textarea class="form-control" id="lrText" rows="4" placeholder="Write lesson content or outline..."></textarea>
                </div>
                <div class="col-md-8 d-none" id="lrUrlWrap">
                  <label class="form-label">Content URL</label>
                  <input type="url" class="form-control" id="lrUrl" placeholder="https://... (YouTube, Vimeo, or any link)" />
                </div>
                <div class="col-12">
                  <label class="form-label">Description (optional)</label>
                  <textarea class="form-control" id="lrDesc" rows="2" placeholder="Short description"></textarea>
                </div>
                <div class="col-12 text-end">
                  <button class="btn btn-primary" id="lrCreateBtn">Create Lesson</button>
                </div>
              </div>
              <hr/>
              <div>
                <h6>All Lessons</h6>
                <div id="lrManageList"></div>
              </div>
            </div>`: ''}
          </div>
        </div>
      </div>
    `;

    await loadCatalog(container);
    await loadPlans(container);
    await loadAssessments(container);
    await loadReports(container);
    const newPlanBtn = container.querySelector('#lrNewPlanBtn');
    if (newPlanBtn) newPlanBtn.addEventListener('click', () => createPlanPrompt(container));
    const addCourseBtn = container.querySelector('#lrAddCourseBtn');
    if (addCourseBtn) addCourseBtn.addEventListener('click', ()=> addCoursePrompt(container));
    if (isHR()) { setupManageHandlers(container); await loadManageList(container); }
  }

  async function addCoursePrompt(container){
    const title = prompt('Enter new course title'); if (!title) return;
    const description = prompt('Optional: description') || '';
    const level = prompt('Optional: level (e.g., Beginner/Intermediate)') || '';
    const ok = await createCourse({ title, description, level });
    window.auth?.showNotification?.(ok? 'Course created' : 'Failed to create course', ok? 'success':'error');
    if (ok) await loadCatalog(container);
  }

  async function createCourse(payload){
    try{
      const res = await fetch('api/learning.php?action=create_course', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
      const js = await res.json();
      return !!js?.success;
    } catch(_){ return false; }
  }

  async function loadReports(container){
    const el = container.querySelector('#lrReports'); if (!el) return;
    el.innerHTML = '<div class="text-muted">Loading reports...</div>';
    try{
      const res = await fetch('api/learning.php?action=reports_course_ratings', { credentials:'same-origin' });
      const js = await res.json(); if (!js?.success) throw new Error();
      const rows = Array.isArray(js.data)? js.data : [];
      if (!rows.length){ el.innerHTML = '<div class="text-muted">No rating data yet.</div>'; return; }
      const html = `
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead><tr><th>Course</th><th class="text-center">★1</th><th class="text-center">★2</th><th class="text-center">★3</th><th class="text-center">★4</th><th class="text-center">★5</th></tr></thead>
            <tbody>
              ${rows.map(r=>{
                const rt = r.ratings || {}; return `<tr>
                  <td>${escapeHtml(r.title||'')}</td>
                  <td class="text-center">${rt[1]||0}</td>
                  <td class="text-center">${rt[2]||0}</td>
                  <td class="text-center">${rt[3]||0}</td>
                  <td class="text-center">${rt[4]||0}</td>
                  <td class="text-center">${rt[5]||0}</td>
                </tr>`;
              }).join('')}
            </tbody>
          </table>
        </div>`;
      el.innerHTML = html;
    }catch(_){ el.innerHTML = '<div class="text-danger">Failed to load reports.</div>'; }
  }

  async function loadCatalog(container){
    const el = container.querySelector('#lrCourses'); if (!el) return;
    el.innerHTML = '<div class="text-muted">Loading catalog...</div>';
    try {
      const res = await fetch('api/learning.php?action=catalog', { credentials:'same-origin' });
      const js = await res.json(); if (!js?.success) throw new Error();
      const lessons = Array.isArray(js.data?.lessons) ? js.data.lessons : [];
      const courses = Array.isArray(js.data?.courses) ? js.data.courses : [];
      if (!lessons.length && !courses.length) { el.innerHTML = '<div class="text-muted">No catalog items yet.</div>'; return; }
      const courseGrid = courses.length ? (`
        <h6 class="mb-2">Courses</h6>
        <div class="row g-3 mb-3">
          ${courses.map(c=>`
            <div class="col-md-4">
              <div class="border rounded p-3 h-100 d-flex flex-column">
                <div class="fw-bold">${escapeHtml(c.title)}</div>
                <div class="small text-muted mt-1">${escapeHtml(c.description||'')}</div>
                <div class="small mt-1">Lessons: ${Number(c.lesson_count||0)}</div>
                <div class="mt-auto pt-2 d-flex gap-2 justify-content-end">
                  <button class="btn btn-sm btn-outline-secondary" data-open-course="${c.id}">Open</button>
                  <button class="btn btn-sm btn-outline-primary" data-add-course-to-plan="${c.id}">Add to Plan</button>
                </div>
              </div>
            </div>`).join('')}
        </div>`): '';
      const lessonGrid = lessons.length ? (`
        <h6 class="mb-2">Lessons</h6>
        <div class="row g-3">
          ${lessons.map(r=>`
            <div class="col-md-4">
              <div class="border rounded p-3 h-100 d-flex flex-column">
                <div class="d-flex align-items-center justify-content-between">
                  <div class="fw-bold">${escapeHtml(r.title)}</div>
                  <span class="badge bg-${r.content_type==='video'?'danger':(r.content_type==='link'?'secondary':'primary')}">${escapeHtml(r.content_type)}</span>
                </div>
                <div class="small text-muted mt-1">${escapeHtml(r.description||'')}</div>
                ${r.content_type!=='text' && r.content_url ? `<a class="mt-2 small" href="${encodeURI(r.content_url)}" target="_blank">Open Resource</a>`: ''}
                <div class="mt-auto pt-2 text-end">
                  <button class="btn btn-sm btn-outline-primary" data-add-lesson-to-plan="${r.id}">Add to Plan</button>
                </div>
              </div>
            </div>`).join('')}
        </div>`): '';
      el.innerHTML = courseGrid + lessonGrid;
      // events
      el.querySelectorAll('[data-open-course]').forEach(btn=>{
        btn.addEventListener('click', ()=>{
          const id = btn.getAttribute('data-open-course');
          openCourse(container, id);
        });
      });
      el.querySelectorAll('[data-add-lesson-to-plan]').forEach(btn=>{
        btn.addEventListener('click', async ()=>{
          const id = btn.getAttribute('data-add-lesson-to-plan');
          await choosePlanAndAdd(container, { type: 'lesson', id });
        });
      });
      el.querySelectorAll('[data-add-course-to-plan]').forEach(btn=>{
        btn.addEventListener('click', async ()=>{
          const id = btn.getAttribute('data-add-course-to-plan');
          await choosePlanAndAdd(container, { type: 'course', id });
        });
      });
    } catch(_) { el.innerHTML = '<div class="text-danger">Failed to load catalog.</div>'; }
  }

  async function openCourse(container, courseId){
    const view = container.querySelector('#lrCourseView'); if (!view) return;
    view.innerHTML = '<div class="text-muted">Loading course...</div>';
    try{
      const res = await fetch(`api/learning.php?action=course&id=${courseId}`, { credentials:'same-origin' });
      const js = await res.json(); if (!js?.success) throw new Error();
      const c = js.data || {}; const lessons = Array.isArray(c.lessons)?c.lessons:[];
      view.innerHTML = `
        <div class="card">
          <div class="card-header d-flex align-items-center justify-content-between">
            <div>
              <div class="fw-semibold">${escapeHtml(c.title||'Course')}</div>
              <div class="small text-muted">${escapeHtml(c.description||'')}</div>
            </div>
            <div class="d-flex gap-2">
              ${isHR()? '<button class="btn btn-sm btn-outline-secondary" id="lrCoursePublishBtn">Publish</button>' : ''}
            </div>
          </div>
          <div class="card-body">
            <div class="d-flex align-items-center justify-content-between mb-2">
              <h6 class="mb-0">Lessons</h6>
              ${isHR()? '<button class="btn btn-sm btn-primary" id="lrCourseAddLessonBtn"><i class="fas fa-plus"></i> Add Lesson</button>' : ''}
            </div>
            ${lessons.length? `<ul class="list-group mb-3">${lessons.map(l=>`<li class=\"list-group-item d-flex align-items-center justify-content-between\"><span>${escapeHtml(l.title)}</span></li>`).join('')}</ul>` : '<div class="text-muted mb-3">No lessons linked.</div>'}
            <div class="d-flex align-items-center gap-2">
              <label class="small">Rate this course</label>
              <select id="lrRateSelect" class="form-select form-select-sm" style="width:auto">
                <option value="">--</option>
                <option>1</option><option>2</option><option>3</option><option>4</option><option>5</option>
              </select>
              <button class="btn btn-sm btn-outline-success" id="lrRateSubmit">Submit</button>
            </div>
          </div>
        </div>`;
      // Publish (HR)
      const pub = view.querySelector('#lrCoursePublishBtn');
      pub?.addEventListener('click', async ()=>{
        try { const r=await fetch(`api/learning.php?action=publish_course&id=${courseId}`,{method:'POST',credentials:'same-origin'}); const j=await r.json(); window.auth?.showNotification?.(j?.success?'Published':'Failed','info'); } catch(_){ window.auth?.showNotification?.('Failed','error'); }
      });
      // Add lesson (HR)
      const add = view.querySelector('#lrCourseAddLessonBtn');
      add?.addEventListener('click', async ()=>{
        let list=[]; try{ const r=await fetch('api/learning.php?action=lessons',{credentials:'same-origin'}); const j=await r.json(); if(j?.success) list=j.data||[]; }catch(_){ }
        if (!list.length){ window.auth?.showNotification?.('No lessons available','warning'); return; }
        const choice = prompt('Enter lesson IDs to add (comma-separated):\n'+list.map(x=>`${x.id}: ${x.title}`).join('\n')); if(!choice) return;
        const ids = choice.split(',').map(s=>parseInt(s,10)).filter(Boolean);
        const existing = lessons.map(x=>x.id);
        const merged = [...new Set(existing.concat(ids))];
        try{ const r=await fetch(`api/learning.php?action=set_course_lessons&id=${courseId}`,{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin', body: JSON.stringify({ lesson_ids: merged })}); const j=await r.json(); if(j?.success){ window.auth?.showNotification?.('Course updated','success'); openCourse(container, courseId);} else { window.auth?.showNotification?.('Failed','error'); } }catch(_){ window.auth?.showNotification?.('Failed','error'); }
      });
      // Rating
      const rateBtn = view.querySelector('#lrRateSubmit');
      rateBtn?.addEventListener('click', async ()=>{
        const val = parseInt(view.querySelector('#lrRateSelect')?.value,10);
        if(!val){ window.auth?.showNotification?.('Select 1-5','warning'); return; }
        try{ const r=await fetch(`api/learning.php?action=add_course_rating&id=${courseId}`,{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin', body: JSON.stringify({ rating: val })}); const j=await r.json(); window.auth?.showNotification?.(j?.success? 'Thanks for rating' : (j?.message||'Failed'), j?.success? 'success':'error'); }catch(_){ window.auth?.showNotification?.('Failed','error'); }
      });
    }catch(_){ view.innerHTML = '<div class="text-danger">Failed to load course.</div>'; }
  }

  async function choosePlanAndAdd(container, item){
    // Fetch my plans
    let plans=[];
    try{ const res=await fetch('api/learning.php?action=plans', {credentials:'same-origin'}); const js=await res.json(); if(js?.success) plans=js.data||[]; }catch(_){ plans=[]; }
    if (!plans.length){
      const title = prompt('No plans found. Enter a title to create a new plan:');
      if (!title) return;
      const created = await createPlan({ title });
      if (!created) return;
      await addItemsToPlan(created.id, [item]);
      window.auth?.showNotification?.('Added to new plan','success');
      await loadPlans(container);
      return;
    }
    const choice = prompt('Enter Plan ID to add to:\n' + plans.map(p=>`${p.id}: ${p.title}`).join('\n') + '\nOr leave blank to create new');
    if (!choice){
      const title = prompt('Enter new plan title:'); if (!title) return;
      const created = await createPlan({ title }); if (!created) return;
      await addItemsToPlan(created.id, [item]);
      window.auth?.showNotification?.('Added to new plan','success');
      await loadPlans(container);
    } else {
      const pid = parseInt(choice,10); if (!pid) return;
      await addItemsToPlan(pid, [item]);
      window.auth?.showNotification?.('Added to plan','success');
      await openPlan(container, pid);
    }
  }

  async function createPlan(payload){
    try{ const res=await fetch('api/learning.php?action=create_plan',{method:'POST',headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload)}); const js=await res.json(); if(js?.success) return js.data; }catch(_){ }
    window.auth?.showNotification?.('Failed to create plan','error'); return null;
  }
  async function addItemsToPlan(planId, items){
    try{ const res=await fetch(`api/learning.php?action=add_plan_items&id=${planId}`,{method:'POST',headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ items })}); const js=await res.json(); return !!js?.success; }catch(_){return false;}
  }

  async function loadPlans(container){
    const el = container.querySelector('#lrPlans'); if (!el) return;
    el.innerHTML = '<div class="text-muted">Loading...</div>';
    try{
      const res = await fetch('api/learning.php?action=plans',{credentials:'same-origin'});
      const js = await res.json();
      if (!js?.success) throw new Error();
      const rows = Array.isArray(js.data) ? js.data : [];
      el.innerHTML = rows.length ? (`
        <div class="row g-2">
          ${rows.map(p=>`
            <div class="col-md-6">
              <div class="border rounded p-2 d-flex align-items-center justify-content-between">
                <div>
                  <div class="fw-semibold">${escapeHtml(p.title)}</div>
                  <div class="small text-muted">${escapeHtml(p.purpose||'')}</div>
                </div>
                <div class="d-flex gap-2">
                  <button class="btn btn-sm btn-outline-primary" data-open-plan="${p.id}">Open</button>
                  <button class="btn btn-sm btn-success" data-enroll-plan="${p.id}">Enroll</button>
                </div>
              </div>
            </div>`).join('')}
        </div>
      `) : '<div class="text-muted">No plans yet.</div>';
      el.querySelectorAll('[data-open-plan]').forEach(b=>b.addEventListener('click', ()=>openPlan(container, b.getAttribute('data-open-plan'))));
      el.querySelectorAll('[data-enroll-plan]').forEach(b=>b.addEventListener('click', async ()=>{ const id=b.getAttribute('data-enroll-plan'); const ok = await enrollPlan(id); window.auth?.showNotification?.(ok?'Enrolled':'Failed','info'); }));
    }catch(_){ el.innerHTML = '<div class="text-danger">Failed to load.</div>'; }
  }

  async function createPlanPrompt(container){
    const title = prompt('Enter new plan title');
    if (!title) return;
    const res = await createPlan({ title });
    if (res) { window.auth?.showNotification?.('Plan created','success'); await loadPlans(container); }
    else { window.auth?.showNotification?.('Failed to create plan','error'); }
  }

  async function enrollPlan(planId){
    try{ const res=await fetch(`api/learning.php?action=enroll_plan&id=${planId}`,{method:'POST', credentials:'same-origin'}); const js=await res.json(); return !!js?.success; }catch(_){ return false; }
  }

  async function openPlan(container, planId){
    const view = container.querySelector('#lrPlanView'); if (!view) return;
    view.innerHTML = '<div class="text-muted">Loading plan...</div>';
    try{
      const res=await fetch(`api/learning.php?action=plan&id=${planId}`,{credentials:'same-origin'}); const js=await res.json(); if(!js?.success) throw new Error();
      const p = js.data;
      const items = Array.isArray(p.items)?p.items:[];
      view.innerHTML = `
        <div class="card">
          <div class="card-header d-flex align-items-center justify-content-between">
            <div>
              <div class="fw-semibold">${escapeHtml(p.title)}</div>
              <div class="small text-muted">${escapeHtml(p.purpose||'')}</div>
            </div>
            <div class="d-flex gap-2">
              <button class="btn btn-sm btn-success" id="lrEnrollPlanBtn">Enroll</button>
              <button class="btn btn-sm btn-outline-primary" id="lrAddLessonBtn">Add Lesson</button>
            </div>
          </div>
          <div class="card-body">
            ${items.length? `<ul class="list-group">${items.map(it=>`<li class="list-group-item d-flex align-items-center justify-content-between"><div>${escapeHtml(it.item_type)} #${it.item_id}</div><button class="btn btn-sm btn-outline-success" data-mark-done="${it.item_id}" data-item-type="${it.item_type}">Mark Completed</button></li>`).join('')}</ul>` : '<div class="text-muted">No items yet.</div>'}
            <hr/>
            <div class="d-flex align-items-center justify-content-between mb-2">
              <h6 class="mb-0">Applicability Rules</h6>
              ${isHR()? '<button class="btn btn-sm btn-secondary" id="lrAddRuleBtn">+ Add Rule</button>' : ''}
            </div>
            <div id="lrRules"></div>
          </div>
        </div>`;
      const enrollBtn = view.querySelector('#lrEnrollPlanBtn'); if (enrollBtn) enrollBtn.addEventListener('click', async ()=>{ const ok=await enrollPlan(planId); window.auth?.showNotification?.(ok?'Enrolled':'Failed','info'); });
      const addLessonBtn = view.querySelector('#lrAddLessonBtn'); if (addLessonBtn) addLessonBtn.addEventListener('click', async ()=>{
        // simple select of available lessons
        let lessons=[]; try{ const r=await fetch('api/learning.php?action=lessons',{credentials:'same-origin'}); const j=await r.json(); if(j?.success) lessons=j.data||[]; }catch(_){ }
        if(!lessons.length){ window.auth?.showNotification?.('No lessons available','warning'); return; }
        const choice = prompt('Enter Lesson ID to add to plan:\n'+lessons.map(l=>`${l.id}: ${l.title}`).join('\n')); if (!choice) return;
        const lid = parseInt(choice,10); if(!lid) return;
        const ok = await addItemsToPlan(planId, [{type:'lesson', id: lid}]); if (ok){ window.auth?.showNotification?.('Added','success'); await openPlan(container, planId); }
      });
      // Plan rules
      await loadPlanRules(view, planId);
      const addRuleBtn = view.querySelector('#lrAddRuleBtn');
      addRuleBtn?.addEventListener('click', ()=> addPlanRulePrompt(view, planId));
      view.querySelectorAll('[data-mark-done]').forEach(btn=>{
        btn.addEventListener('click', async ()=>{
          const lessonId = parseInt(btn.getAttribute('data-mark-done'),10);
          // require enrollment
          const en = await enrollPlan(planId); // idempotent
          const res = await fetch('api/learning.php?action=mark_lesson',{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin', body: JSON.stringify({ plan_id: Number(planId), lesson_id: lessonId, status: 'completed' })});
          const js2 = await res.json();
          window.auth?.showNotification?.(js2?.success?'Marked completed':'Failed','info');
        });
      });
    }catch(_){ view.innerHTML = '<div class="text-danger">Failed to load plan.</div>'; }
  }

  async function loadPlanRules(root, planId){
    const el = root.querySelector('#lrRules'); if (!el) return;
    el.innerHTML = '<div class="text-muted">Loading rules...</div>';
    try{
      const r=await fetch(`api/learning.php?action=plan_rules&id=${planId}`,{credentials:'same-origin'});
      const j=await r.json(); if(!j?.success) throw new Error();
      const rows=Array.isArray(j.data)?j.data:[];
      el.innerHTML = rows.length? (`<ul class="list-group">${rows.map(x=>`<li class="list-group-item d-flex align-items-center justify-content-between"><div><span class="badge bg-info me-2">${escapeHtml(x.rule_type)}</span>${escapeHtml(x.condition_json||'')}</div>${isHR()? `<button class="btn btn-sm btn-outline-danger" data-del-rule="${x.id}">Delete</button>`:''}</li>`).join('')}</ul>`) : '<div class="text-muted">No rules yet.</div>';
      el.querySelectorAll('[data-del-rule]')?.forEach(b=> b.addEventListener('click', async ()=>{
        const rid=b.getAttribute('data-del-rule'); if(!confirm('Delete rule?')) return;
        try{ const rr=await fetch(`api/learning.php?action=delete_plan_rule&id=${rid}`,{method:'POST',credentials:'same-origin'}); const jj=await rr.json(); if(jj?.success){ window.auth?.showNotification?.('Deleted','info'); loadPlanRules(root, planId);} else { window.auth?.showNotification?.('Failed','error'); } }catch(_){ window.auth?.showNotification?.('Failed','error'); }
      }));
    }catch(_){ el.innerHTML = '<div class="text-danger">Failed to load rules.</div>'; }
  }

  function addPlanRulePrompt(root, planId){
    const type = (prompt('Rule type (enrollment/outcome)','enrollment')||'enrollment').toLowerCase();
    const field = prompt('Condition field (e.g., role, designation)'); if(!field) return;
    const value = prompt('Equals value'); if(value===null) return;
    const cond = { [field]: { op: 'eq', value } };
    (async ()=>{
      try{ const r=await fetch(`api/learning.php?action=add_plan_rule&id=${planId}`,{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin', body: JSON.stringify({ rule_type: type, condition: cond })}); const j=await r.json(); if(j?.success){ window.auth?.showNotification?.('Rule added','success'); loadPlanRules(root, planId);} else { window.auth?.showNotification?.(j?.message||'Failed','error'); } }catch(_){ window.auth?.showNotification?.('Failed','error'); }
    })();
  }

  async function loadAssessments(container){
    const el = container.querySelector('#lrAssessList'); if (!el) return;
    el.innerHTML = '<div class="text-muted">Loading assessments...</div>';
    try{
      const res=await fetch('api/learning.php?action=assessments',{credentials:'same-origin'}); const js=await res.json(); if(!js?.success) throw new Error();
      const rows = Array.isArray(js.data)?js.data:[];
      el.innerHTML = rows.length? (`<div class="list-group">
        ${rows.map(a=>`<div class="list-group-item d-flex align-items-center justify-content-between"><div><div class="fw-semibold">${escapeHtml(a.title)}</div><div class="small text-muted">${escapeHtml(a.type)} • ${escapeHtml(a.delivery)}</div></div><div class="d-flex gap-2">${isHR()? `<button class="btn btn-sm btn-outline-secondary" data-view-subs="${a.id}">Submissions</button>`:''}<button class="btn btn-sm btn-outline-primary" data-take-assess="${a.id}">Take</button></div></div>`).join('')}
      </div>`) : '<div class="text-muted">No assessments yet.</div>';
      el.querySelectorAll('[data-take-assess]').forEach(b=>b.addEventListener('click', ()=> takeAssessment(container, b.getAttribute('data-take-assess'))));
      el.querySelectorAll('[data-view-subs]')?.forEach(b=> b.addEventListener('click', ()=> viewSubmissions(container, b.getAttribute('data-view-subs'))));
    }catch(_){ el.innerHTML = '<div class="text-danger">Failed to load.</div>'; }
  }

  async function viewSubmissions(container, assessmentId){
    const view = container.querySelector('#lrAssessView'); if (!view) return;
    view.innerHTML = '<div class="text-muted">Loading submissions...</div>';
    try{
      const r = await fetch(`api/learning.php?action=submissions&assessment_id=${assessmentId}`, { credentials:'same-origin' });
      const j = await r.json(); if (!j?.success) throw new Error();
      const rows = Array.isArray(j.data)? j.data : [];
      view.innerHTML = rows.length? (`
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead><tr><th>Employee</th><th>Submitted</th><th>Status</th><th>Score</th><th>Grade</th><th>Actions</th></tr></thead>
            <tbody>
              ${rows.map(s=>`<tr>
                <td>${escapeHtml((s.first_name||'')+' '+(s.last_name||''))}</td>
                <td>${escapeHtml(s.submitted_at||'')}</td>
                <td><span class="badge bg-${s.status==='graded'?'success':'secondary'}">${escapeHtml(s.status||'')}</span></td>
                <td>${s.score ?? ''}</td>
                <td>${escapeHtml(s.grade||'')}</td>
                <td>${canGrade()? `<button class="btn btn-sm btn-outline-primary" data-grade-id="${s.id}">Grade</button>`:''}</td>
              </tr>`).join('')}
            </tbody>
          </table>
        </div>
      `) : '<div class="text-muted">No submissions yet.</div>';
      if (canGrade()){
        view.querySelectorAll('[data-grade-id]')?.forEach(btn=> btn.addEventListener('click', async ()=>{
          const sid = btn.getAttribute('data-grade-id');
          const scoreStr = prompt('Enter score (leave blank for none)');
          const score = scoreStr?.trim()==='' ? null : Number(scoreStr);
          const grade = prompt('Enter letter grade (optional)') || null;
          const feedback = prompt('Feedback (optional)') || null;
          try{ const rr = await fetch(`api/learning.php?action=grade_submission&id=${sid}`, { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ score, grade, feedback }) }); const jj = await rr.json(); if (jj?.success){ window.auth?.showNotification?.('Graded','success'); viewSubmissions(container, assessmentId);} else { window.auth?.showNotification?.(jj?.message||'Failed','error'); } }catch(_){ window.auth?.showNotification?.('Failed','error'); }
        }));
      }
    }catch(_){ view.innerHTML = '<div class="text-danger">Failed to load submissions.</div>'; }
  }

  async function takeAssessment(container, assessmentId){
    const view = container.querySelector('#lrAssessView'); if (!view) return;
    view.innerHTML = '<div class="text-muted">Loading assessment...</div>';
    try{
      const res=await fetch(`api/learning.php?action=assessment&id=${assessmentId}`,{credentials:'same-origin'}); const js=await res.json(); if(!js?.success) throw new Error();
      const a = js.data; const qs = Array.isArray(a.questions)?a.questions:[];
      const mins = parseInt(a.duration_minutes||0,10);
      view.innerHTML = `
        <div class="card">
          <div class="card-header d-flex align-items-center justify-content-between">
            <div><div class="fw-semibold">${escapeHtml(a.title)}</div><div class="small text-muted">${escapeHtml(a.type)} • ${escapeHtml(a.delivery)}</div></div>
            ${mins? `<div class="text-end"><div class="small text-muted">Total: ${mins} min</div><div class="fw-semibold" id="lrTimer">--:--</div></div>`:''}
          </div>
          <div class="card-body">
            <div class="d-flex align-items-center justify-content-between mb-2">
              <div class="small"><span id="lrAnsCount">0</span> / ${qs.length} Answered</div>
              <div id="lrQNav" class="d-flex flex-wrap gap-1"></div>
            </div>
            <form id="lrAssessForm">
              ${qs.map((q,i)=>`<div class="mb-3" id="sec_${q.id}">${renderQuestion(q,i)}</div>`).join('')}
              <div class="text-end">
                <button class="btn btn-primary" type="submit">Submit</button>
              </div>
            </form>
          </div>
        </div>`;
      const form = view.querySelector('#lrAssessForm');
      form.addEventListener('submit', async (e)=>{
        e.preventDefault();
        const answers = collectAnswers(view, qs);
        try{ const r=await fetch(`api/learning.php?action=submit_assessment&id=${assessmentId}`,{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin', body: JSON.stringify({ answers })}); const s=await r.json(); if(s?.success){ window.auth?.showNotification?.(`Submitted. Score: ${s.data?.score ?? 0}/${s.data?.total_points ?? 0}`,'success'); } else { window.auth?.showNotification?.(s?.message || 'Submit failed','error'); } }catch(_){ window.auth?.showNotification?.('Submit failed','error'); }
      });
      // Question nav + answered count
      const nav = view.querySelector('#lrQNav');
      const ansCount = view.querySelector('#lrAnsCount');
      if (nav){
        nav.innerHTML = qs.map((q,i)=>`<button type="button" class="btn btn-sm btn-outline-secondary" data-go-q="${q.id}">${i+1}</button>`).join('');
        nav.querySelectorAll('[data-go-q]').forEach(b=> b.addEventListener('click', ()=>{ const id=b.getAttribute('data-go-q'); const sec=view.querySelector(`#sec_${id}`); sec?.scrollIntoView({behavior:'smooth', block:'start'}); }));
      }
      const recalc = ()=>{ const answers = collectAnswers(view, qs); if (ansCount) ansCount.textContent = String(answers.length); };
      view.querySelectorAll('input,textarea').forEach(el=> el.addEventListener('input', recalc));
      recalc();
      // Timer
      if (mins){
        const endAt = Date.now() + mins*60*1000;
        const timerEl = view.querySelector('#lrTimer');
        const iv = setInterval(()=>{
          const left = Math.max(0, endAt - Date.now());
          const mm = Math.floor(left/60000); const ss = Math.floor((left%60000)/1000);
          if (timerEl) timerEl.textContent = `${String(mm).padStart(2,'0')}:${String(ss).padStart(2,'0')}`;
          if (left <= 0){ clearInterval(iv); try{ form.requestSubmit(); }catch(_){ form.dispatchEvent(new Event('submit', { cancelable: true })); } }
        }, 500);
      }
    }catch(_){ view.innerHTML = '<div class="text-danger">Failed to load assessment.</div>'; }
  }

  function renderQuestion(q, idx){
    const id = `q_${q.id}`;
    if (q.qtype === 'truefalse'){
      return `<div class="mb-3"><div class="fw-semibold mb-1">${idx+1}. ${escapeHtml(q.question_text)}</div>
        <div class="form-check"><input class="form-check-input" type="radio" name="${id}" value="true" id="${id}_t"><label class="form-check-label" for="${id}_t">True</label></div>
        <div class="form-check"><input class="form-check-input" type="radio" name="${id}" value="false" id="${id}_f"><label class="form-check-label" for="${id}_f">False</label></div>
      </div>`;
    }
    if (q.qtype === 'text'){
      return `<div class="mb-3"><div class="fw-semibold mb-1">${idx+1}. ${escapeHtml(q.question_text)}</div>
        <textarea class="form-control" rows="3" name="${id}"></textarea>
      </div>`;
    }
    // default mcq (single choice)
    let opts = [];
    try{ const arr = JSON.parse(q.options_json||'[]'); if(Array.isArray(arr)) opts = arr; }catch(_){ }
    return `<div class="mb-3"><div class="fw-semibold mb-1">${idx+1}. ${escapeHtml(q.question_text)}</div>
      ${opts.map((op,oi)=>`<div class="form-check"><input class="form-check-input" type="radio" name="${id}" value="${escapeHtml(typeof op==='object'? JSON.stringify(op): String(op))}" id="${id}_${oi}"><label class="form-check-label" for="${id}_${oi}">${escapeHtml(typeof op==='object'? (op.label ?? JSON.stringify(op)) : String(op))}</label></div>`).join('')}
    </div>`;
  }

  function collectAnswers(root, qs){
    const out=[];
    qs.forEach(q=>{
      const id = `q_${q.id}`;
      if (q.qtype === 'truefalse'){
        const el = root.querySelector(`input[name="${id}"]:checked`);
        if (el) out.push({ question_id: q.id, answer: el.value === 'true' });
        return;
      }
      if (q.qtype === 'text'){
        const el = root.querySelector(`textarea[name="${id}"]`);
        out.push({ question_id: q.id, answer_text: el ? el.value : '' });
        return;
      }
      // mcq single
      const el = root.querySelector(`input[name="${id}"]:checked`);
      if (el){
        // try to parse back JSON if needed
        let val = el.value; try{ val = JSON.parse(val); }catch(_){ }
        out.push({ question_id: q.id, answer: val });
      }
    });
    return out;
  }

  function setupManageHandlers(container){
    const typeSel = container.querySelector('#lrType');
    const textWrap = container.querySelector('#lrTextWrap');
    const urlWrap = container.querySelector('#lrUrlWrap');
    typeSel?.addEventListener('change', ()=>{
      const t = typeSel.value;
      if (t === 'text') { textWrap?.classList.remove('d-none'); urlWrap?.classList.add('d-none'); }
      else { textWrap?.classList.add('d-none'); urlWrap?.classList.remove('d-none'); }
    });
    const btn = container.querySelector('#lrCreateBtn');
    btn?.addEventListener('click', async ()=>{
      const title = container.querySelector('#lrTitle')?.value?.trim();
      const content_type = typeSel?.value || 'text';
      const description = container.querySelector('#lrDesc')?.value || '';
      const content_text = content_type === 'text' ? (container.querySelector('#lrText')?.value || '') : '';
      const content_url = content_type !== 'text' ? (container.querySelector('#lrUrl')?.value || '') : '';
      if (!title) { window.auth?.showNotification?.('Title is required','warning'); return; }
      try {
        const res = await fetch('api/learning.php?action=create_lesson', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ title, content_type, description, content_text, content_url }) });
        const js = await res.json();
        if (js?.success){ window.auth?.showNotification?.('Lesson created','success'); await Promise.all([loadCatalog(container), loadManageList(container)]); }
        else window.auth?.showNotification?.(js?.message || 'Failed','error');
      } catch(_) { window.auth?.showNotification?.('Failed','error'); }
    });
  }

  async function loadManageList(container){
    const el = container.querySelector('#lrManageList'); if (!el) return;
    el.innerHTML = '<div class="text-muted">Loading...</div>';
    try {
      const res = await fetch('api/learning.php?action=lessons', { credentials:'same-origin' });
      const js = await res.json();
      if (!js?.success) throw new Error();
      const rows = Array.isArray(js.data) ? js.data : [];
      el.innerHTML = rows.length ? (`
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead><tr><th>Title</th><th>Type</th><th>Status</th><th>Created</th></tr></thead>
            <tbody>
              ${rows.map(r=>`
                <tr>
                  <td>${escapeHtml(r.title)}</td>
                  <td>${escapeHtml(r.content_type)}</td>
                  <td><span class="badge bg-${r.status==='active'?'success':(r.status==='draft'?'secondary':'dark')}">${escapeHtml(r.status)}</span></td>
                  <td>${escapeHtml(r.created_at||'')}</td>
                </tr>`).join('')}
            </tbody>
          </table>
        </div>`) : '<div class="text-muted">No lessons yet.</div>';
    } catch(_) { el.innerHTML = '<div class="text-danger">Failed to load.</div>'; }
  }

  window.pages.learning = { render };
})();
