// HR Settings page (HR Head & HR Officer)
(function(){
  window.pages = window.pages || {};

  function role(){ return (window.auth?.currentUser?.role_slug) || (window.auth?.currentUser?.role) || 'employee'; }
  function isHr(){ return ['hr_head','hr_officer','admin','super_admin'].includes(role()); }
  function notify(msg, type){ try{ window.auth?.showNotification?.(msg, type||'info'); }catch(_){} }
  function esc(s){ try { return String(s).replace(/[&<>"']/g, c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); } catch(_){ return String(s||''); } }

  async function render(container){
    if (!isHr()) { container.innerHTML = '<div class="alert alert-warning">Access denied. HR Settings are available to HR roles only.</div>'; return; }

    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex align-items-center justify-content-between">
          <h5 class="card-title mb-0">HR Settings</h5>
          <small class="text-muted">Grades, Leave Allocations, Leave Types, Payroll Setup</small>
        </div>
        <div class="card-body">
          <ul class="nav nav-tabs" role="tablist">
            <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#hrs-grades" type="button" role="tab">Grades</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#hrs-alloc" type="button" role="tab">Leave Allocations</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#hrs-types" type="button" role="tab">Leave Types</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#hrs-payroll" type="button" role="tab">Payroll Setup</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#hrs-holidays" type="button" role="tab">Holidays</button></li>
          </ul>
          <div class="tab-content mt-3">
            <div class="tab-pane fade show active" id="hrs-grades" role="tabpanel">
              <div id="gradesPanel"></div>
            </div>
            <div class="tab-pane fade" id="hrs-alloc" role="tabpanel">
              <div class="alert alert-info">Manage grade-based annual leave allocations per type.</div>
              <div id="allocPanel"></div>
              <div class="text-end mt-3">
                <button id="allocSaveBtn" class="btn btn-success btn-sm"><i class="fas fa-save"></i> Save Allocations</button>
              </div>
            </div>
            <div class="tab-pane fade" id="hrs-types" role="tabpanel">
              <div id="typesPanel"></div>
            </div>
            <div class="tab-pane fade" id="hrs-payroll" role="tabpanel">
              <div id="payrollPanel"></div>
            </div>
            <div class="tab-pane fade" id="hrs-holidays" role="tabpanel">
              <div id="holidaysPanel"></div>
            </div>
          </div>
        </div>
      </div>`;

    await Promise.all([
      loadGrades(container),
      loadAllocations(container),
      loadLeaveTypes(container),
      loadPayrollSetup(container),
      loadHolidays(container)
    ]);
  }

  // ===== Grades ===== //
  async function loadGrades(container){
    const wrap = container.querySelector('#gradesPanel'); if (!wrap) return;
    wrap.innerHTML = '<div class="text-muted">Loading...</div>';
    try{
      const res = await fetch('api/grades.php', { credentials:'same-origin' });
      const json = await res.json();
      if (!json?.success) throw new Error();
      const rows = Array.isArray(json.data) ? json.data : [];
      wrap.innerHTML = `
        <div class="row g-3">
          <div class="col-lg-5">
            <div class="border rounded p-3 h-100">
              <h6 class="mb-3" id="gradeFormTitle">Create Grade</h6>
              <div class="mb-2"><label class="form-label">Name</label><input id="gName" class="form-control" placeholder="e.g. Grade A" /></div>
              <div class="mb-2"><label class="form-label">Code</label><input id="gCode" class="form-control" placeholder="e.g. GA" /></div>
              <div class="mb-3"><label class="form-label">Base Salary</label><input id="gBase" type="number" class="form-control" placeholder="e.g. 1500" /></div>
              <div class="d-flex gap-2">
                <button class="btn btn-primary" id="gSave">Save</button>
                <button class="btn btn-outline-secondary" id="gReset">Reset</button>
              </div>
              <input type="hidden" id="gId" value="" />
              <hr/>
              <div id="gradeDetail" class="mt-2"></div>
            </div>
          </div>
          <div class="col-lg-7">
            <div class="table-responsive">
              <table class="table table-sm align-middle">
                <thead><tr><th>Name</th><th>Code</th><th class="text-end">Base Salary</th><th></th></tr></thead>
                <tbody id="gradesTBody">
                  ${rows.map(r=>`
                    <tr>
                      <td>${esc(r.name||'')}</td>
                      <td>${esc(r.code||'')}</td>
                      <td class="text-end">${r.base_salary==null?'':Number(r.base_salary)}</td>
                      <td class="text-end">
                        <button class="btn btn-outline-primary btn-sm" data-edit="${r.id}"><i class="fas fa-pen"></i></button>
                        <button class="btn btn-outline-danger btn-sm" data-del="${r.id}"><i class="fas fa-trash"></i></button>
                      </td>
                    </tr>`).join('')}
                </tbody>
              </table>
            </div>
          </div>
        </div>`;

      // Bind actions
      const saveBtn = wrap.querySelector('#gSave');
      const resetBtn = wrap.querySelector('#gReset');
      if (saveBtn && !saveBtn._bound){
        saveBtn._bound = true;
        saveBtn.addEventListener('click', async ()=>{
          const id = wrap.querySelector('#gId').value ? Number(wrap.querySelector('#gId').value) : 0;
          const name = wrap.querySelector('#gName').value.trim();
          const code = wrap.querySelector('#gCode').value.trim();
          const base = wrap.querySelector('#gBase').value;
          const base_salary = base === '' ? null : Number(base);
          if (!name){ notify('Grade name is required','warning'); return; }
          try{
            const res2 = await fetch('api/grades.php?action=save', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ id, name, code, base_salary }) });
            const d2 = await res2.json();
            if (d2?.success){ notify('Saved','success'); loadGrades(container); }
            else { notify(d2?.message || 'Save failed','error'); }
          }catch(_){ notify('Save failed','error'); }
        });
      }
      if (resetBtn && !resetBtn._bound){
        resetBtn._bound = true;
        resetBtn.addEventListener('click', ()=>{ setGradeForm(wrap, null); });
      }

      // Row actions
      wrap.querySelectorAll('[data-edit]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-edit'));
          try{
            const r = await fetch(`api/grades.php?action=detail&id=${id}`, { credentials:'same-origin' });
            const j = await r.json();
            if (j?.success){ setGradeForm(wrap, j.data); }
          }catch(_){}
        });
      });
      wrap.querySelectorAll('[data-del]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-del'));
          if (!confirm('Delete this grade? This cannot be undone.')) return;
          try{
            const r = await fetch(`api/grades.php?action=delete&id=${id}`, { method:'POST', credentials:'same-origin' });
            const j = await r.json();
            if (j?.success){ notify('Deleted','info'); loadGrades(container); }
            else { notify(j?.message || 'Delete failed','error'); }
          }catch(_){ notify('Delete failed','error'); }
        });
      });
    } catch(_){ wrap.innerHTML = '<div class="text-danger">Failed to load grades.</div>'; }
  }

  function setGradeForm(wrap, data){
    wrap.querySelector('#gradeFormTitle').textContent = data ? 'Edit Grade' : 'Create Grade';
    wrap.querySelector('#gId').value = data?.id || '';
    wrap.querySelector('#gName').value = data?.name || '';
    wrap.querySelector('#gCode').value = data?.code || '';
    const base = (data && data.base_salary!=null) ? String(data.base_salary) : '';
    wrap.querySelector('#gBase').value = base;
    const detail = wrap.querySelector('#gradeDetail');
    if (detail){
      if (data?.id){ renderGradeDetail(detail, data.id); }
      else { detail.innerHTML = '<div class="text-muted">Select a grade to manage default allowances and deductions.</div>'; }
    }
  }

  async function renderGradeDetail(detailEl, gradeId){
    detailEl.innerHTML = '<div class="text-muted">Loading grade details...</div>';
    try{
      const [detRes, setupRes] = await Promise.all([
        fetch(`api/grades.php?action=detail&id=${gradeId}`, { credentials:'same-origin' }),
        fetch('api/payroll.php?action=setup_lists', { credentials:'same-origin' })
      ]);
      const det = await detRes.json();
      const setup = await setupRes.json();
      if (!det?.success) throw new Error('detail');
      const grade = det.data || {};
      const allowTypes = setup?.success && Array.isArray(setup.data?.allowance_types) ? setup.data.allowance_types : [];
      const dedTypes = setup?.success && Array.isArray(setup.data?.deduction_types) ? setup.data.deduction_types : [];

      const allowanceRows = Array.isArray(grade.allowances) ? grade.allowances : [];
      const deductionRows = Array.isArray(grade.deductions) ? grade.deductions : [];

      detailEl.innerHTML = `
        <div class="mt-2">
          <h6 class="mb-2">Default Components for <span class="text-primary">${esc(grade.name||'')}</span></h6>
          <div class="row g-3">
            <div class="col-lg-6">
              <div class="border rounded p-2 h-100">
                <div class="d-flex align-items-center justify-content-between mb-2"><strong>Allowances</strong><button class="btn btn-sm btn-outline-primary" id="gaAdd">Add</button></div>
                <div id="gaForm" class="mb-2" style="display:none"></div>
                <div class="table-responsive"><table class="table table-sm align-middle"><thead><tr><th>Type</th><th class="text-end">Amount/Rate</th><th></th></tr></thead><tbody id="gaBody">
                  ${allowanceRows.map(a=>{
                      const mode = (a.calc_mode||'fixed');
                      const val = mode==='fixed' ? Number(a.amount||0) : `${Number(a.rate||0)}% of ${mode==='percent_basic'?'Basic':'Gross'}`;
                      return `
                    <tr data-id="${a.id}" data-typeid="${a.allowance_type_id||''}" data-mode="${esc(mode)}" data-rate="${a.rate==null?'':Number(a.rate)}" data-amount="${a.amount==null?'':Number(a.amount)}">
                      <td>${esc(a.name||a.code||'Type')}</td>
                      <td class="text-end">${esc(val)}</td>
                      <td class="text-end">
                        <button class="btn btn-outline-primary btn-sm" data-ga-edit="${a.id}"><i class="fas fa-pen"></i></button>
                        <button class="btn btn-outline-danger btn-sm" data-ga-del="${a.id}"><i class="fas fa-trash"></i></button>
                      </td>
                    </tr>`;}).join('')}
                </tbody></table></div>
              </div>
            </div>
            <div class="col-lg-6">
              <div class="border rounded p-2 h-100">
                <div class="d-flex align-items-center justify-content-between mb-2"><strong>Deductions</strong><button class="btn btn-sm btn-outline-primary" id="gdAdd">Add</button></div>
                <div id="gdForm" class="mb-2" style="display:none"></div>
                <div class="table-responsive"><table class="table table-sm align-middle"><thead><tr><th>Type</th><th class="text-end">Amount/Rate</th><th></th></tr></thead><tbody id="gdBody">
                  ${deductionRows.map(d=>{
                      const mode = (d.calc_mode||'fixed');
                      const val = mode==='fixed' ? Number(d.amount||0) : `${Number(d.rate||0)}% of ${mode==='percent_basic'?'Basic':'Gross'}`;
                      return `
                    <tr data-id="${d.id}" data-typeid="${d.deduction_type_id||''}" data-mode="${esc(mode)}" data-rate="${d.rate==null?'':Number(d.rate)}" data-amount="${d.amount==null?'':Number(d.amount)}">
                      <td>${esc(d.name||d.code||'Type')}</td>
                      <td class="text-end">${esc(val)}</td>
                      <td class="text-end">
                        <button class="btn btn-outline-primary btn-sm" data-gd-edit="${d.id}"><i class="fas fa-pen"></i></button>
                        <button class="btn btn-outline-danger btn-sm" data-gd-del="${d.id}"><i class="fas fa-trash"></i></button>
                      </td>
                    </tr>`;}).join('')}
                </tbody></table></div>
              </div>
            </div>
          </div>
        </div>`;

      // Forms
      const gaForm = detailEl.querySelector('#gaForm');
      const gdForm = detailEl.querySelector('#gdForm');

      function renderGaForm(data){
        gaForm.style.display = 'block';
        gaForm.innerHTML = `
          <div class="row g-2 align-items-end">
            <div class="col-md-7">
              <label class="form-label">Allowance Type</label>
              <select id="gaType" class="form-select">${allowTypes.map(t=>`<option value="${t.id}" ${Number(data?.allowance_type_id||0)===Number(t.id)?'selected':''}>${esc(t.name)}</option>`).join('')}</select>
            </div>
            <div class="col-md-3">
              <label class="form-label">Mode</label>
              <select id="gaMode" class="form-select">
                <option value="fixed">Fixed Amount</option>
                <option value="percent_basic">% of Basic</option>
                <option value="percent_gross">% of Gross</option>
              </select>
            </div>
            <div class="col-md-3" id="gaAmtWrap">
              <label class="form-label">Amount</label>
              <input id="gaAmt" type="number" step="0.01" class="form-control" value="${data?Number(data.amount||0):''}" />
            </div>
            <div class="col-md-3" id="gaRateWrap" style="display:none">
              <label class="form-label">Rate %</label>
              <input id="gaRate" type="number" step="0.01" class="form-control" value="${data&&data.rate!=null?Number(data.rate):''}" />
            </div>
            <div class="col-md-2 text-end">
              <button class="btn btn-primary btn-sm" id="gaSave">Save</button>
              <button class="btn btn-outline-secondary btn-sm" id="gaCancel">Cancel</button>
              <input type="hidden" id="gaId" value="${data?.id||''}" />
            </div>
          </div>`;
        const modeSel = gaForm.querySelector('#gaMode');
        if (data?.calc_mode) modeSel.value = String(data.calc_mode);
        const toggle = ()=>{
          const m = modeSel.value;
          gaForm.querySelector('#gaAmtWrap').style.display = (m==='fixed')?'block':'none';
          gaForm.querySelector('#gaRateWrap').style.display = (m==='fixed')?'none':'block';
        };
        modeSel.addEventListener('change', toggle); toggle();
        gaForm.querySelector('#gaSave').onclick = async ()=>{
          const payload = {
            id: gaForm.querySelector('#gaId').value ? Number(gaForm.querySelector('#gaId').value) : 0,
            grade_id: gradeId,
            allowance_type_id: Number(gaForm.querySelector('#gaType').value),
            calc_mode: gaForm.querySelector('#gaMode').value,
            amount: Number(gaForm.querySelector('#gaAmt').value||0),
            rate: (function(){ const m = gaForm.querySelector('#gaMode').value; if (m==='fixed') return null; const v = gaForm.querySelector('#gaRate').value; return v===''? null : Number(v); })()
          };
          if (!payload.allowance_type_id){ notify('Choose an allowance type','warning'); return; }
          try{
            const r = await fetch('api/grades.php?action=save_allowance', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
            const j = await r.json();
            if (j?.success){ notify('Saved','success'); renderGradeDetail(detailEl, gradeId); }
            else { notify(j?.message || 'Save failed','error'); }
          }catch(_){ notify('Save failed','error'); }
        };
        gaForm.querySelector('#gaCancel').onclick = ()=>{ gaForm.style.display='none'; gaForm.innerHTML=''; };
      }

      function renderGdForm(data){
        gdForm.style.display = 'block';
        gdForm.innerHTML = `
          <div class="row g-2 align-items-end">
            <div class="col-md-7">
              <label class="form-label">Deduction Type</label>
              <select id="gdType" class="form-select">${dedTypes.map(t=>`<option value="${t.id}" ${Number(data?.deduction_type_id||0)===Number(t.id)?'selected':''}>${esc(t.name)}</option>`).join('')}</select>
            </div>
            <div class="col-md-3">
              <label class="form-label">Mode</label>
              <select id="gdMode" class="form-select">
                <option value="fixed">Fixed Amount</option>
                <option value="percent_basic">% of Basic</option>
                <option value="percent_gross">% of Gross</option>
              </select>
            </div>
            <div class="col-md-3" id="gdAmtWrap">
              <label class="form-label">Amount</label>
              <input id="gdAmt" type="number" step="0.01" class="form-control" value="${data?Number(data.amount||0):''}" />
            </div>
            <div class="col-md-3" id="gdRateWrap" style="display:none">
              <label class="form-label">Rate %</label>
              <input id="gdRate" type="number" step="0.01" class="form-control" value="${data&&data.rate!=null?Number(data.rate):''}" />
            </div>
            <div class="col-md-2 text-end">
              <button class="btn btn-primary btn-sm" id="gdSave">Save</button>
              <button class="btn btn-outline-secondary btn-sm" id="gdCancel">Cancel</button>
              <input type="hidden" id="gdId" value="${data?.id||''}" />
            </div>
          </div>`;
        const modeSel = gdForm.querySelector('#gdMode');
        if (data?.calc_mode) modeSel.value = String(data.calc_mode);
        const toggle = ()=>{
          const m = modeSel.value;
          gdForm.querySelector('#gdAmtWrap').style.display = (m==='fixed')?'block':'none';
          gdForm.querySelector('#gdRateWrap').style.display = (m==='fixed')?'none':'block';
        };
        modeSel.addEventListener('change', toggle); toggle();
        gdForm.querySelector('#gdSave').onclick = async ()=>{
          const payload = {
            id: gdForm.querySelector('#gdId').value ? Number(gdForm.querySelector('#gdId').value) : 0,
            grade_id: gradeId,
            deduction_type_id: Number(gdForm.querySelector('#gdType').value),
            calc_mode: gdForm.querySelector('#gdMode').value,
            amount: Number(gdForm.querySelector('#gdAmt').value||0),
            rate: (function(){ const m = gdForm.querySelector('#gdMode').value; if (m==='fixed') return null; const v = gdForm.querySelector('#gdRate').value; return v===''? null : Number(v); })()
          };
          if (!payload.deduction_type_id){ notify('Choose a deduction type','warning'); return; }
          try{
            const r = await fetch('api/grades.php?action=save_deduction', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
            const j = await r.json();
            if (j?.success){ notify('Saved','success'); renderGradeDetail(detailEl, gradeId); }
            else { notify(j?.message || 'Save failed','error'); }
          }catch(_){ notify('Save failed','error'); }
        };
        gdForm.querySelector('#gdCancel').onclick = ()=>{ gdForm.style.display='none'; gdForm.innerHTML=''; };
      }

      // Bind buttons
      detailEl.querySelector('#gaAdd')?.addEventListener('click', ()=>renderGaForm(null));
      detailEl.querySelectorAll('[data-ga-edit]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', ()=>{
          const tr = btn.closest('tr');
          const id = Number(btn.getAttribute('data-ga-edit'));
          const amount = tr.getAttribute('data-amount');
          const calc_mode = tr.getAttribute('data-mode') || 'fixed';
          const rate = tr.getAttribute('data-rate');
          const allowance_type_id = tr.getAttribute('data-typeid');
          renderGaForm({ id, allowance_type_id: allowance_type_id?Number(allowance_type_id):null, amount: amount===''?null:Number(amount), calc_mode, rate: rate===''?null:Number(rate) });
        });
      });
      detailEl.querySelectorAll('[data-ga-del]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-ga-del'));
          if (!confirm('Delete this grade allowance?')) return;
          try{
            const r = await fetch(`api/grades.php?action=delete_allowance&id=${id}`, { method:'POST', credentials:'same-origin' });
            const j = await r.json();
            if (j?.success){ notify('Deleted','info'); renderGradeDetail(detailEl, gradeId); }
            else { notify(j?.message || 'Delete failed','error'); }
          }catch(_){ notify('Delete failed','error'); }
        });
      });

      detailEl.querySelector('#gdAdd')?.addEventListener('click', ()=>renderGdForm(null));
      detailEl.querySelectorAll('[data-gd-edit]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', ()=>{
          const tr = btn.closest('tr');
          const id = Number(btn.getAttribute('data-gd-edit'));
          const amount = tr.getAttribute('data-amount');
          const calc_mode = tr.getAttribute('data-mode') || 'fixed';
          const rate = tr.getAttribute('data-rate');
          const deduction_type_id = tr.getAttribute('data-typeid');
          renderGdForm({ id, deduction_type_id: deduction_type_id?Number(deduction_type_id):null, amount: amount===''?null:Number(amount), calc_mode, rate: rate===''?null:Number(rate) });
        });
      });
      detailEl.querySelectorAll('[data-gd-del]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-gd-del'));
          if (!confirm('Delete this grade deduction?')) return;
          try{
            const r = await fetch(`api/grades.php?action=delete_deduction&id=${id}`, { method:'POST', credentials:'same-origin' });
            const j = await r.json();
            if (j?.success){ notify('Deleted','info'); renderGradeDetail(detailEl, gradeId); }
            else { notify(j?.message || 'Delete failed','error'); }
          }catch(_){ notify('Delete failed','error'); }
        });
      });
    }catch(_){ detailEl.innerHTML = '<div class="text-danger">Failed to load grade details.</div>'; }
  }

  // ===== Leave Allocations ===== //
  async function loadAllocations(container){
    const panel = container.querySelector('#allocPanel'); if (!panel) return;
    panel.innerHTML = '<div class="text-muted">Loading...</div>';
    try{
      const res = await fetch('api/leave.php?action=allocations', { credentials:'same-origin' });
      const data = await res.json();
      if (!data?.success) throw new Error();
      const types = Array.isArray(data.data?.leave_types) ? data.data.leave_types : [];
      const grades = Array.isArray(data.data?.grades) ? data.data.grades : [];
      const allocRows = Array.isArray(data.data?.allocations) ? data.data.allocations : [];
      const allocMap = new Map(allocRows.map(a => [`${a.grade}|${a.leave_type_id}`, a.days_per_year]));

      panel.innerHTML = `
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead><tr><th>Grade</th>${types.map(t=>`<th>${esc(t.name)}</th>`).join('')}</tr></thead>
            <tbody>
              ${grades.map(g => `
                <tr>
                  <td class="fw-bold">${esc(g)}</td>
                  ${types.map(t => {
                    const key = `${g}|${t.id}`;
                    const val = allocMap.has(key) ? Number(allocMap.get(key)) : '';
                    return `<td><input type="number" class="form-control form-control-sm" min="0" data-alloc="${g}|${t.id}" value="${val}" /></td>`;
                  }).join('')}
                </tr>`).join('')}
            </tbody>
          </table>
        </div>`;

      const saveBtn = container.querySelector('#allocSaveBtn');
      if (saveBtn && !saveBtn._bound){
        saveBtn._bound = true;
        saveBtn.addEventListener('click', async ()=>{
          const inputs = panel.querySelectorAll('[data-alloc]');
          const payload = [];
          inputs.forEach(inp=>{
            const key = inp.getAttribute('data-alloc');
            const [grade, typeId] = key.split('|');
            const v = inp.value;
            if (v !== '') payload.push({ grade, leave_type_id: Number(typeId), days_per_year: Number(v) });
          });
          try{
            const r = await fetch('api/leave.php?action=allocations_save', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ allocations: payload }) });
            const j = await r.json();
            if (j?.success){ notify('Allocations saved','success'); }
            else { notify(j?.message || 'Save failed','error'); }
          }catch(_){ notify('Save failed','error'); }
        });
      }
    } catch(_){ panel.innerHTML = '<div class="text-danger">Failed to load allocations.</div>'; }
  }

  // ===== Leave Types (read-only) ===== //
  async function loadLeaveTypes(container){
    const wrap = container.querySelector('#typesPanel'); if (!wrap) return;
    wrap.innerHTML = '<div class="text-muted">Loading...</div>';
    try{
      const res = await fetch('api/leave.php?action=types', { credentials:'same-origin' });
      const json = await res.json();
      if (!json?.success) throw new Error();
      const rows = Array.isArray(json.data) ? json.data : [];
      wrap.innerHTML = rows.length ? (`
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead><tr><th>Name</th><th>Code</th><th>Annual Days</th><th>Requires Approval</th><th>Paid</th><th>Carry Forward</th></tr></thead>
            <tbody>
              ${rows.map(r=>`
                <tr>
                  <td>${esc(r.name||'')}</td>
                  <td>${esc(r.code||'')}</td>
                  <td>${Number(r.days_per_year||0)}</td>
                  <td>${Number(r.requires_approval||0)?'Yes':'No'}</td>
                  <td>${Number(r.is_paid||0)?'Yes':'No'}</td>
                  <td>${Number(r.carry_forward||0)?'Yes':'No'}</td>
                </tr>`).join('')}
            </tbody>
          </table>
        </div>`) : '<div class="text-muted">No leave types defined.</div>';
    } catch(_){ wrap.innerHTML = '<div class="text-danger">Failed to load leave types.</div>'; }
  }

  // ===== Payroll Setup (Allowance/Deduction Types) ===== //
  async function loadPayrollSetup(container){
    const panel = container.querySelector('#payrollPanel'); if (!panel) return;
    panel.innerHTML = '<div class="text-muted">Loading...</div>';
    try{
      const res = await fetch('api/payroll.php?action=setup_lists', { credentials:'same-origin' });
      const json = await res.json();
      if (!json?.success) throw new Error();
      const allowances = Array.isArray(json.data?.allowance_types) ? json.data.allowance_types : [];
      const deductions = Array.isArray(json.data?.deduction_types) ? json.data.deduction_types : [];

      panel.innerHTML = `
        <div class="row g-3">
          <div class="col-lg-6">
            <div class="border rounded p-3 h-100">
              <div class="d-flex align-items-center justify-content-between mb-2"><h6 class="mb-0">Allowance Types</h6><button class="btn btn-sm btn-primary" id="addAllowanceBtn">Add</button></div>
              <div id="allowanceFormWrap" class="mb-3" style="display:none"></div>
              <div class="table-responsive">
                <table class="table table-sm align-middle"><thead><tr><th>Name</th><th>Code</th><th>Taxable</th><th>Rate %</th><th></th></tr></thead>
                <tbody id="allowanceTBody">
                  ${allowances.map(a=>`
                    <tr>
                      <td>${esc(a.name||'')}</td>
                      <td>${esc(a.code||'')}</td>
                      <td>${Number(a.taxable||0)?'Yes':'No'}</td>
                      <td>${a.taxable_rate==null?'':Number(a.taxable_rate)}</td>
                      <td class="text-end">
                        <button class="btn btn-outline-primary btn-sm" data-edit-allow="${a.id}"><i class="fas fa-pen"></i></button>
                        <button class="btn btn-outline-danger btn-sm" data-del-allow="${a.id}"><i class="fas fa-trash"></i></button>
                      </td>
                    </tr>`).join('')}
                </tbody></table>
              </div>
            </div>
          </div>
          <div class="col-lg-6">
            <div class="border rounded p-3 h-100">
              <div class="d-flex align-items-center justify-content-between mb-2"><h6 class="mb-0">Deduction Types</h6><button class="btn btn-sm btn-primary" id="addDeductionBtn">Add</button></div>
              <div id="deductionFormWrap" class="mb-3" style="display:none"></div>
              <div class="table-responsive">
                <table class="table table-sm align-middle"><thead><tr><th>Name</th><th>Code</th><th></th></tr></thead>
                <tbody id="deductionTBody">
                  ${deductions.map(d=>`
                    <tr>
                      <td>${esc(d.name||'')}</td><td>${esc(d.code||'')}</td>
                      <td class="text-end">
                        <button class="btn btn-outline-primary btn-sm" data-edit-ded="${d.id}"><i class="fas fa-pen"></i></button>
                        <button class="btn btn-outline-danger btn-sm" data-del-ded="${d.id}"><i class="fas fa-trash"></i></button>
                      </td>
                    </tr>`).join('')}
                </tbody></table>
              </div>
            </div>
          </div>
        </div>`;

      // Allowance form helpers
      const allowFormWrap = panel.querySelector('#allowanceFormWrap');
      const dedFormWrap = panel.querySelector('#deductionFormWrap');

      function renderAllowanceForm(data){
        allowFormWrap.style.display = 'block';
        allowFormWrap.innerHTML = `
          <div class="row g-2 align-items-end">
            <div class="col-md-4"><label class="form-label">Name</label><input class="form-control" id="alName" value="${esc(data?.name||'')}" /></div>
            <div class="col-md-3"><label class="form-label">Code</label><input class="form-control" id="alCode" value="${esc(data?.code||'')}" /></div>
            <div class="col-md-3"><label class="form-label">Taxable</label><select id="alTax" class="form-select"><option value="1" ${Number(data?.taxable||1)?'selected':''}>Yes</option><option value="0" ${Number(data?.taxable||1)?'':'selected'}>No</option></select></div>
            <div class="col-md-2"><label class="form-label">Rate %</label><input class="form-control" id="alRate" type="number" step="0.01" value="${data?.taxable_rate==null?'':Number(data.taxable_rate)}" /></div>
            <div class="col-md-12 text-end">
              <button class="btn btn-primary btn-sm" id="alSave">Save</button>
              <button class="btn btn-outline-secondary btn-sm" id="alCancel">Cancel</button>
              <input type="hidden" id="alId" value="${data?.id||''}" />
            </div>
          </div>`;
        allowFormWrap.querySelector('#alSave').onclick = async ()=>{
          const payload = {
            id: allowFormWrap.querySelector('#alId').value ? Number(allowFormWrap.querySelector('#alId').value) : 0,
            name: allowFormWrap.querySelector('#alName').value.trim(),
            code: allowFormWrap.querySelector('#alCode').value.trim(),
            taxable: Number(allowFormWrap.querySelector('#alTax').value),
            taxable_rate: (function(){ const v = allowFormWrap.querySelector('#alRate').value; return v===''? null : Number(v); })()
          };
          if (!payload.name || !payload.code){ notify('Name and Code are required','warning'); return; }
          try{
            const r = await fetch('api/payroll.php?action=save_allowance_type', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
            const j = await r.json();
            if (j?.success){ notify('Saved','success'); loadPayrollSetup(container); }
            else { notify(j?.message || 'Save failed','error'); }
          }catch(_){ notify('Save failed','error'); }
        };
        allowFormWrap.querySelector('#alCancel').onclick = ()=>{ allowFormWrap.style.display = 'none'; allowFormWrap.innerHTML = ''; };
      }

      function renderDeductionForm(data){
        dedFormWrap.style.display = 'block';
        dedFormWrap.innerHTML = `
          <div class="row g-2 align-items-end">
            <div class="col-md-5"><label class="form-label">Name</label><input class="form-control" id="ddName" value="${esc(data?.name||'')}" /></div>
            <div class="col-md-3"><label class="form-label">Code</label><input class="form-control" id="ddCode" value="${esc(data?.code||'')}" /></div>
            <div class="col-md-4 text-end">
              <button class="btn btn-primary btn-sm" id="ddSave">Save</button>
              <button class="btn btn-outline-secondary btn-sm" id="ddCancel">Cancel</button>
              <input type="hidden" id="ddId" value="${data?.id||''}" />
            </div>
          </div>`;
        dedFormWrap.querySelector('#ddSave').onclick = async ()=>{
          const payload = {
            id: dedFormWrap.querySelector('#ddId').value ? Number(dedFormWrap.querySelector('#ddId').value) : 0,
            name: dedFormWrap.querySelector('#ddName').value.trim(),
            code: dedFormWrap.querySelector('#ddCode').value.trim()
          };
          if (!payload.name || !payload.code){ notify('Name and Code are required','warning'); return; }
          try{
            const r = await fetch('api/payroll.php?action=save_deduction_type', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
            const j = await r.json();
            if (j?.success){ notify('Saved','success'); loadPayrollSetup(container); }
            else { notify(j?.message || 'Save failed','error'); }
          }catch(_){ notify('Save failed','error'); }
        };
        dedFormWrap.querySelector('#ddCancel').onclick = ()=>{ dedFormWrap.style.display = 'none'; dedFormWrap.innerHTML = ''; };
      }

      // Bind table buttons
      panel.querySelector('#addAllowanceBtn')?.addEventListener('click', ()=>renderAllowanceForm(null));
      panel.querySelectorAll('[data-edit-allow]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-edit-allow'));
          // We already have row values; but to be safe just pull from DOM row
          const tr = btn.closest('tr');
          renderAllowanceForm({ id, name: tr.children[0].textContent, code: tr.children[1].textContent, taxable: tr.children[2].textContent==='Yes'?1:0, taxable_rate: tr.children[3].textContent===''?null:Number(tr.children[3].textContent) });
        });
      });
      panel.querySelectorAll('[data-del-allow]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-del-allow'));
          if (!confirm('Delete this allowance type?')) return;
          try{
            const r = await fetch(`api/payroll.php?action=delete_allowance_type&id=${id}`, { method:'POST', credentials:'same-origin' });
            const j = await r.json();
            if (j?.success){ notify('Deleted','info'); loadPayrollSetup(container); }
            else { notify(j?.message || 'Delete failed','error'); }
          }catch(_){ notify('Delete failed','error'); }
        });
      });

      panel.querySelector('#addDeductionBtn')?.addEventListener('click', ()=>renderDeductionForm(null));
      panel.querySelectorAll('[data-edit-ded]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', ()=>{
          const id = Number(btn.getAttribute('data-edit-ded'));
          const tr = btn.closest('tr');
          renderDeductionForm({ id, name: tr.children[0].textContent, code: tr.children[1].textContent });
        });
      });
      panel.querySelectorAll('[data-del-ded]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-del-ded'));
          if (!confirm('Delete this deduction type?')) return;
          try{
            const r = await fetch(`api/payroll.php?action=delete_deduction_type&id=${id}`, { method:'POST', credentials:'same-origin' });
            const j = await r.json();
            if (j?.success){ notify('Deleted','info'); loadPayrollSetup(container); }
            else { notify(j?.message || 'Delete failed','error'); }
          }catch(_){ notify('Delete failed','error'); }
        });
      });

    } catch(_){ panel.innerHTML = '<div class="text-danger">Failed to load payroll setup.</div>'; }
  }

  window.pages.hr_settings = { render };
  
  // ===== Holidays (CRUD) ===== //
  async function loadHolidays(container){
    const panel = container.querySelector('#holidaysPanel'); if (!panel) return;
    const now = new Date();
    const y = now.getFullYear();
    panel.innerHTML = '<div class="text-muted">Loading...</div>';
    try{
      const html = `
        <div class="d-flex align-items-center gap-2 mb-2">
          <label class="form-label mb-0">Year</label>
          <input type="number" id="holYear" class="form-control" style="width:120px" value="${y}" />
          <button class="btn btn-sm btn-primary" id="holReload"><i class="fas fa-rotate"></i> Load</button>
          <button class="btn btn-sm btn-success ms-auto" id="holAdd"><i class="fas fa-plus"></i> Add Holiday</button>
        </div>
        <div id="holForm" class="mb-3" style="display:none"></div>
        <div class="table-responsive">
          <table class="table table-sm align-middle">
            <thead><tr><th>Date</th><th>Name</th><th></th></tr></thead>
            <tbody id="holTBody"></tbody>
          </table>
        </div>`;
      panel.innerHTML = html;
      const loadList = async ()=>{
        const year = Number(panel.querySelector('#holYear').value)||y;
        const res = await fetch(`api/holidays.php?action=list&year=${year}`, { credentials:'same-origin' });
        const js = await res.json();
        const rows = (js?.success && Array.isArray(js.data)) ? js.data : [];
        const tb = panel.querySelector('#holTBody');
        tb.innerHTML = rows.length ? rows.map(r=>`
          <tr>
            <td>${esc(r.holiday_date)}</td>
            <td>${esc(r.name||'')}</td>
            <td class="text-end">
              <button class="btn btn-outline-primary btn-sm" data-hol-edit="${r.id}"><i class="fas fa-pen"></i></button>
              <button class="btn btn-outline-danger btn-sm" data-hol-del="${r.id}"><i class="fas fa-trash"></i></button>
            </td>
          </tr>`).join('') : '<tr><td colspan="3" class="text-muted">No holidays configured.</td></tr>';

        // bind row buttons
        tb.querySelectorAll('[data-hol-edit]').forEach(btn=>{
          if (btn._bound) return; btn._bound = true;
          btn.addEventListener('click', ()=>{
            const tr = btn.closest('tr');
            renderHolForm({ id: Number(btn.getAttribute('data-hol-edit')), holiday_date: tr.children[0].textContent.trim(), name: tr.children[1].textContent.trim() });
          });
        });
        tb.querySelectorAll('[data-hol-del]').forEach(btn=>{
          if (btn._bound) return; btn._bound = true;
          btn.addEventListener('click', async ()=>{
            if (!confirm('Delete this holiday?')) return;
            try{
              const r = await fetch('api/holidays.php?action=delete', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ id: Number(btn.getAttribute('data-hol-del')) }) });
              const j = await r.json();
              if (j?.success){ notify('Deleted','info'); loadList(); }
              else { notify(j?.message || 'Delete failed','error'); }
            }catch(_){ notify('Delete failed','error'); }
          });
        });
      };

      function renderHolForm(data){
        const formWrap = panel.querySelector('#holForm');
        formWrap.style.display = 'block';
        formWrap.innerHTML = `
          <div class="row g-2 align-items-end">
            <div class="col-md-3"><label class="form-label">Date</label><input type="date" id="holDate" class="form-control" value="${esc(data?.holiday_date||'')}" /></div>
            <div class="col-md-6"><label class="form-label">Name</label><input type="text" id="holName" class="form-control" value="${esc(data?.name||'')}" /></div>
            <div class="col-md-3 text-end">
              <button class="btn btn-primary btn-sm" id="holSave">Save</button>
              <button class="btn btn-outline-secondary btn-sm" id="holCancel">Cancel</button>
              <input type="hidden" id="holId" value="${data?.id||''}" />
            </div>
          </div>`;
        formWrap.querySelector('#holSave').onclick = async ()=>{
          const payload = {
            id: formWrap.querySelector('#holId').value ? Number(formWrap.querySelector('#holId').value) : 0,
            holiday_date: formWrap.querySelector('#holDate').value,
            name: formWrap.querySelector('#holName').value.trim()
          };
          if (!payload.holiday_date || !payload.name){ notify('Date and Name are required','warning'); return; }
          try{
            const r = await fetch('api/holidays.php?action=save', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
            const j = await r.json();
            if (j?.success){ notify('Saved','success'); formWrap.style.display='none'; formWrap.innerHTML=''; loadList(); }
            else { notify(j?.message || 'Save failed','error'); }
          }catch(_){ notify('Save failed','error'); }
        };
        formWrap.querySelector('#holCancel').onclick = ()=>{ formWrap.style.display='none'; formWrap.innerHTML=''; };
      }

      panel.querySelector('#holReload')?.addEventListener('click', loadList);
      panel.querySelector('#holAdd')?.addEventListener('click', ()=>renderHolForm(null));
      await loadList();
    }catch(_){ panel.innerHTML = '<div class="text-danger">Failed to load holidays.</div>'; }
  }
})();
