// Grades Management Page
(function(){
  window.pages = window.pages || {};

  const state = {
    grades: [],
    aTypes: [],
    dTypes: [],
    employees: [],
  };

  const modeOptions = [
    {v:'fixed', t:'Fixed'},
    {v:'percent_basic', t:'% of Basic'},
    {v:'percent_gross', t:'% of Gross'},
  ];

  const canEdit = () => {
    try { return window.auth?.canEdit('payroll'); } catch(_) { return false; }
  };

  function esc(s){ return (s==null?'':String(s)).replace(/[&<>"']/g,c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); }

  async function fetchSetupLists(){
    try {
      const res = await fetch('api/payroll.php?action=setup_lists', { credentials:'same-origin' });
      const js = await res.json();
      if (js?.success){
        state.aTypes = Array.isArray(js.data?.allowance_types) ? js.data.allowance_types : [];
        state.dTypes = Array.isArray(js.data?.deduction_types) ? js.data.deduction_types : [];
      }
    } catch(_) {}
  }

  async function fetchGrades(){
    const res = await fetch('api/grades.php', { credentials:'same-origin' });
    const js = await res.json();
    state.grades = js?.success && Array.isArray(js.data) ? js.data : [];
  }

  async function fetchGradeDetail(id){
    const res = await fetch(`api/grades.php?action=detail&id=${encodeURIComponent(id)}`, { credentials:'same-origin' });
    const js = await res.json();
    if (!js?.success) throw new Error(js?.message||'failed');
    return js.data;
  }

  async function saveGradeApi(payload){
    const res = await fetch('api/grades.php?action=save', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }

  async function deleteGradeApi(id){
    const res = await fetch(`api/grades.php?action=delete&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }

  async function saveGradeAllowanceApi(payload){
    const res = await fetch('api/grades.php?action=save_allowance', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }
  async function deleteGradeAllowanceApi(id){
    const res = await fetch(`api/grades.php?action=delete_allowance&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }
  async function saveGradeDeductionApi(payload){
    const res = await fetch('api/grades.php?action=save_deduction', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
    return res.json();
  }
  async function deleteGradeDeductionApi(id){
    const res = await fetch(`api/grades.php?action=delete_deduction&id=${encodeURIComponent(id)}`, { method:'POST', credentials:'same-origin' });
    return res.json();
  }

  async function fetchEmployees(){
    const res = await fetch('api/employees.php', { credentials:'same-origin' });
    const js = await res.json();
    state.employees = js?.success && Array.isArray(js.data) ? js.data : [];
  }

  async function bulkMapApi(mappings){
    const res = await fetch('api/grades.php?action=bulk_map', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ mappings }) });
    return res.json();
  }

  function gradeOptionsHtml(selected){
    return [`<option value="">-- Select Grade --</option>`]
      .concat(state.grades.map(g=>`<option value="${g.id}" ${String(selected)===String(g.id)?'selected':''}>${esc(g.name||g.code||('#'+g.id))}${g.code? ' ('+esc(g.code)+')':''}</option>`))
      .join('');
  }

  function typeOptionsHtml(list, selected){
    return [`<option value=""></option>`]
      .concat(list.map(t=>`<option value="${t.id}" ${String(selected)===String(t.id)?'selected':''}>${esc(t.name||t.code||('#'+t.id))}${t.code?' ('+esc(t.code)+')':''}</option>`))
      .join('');
  }

  function modeOptionsHtml(selected){
    return modeOptions.map(m=>`<option value="${m.v}" ${String(selected||'fixed')===m.v?'selected':''}>${m.t}</option>`).join('');
  }

  function renderGradesTable(container){
    const rows = state.grades.map(g=>`
      <tr>
        <td>${esc(g.name||'')}</td>
        <td>${esc(g.code||'')}</td>
        <td>${g.base_salary==null?'':Number(g.base_salary).toFixed(2)}</td>
        <td>
          <button class="btn btn-sm btn-primary" data-edit="${g.id}">Edit</button>
          <button class="btn btn-sm btn-secondary" data-config="${g.id}">Components</button>
          <button class="btn btn-sm btn-outline-danger" data-del="${g.id}">Delete</button>
        </td>
      </tr>`).join('');
    container.querySelector('#gradesTableBody').innerHTML = rows || '<tr><td colspan="4" class="text-center text-muted">No grades</td></tr>';
  }

  function attachGradesTableEvents(container){
    container.querySelector('#addGradeBtn')?.addEventListener('click', ()=> openGradeModal());
    container.querySelector('#gradesTableBody')?.addEventListener('click', async (e)=>{
      const t = e.target;
      const id = t?.dataset?.edit || t?.dataset?.config || t?.dataset?.del;
      if (!id) return;
      if (t.dataset.edit){
        const g = state.grades.find(x=>String(x.id)===String(id));
        openGradeModal(g||{id: parseInt(id,10)});
      } else if (t.dataset.config){
        openComponentsModal(parseInt(id,10));
      } else if (t.dataset.del){
        if (!confirm('Delete this grade?')) return;
        const r = await deleteGradeApi(parseInt(id,10));
        if (r?.success){
          await fetchGrades(); renderGradesTable(container); window.auth?.showNotification?.('Deleted','success');
        } else {
          window.auth?.showNotification?.(r?.message||'Delete failed','error');
        }
      }
    });
  }

  function openGradeModal(grade){
    if (!canEdit()) return;
    const isEdit = !!(grade && grade.id);
    const html = `
      <div class="modal fade" id="gradeModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">${isEdit? 'Edit Grade' : 'Add Grade'}</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
              <div class="mb-3">
                <label class="form-label">Name</label>
                <input type="text" class="form-control" id="gradeName" value="${esc(grade?.name||'')}" />
              </div>
              <div class="mb-3">
                <label class="form-label">Code</label>
                <input type="text" class="form-control" id="gradeCode" value="${esc(grade?.code||'')}" />
              </div>
              <div class="mb-3">
                <label class="form-label">Base Salary</label>
                <input type="number" class="form-control" id="gradeBase" step="0.01" value="${grade?.base_salary??''}" />
              </div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
              <button type="button" class="btn btn-primary" id="saveGradeBtn">Save</button>
            </div>
          </div>
        </div>
      </div>`;
    const div = document.createElement('div'); div.innerHTML = html; document.body.appendChild(div);
    const modalEl = div.querySelector('#gradeModal');
    const modal = bootstrap.Modal.getOrCreateInstance(modalEl);
    modal.show();
    modalEl.querySelector('#saveGradeBtn')?.addEventListener('click', async ()=>{
      const payload = {
        id: grade?.id || undefined,
        name: modalEl.querySelector('#gradeName')?.value?.trim() || '',
        code: modalEl.querySelector('#gradeCode')?.value?.trim() || '',
        base_salary: modalEl.querySelector('#gradeBase')?.value || null,
      };
      const r = await saveGradeApi(payload);
      if (r?.success){
        window.auth?.showNotification?.('Saved','success');
        modal.hide(); div.remove();
        await fetchGrades();
        const container = document.getElementById('gradesPageRoot');
        if (container) renderGradesTable(container);
      } else {
        window.auth?.showNotification?.(r?.message||'Save failed','error');
      }
    });
    modalEl.addEventListener('hidden.bs.modal', ()=> div.remove());
  }

  async function openComponentsModal(gradeId){
    await fetchSetupLists();
    const det = await fetchGradeDetail(gradeId);
    const aRows = (Array.isArray(det.allowances)?det.allowances:[]).map(a=> compRowHtml('a', a)).join('');
    const dRows = (Array.isArray(det.deductions)?det.deductions:[]).map(d=> compRowHtml('d', d)).join('');
    const html = `
      <div class="modal fade" id="gradeCompModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Grade Components</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
              <div class="row g-3">
                <div class="col-md-6">
                  <div class="d-flex justify-content-between align-items-center mb-2">
                    <h6 class="mb-0">Allowances</h6>
                    <button class="btn btn-sm btn-outline-primary" id="addGA">Add</button>
                  </div>
                  <div class="table-responsive"><table class="table table-sm" id="tblGA">
                    <thead><tr><th>Type</th><th>Mode</th><th>Amount</th><th>Rate %</th><th style="width:120px">Actions</th></tr></thead>
                    <tbody>${aRows}</tbody>
                  </table></div>
                </div>
                <div class="col-md-6">
                  <div class="d-flex justify-content-between align-items-center mb-2">
                    <h6 class="mb-0">Deductions</h6>
                    <button class="btn btn-sm btn-outline-primary" id="addGD">Add</button>
                  </div>
                  <div class="table-responsive"><table class="table table-sm" id="tblGD">
                    <thead><tr><th>Type</th><th>Mode</th><th>Amount</th><th>Rate %</th><th style="width:120px">Actions</th></tr></thead>
                    <tbody>${dRows}</tbody>
                  </table></div>
                </div>
              </div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
          </div>
        </div>
      </div>`;
    const shell = document.createElement('div'); shell.innerHTML = html; document.body.appendChild(shell);
    const modalEl = shell.querySelector('#gradeCompModal');
    const modal = bootstrap.Modal.getOrCreateInstance(modalEl); modal.show();

    function compRowHtml(kind, r){
      const isA = kind==='a';
      const types = isA ? state.aTypes : state.dTypes;
      return `<tr data-kind="${isA?'a':'d'}" data-id="${r?.id||''}">
        <td><select class="form-select" data-field="type">${typeOptionsHtml(types, isA? r.allowance_type_id : r.deduction_type_id)}</select></td>
        <td><select class="form-select" data-field="mode">${modeOptionsHtml(r?.calc_mode)}</select></td>
        <td><input type="number" step="0.01" class="form-control" data-field="amount" value="${r?.amount??''}" /></td>
        <td><input type="number" step="0.01" class="form-control" data-field="rate" value="${r?.rate??''}" /></td>
        <td>
          <button class="btn btn-sm btn-primary" data-save>Save</button>
          <button class="btn btn-sm btn-outline-danger" data-del>Delete</button>
        </td>
      </tr>`;
    }

    modalEl.querySelector('#addGA')?.addEventListener('click', ()=>{
      modalEl.querySelector('#tblGA tbody')?.insertAdjacentHTML('beforeend', compRowHtml('a', {}));
    });
    modalEl.querySelector('#addGD')?.addEventListener('click', ()=>{
      modalEl.querySelector('#tblGD tbody')?.insertAdjacentHTML('beforeend', compRowHtml('d', {}));
    });

    modalEl.addEventListener('click', async (e)=>{
      const btn = e.target.closest('button'); if (!btn) return;
      const row = e.target.closest('tr'); if (!row) return;
      const kind = row.getAttribute('data-kind');
      const id = row.getAttribute('data-id');
      if (btn.hasAttribute('data-save')){
        const typeId = row.querySelector('[data-field="type"]').value || '';
        const mode = row.querySelector('[data-field="mode"]').value || 'fixed';
        const amount = row.querySelector('[data-field="amount"]').value || 0;
        const rate = row.querySelector('[data-field="rate"]').value || '';
        const base = { id: id? parseInt(id,10):undefined, grade_id: gradeId, amount: parseFloat(amount), calc_mode: mode, rate: rate===''?null:parseFloat(rate) };
        let resp;
        if (kind==='a') resp = await saveGradeAllowanceApi(Object.assign(base,{ allowance_type_id: parseInt(typeId,10) }));
        else resp = await saveGradeDeductionApi(Object.assign(base,{ deduction_type_id: parseInt(typeId,10) }));
        if (resp?.success){ window.auth?.showNotification?.('Saved','success'); const d=await fetchGradeDetail(gradeId); modalEl.querySelector('#tblGA tbody').innerHTML=(d.allowances||[]).map(x=>compRowHtml('a',x)).join(''); modalEl.querySelector('#tblGD tbody').innerHTML=(d.deductions||[]).map(x=>compRowHtml('d',x)).join(''); }
        else { window.auth?.showNotification?.(resp?.message||'Save failed','error'); }
      } else if (btn.hasAttribute('data-del')){
        if (!id) { row.remove(); return; }
        const api = (kind==='a') ? deleteGradeAllowanceApi : deleteGradeDeductionApi;
        const r = await api(parseInt(id,10));
        if (r?.success){ window.auth?.showNotification?.('Deleted','success'); row.remove(); }
        else { window.auth?.showNotification?.(r?.message||'Delete failed','error'); }
      }
    });

    modalEl.addEventListener('hidden.bs.modal', ()=> shell.remove());
  }

  function renderBulkMapping(container){
    const gradesSel = `<select class="form-select form-select-sm" data-map-grade> ${gradeOptionsHtml('')} </select>`;
    const rows = state.employees.map(e=>{
      const cur = e.grade_id || '';
      return `<tr data-id="${e.id}">
        <td>${esc(e.employee_number||'')}</td>
        <td>${esc(e.full_name || ((e.first_name||'')+' '+(e.last_name||'')).trim())}</td>
        <td>${esc(e.department_name||'')}</td>
        <td>${esc(e.position_title||'')}</td>
        <td><select class="form-select form-select-sm" data-map-grade>${gradeOptionsHtml(cur)}</select></td>
      </tr>`;
    }).join('');
    container.querySelector('#bulkTableBody').innerHTML = rows || '<tr><td colspan="5" class="text-center text-muted">No employees</td></tr>';
  }

  function attachBulkEvents(container){
    container.querySelector('#saveBulkMap')?.addEventListener('click', async ()=>{
      const rows = Array.from(container.querySelectorAll('#bulkTableBody tr'));
      const mappings = rows.map(tr=>{
        const eid = parseInt(tr.getAttribute('data-id'),10);
        const gidVal = tr.querySelector('[data-map-grade]')?.value || '';
        return { employee_id: eid, grade_id: gidVal? parseInt(gidVal,10) : null };
      });
      const out = await bulkMapApi(mappings);
      if (out?.success){
        window.auth?.showNotification?.('Bulk mapping saved','success');
        try {
          await fetchEmployees();
          renderBulkMapping(container);
          renderDistribution(container);
        } catch(_){}
      }
      else { window.auth?.showNotification?.(out?.message||'Bulk mapping failed','error'); }
    });
  }

  function renderDistribution(container){
    const content = container.querySelector('#distContentGrades');
    if (!content) return;
    // Build map grade_id -> employees
    const empByGrade = new Map();
    (state.grades||[]).forEach(g=> empByGrade.set(String(g.id), []));
    const unassigned = [];
    (state.employees||[]).forEach(e => {
      const key = e.grade_id != null ? String(e.grade_id) : null;
      if (key && empByGrade.has(key)) empByGrade.get(key).push(e); else unassigned.push(e);
    });
    // Sort grades by name/code
    const gradesSorted = (state.grades||[]).slice().sort((a,b)=>{
      const an = (a.name||a.code||'').toLowerCase();
      const bn = (b.name||b.code||'').toLowerCase();
      return an.localeCompare(bn);
    });
    let html = '';
    gradesSorted.forEach(g => {
      const list = empByGrade.get(String(g.id)) || [];
      const title = esc(g.name||g.code||('#'+g.id)) + (g.code? ` <span class="text-muted">(${esc(g.code)})</span>` : '');
      const rows = list.length ? `<ul class="list-group list-group-flush">${list.map(e=>`<li class="list-group-item p-2">${esc(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())} <span class="text-muted">${esc(e.employee_number||'')}</span></li>`).join('')}</ul>` : '<div class="p-3 text-muted">No employees</div>';
      html += `<div class="card mb-3">
        <div class="card-header d-flex justify-content-between align-items-center">
          <strong>${title}</strong>
          <span class="badge bg-primary">${list.length}</span>
        </div>
        ${rows}
      </div>`;
    });
    const showUn = container.querySelector('#showUnassignedGrades');
    if (showUn?.checked) {
      html += `<div class="card mb-3 border-warning">
        <div class="card-header d-flex justify-content-between align-items-center">
          <strong>Unassigned</strong>
          <span class="badge bg-warning text-dark">${unassigned.length}</span>
        </div>
        ${unassigned.length ? `<ul class="list-group list-group-flush">${unassigned.map(e=>`<li class="list-group-item p-2">${esc(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())}</li>`).join('')}</ul>` : '<div class="p-3 text-muted">None</div>'}
      </div>`;
    }
    content.innerHTML = html || '<div class="text-muted">No grades</div>';
  }

  function initDistributionTab(container){
    const reloadBtn = container.querySelector('#distReloadGrades');
    const showUn = container.querySelector('#showUnassignedGrades');
    const refresh = async ()=>{ try { await fetchEmployees(); } catch(_){} renderDistribution(container); };
    reloadBtn?.addEventListener('click', refresh);
    showUn?.addEventListener('change', ()=> renderDistribution(container));
    // initial
    renderDistribution(container);
  }

  async function render(container){
    const root = document.createElement('div');
    root.id = 'gradesPageRoot';
    root.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Grades</h5>
          ${canEdit()? '<button class="btn btn-sm btn-primary" id="addGradeBtn"><i class="fas fa-plus"></i> Add Grade</button>' : ''}
        </div>
        <div class="card-body">
          <ul class="nav nav-tabs" role="tablist">
            <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#tab-grades" type="button" role="tab">Grades</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-bulk" type="button" role="tab">Bulk Mapping</button></li>
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-dist" type="button" role="tab">Distribution</button></li>
          </ul>
          <div class="tab-content mt-3">
            <div class="tab-pane fade show active" id="tab-grades" role="tabpanel">
              <div class="table-responsive">
                <table class="table table-sm">
                  <thead><tr><th>Name</th><th>Code</th><th>Base Salary</th><th style="width:220px">Actions</th></tr></thead>
                  <tbody id="gradesTableBody"></tbody>
                </table>
              </div>
            </div>
            <div class="tab-pane fade" id="tab-bulk" role="tabpanel">
              <div class="d-flex justify-content-end mb-2">
                <button class="btn btn-sm btn-primary" id="saveBulkMap">Save Mapping</button>
              </div>
              <div class="table-responsive">
                <table class="table table-sm">
                  <thead><tr><th>#</th><th>Name</th><th>Department</th><th>Position</th><th>Grade</th></tr></thead>
                  <tbody id="bulkTableBody"></tbody>
                </table>
              </div>
            </div>
            <div class="tab-pane fade" id="tab-dist" role="tabpanel">
              <div class="d-flex justify-content-between align-items-center mb-2">
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" id="showUnassignedGrades" />
                  <label class="form-check-label" for="showUnassignedGrades">Show unassigned</label>
                </div>
                <button class="btn btn-sm btn-outline-secondary" id="distReloadGrades"><i class="fas fa-rotate"></i> Reload</button>
              </div>
              <div id="distContentGrades">Loading...</div>
            </div>
          </div>
        </div>
      </div>`;
    container.innerHTML=''; container.appendChild(root);

    await fetchSetupLists();
    await fetchGrades();
    renderGradesTable(root);
    attachGradesTableEvents(root);

    await fetchEmployees();
    renderBulkMapping(root);
    attachBulkEvents(root);
    initDistributionTab(root);
  }

  window.pages['grades'] = { render };
})();
