// Expenses / Imprest / Reimbursement / Overtime / Per-diem module
(function(){
  window.pages = window.pages || {};

  const esc = s=>{ try{ return String(s).replace(/[&<>"']/g,c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); }catch(_){ return String(s||''); } };
  const notify = (m,t)=>{ try{ window.auth?.showNotification?.(m,t||'info'); }catch(_){} };
  const role = ()=> (window.auth?.currentUser?.role_slug) || (window.auth?.currentUser?.role) || 'employee';
  const canApprove = ()=> ['manager','hr_head','admin','super_admin'].includes(role());

  async function uploadFilesForExpense(expenseId, files){
    if (!files || !files.length) return;
    for (let i=0;i<files.length;i++){
      const fd = new FormData(); fd.append('expense_id', String(expenseId)); fd.append('file', files[i]);
      await fetch('api/expenses.php?action=upload_file', { method:'POST', body: fd, credentials:'same-origin' });
    }
  }

  window.pages.expenses = {
    async render(container){
      container.innerHTML = `
        <ul class="nav nav-tabs" role="tablist">
          <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#exp-submit" type="button" role="tab"><i class="fas fa-file-circle-plus"></i> Submit</button></li>
          <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#exp-mine" type="button" role="tab"><i class="fas fa-list"></i> My Requests</button></li>
          ${canApprove()?`<li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#exp-approvals" type="button" role="tab"><i class="fas fa-check-double"></i> Approvals</button></li>`:''}
        </ul>
        <div class="tab-content mt-3">
          <div class="tab-pane fade show active" id="exp-submit" role="tabpanel">
            <div class="card">
              <div class="card-header"><h6 class="mb-0">Submit Request</h6></div>
              <div class="card-body">
                <div class="row g-3">
                  <div class="col-md-3">
                    <label class="form-label">Type</label>
                    <select id="expType" class="form-select">
                      <option value="reimbursement">Reimbursement</option>
                      <option value="imprest">Imprest</option>
                      <option value="per_diem">Per-diem</option>
                      <option value="overtime">Overtime</option>
                      <option value="other">Other</option>
                    </select>
                  </div>
                  <div class="col-md-3">
                    <label class="form-label">Date</label>
                    <input id="expDate" type="date" class="form-control" />
                  </div>
                  <div class="col-md-3" id="amountWrap">
                    <label class="form-label">Amount</label>
                    <input id="expAmount" type="number" step="0.01" class="form-control" placeholder="0.00" />
                  </div>
                  <div class="col-md-3" id="currWrap">
                    <label class="form-label">Currency</label>
                    <input id="expCurrency" class="form-control" value="USD" />
                  </div>
                  <div class="col-md-3 d-none" id="hoursWrap">
                    <label class="form-label">Hours (overtime)</label>
                    <input id="expHours" type="number" step="0.25" class="form-control" placeholder="0" />
                  </div>
                  <div class="col-12">
                    <label class="form-label">Description</label>
                    <textarea id="expDesc" rows="3" class="form-control" placeholder="Describe the request (purpose, trip, task, etc.)"></textarea>
                  </div>
                  <div class="col-md-6">
                    <label class="form-label">Receipts/Attachments</label>
                    <input id="expFiles" type="file" class="form-control" multiple accept=".pdf,.doc,.docx,.png,.jpg,.jpeg,.txt,.xls,.xlsx" />
                    <div class="form-text">You can attach receipts; they will be uploaded after submission.</div>
                  </div>
                  <div class="col-12 text-end">
                    <button class="btn btn-primary" id="expSubmitBtn"><i class="fas fa-paper-plane"></i> Submit</button>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div class="tab-pane fade" id="exp-mine" role="tabpanel">
            <div class="card">
              <div class="card-header d-flex justify-content-between align-items-center">
                <h6 class="mb-0">Requests</h6>
                <div class="d-flex gap-2">
                  <select id="mineStatus" class="form-select form-select-sm" style="min-width:140px">
                    <option value="">All</option>
                    <option value="pending">pending</option>
                    <option value="approved">approved</option>
                    <option value="rejected">rejected</option>
                  </select>
                  <select id="mineType" class="form-select form-select-sm" style="min-width:160px">
                    <option value="">All types</option>
                    <option value="reimbursement">Reimbursement</option>
                    <option value="imprest">Imprest</option>
                    <option value="per_diem">Per-diem</option>
                    <option value="overtime">Overtime</option>
                    <option value="other">Other</option>
                  </select>
                  <button class="btn btn-sm btn-outline-primary" id="mineApply">Apply</button>
                </div>
              </div>
              <div class="card-body">
                <div class="table-responsive">
                  <table class="table table-sm align-middle">
                    <thead><tr><th>Type</th><th>Date</th><th class="text-end">Amount</th><th>Status</th><th>Description</th><th>Files</th><th></th></tr></thead>
                    <tbody id="mineTBody"><tr><td colspan="7" class="text-muted">Loading...</td></tr></tbody>
                  </table>
                </div>
              </div>
            </div>
          </div>
          ${canApprove()?`
          <div class="tab-pane fade" id="exp-approvals" role="tabpanel">
            <div class="card">
              <div class="card-header d-flex justify-content-between align-items-center">
                <h6 class="mb-0">Pending Approvals</h6>
                <div><button class="btn btn-sm btn-outline-secondary" id="apRefresh"><i class="fas fa-rotate"></i></button></div>
              </div>
              <div class="card-body">
                <div class="table-responsive">
                  <table class="table table-sm align-middle">
                    <thead><tr><th>Employee</th><th>Type</th><th>Date</th><th class="text-end">Amount/Hours</th><th>Description</th><th>Files</th><th></th></tr></thead>
                    <tbody id="apTBody"><tr><td colspan="7" class="text-muted">Loading...</td></tr></tbody>
                  </table>
                </div>
              </div>
            </div>
          </div>`:''}
        </div>
      `;

      // defaults
      const today = new Date(); const y = today.getFullYear(); const m = String(today.getMonth()+1).padStart(2,'0'); const d = String(today.getDate()).padStart(2,'0');
      const dateInp = container.querySelector('#expDate'); if (dateInp) dateInp.value = `${y}-${m}-${d}`;

      // type toggles
      const typeSel = container.querySelector('#expType');
      const hoursWrap = container.querySelector('#hoursWrap');
      const amtWrap = container.querySelector('#amountWrap');
      const currWrap = container.querySelector('#currWrap');
      function syncFields(){
        const t = typeSel.value;
        if (t === 'overtime') { hoursWrap.classList.remove('d-none'); amtWrap.classList.add('d-none'); currWrap.classList.add('d-none'); }
        else { hoursWrap.classList.add('d-none'); amtWrap.classList.remove('d-none'); currWrap.classList.remove('d-none'); }
      }
      typeSel.addEventListener('change', syncFields); syncFields();

      // submit handler
      container.querySelector('#expSubmitBtn')?.addEventListener('click', async ()=>{
        const t = typeSel.value;
        const payload = {
          type: t,
          request_date: container.querySelector('#expDate').value || null,
          description: container.querySelector('#expDesc').value || null,
          amount: t==='overtime'? null : Number(container.querySelector('#expAmount').value || 0),
          hours: t==='overtime'? Number(container.querySelector('#expHours').value || 0) : null,
          currency: t==='overtime'? null : (container.querySelector('#expCurrency').value || 'USD')
        };
        try{
          const btn = container.querySelector('#expSubmitBtn'); const orig = btn.innerHTML; btn.disabled = true; btn.innerHTML = '<span class="spinner"></span> Submitting...';
          const r = await fetch('api/expenses.php?action=create', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
          const j = await r.json();
          if (j?.success){
            const files = container.querySelector('#expFiles')?.files;
            await uploadFilesForExpense(j.data?.id, files);
            notify('Request submitted','success');
            // reset form
            container.querySelector('#expDesc').value = '';
            container.querySelector('#expAmount').value = '';
            container.querySelector('#expHours').value = '';
            container.querySelector('#expFiles').value = '';
            loadMine(container);
          } else {
            notify(j?.message || 'Submit failed','error');
          }
          btn.disabled = false; btn.innerHTML = orig;
        }catch(e){ notify('Submit failed','error'); }
      });

      // mine list & approvals
      container.querySelector('#mineApply')?.addEventListener('click', ()=>loadMine(container));
      if (canApprove()) container.querySelector('#apRefresh')?.addEventListener('click', ()=>loadApprovals(container));

      // initial loads
      loadMine(container);
      if (canApprove()) loadApprovals(container);
    }
  };

  async function loadMine(container){
    const tb = container.querySelector('#mineTBody'); if (!tb) return; tb.innerHTML = '<tr><td colspan="7" class="text-muted">Loading...</td></tr>';
    try{
      const status = container.querySelector('#mineStatus')?.value || '';
      const type = container.querySelector('#mineType')?.value || '';
      const qs = new URLSearchParams(); if (status) qs.set('status', status); if (type) qs.set('type', type);
      const r = await fetch('api/expenses.php' + (qs.toString()?('?'+qs.toString()):''), { credentials:'same-origin' });
      const j = await r.json(); const rows = j?.success ? (j.data||[]) : [];
      tb.innerHTML = rows.length ? rows.map(row=>{
        const amt = row.type==='overtime' ? (row.hours||0)+' h' : (row.amount? Number(row.amount).toFixed(2)+' '+(row.currency||'') : '-');
        const files = (row.attachments||[]).map(p=>`<a href="${esc(p)}" target="_blank">file</a>`).join(' ');
        const attach = `<label class="btn btn-sm btn-outline-secondary mb-0">Attach<input type="file" class="d-none exp-file-input" data-exp="${row.id}" accept=".pdf,.doc,.docx,.png,.jpg,.jpeg,.txt,.xls,.xlsx"/></label>`;
        return `<tr>
          <td class="text-capitalize">${esc(row.type)}</td>
          <td>${esc(row.request_date||'')}</td>
          <td class="text-end">${esc(amt)}</td>
          <td><span class="badge ${row.status==='approved'?'bg-success':(row.status==='rejected'?'bg-danger':'bg-secondary')}">${esc(row.status)}</span></td>
          <td style="max-width:280px">${esc(row.description||'')}</td>
          <td>${files}</td>
          <td class="text-end">${attach}</td>
        </tr>`;
      }).join('') : '<tr><td colspan="7" class="text-muted">No requests found.</td></tr>';

      // bind file uploads
      tb.querySelectorAll('.exp-file-input')?.forEach(inp=>{
        if (inp._b) return; inp._b = true;
        inp.addEventListener('change', async ()=>{
          const id = Number(inp.getAttribute('data-exp')); const f = inp.files && inp.files[0]; if (!id || !f) return;
          const fd = new FormData(); fd.append('expense_id', String(id)); fd.append('file', f);
          try{ await fetch('api/expenses.php?action=upload_file', { method:'POST', body: fd, credentials:'same-origin' }); loadMine(container); }catch(_){ notify('Upload failed','error'); }
        });
      });
    }catch(_){ tb.innerHTML = '<tr><td colspan="7" class="text-danger">Failed to load.</td></tr>'; }
  }

  async function loadApprovals(container){
    const tb = container.querySelector('#apTBody'); if (!tb) return; tb.innerHTML = '<tr><td colspan="7" class="text-muted">Loading...</td></tr>';
    try{
      const r = await fetch('api/expenses.php?status=pending&for_approval=1', { credentials:'same-origin' });
      const j = await r.json(); const rows = j?.success ? (j.data||[]) : [];
      tb.innerHTML = rows.length ? rows.map(row=>{
        const amt = row.type==='overtime' ? (row.hours||0)+' h' : (row.amount? Number(row.amount).toFixed(2)+' '+(row.currency||'') : '-');
        const files = (row.attachments||[]).map(p=>`<a href="${esc(p)}" target="_blank">file</a>`).join(' ');
        const actions = `
          <div class="d-flex gap-1">
            <input type="text" class="form-control form-control-sm" placeholder="Notes / reason" data-ap-notes="${row.id}" style="max-width:220px"/>
            <button class="btn btn-sm btn-outline-success" data-ap-approve="${row.id}"><i class="fas fa-check"></i></button>
            <button class="btn btn-sm btn-outline-danger" data-ap-reject="${row.id}"><i class="fas fa-xmark"></i></button>
          </div>`;
        return `<tr>
          <td>${esc(row.employee_name||'')}</td>
          <td class="text-capitalize">${esc(row.type)}</td>
          <td>${esc(row.request_date||'')}</td>
          <td class="text-end">${esc(amt)}</td>
          <td style="max-width:260px">${esc(row.description||'')}</td>
          <td>${files}</td>
          <td class="text-end">${actions}</td>
        </tr>`;
      }).join('') : '<tr><td colspan="7" class="text-muted">No pending approvals.</td></tr>';

      // bind actions
      tb.querySelectorAll('[data-ap-approve]')?.forEach(btn=>{
        if (btn._b) return; btn._b = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-ap-approve')); const notes = tb.querySelector(`[data-ap-notes="${id}"]`)?.value || null;
          try{
            const r = await fetch(`api/expenses.php?action=approve&id=${id}`, { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ notes }) });
            const j = await r.json(); if (j?.success){ notify('Approved','success'); loadApprovals(container); } else notify(j?.message||'Approve failed','error');
          }catch(_){ notify('Approve failed','error'); }
        });
      });
      tb.querySelectorAll('[data-ap-reject]')?.forEach(btn=>{
        if (btn._b) return; btn._b = true;
        btn.addEventListener('click', async ()=>{
          const id = Number(btn.getAttribute('data-ap-reject')); const reason = tb.querySelector(`[data-ap-notes="${id}"]`)?.value || '';
          if (!reason){ notify('Enter rejection reason','warning'); return; }
          try{
            const r = await fetch(`api/expenses.php?action=reject&id=${id}`, { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ rejection_reason: reason }) });
            const j = await r.json(); if (j?.success){ notify('Rejected','info'); loadApprovals(container); } else notify(j?.message||'Reject failed','error');
          }catch(_){ notify('Reject failed','error'); }
        });
      });
    }catch(_){ tb.innerHTML = '<tr><td colspan="7" class="text-danger">Failed to load.</td></tr>'; }
  }
})();
