// Employees page module
(function () {
  window.pages = window.pages || {};

  function canEditEmployees() {
    try {
      return window.auth?.canEdit('employees');
    } catch (_) {
      return false;
    }
  }

  // Column chooser configuration
  const COLUMN_PREFS_KEY = 'employees_table_columns_v1';
  const ALL_COLUMNS = [
    {
      key: 'employee_number',
      label: '#',
      defaultVisible: true,
      render: (emp) => escapeHtml(emp.employee_number || '-')
    },
    {
      key: 'full_name',
      label: 'Name',
      defaultVisible: true,
      render: (emp) => escapeHtml(emp.full_name || `${emp.first_name || ''} ${emp.last_name || ''}`.trim())
    },
    {
      key: 'department_name',
      label: 'Department',
      defaultVisible: true,
      render: (emp) => escapeHtml(emp.department_name || '-')
    },
    {
      key: 'position_title',
      label: 'Position',
      defaultVisible: true,
      render: (emp) => escapeHtml(emp.position_title || '-')
    },
    {
      key: 'hire_date',
      label: 'Hire Date',
      defaultVisible: true,
      render: (emp) => escapeHtml(emp.hire_date || '')
    },
    {
      key: 'email',
      label: 'Email',
      defaultVisible: true,
      render: (emp) => escapeHtml(emp.email || '-')
    },
    {
      key: 'phone',
      label: 'Phone',
      defaultVisible: true,
      render: (emp) => escapeHtml(emp.phone || '-')
    },
    {
      key: 'employment_status',
      label: 'Status',
      defaultVisible: true,
      render: (emp) => statusBadge(emp.employment_status || 'active')
    },
    {
      key: 'branch_name',
      label: 'Branch',
      defaultVisible: false,
      render: (emp) => escapeHtml(emp.branch_name || '-')
    },
    {
      key: 'manager_full_name',
      label: 'Manager',
      defaultVisible: false,
      render: (emp) => escapeHtml(emp.manager_full_name || '-')
    }
  ];

  function getColumnPrefs() {
    try {
      const raw = localStorage.getItem(COLUMN_PREFS_KEY);
      if (!raw) return null;
      const obj = JSON.parse(raw);
      return obj && typeof obj === 'object' ? obj : null;
    } catch (_) { return null; }
  }

  function saveColumnPrefs(prefs) {
    try { localStorage.setItem(COLUMN_PREFS_KEY, JSON.stringify(prefs)); } catch (_) {}
  }

  function getVisibleColumns() {
    const prefs = getColumnPrefs();
    return ALL_COLUMNS.filter(col => {
      const v = prefs && Object.prototype.hasOwnProperty.call(prefs, col.key) ? !!prefs[col.key] : col.defaultVisible;
      return v;
    });
  }

  function setColumnVisibility(key, visible) {
    const prefs = getColumnPrefs() || {};
    prefs[key] = !!visible;
    // Ensure at least one column remains visible
    const countVisible = ALL_COLUMNS.filter(c => (c.key === key ? !!visible : (prefs.hasOwnProperty(c.key) ? !!prefs[c.key] : c.defaultVisible))).length;
    if (countVisible === 0) return; // do not allow hide all
    saveColumnPrefs(prefs);
  }

  function buildColumnChooserMenu(container) {
    const menu = container.querySelector('#colChooserMenu');
    if (!menu) return;
    const prefs = getColumnPrefs() || {};
    menu.innerHTML = ALL_COLUMNS.map(col => {
      const checked = (prefs.hasOwnProperty(col.key) ? !!prefs[col.key] : col.defaultVisible) ? 'checked' : '';
      return `
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="col-${col.key}" data-col-key="${col.key}" ${checked} />
          <label class="form-check-label" for="col-${col.key}">${escapeHtml(col.label)}</label>
        </div>`;
    }).join('');

    menu.querySelectorAll('[data-col-key]').forEach(input => {
      input.addEventListener('change', () => {
        const key = input.getAttribute('data-col-key');
        const wantVisible = input.checked;
        // Prevent unchecking the last visible column
        const visCols = getVisibleColumns();
        if (!wantVisible && visCols.length <= 1) {
          input.checked = true; return;
        }
        setColumnVisibility(key, wantVisible);
        renderTableHead(container);
        loadEmployees(container);
      });
    });
  }

  function renderTableHead(container) {
    const headRow = container.querySelector('#empTableHead');
    if (!headRow) return;
    const cols = getVisibleColumns();
    const headerHtml = cols.map(c => `<th>${escapeHtml(c.label)}</th>`).join('') + `<th style="width: 200px;">Actions</th>`;
    headRow.innerHTML = headerHtml;
  }

  // Stubs will be replaced with full implementations in subsequent updates
  async function prepareCreateModal() {
    const form = document.getElementById('employeeModalForm');
    if (form) form.reset();
    try { await Promise.all([loadOnboardingListsIfNeeded(), loadJobOptions()]); } catch (_) {}
    const docList = document.getElementById('docList'); if (docList) docList.innerHTML = '';
    const skillList = document.getElementById('skillList'); if (skillList) skillList.innerHTML = '';
    const histList = document.getElementById('histList'); if (histList) histList.innerHTML = '';
    const certList = document.getElementById('certList'); if (certList) certList.innerHTML = '';
    // Add an initial row for each section for UX
    addDocRow();
    addSkillRow();
    addHistRow();
    addCertRow();
    // Wire add buttons (avoid duplicate listeners by setting onclick)
    const addDocBtn = document.getElementById('addDocRow'); if (addDocBtn) addDocBtn.onclick = ()=> addDocRow();
    const addSkillBtn = document.getElementById('addSkillRow'); if (addSkillBtn) addSkillBtn.onclick = ()=> addSkillRow();
    const addHistBtn = document.getElementById('addHistRow'); if (addHistBtn) addHistBtn.onclick = ()=> addHistRow();
    const addCertBtn = document.getElementById('addCertRow'); if (addCertBtn) addCertBtn.onclick = ()=> addCertRow();
    const saveBtn = document.getElementById('saveEmployeeBtn');
    if (saveBtn) { saveBtn.textContent = 'Create Employee'; saveBtn.dataset.mode = 'create'; saveBtn.dataset.id = ''; }
    const modalEl = document.getElementById('employeeModal');
    if (modalEl) { const t = modalEl.querySelector('.modal-title'); if (t) t.textContent = 'Create Employee'; }
    const idFile = document.getElementById('idImageFile'); if (idFile) idFile.value = '';
    const idImg = document.getElementById('idImagePreview'); if (idImg) { idImg.src = ''; idImg.style.display = 'none'; }
  }

  async function startEditEmployee(id) {
    const res = await fetch(`api/employees.php?id=${encodeURIComponent(id)}`, { credentials: 'same-origin' });
    const data = await res.json();
    if (!data?.success) throw new Error('not found');
    const emp = data.data;
    try { await Promise.all([loadOnboardingListsIfNeeded(), loadJobOptions()]); } catch (_) {}
    const modalEl = document.getElementById('employeeModal');
    const form = document.getElementById('employeeModalForm');
    if (!form || !modalEl) return;
    form.reset();
    // Fill scalar inputs
    Array.from(form.querySelectorAll('[name]')).forEach((el) => {
      const n = el.getAttribute('name');
      if (!(n in emp)) return;
      let v = emp[n];
      if (el.type === 'datetime-local' && typeof v === 'string') v = v.replace(' ', 'T').slice(0,16);
      el.value = v ?? '';
    });
    // Populate dynamic sections
    ['docList','skillList','histList','certList'].forEach(pid => { const el = document.getElementById(pid); if (el) el.innerHTML=''; });
    (Array.isArray(emp.documents) && emp.documents.length ? emp.documents : [{}]).forEach(d => addDocRow(d));
    (Array.isArray(emp.skills) && emp.skills.length ? emp.skills : [{}]).forEach(s => addSkillRow(s));
    (Array.isArray(emp.employment_history) && emp.employment_history.length ? emp.employment_history : [{}]).forEach(h => addHistRow(h));
    (Array.isArray(emp.certificates) && emp.certificates.length ? emp.certificates : [{}]).forEach(c => addCertRow(c));
    // Wire add-row buttons
    document.getElementById('addDocRow')?.addEventListener('click', ()=> addDocRow());
    document.getElementById('addSkillRow')?.addEventListener('click', ()=> addSkillRow());
    document.getElementById('addHistRow')?.addEventListener('click', ()=> addHistRow());
    document.getElementById('addCertRow')?.addEventListener('click', ()=> addCertRow());
    // Title & save button
    const titleEl = modalEl.querySelector('.modal-title'); if (titleEl) titleEl.textContent = 'Edit Employee';
    const saveBtn = document.getElementById('saveEmployeeBtn');
    if (saveBtn) { saveBtn.dataset.mode = 'edit'; saveBtn.dataset.id = String(id); saveBtn.textContent = 'Update Employee'; }
    // Sync top department selector with job tab selector
    try {
      const depSel = document.getElementById('empDepartment');
      const depTop = document.getElementById('empDepartmentTop');
      if (depSel && depTop) { depTop.value = depSel.value; }
      const brSel = document.getElementById('empBranch');
      const brTop = document.getElementById('empBranchTop');
      if (brSel && brTop) { brTop.value = brSel.value; }
    } catch(_){}
    // Show ID image preview if exists
    try {
      const img = document.getElementById('idImagePreview');
      const pic = emp.profile_pic || emp.photo; // prefer profile_pic if present, else photo
      if (img && pic) { img.src = pic; img.style.display = 'inline-block'; }
    } catch(_){}
    bootstrap.Modal.getOrCreateInstance(modalEl).show();
  }

  async function openProfileModal(id) {
    const res = await fetch(`api/employees.php?id=${encodeURIComponent(id)}`, { credentials: 'same-origin' });
    const data = await res.json();
    if (!data?.success) throw new Error('not found');
    const emp = data.data;
    // Ensure lists (including grades) are loaded for controls
    try { await loadOnboardingListsIfNeeded(); } catch(_) {}
    const body = document.getElementById('employeeProfileBody');
    if (!body) return;
    const fullName = emp.full_name || `${emp.first_name || ''} ${emp.last_name || ''}`.trim();
    const canEditPay = !!canEditEmployees();
    const docs = Array.isArray(emp.documents) ? emp.documents : [];
    const skills = Array.isArray(emp.skills) ? emp.skills : [];
    const hist = Array.isArray(emp.employment_history) ? emp.employment_history : [];
    const certs = Array.isArray(emp.certificates) ? emp.certificates : [];
    const isImgMime = (m) => typeof m === 'string' && m.startsWith('image/');
    const isImgPath = (p) => /\.(png|jpe?g|gif|webp|bmp)$/i.test(p || '');
    const docsHtml = docs.length ? `<ul class="list-group list-group-flush">${docs.map(d=>{
      const img = (isImgMime(d.mime_type) || isImgPath(d.file_path)) && d.file_path ? `<img src="${escapeHtml(d.file_path)}" alt="" style="height:40px;width:auto;margin-right:8px;border-radius:4px;border:1px solid #eee;" />` : '';
      const link = d.file_path ? `<a href="${escapeHtml(d.file_path)}" target="_blank" class="ms-2">view</a>` : '';
      return `<li class="list-group-item p-2 d-flex align-items-center">${img}<div><div>${escapeHtml(d.document_type||'')} — ${escapeHtml(d.title||'')}</div>${link}</div></li>`;
    }).join('')}</ul>` : '<div class="text-muted">No documents</div>';
    const certsHtml = certs.length ? `<ul class="list-group list-group-flush">${certs.map(c=>{
      const img = isImgPath(c.file_path) && c.file_path ? `<img src="${escapeHtml(c.file_path)}" alt="" style="height:40px;width:auto;margin-right:8px;border-radius:4px;border:1px solid #eee;" />` : '';
      const link = c.file_path ? `<a href="${escapeHtml(c.file_path)}" target="_blank" class="ms-2">view</a>` : '';
      return `<li class="list-group-item p-2 d-flex align-items-center">${img}<div><div>${escapeHtml(c.certificate_name||('#'+c.certificate_id))} — ${escapeHtml(c.status||'')}</div>${link}</div></li>`;
    }).join('')}</ul>` : '<div class="text-muted">No certificates</div>';
    const managerFullName = `${emp.manager_first_name || ''} ${emp.manager_last_name || ''}`.trim();
    body.innerHTML = `
      <ul class="nav nav-tabs" role="tablist">
        <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#pv-personal" type="button" role="tab">Personal</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-contact" type="button" role="tab">Contact & Address</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-ids" type="button" role="tab">IDs</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-education" type="button" role="tab">Education</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-job" type="button" role="tab">Job</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-docs" type="button" role="tab">Documents (${docs.length})</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-skills" type="button" role="tab">Skills (${skills.length})</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-history" type="button" role="tab">History (${hist.length})</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-certs" type="button" role="tab">Certificates (${certs.length})</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-nok" type="button" role="tab">Next of Kin</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-payment" type="button" role="tab">Payment Details</button></li>
        <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#pv-payroll" type="button" role="tab">Payroll Components</button></li>
      </ul>
      <div class="tab-content mt-3">
        <div class="tab-pane fade show active" id="pv-personal" role="tabpanel">
          <div class="row g-2">
            <div class="col-md-3"><p class="mb-1"><strong>Salutation:</strong> ${escapeHtml(emp.salutation || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>First Name:</strong> ${escapeHtml(emp.first_name || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Middle Name:</strong> ${escapeHtml(emp.middle_name || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Last Name:</strong> ${escapeHtml(emp.last_name || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Employee #:</strong> ${escapeHtml(emp.employee_number || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Hire Date:</strong> ${escapeHtml(emp.hire_date || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Birth Date:</strong> ${escapeHtml(emp.date_of_birth || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Gender:</strong> ${escapeHtml(emp.gender || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Marital Status:</strong> ${escapeHtml(emp.marital_status || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Nationality:</strong> ${escapeHtml(emp.nationality || '')}</p></div>
            <div class="col-md-3"><p class="mb-1"><strong>Official Title:</strong> ${escapeHtml(emp.title || '')}</p></div>
          </div>
        </div>
        <div class="tab-pane fade" id="pv-contact" role="tabpanel">
          <div class="row g-2">
            <div class="col-md-4"><p class="mb-1"><strong>Email:</strong> ${escapeHtml(emp.email || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Official Email:</strong> ${escapeHtml(emp.official_email || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Phone:</strong> ${escapeHtml(emp.phone || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Mobile:</strong> ${escapeHtml(emp.mobile || '')}</p></div>
            <div class="col-md-8"><p class="mb-1"><strong>Address:</strong> ${escapeHtml(emp.address || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Digital Address:</strong> ${escapeHtml(emp.digital_address || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Postal Address:</strong> ${escapeHtml(emp.postal_address || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>House #:</strong> ${escapeHtml(emp.house_number || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Landmark:</strong> ${escapeHtml(emp.land_mark || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Region:</strong> ${escapeHtml(emp.region || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Religion:</strong> ${escapeHtml(emp.religion || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Location:</strong> ${escapeHtml(emp.location || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Staff Location:</strong> ${escapeHtml(emp.staff_location || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Residential Ownership:</strong> ${escapeHtml(emp.residential_ownership || '')}</p></div>
          </div>
        </div>
        <div class="tab-pane fade" id="pv-ids" role="tabpanel">
          <div class="row g-2">
            <div class="col-md-4"><p class="mb-1"><strong>ID Type:</strong> ${escapeHtml(emp.id_type || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>ID Number:</strong> ${escapeHtml(emp.id_number || '')}</p></div>
            <div class="col-md-4">
              <p class="mb-1"><strong>ID Image:</strong></p>
              ${emp.photo ? `<img src="${escapeHtml(emp.photo)}" alt="ID" style="max-height:80px;border:1px solid #eee;border-radius:4px;" />` : '<span class="text-muted">None</span>'}
            </div>
          </div>
        </div>
        <div class="tab-pane fade" id="pv-education" role="tabpanel">
          <div class="row g-2">
            <div class="col-md-4"><p class="mb-1"><strong>Certificate Type:</strong> ${escapeHtml(emp.certificate_type || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>First Degree Program:</strong> ${escapeHtml(emp.first_degree_program || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>First Degree Class:</strong> ${escapeHtml(emp.first_degree_class || '')}</p></div>
            <div class="col-md-6"><p class="mb-1"><strong>First Degree Institution:</strong> ${escapeHtml(emp.first_degree_institution || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Second Degree:</strong> ${escapeHtml(emp.second_degree || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Second Degree Program:</strong> ${escapeHtml(emp.second_degree_program || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Second Degree Class:</strong> ${escapeHtml(emp.second_degree_class || '')}</p></div>
            <div class="col-md-6"><p class="mb-1"><strong>Second Degree Institution:</strong> ${escapeHtml(emp.second_degree_institution || '')}</p></div>
            <div class="col-md-6"><p class="mb-1"><strong>Professional Certificate:</strong> ${escapeHtml(emp.professional_certificate || '')}</p></div>
            <div class="col-md-6"><p class="mb-1"><strong>Professional Institution:</strong> ${escapeHtml(emp.professional_institution || '')}</p></div>
            <div class="col-md-6"><p class="mb-1"><strong>Professional Results:</strong> ${escapeHtml(emp.professional_results || '')}</p></div>
            <div class="col-12"><p class="mb-1"><strong>Academic Program:</strong> ${escapeHtml(emp.academic_program || '')}</p></div>
          </div>
        </div>
        <div class="tab-pane fade" id="pv-job" role="tabpanel">
          <div class="row g-2">
            <div class="col-md-4"><p class="mb-1"><strong>Department:</strong> ${escapeHtml(emp.department_name || '-')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Position:</strong> ${escapeHtml(emp.position_title || '-')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Branch:</strong> ${escapeHtml(emp.branch_name || '-')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Manager:</strong> ${escapeHtml(managerFullName || '-')}</p></div>
            <!-- Grade Mapping Control -->
            ${(()=>{
              const grades = Array.isArray(_lists.grades) ? _lists.grades : [];
              const sel = `<select class=\"form-select\" id=\"empGradeSelect\"><option value=\"\">-- Select Grade --</option>${grades.map(g=>`<option value="${g.id}" ${String(emp.grade_id||'')===String(g.id)?'selected':''}>${escapeHtml(g.name||g.code||('#'+g.id))}${g.code? ' ('+escapeHtml(g.code)+')':''}</option>`).join('')}</select>`;
              const cur = (function(){ const g = grades.find(x=>String(x.id)===String(emp.grade_id||'')); return g ? (g.name || g.code || ('#'+g.id)) : (emp.current_grade || emp.starting_grade || '-'); })();
              return canEditPay
                ? `<div class=\"col-md-6\"><label class=\"form-label\">Grade</label>${sel}</div><div class=\"col-md-6 text-end\"><button type=\"button\" class=\"btn btn-primary mt-4\" id=\"btnSaveEmpGrade\">Save Grade</button></div>`
                : `<div class=\"col-md-4\"><p class=\"mb-1\"><strong>Grade:</strong> ${escapeHtml(cur||'-')}</p></div>`;
            })()}
            <div class="col-md-4"><p class="mb-1"><strong>Contract Type:</strong> ${escapeHtml(emp.contract_type || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Status:</strong> ${escapeHtml(emp.employment_status || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Salary:</strong> ${escapeHtml(emp.salary || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Currency:</strong> ${escapeHtml(emp.currency || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Probation End:</strong> ${escapeHtml(emp.probation_end_date || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Department Group:</strong> ${escapeHtml(emp.department_group || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Department Unit:</strong> ${escapeHtml(emp.department_unit || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Branch Type:</strong> ${escapeHtml(emp.branch_type || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Starting Grade:</strong> ${escapeHtml(emp.starting_grade || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Current Grade:</strong> ${escapeHtml(emp.current_grade || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Joined Dept Date:</strong> ${escapeHtml(emp.joined_department_date || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Last Promotion Date:</strong> ${escapeHtml(emp.last_promotion_date || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Confirmation:</strong> ${escapeHtml(emp.confirmation || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Service Length:</strong> ${escapeHtml(emp.service_length || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Staff Status:</strong> ${escapeHtml(emp.staff_status || '')}</p></div>
          </div>
        </div>
        <div class="tab-pane fade" id="pv-docs" role="tabpanel">
          ${docsHtml}
        </div>
        <div class="tab-pane fade" id="pv-skills" role="tabpanel">
          ${skills.length ? `<ul class="list-group list-group-flush">${skills.map(s=>`<li class="list-group-item p-2">${escapeHtml(s.skill_name||('#'+s.skill_id))} — ${escapeHtml(s.proficiency_level||'')}</li>`).join('')}</ul>` : '<div class="text-muted">No skills</div>'}
        </div>
        <div class="tab-pane fade" id="pv-history" role="tabpanel">
          ${hist.length ? `<ul class="list-group list-group-flush">${hist.map(h=>`<li class="list-group-item p-2">${escapeHtml(h.start_date||'')} → ${escapeHtml(h.end_date||'')}</li>`).join('')}</ul>` : '<div class="text-muted">No history</div>'}
        </div>
        <div class="tab-pane fade" id="pv-certs" role="tabpanel">
          ${certsHtml}
        </div>
        <div class="tab-pane fade" id="pv-nok" role="tabpanel">
          <div class="row g-2">
            <div class="col-md-6"><p class="mb-1"><strong>NOK Name:</strong> ${escapeHtml(emp.nok_name || '')}</p></div>
            <div class="col-md-6"><p class="mb-1"><strong>Relationship:</strong> ${escapeHtml(emp.relationship || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>NOK Contact:</strong> ${escapeHtml(emp.nok_contact || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>NOK Postal Address:</strong> ${escapeHtml(emp.nok_postal_address || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>NOK Home Address:</strong> ${escapeHtml(emp.nok_home_address || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>NOK Date:</strong> ${escapeHtml(emp.nok_date || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Emergency Contact Name:</strong> ${escapeHtml(emp.emergency_contact_name || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Emergency Contact Phone:</strong> ${escapeHtml(emp.emergency_contact_phone || '')}</p></div>
            <div class="col-md-4"><p class="mb-1"><strong>Emergency Relationship:</strong> ${escapeHtml(emp.emergency_contact_relationship || '')}</p></div>
          </div>
        </div>
        <div class="tab-pane fade" id="pv-payment" role="tabpanel">
          <div class="row g-3">
            <div class="col-md-6">
              <label class="form-label">Bank Code</label>
              <input type="text" class="form-control" id="empBankCode" value="${escapeHtml(emp.bank_code||'')}" ${canEditPay? '':'disabled'} />
            </div>
            <div class="col-md-6">
              <label class="form-label">Bank Account Number</label>
              <input type="text" class="form-control" id="empBankAccount" value="${escapeHtml(emp.bank_account||'')}" ${canEditPay? '':'disabled'} />
            </div>
            <div class="col-md-6">
              <label class="form-label">MoMo Provider (Paystack code)</label>
              <input type="text" class="form-control" id="empMomoProvider" value="${escapeHtml(emp.momo_provider||'')}" ${canEditPay? '':'disabled'} />
            </div>
            <div class="col-md-6">
              <label class="form-label">MoMo Number</label>
              <input type="text" class="form-control" id="empMomoNumber" value="${escapeHtml(emp.momo_number||'')}" ${canEditPay? '':'disabled'} />
            </div>
            ${canEditPay ? `<div class="col-12 text-end"><button type="button" class="btn btn-primary" id="btnSavePaymentDefaults">Save</button></div>` : ''}
            <div class="col-12"><div class="form-text">These defaults will be used for Paystack payouts if not provided at payout time.</div></div>
          </div>
        </div>
        <div class="tab-pane fade" id="pv-payroll" role="tabpanel">
          <div class="row g-3">
            <div class="col-md-6">
              <div class="d-flex justify-content-between align-items-center mb-2">
                <h6 class="mb-0">Allowances</h6>
                <button type="button" class="btn btn-sm btn-outline-primary" id="empAddAllowance">Add</button>
              </div>
              <div class="table-responsive">
                <table class="table table-sm" id="empAllowancesTbl">
                  <thead><tr><th>Type</th><th>Mode</th><th>Amount</th><th>Rate %</th><th style="width:160px">Actions</th></tr></thead>
                  <tbody></tbody>
                </table>
              </div>
            </div>
            <div class="col-md-6">
              <div class="d-flex justify-content-between align-items-center mb-2">
                <h6 class="mb-0">Deductions</h6>
                <button type="button" class="btn btn-sm btn-outline-primary" id="empAddDeduction">Add</button>
              </div>
              <div class="table-responsive">
                <table class="table table-sm" id="empDeductionsTbl">
                  <thead><tr><th>Type</th><th>Mode</th><th>Amount</th><th>Rate %</th><th style="width:160px">Actions</th></tr></thead>
                  <tbody></tbody>
                </table>
              </div>
            </div>
          </div>
        </div>
      </div>`;
    const modalEl = document.getElementById('employeeProfileModal');
    if (modalEl) bootstrap.Modal.getOrCreateInstance(modalEl).show();
    // Initialize payroll components tab
    try { await initEmployeePayrollComponentsTab(emp.id); } catch(_) {}
    // Wire payment defaults save
    const savePayBtn = document.getElementById('btnSavePaymentDefaults');
    if (savePayBtn) {
      savePayBtn.addEventListener('click', async ()=>{
        const body = {
          bank_code: document.getElementById('empBankCode')?.value || '',
          bank_account: document.getElementById('empBankAccount')?.value || '',
          momo_provider: document.getElementById('empMomoProvider')?.value || '',
          momo_number: document.getElementById('empMomoNumber')?.value || ''
        };
        try {
          const res = await fetch(`api/employees.php?id=${encodeURIComponent(id)}`, { method:'PUT', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(body) });
          const data = await res.json();
          if (data?.success) { window.auth?.showNotification?.('Saved', 'success'); }
          else { window.auth?.showNotification?.(data?.message || 'Failed to save', 'error'); }
        } catch (_) { window.auth?.showNotification?.('Failed to save', 'error'); }
      });
    }
  }

  // Onboarding data helpers (cached lists for dynamic sections)
  const _lists = {
    document_types: ['contract','id_copy','passport','certificate','resume','other'],
    skills: [],
    proficiency_levels: ['beginner','intermediate','advanced','expert'],
    certificates: [],
    certificate_status: ['active','expired','revoked'],
    departments: [],
    positions: [],
    branches: []
  };

  async function loadOnboardingListsIfNeeded() {
    try {
      const res = await fetch('api/employees.php?action=lists', { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success) {
        if (Array.isArray(data.data?.document_types)) _lists.document_types = data.data.document_types;
        if (Array.isArray(data.data?.skills)) _lists.skills = data.data.skills;
        if (Array.isArray(data.data?.certificates)) _lists.certificates = data.data.certificates;
        if (Array.isArray(data.data?.proficiency_levels)) _lists.proficiency_levels = data.data.proficiency_levels;
        if (Array.isArray(data.data?.certificate_status)) _lists.certificate_status = data.data.certificate_status;
        if (Array.isArray(data.data?.grades)) _lists.grades = data.data.grades;
      }
    } catch (_) {}
    // Organization lists for history rows
    try {
      const [deps, poss, brs] = await Promise.all([
        fetch('api/organization.php?entity=departments', { credentials: 'same-origin' }).then(r=>r.json()).catch(()=>null),
        fetch('api/organization.php?entity=positions', { credentials: 'same-origin' }).then(r=>r.json()).catch(()=>null),
        fetch('api/organization.php?entity=branches', { credentials: 'same-origin' }).then(r=>r.json()).catch(()=>null),
      ]);
      _lists.departments = deps?.success ? (Array.isArray(deps.data)?deps.data:[]) : _lists.departments;
      _lists.positions = poss?.success ? (Array.isArray(poss.data)?poss.data:[]) : _lists.positions;
      _lists.branches = brs?.success ? (Array.isArray(brs.data)?brs.data:[]) : _lists.branches;
    } catch (_) {}
  }

  function addDocRow(values = {}) {
    const wrap = document.getElementById('docList'); if (!wrap) return;
    const row = document.createElement('div');
    row.className = 'card p-3';
    const typeOpts = _lists.document_types.map(t=>`<option value="${t}" ${values.document_type===t?'selected':''}>${t}</option>`).join('');
    row.innerHTML = `
      <div class="row g-2 align-items-end">
        <div class="col-md-3">
          <label class="form-label">Type</label>
          <select class="form-select" data-field="document_type">${typeOpts}</select>
        </div>
        <div class="col-md-5">
          <label class="form-label">Title</label>
          <input type="text" class="form-control" data-field="title" value="${escapeHtml(values.title||'')}" />
        </div>
        <div class="col-md-3">
          <label class="form-label">File</label>
          <input type="file" class="form-control" data-field="file" accept=".pdf,image/*" />
          ${values.file_path ? `<div class="form-text">Current: <a href="${escapeHtml(values.file_path)}" target="_blank">view</a></div>` : ''}
          <input type="hidden" data-field="file_path" value="${escapeHtml(values.file_path||'')}" />
        </div>
        <div class="col-md-1 text-end">
          <button type="button" class="btn btn-sm btn-outline-danger" data-remove-row>Remove</button>
        </div>
      </div>`;
    wrap.appendChild(row);
    row.querySelector('[data-remove-row]')?.addEventListener('click', ()=> row.remove());
  }

  function addSkillRow(values = {}) {
    const wrap = document.getElementById('skillList'); if (!wrap) return;
    const skillOpts = _lists.skills.map(s=>`<option value="${s.id}" ${String(values.skill_id)===String(s.id)?'selected':''}>${escapeHtml(s.name)}</option>`).join('');
    const profOpts = _lists.proficiency_levels.map(p=>`<option ${values.proficiency_level===p?'selected':''}>${p}</option>`).join('');
    const row = document.createElement('div');
    row.className = 'card p-3';
    row.innerHTML = `
      <div class="row g-2 align-items-end">
        <div class="col-md-4">
          <label class="form-label">Skill</label>
          <select class="form-select" data-field="skill_id"><option value=""></option>${skillOpts}</select>
        </div>
        <div class="col-md-3">
          <label class="form-label">Proficiency</label>
          <select class="form-select" data-field="proficiency_level">${profOpts}</select>
        </div>
        <div class="col-md-2">
          <label class="form-label">Years</label>
          <input type="number" class="form-control" data-field="years_of_experience" value="${values.years_of_experience??''}" />
        </div>
        <div class="col-md-3 text-end">
          <button type="button" class="btn btn-sm btn-outline-danger" data-remove-row>Remove</button>
        </div>
        <div class="col-md-3 form-check mt-2">
          <input class="form-check-input" type="checkbox" data-field="certified" ${values.certified? 'checked':''} />
          <label class="form-check-label">Certified</label>
        </div>
        <div class="col-md-4">
          <label class="form-label">Certification Name</label>
          <input type="text" class="form-control" data-field="certification_name" value="${escapeHtml(values.certification_name||'')}" />
        </div>
        <div class="col-md-3">
          <label class="form-label">Certification Date</label>
          <input type="date" class="form-control" data-field="certification_date" value="${values.certification_date||''}" />
        </div>
        <div class="col-12">
          <label class="form-label">Notes</label>
          <input type="text" class="form-control" data-field="notes" value="${escapeHtml(values.notes||'')}" />
        </div>
      </div>`;
    wrap.appendChild(row);
    row.querySelector('[data-remove-row]')?.addEventListener('click', ()=> row.remove());
  }

  function addHistRow(values = {}) {
    const wrap = document.getElementById('histList'); if (!wrap) return;
    const depOpts = _lists.departments.map(d=>`<option value="${d.id}" ${String(values.department_id)===String(d.id)?'selected':''}>${escapeHtml(d.name)}</option>`).join('');
    const posOpts = _lists.positions.map(p=>`<option value="${p.id}" ${String(values.position_id)===String(p.id)?'selected':''}>${escapeHtml(p.title||p.name||('#'+p.id))}</option>`).join('');
    const brOpts  = _lists.branches.map(b=>`<option value="${b.id}" ${String(values.branch_id)===String(b.id)?'selected':''}>${escapeHtml(b.name)}</option>`).join('');
    const row = document.createElement('div');
    row.className = 'card p-3';
    row.innerHTML = `
      <div class="row g-2 align-items-end">
        <div class="col-md-3">
          <label class="form-label">Department</label>
          <select class="form-select" data-field="department_id"><option value=""></option>${depOpts}</select>
        </div>
        <div class="col-md-3">
          <label class="form-label">Position</label>
          <select class="form-select" data-field="position_id"><option value=""></option>${posOpts}</select>
        </div>
        <div class="col-md-3">
          <label class="form-label">Branch</label>
          <select class="form-select" data-field="branch_id"><option value=""></option>${brOpts}</select>
        </div>
        <div class="col-md-3 text-end">
          <button type="button" class="btn btn-sm btn-outline-danger" data-remove-row>Remove</button>
        </div>
        <div class="col-md-3">
          <label class="form-label">Start Date</label>
          <input type="date" class="form-control" data-field="start_date" value="${values.start_date||''}" />
        </div>
        <div class="col-md-3">
          <label class="form-label">End Date</label>
          <input type="date" class="form-control" data-field="end_date" value="${values.end_date||''}" />
        </div>
        <div class="col-md-6">
          <label class="form-label">Notes</label>
          <input type="text" class="form-control" data-field="notes" value="${escapeHtml(values.notes||'')}" />
        </div>
      </div>`;
    wrap.appendChild(row);
    row.querySelector('[data-remove-row]')?.addEventListener('click', ()=> row.remove());
  }

  function addCertRow(values = {}) {
    const wrap = document.getElementById('certList'); if (!wrap) return;
    const certOpts = _lists.certificates.map(c=>`<option value="${c.id}" ${String(values.certificate_id)===String(c.id)?'selected':''}>${escapeHtml(c.name)}</option>`).join('');
    const statusOpts = _lists.certificate_status.map(s=>`<option ${values.status===s?'selected':''}>${s}</option>`).join('');
    const row = document.createElement('div');
    row.className = 'card p-3';
    row.innerHTML = `
      <div class="row g-2 align-items-end">
        <div class="col-md-4">
          <label class="form-label">Certificate</label>
          <select class="form-select" data-field="certificate_id"><option value=""></option>${certOpts}</select>
        </div>
        <div class="col-md-2">
          <label class="form-label">Certificate #</label>
          <input type="text" class="form-control" data-field="certificate_number" value="${escapeHtml(values.certificate_number||'')}" />
        </div>
        <div class="col-md-2">
          <label class="form-label">Issue</label>
          <input type="date" class="form-control" data-field="issue_date" value="${values.issue_date||''}" />
        </div>
        <div class="col-md-2">
          <label class="form-label">Expiry</label>
          <input type="date" class="form-control" data-field="expiry_date" value="${values.expiry_date||''}" />
        </div>
        <div class="col-md-2">
          <label class="form-label">Status</label>
          <select class="form-select" data-field="status">${statusOpts}</select>
        </div>
        <div class="col-md-6">
          <label class="form-label">File</label>
          <input type="file" class="form-control" data-field="file" accept=".pdf,image/*" />
          ${values.file_path ? `<div class="form-text">Current: <a href="${escapeHtml(values.file_path)}" target="_blank">view</a></div>` : ''}
          <input type="hidden" data-field="file_path" value="${escapeHtml(values.file_path||'')}" />
        </div>
        <div class="col-12 text-end">
          <button type="button" class="btn btn-sm btn-outline-danger" data-remove-row>Remove</button>
        </div>
      </div>`;
    wrap.appendChild(row);
    row.querySelector('[data-remove-row]')?.addEventListener('click', ()=> row.remove());
  }

  function collectDocuments() {
    const wrap = document.getElementById('docList'); if (!wrap) return [];
    return Array.from(wrap.children).map(card => {
      const get = (sel) => card.querySelector(`[data-field="${sel}"]`);
      const file = get('file')?.files?.[0] || null;
      return {
        document_type: get('document_type')?.value || '',
        title: get('title')?.value || '',
        file_path: get('file_path')?.value || null,
        __file: file
      };
    }).filter(d => d.document_type && d.title);
  }

  function collectSkills(){
    const wrap = document.getElementById('skillList'); if (!wrap) return [];
    return Array.from(wrap.children).map(card => {
      const get = (sel) => card.querySelector(`[data-field="${sel}"]`);
      const skill_id = get('skill_id')?.value || '';
      if (!skill_id) return null;
      return {
        skill_id: skill_id,
        proficiency_level: get('proficiency_level')?.value || 'beginner',
        years_of_experience: get('years_of_experience')?.value ? parseInt(get('years_of_experience').value, 10) : null,
        certified: get('certified')?.checked ? 1 : 0,
        certification_name: get('certification_name')?.value || null,
        certification_date: get('certification_date')?.value || null,
        notes: get('notes')?.value || null,
      };
    }).filter(Boolean);
  }

  function collectHistory(){
    const wrap = document.getElementById('histList'); if (!wrap) return [];
    return Array.from(wrap.children).map(card => {
      const get = (sel) => card.querySelector(`[data-field="${sel}"]`);
      const dept = get('department_id')?.value || '';
      const pos = get('position_id')?.value || '';
      const branch = get('branch_id')?.value || '';
      const start = get('start_date')?.value || '';
      const end = get('end_date')?.value || '';
      if (!dept && !pos && !branch && !start && !end) return null; // skip empty row
      return {
        department_id: dept || null,
        position_id: pos || null,
        branch_id: branch || null,
        start_date: start || null,
        end_date: end || null,
        notes: get('notes')?.value || null,
      };
    }).filter(Boolean);
  }

  function collectCertificates(){
    const wrap = document.getElementById('certList'); if (!wrap) return [];
    return Array.from(wrap.children).map(card => {
      const get = (sel) => card.querySelector(`[data-field="${sel}"]`);
      const cid = get('certificate_id')?.value || '';
      if (!cid) return null;
      const file = get('file')?.files?.[0] || null;
      return {
        certificate_id: cid,
        certificate_number: get('certificate_number')?.value || null,
        issue_date: get('issue_date')?.value || null,
        expiry_date: get('expiry_date')?.value || null,
        status: get('status')?.value || null,
        file_path: get('file_path')?.value || null,
        __file: file
      };
    }).filter(Boolean);
  }

  async function render(container) {
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Employees</h5>
          <div>
            <input type="text" id="empSearch" class="form-control form-control-sm d-inline-block" placeholder="Search employees..." style="width: 220px;" />
            ${canEditEmployees() ? `<button class="btn btn-sm btn-primary ms-2" id="addEmployeeBtn"><i class="fas fa-plus"></i> Add Employee</button>` : ''}
            ${canEditEmployees() ? `<button class="btn btn-sm btn-outline-secondary ms-2" id="bulkUploadBtn"><i class="fas fa-file-upload"></i> Bulk Upload</button>` : ''}
          </div>
        </div>
        <div class="card-body">
          ${canEditEmployees() ? `
          <!-- Create Employee Modal trigger is the Add Employee button -->
          <!-- Modal -->
          <div class="modal fade" id="employeeModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog modal-xl modal-dialog-scrollable">
              <div class="modal-content">
                <div class="modal-header">
                  <h5 class="modal-title">Create Employee</h5>
                  <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                  <form id="employeeModalForm">
                    <ul class="nav nav-tabs" role="tablist">
                      <li class="nav-item" role="presentation">
                        <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#tab-personal" type="button" role="tab">Personal</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-contact" type="button" role="tab">Contact & Address</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-ids" type="button" role="tab">IDs</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-education" type="button" role="tab">Education</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-job" type="button" role="tab">Job</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-nok" type="button" role="tab">Next of Kin</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-docs" type="button" role="tab">Documents</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-skills" type="button" role="tab">Skills</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-history" type="button" role="tab">Employment History</button>
                      </li>
                      <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-certs" type="button" role="tab">Certificates</button>
                      </li>
                    </ul>

                    <div class="tab-content mt-3">
                      <div class="tab-pane fade show active" id="tab-personal" role="tabpanel">
                        <div class="row g-3">
                          <div class="col-md-2">
                            <label class="form-label">Salutation</label>
                            <input type="text" name="salutation" class="form-control" placeholder="Mr/Ms/Dr" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">First Name *</label>
                            <input type="text" name="first_name" class="form-control" required />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Middle Name</label>
                            <input type="text" name="middle_name" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Last Name *</label>
                            <input type="text" name="last_name" class="form-control" required />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Employee # *</label>
                            <input type="text" name="employee_number" class="form-control" required />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Hire Date *</label>
                            <input type="date" name="hire_date" class="form-control" required />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Birth Date</label>
                            <input type="date" name="date_of_birth" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Gender</label>
                            <select name="gender" class="form-select">
                              <option value="">Select</option>
                              <option>Male</option>
                              <option>Female</option>
                              <option>Other</option>
                            </select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Marital Status</label>
                            <select name="marital_status" class="form-select">
                              <option value="">Select</option>
                              <option>Single</option>
                              <option>Married</option>
                              <option>Divorced</option>
                              <option>Widowed</option>
                            </select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Nationality</label>
                            <input type="text" name="nationality" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Official Title</label>
                            <input type="text" name="title" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Department</label>
                            <select id="empDepartmentTop" class="form-select"></select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Branch</label>
                            <select id="empBranchTop" class="form-select"></select>
                          </div>
                        </div>
                      </div>
                      <div class="tab-pane fade" id="tab-docs" role="tabpanel">
                        <div class="d-flex justify-content-end mb-2">
                          <button type="button" class="btn btn-sm btn-outline-primary" id="addDocRow"><i class="fas fa-plus"></i> Add Document</button>
                        </div>
                        <div id="docList" class="vstack gap-2"></div>
                      </div>
                      <div class="tab-pane fade" id="tab-skills" role="tabpanel">
                        <div class="d-flex justify-content-end mb-2">
                          <button type="button" class="btn btn-sm btn-outline-primary" id="addSkillRow"><i class="fas fa-plus"></i> Add Skill</button>
                        </div>
                        <div id="skillList" class="vstack gap-2"></div>
                      </div>
                      <div class="tab-pane fade" id="tab-history" role="tabpanel">
                        <div class="d-flex justify-content-end mb-2">
                          <button type="button" class="btn btn-sm btn-outline-primary" id="addHistRow"><i class="fas fa-plus"></i> Add History</button>
                        </div>
                        <div id="histList" class="vstack gap-2"></div>
                      </div>
                      <div class="tab-pane fade" id="tab-certs" role="tabpanel">
                        <div class="d-flex justify-content-end mb-2">
                          <button type="button" class="btn btn-sm btn-outline-primary" id="addCertRow"><i class="fas fa-plus"></i> Add Certificate</button>
                        </div>
                        <div id="certList" class="vstack gap-2"></div>
                      </div>

                      <div class="tab-pane fade" id="tab-contact" role="tabpanel">
                        <div class="row g-3">
                          <div class="col-md-4">
                            <label class="form-label">Email</label>
                            <input type="email" name="email" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Official Email</label>
                            <input type="email" name="official_email" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Phone</label>
                            <input type="text" name="phone" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Mobile</label>
                            <input type="text" name="mobile" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Digital Address</label>
                            <input type="text" name="digital_address" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Postal Address</label>
                            <input type="text" name="postal_address" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">House #</label>
                            <input type="text" name="house_number" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Landmark</label>
                            <input type="text" name="land_mark" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Region</label>
                            <select name="region" id="regionSelect" class="form-select">
                              <option value="">-- Select Region --</option>
                              <option>Greater Accra</option>
                              <option>Ashanti</option>
                              <option>Western</option>
                              <option>Western North</option>
                              <option>Central</option>
                              <option>Eastern</option>
                              <option>Volta</option>
                              <option>Oti</option>
                              <option>Northern</option>
                              <option>North East</option>
                              <option>Savannah</option>
                              <option>Upper East</option>
                              <option>Upper West</option>
                              <option>Bono</option>
                              <option>Bono East</option>
                              <option>Ahafo</option>
                            </select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Religion</label>
                            <select name="religion" id="religionSelect" class="form-select">
                              <option value="">-- Select Religion --</option>
                              <option>Christian</option>
                              <option>Muslim</option>
                              <option>Traditional</option>
                              <option>Others</option>
                            </select>
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Location</label>
                            <input type="text" name="location" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Staff Location</label>
                            <input type="text" name="staff_location" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Residential Ownership</label>
                            <input type="text" name="residential_ownership" class="form-control" />
                          </div>
                        </div>
                      </div>

                      <div class="tab-pane fade" id="tab-ids" role="tabpanel">
                        <div class="row g-3">
                          <div class="col-md-4">
                            <label class="form-label">ID Type</label>
                            <input type="text" name="id_type" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">ID Number</label>
                            <input type="text" name="id_number" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">ID Image</label>
                            <input type="file" id="idImageFile" accept="image/*" class="form-control" />
                            <input type="hidden" name="profile_pic" />
                            <div class="mt-2" id="idImagePreviewWrap">
                              <img id="idImagePreview" alt="ID preview" style="max-height: 70px; display: none;" />
                            </div>
                          </div>
                        </div>
                      </div>

                      <div class="tab-pane fade" id="tab-education" role="tabpanel">
                        <div class="row g-3">
                          <div class="col-md-4">
                            <label class="form-label">Certificate Type</label>
                            <input type="text" name="certificate_type" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">First Degree Program</label>
                            <input type="text" name="first_degree_program" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">First Degree Class</label>
                            <input type="text" name="first_degree_class" class="form-control" />
                          </div>
                          <div class="col-md-6">
                            <label class="form-label">First Degree Institution</label>
                            <input type="text" name="first_degree_institution" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Second Degree</label>
                            <input type="text" name="second_degree" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Second Degree Program</label>
                            <input type="text" name="second_degree_program" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Second Degree Class</label>
                            <input type="text" name="second_degree_class" class="form-control" />
                          </div>
                          <div class="col-md-6">
                            <label class="form-label">Second Degree Institution</label>
                            <input type="text" name="second_degree_institution" class="form-control" />
                          </div>
                          <div class="col-md-6">
                            <label class="form-label">Professional Certificate</label>
                            <input type="text" name="professional_certificate" class="form-control" />
                          </div>
                          <div class="col-md-6">
                            <label class="form-label">Professional Institution</label>
                            <input type="text" name="professional_institution" class="form-control" />
                          </div>
                          <div class="col-md-6">
                            <label class="form-label">Professional Results</label>
                            <input type="text" name="professional_results" class="form-control" />
                          </div>
                          <div class="col-12">
                            <label class="form-label">Academic Program</label>
                            <textarea name="academic_program" class="form-control" rows="2"></textarea>
                          </div>
                        </div>
                      </div>

                      <div class="tab-pane fade" id="tab-job" role="tabpanel">
                        <div class="row g-3">
                          <div class="col-md-4">
                            <label class="form-label">Job Title</label>
                            <input type="text" name="job_title" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Designation</label>
                            <input type="text" name="designation" class="form-control" />
                          </div>
                              <div class="col-md-4">
                            <label class="form-label">Employment Status</label>
                            <select name="employment_status" class="form-select">
                              <option value="active">active</option>
                              <option value="inactive">inactive</option>
                              <option value="resigned">resigned</option>
                              <option value="retired">retired</option>
                              <option value="terminated">terminated</option>
                              <option value="exited">exited</option>
                            </select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Department</label>
                            <select name="department_id" id="empDepartment" class="form-select"></select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Position</label>
                            <select name="position_id" id="empPosition" class="form-select"></select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Grade</label>
                            <select name="grade_id" id="empGrade" class="form-select"></select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Branch</label>
                            <select name="branch_id" id="empBranch" class="form-select"></select>
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Manager</label>
                            <select name="manager_id" id="empManager" class="form-select"></select>
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Department Group</label>
                            <input type="text" name="department_group" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Department Unit</label>
                            <input type="text" name="department_unit" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Branch Type</label>
                            <input type="text" name="branch_type" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Starting Grade</label>
                            <input type="text" name="starting_grade" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Current Grade</label>
                            <input type="text" name="current_grade" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Joined Dept Date</label>
                            <input type="datetime-local" name="joined_department_date" class="form-control" />
                          </div>
                          <div class="col-md-3">
                            <label class="form-label">Last Promotion Date</label>
                            <input type="date" name="last_promotion_date" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Confirmation</label>
                            <input type="text" name="confirmation" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Service Length</label>
                            <input type="text" name="service_length" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Staff Status</label>
                            <input type="text" name="staff_status" class="form-control" />
                          </div>
                        </div>
                      </div>

                      <div class="tab-pane fade" id="tab-nok" role="tabpanel">
                        <div class="row g-3">
                          <div class="col-md-6">
                            <label class="form-label">Next of Kin Name</label>
                            <input type="text" name="nok_name" class="form-control" />
                          </div>
                          <div class="col-md-6">
                            <label class="form-label">Relationship</label>
                            <input type="text" name="relationship" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">NOK Contact</label>
                            <input type="text" name="nok_contact" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">NOK Postal Address</label>
                            <input type="text" name="nok_postal_address" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">NOK Home Address</label>
                            <input type="text" name="nok_home_address" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">NOK Date</label>
                            <input type="date" name="nok_date" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Emergency Contact Name</label>
                            <input type="text" name="emergency_contact_name" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Emergency Contact Phone</label>
                            <input type="text" name="emergency_contact_phone" class="form-control" />
                          </div>
                          <div class="col-md-4">
                            <label class="form-label">Emergency Relationship</label>
                            <input type="text" name="emergency_contact_relationship" class="form-control" />
                          </div>
                        </div>
                      </div>
                    </div>
                  </form>
                </div>
                <div class="modal-footer">
                  <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
                  <button type="button" class="btn btn-primary" id="saveEmployeeBtn">Create Employee</button>
                </div>
              </div>
            </div>
          </div>` : ''}

          <!-- Bulk Upload Modal -->
          <div class="modal fade" id="bulkUploadModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog modal-xl modal-dialog-scrollable">
              <div class="modal-content">
                <div class="modal-header">
                  <h5 class="modal-title">Bulk Upload Employees (CSV/Excel)</h5>
                  <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                  <div class="row g-3">
                    <div class="col-md-6">
                      <label class="form-label">Select File (.csv, .xlsx, .xls)</label>
                      <input type="file" id="bulkFile" class="form-control" accept=".csv,.xlsx,.xls" />
                      <div class="form-text">Download template: <a href="#" id="bulkTemplateLink">CSV Template</a></div>
                    </div>
                    <div class="col-md-6 text-end">
                      <button class="btn btn-sm btn-outline-secondary" id="bulkClearBtn">Clear</button>
                    </div>
                    <div class="col-12">
                      <div id="bulkPreview" class="table-responsive" style="max-height: 420px; overflow:auto; border:1px solid #eee; border-radius:6px;"></div>
                    </div>
                  </div>
                </div>
                <div class="modal-footer">
                  <div class="me-auto text-muted small">Required columns: employee_number, first_name, last_name, hire_date. Optional: email, phone, gender, date_of_birth, department, position, branch, manager_employee_number.</div>
                  <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
                  <button type="button" class="btn btn-primary" id="startBulkUploadBtn">Start Upload</button>
                </div>
              </div>
            </div>
          </div>

          <!-- Read-only Employee Profile Modal (container only; content is rendered dynamically) -->
          <div class="modal fade" id="employeeProfileModal" tabindex="-1" aria-hidden="true">
            <div class="modal-dialog modal-xl modal-dialog-scrollable">
              <div class="modal-content">
                <div class="modal-header">
                  <h5 class="modal-title">Employee Profile</h5>
                  <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="employeeProfileBody"></div>
                <div class="modal-footer">
                  <button type="button" class="btn btn-light" data-bs-dismiss="modal">Close</button>
                </div>
              </div>
            </div>
          </div>

          <div class="d-flex justify-content-between align-items-center mb-2">
            <ul class="nav nav-pills gap-1 flex-wrap" id="empStatusTabs">
              <li class="nav-item"><button type="button" class="nav-link" data-status="all">All</button></li>
              <li class="nav-item"><button type="button" class="nav-link active" data-status="active">Active</button></li>
              <li class="nav-item"><button type="button" class="nav-link" data-status="resigned">Resigned</button></li>
              <li class="nav-item"><button type="button" class="nav-link" data-status="retired">Retired</button></li>
              <li class="nav-item"><button type="button" class="nav-link" data-status="terminated">Terminated</button></li>
              <li class="nav-item"><button type="button" class="nav-link" data-status="inactive">Inactive</button></li>
              <li class="nav-item"><button type="button" class="nav-link" data-status="exited">Exited</button></li>
            </ul>
          </div>

          <div class="table-responsive">
            <table class="table mb-0">
              <thead>
                <tr id="empTableHead"></tr>
              </thead>
              <tbody id="empTableBody">
                <tr><td colspan="1">Loading...</td></tr>
              </tbody>
            </table>
          </div>
        </div>
      </div>
    `;

    const addBtn = container.querySelector('#addEmployeeBtn');
    if (addBtn) {
      addBtn.addEventListener('click', async () => {
        const modalEl = document.getElementById('employeeModal');
        if (!modalEl) return;
        await prepareCreateModal();
        bootstrap.Modal.getOrCreateInstance(modalEl).show();
      });
    }

    // Bulk Upload wiring
    const bulkBtn = container.querySelector('#bulkUploadBtn');
    if (bulkBtn) {
      bulkBtn.addEventListener('click', () => {
        const m = document.getElementById('bulkUploadModal');
        const prev = document.getElementById('bulkPreview'); if (prev) prev.innerHTML = '';
        const file = document.getElementById('bulkFile'); if (file) file.value='';
        bootstrap.Modal.getOrCreateInstance(m).show();
      });
    }
    const bulkTemplate = container.querySelector('#bulkTemplateLink');
    if (bulkTemplate) {
      bulkTemplate.addEventListener('click', (e)=>{
        e.preventDefault();
        const csv = bulkTemplateCSV();
        const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a'); a.href=url; a.download='employees_template.csv'; a.click();
        setTimeout(()=>URL.revokeObjectURL(url), 1000);
      });
    }
    const bulkClearBtn = container.querySelector('#bulkClearBtn');
    bulkClearBtn?.addEventListener('click', ()=>{
      const prev = document.getElementById('bulkPreview'); if (prev) prev.innerHTML = '';
      const file = document.getElementById('bulkFile'); if (file) file.value='';
    });

    const bulkFile = container.querySelector('#bulkFile');
    if (bulkFile && !bulkFile._wired) {
      bulkFile._wired = true;
      bulkFile.addEventListener('change', handleBulkFileChange);
    }
    const startBulk = container.querySelector('#startBulkUploadBtn');
    if (startBulk && !startBulk._wired) {
      startBulk._wired = true;
      startBulk.addEventListener('click', startBulkUpload);
    }

    // Build column chooser UI and table header
    buildColumnChooserMenu(container);
    renderTableHead(container);
    // Adjust initial loading colspan
    const initialCols = getVisibleColumns().length + 1; // + Actions
    const tb = container.querySelector('#empTableBody');
    if (tb) tb.innerHTML = `<tr><td colspan="${initialCols}">Loading...</td></tr>`;

    // Status tabs default
    container.dataset.empStatus = container.dataset.empStatus || 'active';
    const tabs = container.querySelector('#empStatusTabs');
    if (tabs && !tabs._wired){
      tabs._wired = true;
      tabs.addEventListener('click', (ev)=>{
        const b = ev.target.closest('[data-status]'); if (!b) return;
        const status = b.getAttribute('data-status') || 'active';
        container.dataset.empStatus = status;
        tabs.querySelectorAll('.nav-link').forEach(x=> x.classList.remove('active'));
        b.classList.add('active');
        const searchVal = container.querySelector('#empSearch')?.value?.trim() || '';
        loadEmployees(container, searchVal, status);
      });
    }

    const searchInput = container.querySelector('#empSearch');
    if (searchInput) {
      let debounceTimer;
      searchInput.addEventListener('input', () => {
        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(() => loadEmployees(container, searchInput.value.trim(), container.dataset.empStatus || 'active'), 300);
      });
    }

    const saveBtn = container.querySelector('#saveEmployeeBtn');
    if (saveBtn) {
      saveBtn.addEventListener('click', async () => {
        const form = document.getElementById('employeeModalForm');
        if (!form) return;
        const body = formToJSON(new FormData(form));
        // onboarding arrays (populated in a later step)
        body.documents = collectDocuments();
        body.skills = collectSkills();
        body.employment_history = collectHistory();
        body.certificates = collectCertificates();
        // Upload files first (ID image, documents, certificates)
        try {
          // ID image
          const idFile = document.getElementById('idImageFile')?.files?.[0] || null;
          if (idFile) {
            const up = await uploadFile(idFile);
            body.profile_pic = up.path;
          }
          // Documents
          for (const d of body.documents) {
            if (d.__file) {
              const up = await uploadFile(d.__file);
              d.file_path = up.path;
              d.file_size = up.size;
              d.mime_type = up.mime_type;
              delete d.__file;
            } else {
              delete d.__file;
            }
          }
          // Certificates
          for (const c of body.certificates) {
            if (c.__file) {
              const up = await uploadFile(c.__file);
              c.file_path = up.path;
              delete c.__file;
            } else {
              delete c.__file;
            }
          }
        } catch (e) {
          window.auth?.showNotification?.('File upload failed: ' + (e?.message || ''), 'error');
          return;
        }
        const mode = saveBtn.dataset.mode || 'create';
        try {
          let res;
          if (mode === 'edit' && saveBtn.dataset.id) {
            res = await fetch(`api/employees.php?id=${encodeURIComponent(saveBtn.dataset.id)}`, {
              method: 'PUT',
              headers: { 'Content-Type': 'application/json' },
              credentials: 'same-origin',
              body: JSON.stringify(body)
            });
          } else {
            res = await fetch('api/employees.php', {
              method: 'POST',
              headers: { 'Content-Type': 'application/json' },
              credentials: 'same-origin',
              body: JSON.stringify(body)
            });
          }
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.(mode === 'edit' ? 'Employee updated' : 'Employee created', 'success');
            const modalEl = document.getElementById('employeeModal');
            if (modalEl) bootstrap.Modal.getOrCreateInstance(modalEl).hide();
            saveBtn.dataset.mode = 'create';
            saveBtn.dataset.id = '';
            await loadEmployees(container, '', container.dataset.empStatus || 'active');
          } else {
            window.auth?.showNotification?.(data?.message || (mode === 'edit' ? 'Failed to update employee' : 'Failed to create employee'), 'error');
          }
        } catch (e) {
          window.auth?.showNotification?.(mode === 'edit' ? 'Failed to update employee' : 'Failed to create employee', 'error');
        }
      });
    }

    // Delegate actions (view/edit/delete)
    container.addEventListener('click', async (e) => {
      const btn = e.target.closest('[data-action]');
      if (!btn) return;
      const action = btn.getAttribute('data-action');
      if (action === 'view-employee') {
        const id = btn.getAttribute('data-id');
        if (!id) return;
        try { await openProfileModal(id); } catch (_) { window.auth?.showNotification?.('Failed to load employee', 'error'); }
      } else if (action === 'edit-employee') {
        const id = btn.getAttribute('data-id');
        if (!id) return;
        try { await startEditEmployee(id); } catch (_) { window.auth?.showNotification?.('Failed to load employee', 'error'); }
      } else if (action === 'delete-employee') {
        const id = btn.getAttribute('data-id');
        if (!id) return;
        if (!confirm('Are you sure you want to delete this employee?')) return;
        try {
          const res = await fetch(`api/employees.php?id=${encodeURIComponent(id)}`, {
            method: 'DELETE',
            credentials: 'same-origin'
          });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Employee deleted', 'success');
            await loadEmployees(container);
          } else {
            window.auth?.showNotification?.(data?.message || 'Failed to delete employee', 'error');
          }
        } catch (_) {
          window.auth?.showNotification?.('Failed to delete employee', 'error');
        }
      } else if (action === 'request-exit') {
        const id = btn.getAttribute('data-id');
        if (!id) return;
        // Quick modal-less prompt flow; can be enhanced later
        const proposed = prompt('Enter proposed last working day (YYYY-MM-DD):', '');
        if (proposed === null) return; // cancelled
        const reason = prompt('Enter reason for exit:', '');
        if (reason === null) return;
        const exitType = prompt('Exit type (e.g., resignation, termination, retirement):', 'resignation');
        try {
          const res = await fetch('api/exits.php?action=request', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'same-origin',
            body: JSON.stringify({ employee_id: Number(id), proposed_last_day: proposed || null, reason: reason || null, exit_type: exitType || null })
          });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Exit request submitted', 'success');
          } else {
            window.auth?.showNotification?.(data?.message || 'Failed to submit exit request', 'error');
          }
        } catch (_) {
          window.auth?.showNotification?.('Failed to submit exit request', 'error');
        }
      }
    });

    await loadEmployees(container, '', container.dataset.empStatus || 'active');
  }

  async function loadEmployees(container, search = '', status = '') {
    const tbody = container.querySelector('#empTableBody');
    if (!tbody) return;
    const visibleCols = getVisibleColumns();
    const colSpan = visibleCols.length + 1; // + Actions
    tbody.innerHTML = `<tr><td colspan="${colSpan}">Loading...</td></tr>`;

    try {
      const qs = new URLSearchParams();
      if (search) qs.set('search', search);
      if (status && status !== 'all') qs.set('status', status);
      const url = `api/employees.php${qs.toString() ? '?' + qs.toString() : ''}`;
      const res = await fetch(url, { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success && Array.isArray(data.data)) {
        if (data.data.length === 0) {
          tbody.innerHTML = `<tr><td colspan="${colSpan}" class="text-center text-muted">No employees found</td></tr>`;
        } else {
          tbody.innerHTML = data.data.map((emp) => employeeRow(emp, visibleCols)).join('');
        }
      } else {
        tbody.innerHTML = `<tr><td colspan="${colSpan}" class="text-danger">Failed to load employees</td></tr>`;
      }
    } catch (e) {
      tbody.innerHTML = `<tr><td colspan="${colSpan}" class="text-danger">Failed to load employees</td></tr>`;
    }
  }

  function employeeRow(emp, cols) {
    const canDelete = (window.auth?.currentUser?.role_slug === 'admin' || window.auth?.currentUser?.role_slug === 'super_admin');
    const canEdit = !!canEditEmployees?.() || window.auth?.canEdit?.('employees');
    const st = String(emp.employment_status || 'active').toLowerCase();
    const canRequestExit = (window.auth?.canEdit?.('employees') && (st === 'active' || st === 'inactive'));
    const tds = cols.map(col => `<td>${col.render(emp)}</td>`).join('');
    return `
      <tr>
        ${tds}
        <td>
          <button class="btn btn-sm btn-outline-secondary me-1" data-action="view-employee" data-id="${emp.id}"><i class="fas fa-eye"></i></button>
          ${canEdit ? `<button class="btn btn-sm btn-outline-primary me-1" data-action="edit-employee" data-id="${emp.id}"><i class="fas fa-edit"></i></button>` : ''}
          ${canRequestExit ? `<button class="btn btn-sm btn-outline-warning me-1" title="Initiate Exit" data-action="request-exit" data-id="${emp.id}"><i class="fas fa-person-walking-arrow-right"></i></button>` : ''}
          ${canDelete ? `<button class="btn btn-sm btn-danger" data-action="delete-employee" data-id="${emp.id}"><i class="fas fa-trash"></i></button>` : ''}
        </td>
      </tr>`;
  }

  function statusBadge(status) {
    const map = {
      'active': 'status-badge status-active',
      'pending': 'status-badge status-pending',
      'approved': 'status-badge status-approved',
      'rejected': 'status-badge status-rejected',
      'inactive': 'status-badge status-inactive',
      'terminated': 'status-badge status-inactive',
      'resigned': 'status-badge status-inactive',
      'retired': 'status-badge status-inactive',
      'exited': 'status-badge status-inactive'
    };
    const cls = map[status] || 'status-badge';
    return `<span class="${cls}">${escapeHtml(status)}</span>`;
  }

  function formToJSON(fd) {
    const obj = {};
    fd.forEach((v, k) => { obj[k] = v; });
    return obj;
  }

  async function loadJobOptions() {
    try {
      const [deps, poss, brs, emps] = await Promise.all([
        fetch('api/organization.php?entity=departments', { credentials: 'same-origin' }).then(r=>r.json()).catch(()=>null),
        fetch('api/organization.php?entity=positions', { credentials: 'same-origin' }).then(r=>r.json()).catch(()=>null),
        fetch('api/organization.php?entity=branches', { credentials: 'same-origin' }).then(r=>r.json()).catch(()=>null),
        fetch('api/employees.php', { credentials: 'same-origin' }).then(r=>r.json()).catch(()=>null),
      ]);
      const depSel = document.getElementById('empDepartment');
      const depSelTop = document.getElementById('empDepartmentTop');
      const posSel = document.getElementById('empPosition');
      const brSel  = document.getElementById('empBranch');
      const brSelTop = document.getElementById('empBranchTop');
      const manSel = document.getElementById('empManager');
      const gradeSel = document.getElementById('empGrade');
      if (depSel) {
        depSel.innerHTML = '<option value="">-- Select --</option>' + ((deps?.success && Array.isArray(deps.data)) ? deps.data.map(d=>`<option value="${d.id}">${escapeHtml(d.name)}</option>`).join('') : '');
      }
      if (depSelTop) {
        depSelTop.innerHTML = '<option value="">-- Select --</option>' + ((deps?.success && Array.isArray(deps.data)) ? deps.data.map(d=>`<option value="${d.id}">${escapeHtml(d.name)}</option>`).join('') : '');
        // keep both in sync
        depSelTop.onchange = ()=>{ if (depSel) depSel.value = depSelTop.value; };
        if (depSel) depSel.onchange = ()=>{ depSelTop.value = depSel.value; };
      }
      if (posSel) {
        posSel.innerHTML = '<option value="">-- Select --</option>' + ((poss?.success && Array.isArray(poss.data)) ? poss.data.map(p=>`<option value="${p.id}">${escapeHtml(p.title || p.name || ('#'+p.id))}</option>`).join('') : '');
      }
      if (brSel) {
        brSel.innerHTML = '<option value="">-- Select --</option>' + ((brs?.success && Array.isArray(brs.data)) ? brs.data.map(b=>`<option value="${b.id}">${escapeHtml(b.name)}</option>`).join('') : '');
      }
      if (brSelTop) {
        brSelTop.innerHTML = '<option value="">-- Select --</option>' + ((brs?.success && Array.isArray(brs.data)) ? brs.data.map(b=>`<option value="${b.id}">${escapeHtml(b.name)}</option>`).join('') : '');
        // keep both in sync
        brSelTop.onchange = ()=>{ if (brSel) brSel.value = brSelTop.value; };
        if (brSel) brSel.onchange = ()=>{ brSelTop.value = brSel.value; };
      }
      if (manSel) {
        manSel.innerHTML = '<option value="">-- None --</option>' + ((emps?.success && Array.isArray(emps.data)) ? emps.data.filter(e=> (e.employment_status || 'active') === 'active').map(e=>`<option value="${e.id}">${escapeHtml(e.full_name || (e.first_name + ' ' + e.last_name))}</option>`).join('') : '');
      }
      if (gradeSel) {
        const grades = Array.isArray(_lists?.grades) ? _lists.grades : [];
        gradeSel.innerHTML = '<option value="">-- Select --</option>' + grades.map(g=>`<option value="${g.id}">${escapeHtml(g.name)}${g.code? ' ('+escapeHtml(g.code)+')':''}</option>`).join('');
      }
    } catch (_) {}
  }

  function escapeHtml(s) {
    return String(s ?? '').replace(/[&<>"']/g, (c) => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c]));
  }

  async function uploadFile(file) {
    const fd = new FormData();
    fd.append('file', file);
    const res = await fetch('api/upload.php', { method: 'POST', credentials: 'same-origin', body: fd });
    const data = await res.json();
    if (!data?.success) throw new Error(data?.message || 'Upload failed');
    return data.data;
  }

  // ID image preview listener (delegated when modal opens)
  document.addEventListener('change', (e) => {
    const input = e.target;
    if (!(input instanceof HTMLInputElement)) return;
    if (input.id === 'idImageFile' && input.files && input.files[0]) {
      const img = document.getElementById('idImagePreview');
      if (img) {
        img.src = URL.createObjectURL(input.files[0]);
        img.style.display = 'inline-block';
      }
    }
  }, true);

  // ===== Bulk Upload helpers =====
  let _bulkRows = [];

  function bulkTemplateCSV(){
    const headers = ['employee_number','first_name','middle_name','last_name','hire_date','email','phone','gender','date_of_birth','department','position','branch','manager_employee_number'];
    return headers.join(',') + '\n' + 'EMP001,John,,Doe,2024-01-15,john@example.com,555-1234,Male,1990-05-01,IT,Developer,Head Office,EMP010\n';
  }

  async function handleBulkFileChange(evt){
    const file = evt.target.files && evt.target.files[0];
    if (!file) return;
    const preview = document.getElementById('bulkPreview');
    preview.innerHTML = '<div class="p-3">Parsing...</div>';
    try {
      let rows = [];
      if (window.XLSX) {
        const data = await file.arrayBuffer();
        const wb = XLSX.read(data, { type: 'array' });
        const wsname = wb.SheetNames[0];
        const ws = wb.Sheets[wsname];
        const json = XLSX.utils.sheet_to_json(ws, { header: 1, defval: '' });
        // Header validation (first row)
        const headerRaw = (json[0] || []).map(h=>String(h||''));
        const check = validateUploadHeaders(headerRaw);
        if (!check.ok) {
          preview.innerHTML = headerErrorHtml(check, headerRaw);
          _bulkRows = [];
          return;
        }
        rows = normalizeSheet(json);
      } else {
        const text = await file.text();
        // Header validation from first non-empty line
        const firstLine = (text.split(/\r?\n/).find(l=>l.trim()!=='') || '');
        const headerRaw = firstLine.split(',').map(s=>s.trim());
        const check = validateUploadHeaders(headerRaw);
        if (!check.ok) {
          preview.innerHTML = headerErrorHtml(check, headerRaw);
          _bulkRows = [];
          return;
        }
        rows = parseCSVText(text);
      }
      if (!rows.length) { preview.innerHTML = '<div class="p-3 text-muted">No rows found</div>'; _bulkRows = []; return; }
      if (rows.length > 1000) rows = rows.slice(0, 1000); // limit for safety
      _bulkRows = rows;
      preview.innerHTML = renderRowsPreview(rows);
    } catch (e) {
      preview.innerHTML = `<div class="alert alert-danger">Failed to parse file: ${escapeHtml(e?.message||String(e))}</div>`;
      _bulkRows = [];
    }
  }

  function normalizeSheet(sheetRows){
    if (!Array.isArray(sheetRows) || sheetRows.length === 0) return [];
    const header = (sheetRows[0] || []).map(h=>normalizeHeader(String(h||'')));
    const out = [];
    for (let i=1;i<sheetRows.length;i++){
      const row = sheetRows[i]; if (!row || row.length===0) continue;
      const obj = {};
      header.forEach((h,idx)=>{ if(!h) return; obj[h] = row[idx] ?? ''; });
      if (String(obj.employee_number||'').trim()==='' && String(obj.first_name||'').trim()==='') continue;
      out.push(obj);
    }
    return out;
  }

  function parseCSVText(text){
    // naive CSV split; recommend using XLSX path for robust parsing
    const lines = text.split(/\r?\n/).filter(l=>l.trim()!=='');
    if (!lines.length) return [];
    const header = lines[0].split(',').map(h=>normalizeHeader(h));
    const out = [];
    for (let i=1;i<lines.length;i++){
      const cols = lines[i].split(',');
      const obj = {};
      header.forEach((h,idx)=>{ if(!h) return; obj[h] = cols[idx] ?? ''; });
      if (!obj.employee_number && !obj.first_name) continue;
      out.push(obj);
    }
    return out;
  }

  function normalizeHeader(h){
    const raw = String(h||'');
    let t = raw.trim().toLowerCase();
    if (!t) return '';
    // unify common words
    t = t.replace(/\bno\.?\b/g, 'number'); // employee no. -> employee number
    t = t.replace(/\bemp\b/g, 'employee');
    t = t.replace(/\btel\b/g, 'phone');
    // strip punctuation to spaces, collapse, then snake_case
    t = t.replace(/[^a-z0-9]+/g, ' ').replace(/\s+/g, ' ').trim();
    const s = t.replace(/\s+/g, '_');
    const alias = {
      // identifiers
      'employee_number':'employee_number','employee_no':'employee_number','employeenumber':'employee_number','emp_number':'employee_number','emp_no':'employee_number','empno':'employee_number',
      'staff_no':'employee_number','staff_number':'employee_number','staffno':'employee_number','staffnumber':'employee_number','employee_id':'employee_number','employeeid':'employee_number',
      // names
      'firstname':'first_name','first_name':'first_name','first':'first_name',
      'middlename':'middle_name','middle_name':'middle_name',
      'lastname':'last_name','last_name':'last_name','surname':'last_name',
      // dates
      'hire_date':'hire_date','date_hired':'hire_date','hiredate':'hire_date',
      'date_of_birth':'date_of_birth','dob':'date_of_birth',
      // org
      'department_name':'department','department':'department','dept':'department',
      'position_title':'position','job_title':'position','position':'position',
      'branch_name':'branch','branch':'branch',
      // manager ref
      'manager_emp_no':'manager_employee_number','manager_number':'manager_employee_number','manager_employee_number':'manager_employee_number','manager_emp_number':'manager_employee_number'
    };
    return alias[s] || s;
  }

  function validateUploadHeaders(headerRaw){
    const normalized = (headerRaw||[]).map(normalizeHeader).filter(Boolean);
    const required = ['employee_number'];
    const missing = required.filter(r => !normalized.includes(r));
    return { ok: missing.length === 0, missing, normalized };
  }

  function headerErrorHtml(check, headerRaw){
    const expected = 'employee_number (accepted: "Employee No", "EMPLOYEE NO.", "Emp No", "Staff No")';
    const rawLine = (headerRaw||[]).map(h=>escapeHtml(String(h||''))).join(', ');
    return `
      <div class="alert alert-danger">
        <div><strong>Header error:</strong> Missing required header(s): ${check.missing.map(escapeHtml).join(', ')}</div>
        <div class="mt-1">Expected at minimum: <code>${expected}</code></div>
        <div class="mt-2 small text-muted">Detected header row: ${rawLine || '(empty)'}</div>
      </div>
    `;
  }

  function renderRowsPreview(rows){
    const cols = Array.from(new Set(rows.flatMap(r=>Object.keys(r))));
    const head = '<thead><tr>'+cols.map(c=>`<th>${escapeHtml(c)}</th>`).join('')+'</tr></thead>';
    const body = '<tbody>'+rows.slice(0,50).map(r=>'<tr>'+cols.map(c=>`<td>${escapeHtml(String(r[c]??''))}</td>`).join('')+'</tr>').join('') + (rows.length>50?`<tr><td colspan="${cols.length}" class="text-muted">…and ${rows.length-50} more</td></tr>`:'') + '</tbody>';
    return `<table class="table table-sm">${head}${body}</table>`;
  }

  async function startBulkUpload(){
    if (!_bulkRows || _bulkRows.length===0) { window.auth?.showNotification?.('No rows to upload', 'warning'); return; }
    const btn = document.getElementById('startBulkUploadBtn'); if (btn) btn.disabled = true;
    try {
      const res = await fetch('api/employees.php?action=bulk_create', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'same-origin',
        body: JSON.stringify({ rows: _bulkRows })
      });
      const data = await res.json();
      if (data?.success) {
        const m = document.getElementById('bulkUploadModal');
        document.getElementById('bulkPreview').innerHTML = uploadResultSummary(data.data);
        await loadEmployees(document.querySelector('#pageContent'));
        window.auth?.showNotification?.(`Imported ${data.data.success_count} of ${data.data.total} rows`, 'success');
      } else {
        window.auth?.showNotification?.(data?.message || 'Bulk upload failed', 'error');
      }
    } catch (e) {
      window.auth?.showNotification?.('Bulk upload failed', 'error');
    } finally { if (btn) btn.disabled = false; }
  }

  function uploadResultSummary(r){
    const rows = Array.isArray(r?.rows) ? r.rows : [];
    return `<div class="alert alert-info">Success: ${r.success_count}/${r.total}. Failures: ${r.fail_count}.</div>` +
      `<div class="table-responsive"><table class="table table-sm"><thead><tr><th>#</th><th>Status</th><th>Message</th><th>Employee #</th></tr></thead><tbody>`+
      rows.slice(0,200).map((x,i)=>`<tr><td>${i+1}</td><td>${x.success?'OK':'ERR'}</td><td>${escapeHtml(x.message||'')}</td><td>${escapeHtml(x.employee_number||'')}</td></tr>`).join('')+
      (rows.length>200?`<tr><td colspan="4" class="text-muted">…and ${rows.length-200} more</td></tr>`:'')+
      `</tbody></table></div>`;
  }

  // ===== Employee Payroll Components (Allowances & Deductions) =====
  const _empCompState = { empId: null, allowances: [], deductions: [], aTypes: [], dTypes: [] };

  function modeLabel(m){
    const mm = String(m||'fixed');
    if (mm==='percent_basic') return '% Basic';
    if (mm==='percent_gross') return '% Gross';
    return 'Fixed';
  }

  function modeOptionsHtml(selected){
    const opts = [
      {v:'fixed', t:'Fixed'},
      {v:'percent_basic', t:'% of Basic'},
      {v:'percent_gross', t:'% of Gross'}
    ];
    return opts.map(o=>`<option value="${o.v}" ${selected===o.v?'selected':''}>${o.t}</option>`).join('');
  }

  function typeOptionsHtml(types, selected){
    return (types||[]).map(t=>`<option value="${t.id}" ${String(selected)===String(t.id)?'selected':''}>${escapeHtml(t.name || t.code || ('#'+t.id))}${t.code? ' ('+escapeHtml(t.code)+')':''}</option>`).join('');
  }

  async function loadComponentTypes(){
    try {
      const res = await fetch('api/payroll.php?action=setup_lists', { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success) {
        _empCompState.aTypes = Array.isArray(data.data?.allowance_types) ? data.data.allowance_types : [];
        _empCompState.dTypes = Array.isArray(data.data?.deduction_types) ? data.data.deduction_types : [];
      }
    } catch(_) {}
  }

  async function fetchEmployeeComponents(empId){
    const res = await fetch(`api/employees.php?action=components&id=${encodeURIComponent(empId)}`, { credentials: 'same-origin' });
    const data = await res.json();
    if (!data?.success) throw new Error(data?.message || 'Failed to load components');
    _empCompState.allowances = Array.isArray(data.data?.allowances) ? data.data.allowances : [];
    _empCompState.deductions = Array.isArray(data.data?.deductions) ? data.data.deductions : [];
  }

  function renderAllowances(){
    const tbody = document.querySelector('#empAllowancesTbl tbody');
    if (!tbody) return;
    const rows = _empCompState.allowances;
    tbody.innerHTML = rows.length ? rows.map(a=>{
      const isPct = a.calc_mode==='percent_basic' || a.calc_mode==='percent_gross';
      const amt = a.calc_mode==='fixed' ? Number(a.amount||0) : '';
      const rate = isPct ? Number(a.rate||0) : '';
      const type = escapeHtml(a.name || a.code || ('#'+a.allowance_type_id));
      return `<tr data-id="${a.id}" data-kind="allowance">
        <td>${type}</td>
        <td>${modeLabel(a.calc_mode)}</td>
        <td>${amt!==''? amt.toFixed(2): ''}</td>
        <td>${rate!==''? rate: ''}</td>
        <td>
          <button class="btn btn-sm btn-outline-primary me-1" data-action="edit-comp">Edit</button>
          <button class="btn btn-sm btn-outline-danger" data-action="delete-comp">Delete</button>
        </td>
      </tr>`;
    }).join('') : '<tr><td colspan="5" class="text-muted">No allowances</td></tr>';
  }

  function renderDeductions(){
    const tbody = document.querySelector('#empDeductionsTbl tbody');
    if (!tbody) return;
    const rows = _empCompState.deductions;
    tbody.innerHTML = rows.length ? rows.map(d=>{
      const isPct = d.calc_mode==='percent_basic' || d.calc_mode==='percent_gross';
      const amt = d.calc_mode==='fixed' ? Number(d.amount||0) : '';
      const rate = isPct ? Number(d.rate||0) : '';
      const type = escapeHtml(d.name || d.code || ('#'+d.deduction_type_id));
      return `<tr data-id="${d.id}" data-kind="deduction">
        <td>${type}</td>
        <td>${modeLabel(d.calc_mode)}</td>
        <td>${amt!==''? amt.toFixed(2): ''}</td>
        <td>${rate!==''? rate: ''}</td>
        <td>
          <button class="btn btn-sm btn-outline-primary me-1" data-action="edit-comp">Edit</button>
          <button class="btn btn-sm btn-outline-danger" data-action="delete-comp">Delete</button>
        </td>
      </tr>`;
    }).join('') : '<tr><td colspan="5" class="text-muted">No deductions</td></tr>';
  }

  function editRowHtml(kind, rec = {}){
    const isAllowance = kind==='allowance';
    const types = isAllowance ? _empCompState.aTypes : _empCompState.dTypes;
    const typeIdKey = isAllowance ? 'allowance_type_id' : 'deduction_type_id';
    const mode = rec.calc_mode || 'fixed';
    const amount = Number(rec.amount || 0);
    const rate = rec.rate ?? '';
    return `<tr data-id="${rec.id || 0}" data-kind="${kind}" data-editing="1">
      <td>
        <select class="form-select form-select-sm" data-field="type_id">${typeOptionsHtml(types, rec[typeIdKey] || '')}</select>
      </td>
      <td>
        <select class="form-select form-select-sm" data-field="calc_mode">${modeOptionsHtml(mode)}</select>
      </td>
      <td>
        <input type="number" step="0.01" class="form-control form-control-sm" data-field="amount" value="${mode==='fixed' ? String(amount) : ''}" ${mode==='fixed' ? '' : 'disabled'} />
      </td>
      <td>
        <input type="number" step="0.01" class="form-control form-control-sm" data-field="rate" value="${mode!=='fixed' ? String(rate||0) : ''}" ${mode!=='fixed' ? '' : 'disabled'} />
      </td>
      <td>
        <button class="btn btn-sm btn-success me-1" data-action="save-comp">Save</button>
        <button class="btn btn-sm btn-secondary" data-action="cancel-comp">Cancel</button>
      </td>
    </tr>`;
  }

  function switchToEditRow(tr){
    const kind = tr.getAttribute('data-kind');
    const id = parseInt(tr.getAttribute('data-id')||'0',10) || 0;
    let rec = {};
    if (id>0){
      const list = kind==='allowance'? _empCompState.allowances : _empCompState.deductions;
      rec = list.find(x=>String(x.id)===String(id)) || {};
    }
    const html = editRowHtml(kind, rec);
    const newTr = document.createElement('tr');
    newTr.innerHTML = html;
    tr.replaceWith(newTr.firstElementChild);
  }

  async function initEmployeePayrollComponentsTab(empId){
    _empCompState.empId = empId;
    const tab = document.getElementById('pv-payroll');
    if (!tab) return;
    if (tab.dataset.init === '1') {
      // Refresh on reopen
      await fetchEmployeeComponents(empId);
      renderAllowances();
      renderDeductions();
      return;
    }
    tab.dataset.init = '1';
    await loadComponentTypes();
    await fetchEmployeeComponents(empId);
    renderAllowances();
    renderDeductions();

    // Add buttons
    const addA = document.getElementById('empAddAllowance');
    const addD = document.getElementById('empAddDeduction');
    if (addA) addA.addEventListener('click', ()=>{
      const tbody = document.querySelector('#empAllowancesTbl tbody');
      if (!tbody) return;
      const hasEdit = tbody.querySelector('tr[data-editing="1"]');
      if (hasEdit) return;
      const row = document.createElement('tbody');
      row.innerHTML = editRowHtml('allowance', {});
      tbody.prepend(row.firstElementChild);
    });
    if (addD) addD.addEventListener('click', ()=>{
      const tbody = document.querySelector('#empDeductionsTbl tbody');
      if (!tbody) return;
      const hasEdit = tbody.querySelector('tr[data-editing="1"]');
      if (hasEdit) return;
      const row = document.createElement('tbody');
      row.innerHTML = editRowHtml('deduction', {});
      tbody.prepend(row.firstElementChild);
    });

    // Delegated handlers inside payroll tab
    tab.addEventListener('change', (e)=>{
      const t = e.target;
      if (!(t instanceof HTMLSelectElement || t instanceof HTMLInputElement)) return;
      const tr = t.closest('tr[data-editing="1"]'); if (!tr) return;
      if (t.getAttribute('data-field') === 'calc_mode'){
        const cm = t.value;
        const amt = tr.querySelector('input[data-field="amount"]');
        const rate = tr.querySelector('input[data-field="rate"]');
        if (amt) { amt.disabled = (cm !== 'fixed'); if (cm==='fixed') amt.value = amt.value || '0.00'; else amt.value=''; }
        if (rate) { rate.disabled = (cm === 'fixed'); if (cm!=='fixed') rate.value = rate.value || '0'; else rate.value=''; }
      }
    });

    tab.addEventListener('click', async (e)=>{
      const btn = e.target.closest('button'); if (!btn) return;
      const action = btn.getAttribute('data-action');
      const row = btn.closest('tr');
      if (action === 'edit-comp') {
        if (row) switchToEditRow(row);
      } else if (action === 'cancel-comp') {
        await fetchEmployeeComponents(empId);
        renderAllowances();
        renderDeductions();
      } else if (action === 'save-comp') {
        if (!row) return;
        const kind = row.getAttribute('data-kind');
        const id = parseInt(row.getAttribute('data-id')||'0',10) || 0;
        const typeId = parseInt(row.querySelector('[data-field="type_id"]').value||'0',10) || 0;
        const calcMode = row.querySelector('[data-field="calc_mode"]').value || 'fixed';
        const amtEl = row.querySelector('input[data-field="amount"]');
        const rateEl = row.querySelector('input[data-field="rate"]');
        const payload = { employee_id: Number(empId), calc_mode: calcMode };
        if (id>0) payload.id = id;
        if (kind === 'allowance') payload.allowance_type_id = typeId; else payload.deduction_type_id = typeId;
        if (calcMode === 'fixed') {
          const a = parseFloat(amtEl?.value || '0');
          if (isNaN(a)) { window.auth?.showNotification?.('Enter a valid amount', 'warning'); return; }
          payload.amount = a;
          payload.rate = null;
        } else {
          const r = parseFloat(rateEl?.value || '0');
          if (isNaN(r) || r <= 0) { window.auth?.showNotification?.('Enter a valid percentage rate', 'warning'); return; }
          payload.amount = 0;
          payload.rate = r;
        }
        try {
          const url = kind==='allowance' ? 'api/employees.php?action=save_allowance' : 'api/employees.php?action=save_deduction';
          const res = await fetch(url, { method: 'POST', headers: { 'Content-Type': 'application/json' }, credentials: 'same-origin', body: JSON.stringify(payload) });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Saved', 'success');
            await fetchEmployeeComponents(empId);
            renderAllowances();
            renderDeductions();
          } else {
            window.auth?.showNotification?.(data?.message || 'Save failed', 'error');
          }
        } catch (_) {
          window.auth?.showNotification?.('Save failed', 'error');
        }
      } else if (action === 'delete-comp') {
        const kind = row?.getAttribute('data-kind');
        const id = row ? (row.getAttribute('data-id')||'') : '';
        if (!id) return;
        if (!confirm('Delete this item?')) return;
        try {
          const url = kind==='allowance' ? `api/employees.php?action=delete_allowance&id=${encodeURIComponent(id)}` : `api/employees.php?action=delete_deduction&id=${encodeURIComponent(id)}`;
          const res = await fetch(url, { method: 'POST', credentials: 'same-origin' });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Deleted', 'success');
            await fetchEmployeeComponents(empId);
            renderAllowances();
            renderDeductions();
          } else {
            window.auth?.showNotification?.(data?.message || 'Delete failed', 'error');
          }
        } catch (_) {
          window.auth?.showNotification?.('Delete failed', 'error');
        }
      }
    });
  }

  window.pages.employees = { render };
})();
