// Dashboard page module
(function () {
  window.pages = window.pages || {};

  async function render(container) {
    const user = window.auth?.currentUser || null;
    const firstName = user?.first_name || 'User';

    container.innerHTML = `
      <div class="row">
        <div class="col-12">
          <div class="card mb-4">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h5 class="card-title mb-0">Welcome, ${firstName}</h5>
              <div>
                <button class="btn btn-sm btn-primary" id="refreshDashboardBtn">
                  <i class="fas fa-rotate"></i> Refresh
                </button>
              </div>
            </div>
            <div class="card-body">
              <p class="mb-0">This is your SmartQuantumHR dashboard. Quick stats and shortcuts appear here. Use the sidebar to navigate.</p>
            </div>
          </div>
        </div>
      </div>

      <div class="row g-3">
        <div class="col-md-3 col-sm-6">
          <div class="stats-card">
            <div class="icon" style="background: var(--primary-color)"><i class="fas fa-users"></i></div>
            <div class="value" id="statEmployees">--</div>
            <div class="label">Employees</div>
          </div>
        </div>
        <div class="col-md-3 col-sm-6">
          <div class="stats-card">
            <div class="icon" style="background: var(--success-color)"><i class="fas fa-star"></i></div>
            <div class="value" id="statAppraisals">--</div>
            <div class="label">Appraisals</div>
          </div>
        </div>
        <div class="col-md-3 col-sm-6">
          <div class="stats-card">
            <div class="icon" style="background: var(--warning-color)"><i class="fas fa-calendar-alt"></i></div>
            <div class="value" id="statLeave">--</div>
            <div class="label">Open Leave</div>
          </div>
        </div>
        <div class="col-md-3 col-sm-6">
          <div class="stats-card">
            <div class="icon" style="background: var(--info-color)"><i class="fas fa-graduation-cap"></i></div>
            <div class="value" id="statTraining">--</div>
            <div class="label">Trainings</div>
          </div>
        </div>
      </div>

      <div class="row mt-3">
        <div class="col-lg-4">
          <div class="card h-100">
            <div class="card-header"><h6 class="mb-0">My Profile</h6></div>
            <div class="card-body" id="profileCard">
              <div class="d-flex align-items-center gap-3">
                <img class="rounded-circle" src="assets/img/avatar-default.png" width="56" height="56" alt="avatar">
                <div>
                  <div class="fw-bold" id="pfName">--</div>
                  <div class="text-muted small" id="pfRoles">--</div>
                  <div class="text-muted small" id="pfDept">--</div>
                </div>
              </div>
            </div>
          </div>
        </div>
        <div class="col-lg-4">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Today's Birthdays</h6><small class="text-muted" id="bdCount"></small></div>
            <div class="card-body p-0">
              <ul class="list-group list-group-flush" id="bdList"><li class="list-group-item text-muted">Loading...</li></ul>
            </div>
          </div>
        </div>
        <div class="col-lg-4">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Work Anniversaries</h6><small class="text-muted" id="anCount"></small></div>
            <div class="card-body p-0">
              <ul class="list-group list-group-flush" id="anList"><li class="list-group-item text-muted">Loading...</li></ul>
            </div>
          </div>
        </div>
      </div>

      <div class="row mt-3">
        <div class="col-lg-7">
          <div class="card">
            <div class="card-header"><h6 class="mb-0">Department Distribution</h6></div>
            <div class="card-body">
              <canvas id="deptChart" height="140"></canvas>
            </div>
          </div>
        </div>
        <div class="col-lg-5">
          <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">New Hires (last 6 months)</h6>
              <small class="text-muted" id="hiresTotal"></small>
            </div>
            <div class="card-body">
              <canvas id="hiresChart" height="140"></canvas>
            </div>
          </div>
        </div>
      </div>

      <div class="row">
        <div class="col-12">
          <div class="card">
            <div class="card-header"><h6 class="mb-0">Insights</h6></div>
            <div class="card-body">
              <ul id="insightsList" class="mb-0"></ul>
            </div>
          </div>
        </div>
      </div>

      <div class="row mt-3">
        <div class="col-12">
          <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Company Holidays</h6>
              <div>
                <button class="btn btn-sm btn-outline-secondary" id="calPrev"><i class="fas fa-chevron-left"></i></button>
                <span class="mx-2" id="calTitle"></span>
                <button class="btn btn-sm btn-outline-secondary" id="calNext"><i class="fas fa-chevron-right"></i></button>
              </div>
            </div>
            <div class="card-body" id="holidayCalendarWrap"></div>
          </div>
        </div>
      </div>
    `;

    // Load analytics-driven stats and charts
    loadAnalytics().catch(() => {/* ignore for now */});

    // Init holidays calendar
    initHolidayCalendar();

    const btn = document.getElementById('refreshDashboardBtn');
    if (btn) {
      btn.addEventListener('click', () => loadAnalytics());
    }

  async function loadPeopleWidgets(){
    try{
      const res = await fetch('api/analytics.php?action=people', { credentials:'same-origin' });
      const js = await res.json();
      if (!js?.success) return;
      const p = js.data?.profile || {};
      const nm = [p.first_name, p.last_name].filter(Boolean).join(' ') || (p.username||'--');
      const roles = Array.isArray(p.roles) ? p.roles.map(r=>r.name).join(', ') : '--';
      const dept = p.department_name ? `Department: ${p.department_name}` : 'Department: --';
      const elN = document.getElementById('pfName'); if (elN) elN.textContent = nm;
      const elR = document.getElementById('pfRoles'); if (elR) elR.textContent = roles || '--';
      const elD = document.getElementById('pfDept'); if (elD) elD.textContent = dept;

      // Birthdays
      const bRows = Array.isArray(js.data?.birthdays_today) ? js.data.birthdays_today : [];
      const bdList = document.getElementById('bdList');
      const bdCount = document.getElementById('bdCount');
      if (bdCount) bdCount.textContent = bRows.length ? String(bRows.length) : '';
      if (bdList) bdList.innerHTML = bRows.length ? bRows.map(b=>{
        const name = `${escapeHtml(b.first_name||'')} ${escapeHtml(b.last_name||'')}`.trim();
        const dept = escapeHtml(b.department_name||'');
        const turning = '';
        const wishBtn = b.user_id ? `<button class="btn btn-sm btn-outline-primary" data-wish="${b.user_id}" data-type="birthday" data-name="${escapeHtml(name)}">Congratulate</button>` : '';
        return `<li class="list-group-item d-flex justify-content-between align-items-center"><div><div class="fw-semibold">${name}</div><div class="small text-muted">${dept}${turning}</div></div><div>${wishBtn}</div></li>`;
      }).join('') : '<li class="list-group-item text-muted">No birthdays today.</li>';

      // Anniversaries
      const aRows = Array.isArray(js.data?.anniversaries_today) ? js.data.anniversaries_today : [];
      const anList = document.getElementById('anList');
      const anCount = document.getElementById('anCount');
      if (anCount) anCount.textContent = aRows.length ? String(aRows.length) : '';
      if (anList) anList.innerHTML = aRows.length ? aRows.map(a=>{
        const name = `${escapeHtml(a.first_name||'')} ${escapeHtml(a.last_name||'')}`.trim();
        const dept = escapeHtml(a.department_name||'');
        const years = a.years != null ? ` • ${Number(a.years)} year(s)` : '';
        const wishBtn = a.user_id ? `<button class="btn btn-sm btn-outline-primary" data-wish="${a.user_id}" data-type="anniversary" data-name="${escapeHtml(name)}">Congratulate</button>` : '';
        return `<li class="list-group-item d-flex justify-content-between align-items-center"><div><div class="fw-semibold">${name}</div><div class="small text-muted">${dept}${years}</div></div><div>${wishBtn}</div></li>`;
      }).join('') : '<li class="list-group-item text-muted">No anniversaries today.</li>';

      // Wire wish buttons
      document.querySelectorAll('[data-wish]').forEach(btn=>{
        if (btn._wishBound) return; btn._wishBound = true;
        btn.addEventListener('click', async ()=>{
          const uid = Number(btn.getAttribute('data-wish'));
          const type = btn.getAttribute('data-type');
          const name = btn.getAttribute('data-name');
          const placeholder = type==='birthday' ? `Write a birthday wish to ${name}` : `Write an anniversary message to ${name}`;
          const msg = prompt(placeholder, type==='birthday' ? 'Happy Birthday! Wishing you a wonderful year ahead.' : 'Happy Work Anniversary! Thank you for your dedication.');
          if (!msg) return;
          try{
            const r = await fetch('api/messages.php?action=send', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ recipient_user_id: uid, subject: type==='birthday'?'Happy Birthday':'Happy Work Anniversary', content: msg }) });
            const j = await r.json();
            if (j?.success) { window.auth?.showNotification?.('Message sent','success'); }
            else { window.auth?.showNotification?.(j?.message||'Failed to send','error'); }
          }catch(_){ window.auth?.showNotification?.('Failed to send','error'); }
        });
      });
    }catch(_){ /* ignore */ }
  }

    // Load people sections
    loadPeopleWidgets().catch(()=>{});
  }

  async function loadAnalytics() {
    try {
      const res = await fetch('api/analytics.php', { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success && data.data) {
        const m = data.data;
        // Stats
        document.getElementById('statEmployees').textContent = m.headcount?.active ?? '--';
        document.getElementById('statAppraisals').textContent = m.workflows?.appraisals_in_progress ?? '--';
        document.getElementById('statLeave').textContent = m.workflows?.leave_pending ?? '--';
        document.getElementById('statTraining').textContent = m.workflows?.training_active ?? '--';

        // Charts
        renderDeptChart(m.department_distribution || []);
        renderHiresChart(m.hires_trend || []);
      }
    } catch {}

    // Insights
    try {
      const res2 = await fetch('api/analytics.php?action=insights', { credentials: 'same-origin' });
      const data2 = await res2.json();
      const ul = document.getElementById('insightsList');
      if (ul) {
        ul.innerHTML = '';
        if (data2?.success && Array.isArray(data2.data?.insights)) {
          data2.data.insights.forEach(line => {
            const li = document.createElement('li');
            li.textContent = line;
            ul.appendChild(li);
          });
        } else {
          const li = document.createElement('li'); li.textContent = 'No insights available.'; ul.appendChild(li);
        }
      }
    } catch {}
  }

  function renderDeptChart(rows) {
    const ctx = document.getElementById('deptChart');
    if (!ctx) return;
    const labels = rows.map(r => r.name);
    const values = rows.map(r => Number(r.count || 0));
    if (!labels.length) {
      ctx.replaceWith(document.createElement('div'));
      return;
    }
    const colors = labels.map((_, i) => `hsl(${(i*47)%360} 70% 55%)`);
    new Chart(ctx, {
      type: 'bar',
      data: { labels, datasets: [{ label: 'Employees', data: values, backgroundColor: colors }] },
      options: { responsive: true, plugins: { legend: { display: false } }, scales: { y: { beginAtZero: true } } }
    });
  }

  function renderHiresChart(rows) {
    const ctx = document.getElementById('hiresChart');
    if (!ctx) return;
    const labels = rows.map(r => r.ym);
    const values = rows.map(r => Number(r.hires || 0));
    const total = values.reduce((a,b)=>a+b,0);
    const totalEl = document.getElementById('hiresTotal');
    if (totalEl) totalEl.textContent = `Total: ${total}`;
    if (!labels.length) {
      ctx.replaceWith(document.createElement('div'));
      return;
    }
    new Chart(ctx, {
      type: 'line',
      data: { labels, datasets: [{ label: 'Hires', data: values, borderColor: 'rgba(79,70,229,0.9)', backgroundColor: 'rgba(79,70,229,0.2)', fill: true, tension: 0.3 }] },
      options: { responsive: true, plugins: { legend: { display: false } }, scales: { y: { beginAtZero: true } } }
    });
  }

  window.pages.dashboard = { render };

  // ===== Holidays Calendar ===== //
  let calState = null; // {year, month}
  function initHolidayCalendar(){
    try{
      const wrap = document.getElementById('holidayCalendarWrap');
      if (!wrap) return;
      const now = new Date();
      calState = { year: now.getFullYear(), month: now.getMonth() }; // 0-based
      document.getElementById('calPrev')?.addEventListener('click', ()=>{ shiftMonth(-1); });
      document.getElementById('calNext')?.addEventListener('click', ()=>{ shiftMonth(1); });
      renderHolidayCalendar();
    }catch(_){/* ignore */}
  }

  function shiftMonth(delta){
    if (!calState) return;
    let y = calState.year; let m = calState.month + delta;
    while (m < 0) { m += 12; y--; }
    while (m > 11) { m -= 12; y++; }
    calState.year = y; calState.month = m;
    renderHolidayCalendar();
  }

  async function renderHolidayCalendar(){
    const wrap = document.getElementById('holidayCalendarWrap');
    const title = document.getElementById('calTitle');
    if (!wrap || !calState || !title) return;
    const y = calState.year, m = calState.month; // 0-based
    const first = new Date(y, m, 1);
    const last = new Date(y, m + 1, 0);
    title.textContent = first.toLocaleString(undefined, { month: 'long', year: 'numeric' });

    // Load holidays for this month range
    const from = `${y}-${String(m+1).padStart(2,'0')}-01`;
    const to = `${y}-${String(m+1).padStart(2,'0')}-${String(last.getDate()).padStart(2,'0')}`;
    let holidays = {};
    let eventsByDate = {};
    try{
      const res = await fetch(`api/holidays.php?action=range&from=${from}&to=${to}`, { credentials:'same-origin' });
      const js = await res.json();
      if (js?.success && Array.isArray(js.data)) {
        js.data.forEach(r => { holidays[r.holiday_date] = r.name || 'Holiday'; });
      }
      // load planner events for same range
      try {
        const evRes = await fetch(`api/calendar.php?action=range&from=${from}&to=${to}`, { credentials:'same-origin' });
        const evJs = await evRes.json();
        if (evJs?.success && Array.isArray(evJs.data)){
          evJs.data.forEach(ev=>{
            try{
              const d = new Date(String(ev.start_at||'').replace(' ','T'));
              if (!isNaN(d)){ const key = `${d.getFullYear()}-${String(d.getMonth()+1).padStart(2,'0')}-${String(d.getDate()).padStart(2,'0')}`;
                (eventsByDate[key] = eventsByDate[key] || []).push({ title: ev.title||'Event' });
              }
            }catch(_){ /* ignore */ }
          });
        }
      } catch(_) { /* ignore */ }
    }catch(_){ /* ignore */ }

    // Build grid Mon..Sun
    const startDow = (first.getDay() + 6) % 7; // 0=Mon
    const daysInMonth = last.getDate();
    const cells = [];
    for (let i=0;i<startDow;i++) cells.push(null); // leading blanks
    for (let d=1; d<=daysInMonth; d++) cells.push(new Date(y, m, d));
    while (cells.length % 7 !== 0) cells.push(null);

    const head = ['Mon','Tue','Wed','Thu','Fri','Sat','Sun'];
    let html = '<div class="table-responsive"><table class="table table-bordered table-sm mb-0"><thead><tr>'
      + head.map(h=>`<th class="text-center">${h}</th>`).join('')
      + '</tr></thead><tbody>';
    for (let i=0; i<cells.length; i+=7){
      html += '<tr>';
      for (let j=0;j<7;j++){
        const dt = cells[i+j];
        if (!dt) { html += '<td class="bg-light"></td>'; continue; }
        const key = `${dt.getFullYear()}-${String(dt.getMonth()+1).padStart(2,'0')}-${String(dt.getDate()).padStart(2,'0')}`;
        const isHol = Object.prototype.hasOwnProperty.call(holidays, key);
        const isWeekend = dt.getDay()===0 || dt.getDay()===6;
        const badgeHol = isHol ? `<div class="badge bg-danger w-100">${escapeHtml(holidays[key])}</div>` : (isWeekend ? '<div class="badge bg-secondary w-100">Weekend</div>' : '');
        const evs = eventsByDate[key] || [];
        let evHtml = '';
        if (evs.length){
          const maxShow = 2;
          evs.slice(0,maxShow).forEach(e=>{ evHtml += `<div class="badge bg-primary w-100 text-truncate" title="${escapeHtml(e.title)}">${escapeHtml(e.title)}</div>`; });
          if (evs.length > maxShow){ evHtml += `<div class="small text-primary">+${evs.length-maxShow} more</div>`; }
        }
        html += `<td style="vertical-align:top;min-width:116px"><div class="small fw-bold">${dt.getDate()}</div>${badgeHol}${evHtml}</td>`;
      }
      html += '</tr>';
    }
    html += '</tbody></table></div>';
    wrap.innerHTML = html;
  }
  
  // helpers
  function escapeHtml(s){
    try { return String(s).replace(/[&<>"']/g, c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); } catch(_) { return String(s||''); }
  }
})();
