// Branches management page module
(function () {
  window.pages = window.pages || {};

  function canManage() {
    const role = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role;
    return ['super_admin','admin','hr_head','hr_officer'].includes(role);
  }

  // Caches
  let _empCache = null;
  let _branchCache = null;

  async function ensureCaches() {
    if (!_branchCache) await cacheBranches();
    if (!_empCache) await cacheEmployees();
  }

  async function cacheBranches() {
    try {
      const res = await fetch('api/organization.php?entity=branches', { credentials: 'same-origin' });
      const data = await res.json();
      _branchCache = data?.success && Array.isArray(data.data) ? data.data : [];
    } catch (_) { _branchCache = []; }
  }

  async function cacheEmployees() {
    try {
      const res = await fetch('api/employees.php', { credentials: 'same-origin' });
      const data = await res.json();
      _empCache = data?.success && Array.isArray(data.data) ? data.data : [];
    } catch (_) { _empCache = []; }
  }

  function escapeHtml(s){ return String(s ?? '').replace(/[&<>"']/g, c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c])); }
  function statusBadge(status){ const map={active:'status-badge status-active', inactive:'status-badge status-inactive'}; return `<span class="${map[status]||'status-badge'}">${escapeHtml(status||'')}</span>`; }

  async function render(container) {
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Branches</h5>
          ${canManage() ? `<button class="btn btn-sm btn-primary" id="toggleCreate"><i class="fas fa-plus"></i> New Branch</button>` : ''}
        </div>
        <div class="card-body">
          <ul class="nav nav-tabs" role="tablist">
            <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#tab-branch-list" type="button" role="tab"><i class="fas fa-code-branch"></i> Branches</button></li>
            ${canManage() ? `<li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-map" type="button" role="tab"><i class="fas fa-user-tag"></i> Map Employees</button></li>` : ''}
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-dist" type="button" role="tab"><i class="fas fa-diagram-project"></i> Distribution</button></li>
          </ul>
          <div class="tab-content mt-3">
            <div class="tab-pane fade show active" id="tab-branch-list" role="tabpanel">
              ${canManage() ? `
              <div id="createBox" class="mb-4" style="display:none;">
                <form id="createBranchForm" class="row g-3">
                  <div class="col-md-3">
                    <label class="form-label">Name</label>
                    <input type="text" name="name" class="form-control" required />
                  </div>
                  <div class="col-md-2">
                    <label class="form-label">Code</label>
                    <input type="text" name="code" class="form-control" />
                  </div>
                  <div class="col-md-4">
                    <label class="form-label">Location</label>
                    <input type="text" name="address" class="form-control" />
                  </div>
                  <div class="col-md-3">
                    <label class="form-label">Branch Manager</label>
                    <select name="manager_id" id="branchManagerSelect" class="form-select"><option value="">- None -</option></select>
                  </div>
                  <div class="col-12 d-flex justify-content-end">
                    <button type="button" class="btn btn-light me-2" id="cancelCreate">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create</button>
                  </div>
                </form>
                <hr/>
              </div>` : ''}

              <div class="table-responsive">
                <table class="table mb-0">
                  <thead>
                    <tr>
                      <th>Name</th>
                      <th>Code</th>
                      <th>Manager</th>
                      <th>Status</th>
                      <th style="width: 160px;">Actions</th>
                    </tr>
                  </thead>
                  <tbody id="branchTableBody"><tr><td colspan="5">Loading...</td></tr></tbody>
                </table>
              </div>
            </div>

            <div class="tab-pane fade" id="tab-map" role="tabpanel">
              ${canManage() ? `
              <div id="mapWrap">
                <div class="row g-3 align-items-end">
                  <div class="col-md-4">
                    <label class="form-label">Target Branch</label>
                    <select id="mapBranchSelect" class="form-select"></select>
                  </div>
                  <div class="col-md-8 text-end">
                    <button class="btn btn-sm btn-secondary me-2" id="mapReload"><i class="fas fa-rotate"></i> Reload</button>
                    <button class="btn btn-sm btn-primary" id="assignSelected"><i class="fas fa-user-check"></i> Assign Selected</button>
                  </div>
                  <div class="col-12">
                    <input id="mapSearch" class="form-control" placeholder="Search employees by name, email or #..." />
                  </div>
                  <div class="col-12">
                    <div class="table-responsive">
                      <table class="table table-sm align-middle">
                        <thead>
                          <tr>
                            <th style="width:28px"><input type="checkbox" id="mapSelectAll" /></th>
                            <th>#</th>
                            <th>Name</th>
                            <th>Branch</th>
                            <th>Position</th>
                            <th>Status</th>
                          </tr>
                        </thead>
                        <tbody id="mapEmpTableBody"><tr><td colspan="6">Loading...</td></tr></tbody>
                      </table>
                    </div>
                  </div>
                </div>
              </div>` : '<div class="text-muted">You do not have permission to map employees.</div>'}
            </div>

            <div class="tab-pane fade" id="tab-dist" role="tabpanel">
              <div class="d-flex justify-content-between align-items-center mb-2">
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" id="showUnassigned" />
                  <label class="form-check-label" for="showUnassigned">Show unassigned</label>
                </div>
                <button class="btn btn-sm btn-outline-secondary" id="distReload"><i class="fas fa-rotate"></i> Reload</button>
              </div>
              <div id="distContent">Loading...</div>
            </div>
          </div>
        </div>
      </div>

      <!-- Edit Branch Modal -->
      <div class="modal fade" id="branchEditModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Edit Branch</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
              <form id="editBranchForm" class="row g-3">
                <div class="col-md-3">
                  <label class="form-label">Name</label>
                  <input type="text" name="name" class="form-control" required />
                </div>
                <div class="col-md-2">
                  <label class="form-label">Code</label>
                  <input type="text" name="code" class="form-control" />
                </div>
                <div class="col-md-4">
                  <label class="form-label">Location</label>
                  <input type="text" name="address" class="form-control" />
                </div>
                <div class="col-md-3">
                  <label class="form-label">Branch Manager</label>
                  <select name="manager_id" id="editBranchManagerSelect" class="form-select"><option value="">- None -</option></select>
                </div>
                <div class="col-md-3">
                  <label class="form-label">Status</label>
                  <select name="status" class="form-select">
                    <option value="active">active</option>
                    <option value="inactive">inactive</option>
                  </select>
                </div>
              </form>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-light" data-bs-dismiss="modal">Close</button>
              <button type="button" class="btn btn-primary" id="saveBranchBtn">Save</button>
            </div>
          </div>
        </div>
      </div>
    `;

    if (canManage()) {
      const toggleBtn = container.querySelector('#toggleCreate');
      const box = container.querySelector('#createBox');
      const cancelBtn = container.querySelector('#cancelCreate');
      toggleBtn?.addEventListener('click', async () => {
        box.style.display = box.style.display === 'none' ? 'block' : 'none';
        if (box.style.display === 'block') {
          try { await ensureCaches(); populateManagerSelect(container.querySelector('#branchManagerSelect')); } catch (_) {}
        }
      });
      cancelBtn?.addEventListener('click', () => { box.style.display = 'none'; });

      const form = container.querySelector('#createBranchForm');
      form?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const body = formToJSON(new FormData(form));
        try {
          const res = await fetch('api/organization.php?entity=branches', {
            method: 'POST', headers: { 'Content-Type': 'application/json' }, credentials: 'same-origin', body: JSON.stringify(body)
          });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Branch created', 'success');
            form.reset(); box.style.display = 'none'; await loadBranches(container); try { await cacheBranches(); } catch(_){ }
          } else { window.auth?.showNotification?.(data?.message || 'Failed to create branch', 'error'); }
        } catch (_) { window.auth?.showNotification?.('Failed to create branch', 'error'); }
      });
    }

    // Delegate actions (edit / soft delete)
    container.addEventListener('click', async (e) => {
      const btn = e.target.closest('[data-action]'); if (!btn) return;
      const action = btn.getAttribute('data-action'); const id = btn.getAttribute('data-id');
      if (action === 'deactivate' && id) {
        if (!confirm('Deactivate this branch?')) return;
        try {
          const res = await fetch(`api/organization.php?entity=branches&id=${encodeURIComponent(id)}`, { method: 'DELETE', credentials: 'same-origin' });
          const data = await res.json();
          if (data?.success) { window.auth?.showNotification?.('Branch deactivated', 'success'); await loadBranches(container); try { await cacheBranches(); } catch(_){} }
          else { window.auth?.showNotification?.(data?.message || 'Action failed', 'error'); }
        } catch (_) { window.auth?.showNotification?.('Action failed', 'error'); }
      } else if (action === 'edit' && id) {
        try { await ensureCaches(); await openEditBranch(container, parseInt(id,10)); } catch (_) {}
      }
    });

    // Initial loads
    await loadBranches(container);
    try { await ensureCaches(); } catch(_){ }
    initMapTab(container);
    initDistributionTab(container);
  }

  function formToJSON(fd){ const obj={}; fd.forEach((v,k)=>{ obj[k]=v; }); return obj; }

  async function loadBranches(container){
    const tbody = container.querySelector('#branchTableBody');
    tbody.innerHTML = `<tr><td colspan="5">Loading...</td></tr>`;
    try {
      const res = await fetch('api/organization.php?entity=branches', { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success && Array.isArray(data.data)) {
        if (data.data.length === 0) { tbody.innerHTML = `<tr><td colspan="5" class="text-muted text-center">No branches</td></tr>`; return; }
        tbody.innerHTML = data.data.map((b)=> branchRow(b)).join('');
      } else { tbody.innerHTML = `<tr><td colspan="5" class="text-danger">Failed to load branches</td></tr>`; }
    } catch (_) { tbody.innerHTML = `<tr><td colspan="5" class="text-danger">Failed to load branches</td></tr>`; }
  }

  function branchRow(b){
    const status = b.status || 'active';
    const controls = canManage() ? `
      <div class="btn-group">
        <button class="btn btn-sm btn-outline-primary" data-action="edit" data-id="${b.id}"><i class="fas fa-pen"></i></button>
        <button class="btn btn-sm btn-danger" data-action="deactivate" data-id="${b.id}"><i class="fas fa-ban"></i></button>
      </div>` : '';
    return `
      <tr>
        <td>${escapeHtml(b.name || '')}</td>
        <td>${escapeHtml(b.code || '')}</td>
        <td>${escapeHtml(b.manager_name || '-')}</td>
        <td>${statusBadge(status)}</td>
        <td>${controls}</td>
      </tr>`;
  }

  function populateManagerSelect(selectEl){
    if (!selectEl) return;
    const opts = ['<option value="">- None -</option>']
      .concat((_empCache||[]).map(e=>`<option value="${e.id}">${escapeHtml(e.full_name || ((e.first_name||'')+' '+(e.last_name||'')))}</option>`));
    selectEl.innerHTML = opts.join('');
  }

  async function openEditBranch(container, id){
    const res = await fetch(`api/organization.php?entity=branches&id=${encodeURIComponent(id)}`, { credentials:'same-origin' });
    const data = await res.json();
    if (!data?.success) { window.auth?.showNotification?.('Branch not found', 'error'); return; }
    const b = data.data;
    const form = container.querySelector('#editBranchForm'); if (!form) return;
    // Populate
    form.querySelector('[name="name"]').value = b.name || '';
    form.querySelector('[name="code"]').value = b.code || '';
    form.querySelector('[name="address"]').value = b.address || '';
    populateManagerSelect(container.querySelector('#editBranchManagerSelect'));
    const mgrSel = container.querySelector('#editBranchManagerSelect'); if (mgrSel) mgrSel.value = b.manager_id ? String(b.manager_id) : '';
    form.querySelector('[name="status"]').value = b.status || 'active';
    const saveBtn = container.querySelector('#saveBranchBtn'); if (saveBtn) saveBtn.dataset.id = String(id);
    const modal = bootstrap.Modal.getOrCreateInstance(container.querySelector('#branchEditModal')); modal.show();
    if (!saveBtn._wired){
      saveBtn._wired = true;
      saveBtn.addEventListener('click', async ()=>{
        const payload = formToJSON(new FormData(form));
        try {
          const r = await fetch(`api/organization.php?entity=branches&id=${encodeURIComponent(saveBtn.dataset.id)}`, {
            method:'PUT', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload)
          });
          const out = await r.json();
          if (out?.success){ window.auth?.showNotification?.('Branch updated', 'success'); modal.hide(); await loadBranches(container); try { await cacheBranches(); } catch(_){ } }
          else { window.auth?.showNotification?.(out?.message || 'Update failed', 'error'); }
        } catch(_){ window.auth?.showNotification?.('Update failed', 'error'); }
      });
    }
  }

  function initMapTab(container){
    const mapBranchSel = container.querySelector('#mapBranchSelect');
    const mapBody = container.querySelector('#mapEmpTableBody');
    const mapSearch = container.querySelector('#mapSearch');
    const selectAll = container.querySelector('#mapSelectAll');
    const reloadBtn = container.querySelector('#mapReload');
    const assignBtn = container.querySelector('#assignSelected');
    if (!mapBranchSel || !mapBody) return;
    const render = ()=>{
      mapBranchSel.innerHTML = (_branchCache||[]).map(b=>`<option value="${b.id}">${escapeHtml(b.name)}</option>`).join('');
      const term = (mapSearch?.value||'').toLowerCase();
      const rows = (_empCache||[])
        .filter(e=>{ const t = `${e.employee_number||''} ${(e.full_name||((e.first_name||'')+' '+(e.last_name||''))).toLowerCase()} ${(e.email||'').toLowerCase()}`.toLowerCase(); return !term || t.includes(term); })
        .map(e=>{
          const br = e.branch_name || '-';
          return `<tr>
            <td><input type="checkbox" class="mapRowChk" data-id="${e.id}"/></td>
            <td>${escapeHtml(e.employee_number||'')}</td>
            <td>${escapeHtml(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())}</td>
            <td>${escapeHtml(br)}</td>
            <td>${escapeHtml(e.position_title||'-')}</td>
            <td>${statusBadge(e.employment_status||'active')}</td>
          </tr>`;
        }).join('');
      mapBody.innerHTML = rows || '<tr><td colspan="6" class="text-muted">No employees</td></tr>';
    };
    const refresh = async ()=>{ await ensureCaches(); render(); };
    mapSearch?.addEventListener('input', render);
    selectAll?.addEventListener('change', ()=>{ mapBody.querySelectorAll('.mapRowChk').forEach(ch=>{ ch.checked = !!selectAll.checked; }); });
    reloadBtn?.addEventListener('click', refresh);
    assignBtn?.addEventListener('click', async ()=>{
      const targetId = mapBranchSel.value; if (!targetId) { window.auth?.showNotification?.('Select a target branch','warning'); return; }
      const ids = Array.from(mapBody.querySelectorAll('.mapRowChk:checked')).map(ch=>ch.getAttribute('data-id'));
      if (!ids.length) { window.auth?.showNotification?.('Select at least one employee','warning'); return; }
      assignBtn.disabled = true;
      try {
        for (let i=0;i<ids.length;i++){
          const r = await fetch(`api/employees.php?id=${encodeURIComponent(ids[i])}`, { method:'PUT', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ branch_id: targetId }) });
          await r.json();
        }
        window.auth?.showNotification?.('Employees assigned','success'); await cacheEmployees(); render(); await rebuildDistribution(container);
      } catch(_) { window.auth?.showNotification?.('Assignment failed','error'); } finally { assignBtn.disabled = false; }
    });
    // initial
    refresh();
  }

  function initDistributionTab(container){
    const reloadBtn = container.querySelector('#distReload');
    const showUn = container.querySelector('#showUnassigned');
    const content = container.querySelector('#distContent');
    const render = ()=>{
      const empByBranch = new Map();
      (_branchCache||[]).forEach(b=>empByBranch.set(String(b.id), []));
      const unassigned = [];
      (_empCache||[]).forEach(e=>{ const key = e.branch_id ? String(e.branch_id) : null; if (key && empByBranch.has(key)) empByBranch.get(key).push(e); else unassigned.push(e); });
      let html = '';
      (_branchCache||[]).forEach(b=>{
        const list = empByBranch.get(String(b.id));
        html += `<div class="card mb-3">
          <div class="card-header d-flex justify-content-between align-items-center">
            <strong>${escapeHtml(b.name)}</strong>
            <span class="badge bg-primary">${list.length}</span>
          </div>
          ${list.length ? `<ul class="list-group list-group-flush">${list.map(e=>`<li class="list-group-item p-2">${escapeHtml(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())} <span class="text-muted">${escapeHtml(e.employee_number||'')}</span></li>`).join('')}</ul>` : '<div class="p-3 text-muted">No employees</div>'}
        </div>`;
      });
      if (showUn?.checked) {
        html += `<div class="card mb-3 border-warning">
          <div class="card-header d-flex justify-content-between align-items-center">
            <strong>Unassigned</strong>
            <span class="badge bg-warning text-dark">${unassigned.length}</span>
          </div>
          ${unassigned.length ? `<ul class="list-group list-group-flush">${unassigned.map(e=>`<li class="list-group-item p-2">${escapeHtml(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())}</li>`).join('')}</ul>` : '<div class="p-3 text-muted">None</div>'}
        </div>`;
      }
      content.innerHTML = html || '<div class="text-muted">No branches</div>';
    };
    const refresh = async ()=>{ await ensureCaches(); render(); };
    reloadBtn?.addEventListener('click', refresh);
    showUn?.addEventListener('change', render);
    refresh();
  }

  async function rebuildDistribution(container){
    try { await ensureCaches(); } catch(_){ }
    const content = container.querySelector('#distContent'); if (content) { const reloadBtn = container.querySelector('#distReload'); reloadBtn?.click(); if (!reloadBtn) { const evt = new Event('change'); container.querySelector('#showUnassigned')?.dispatchEvent(evt); } }
  }

  window.pages.branches = { render };
})();
