// Authentication System for SuperbHR Platform

class AuthManager {
  constructor() {
    this.currentUser = null;
    this.userRoles = {
      'super_admin': { level: 10, name: 'Super Admin' },
      'admin': { level: 9, name: 'Admin' },
      'hr_head': { level: 8, name: 'HR Head' },
      'hr_officer': { level: 6, name: 'HR Officer' },
      'manager': { level: 5, name: 'Manager' },
      'employee': { level: 1, name: 'Employee' }
    };
    // Idle/session keepalive state
    this.idleTimeoutMs = 10 * 60 * 1000; // 10 minutes
    this._idleTimerId = null;
    this._idleSetup = false;
    this._lastActivity = Date.now();
    this._lastTouchSent = 0;
    this._heartbeatIntervalId = null;
    this._activityHandler = null;
    this._activityEvents = ['mousemove','keydown','click','scroll','touchstart'];
    this.init();
  }

  init() {
    // Check if user is already logged in
    const savedUser = localStorage.getItem('superbhr_user');
    if (savedUser) {
      this.currentUser = JSON.parse(savedUser);
      this.showMainApp();
    } else {
      this.showLoginPage();
    }

    // Verify session with backend to keep in sync
    this.verifySession();

    // Setup login form handler
    this.setupLoginForm();
  }

  showMainApp() {
    document.getElementById('loginPage').style.display = 'none';
    document.getElementById('mainApp').style.display = 'flex';
    
    // Update user info in navbar
    this.updateUserInfo();
    
    // Generate sidebar menu based on role
    this.generateSidebarMenu();
    
    // Load role-based dashboard
    const roleSlug = this.currentUser.role_slug || this.currentUser.role;
    let defaultPage;
    if (roleSlug === 'super_admin') defaultPage = 'superadmin_dashboard';
    else if (['admin','hr_head','hr_officer'].includes(roleSlug)) defaultPage = 'dashboard_hr';
    else defaultPage = 'dashboard_staff';
    // Reset and start idle/keepalive watchers to avoid stale timers from previous session
    this.clearIdleDetection();
    this._lastActivity = Date.now();
    this._lastTouchSent = 0;
    this.setupIdleDetection();
    window.app.showPage(defaultPage);
  }

  showLoginPage() {
    const lp = document.getElementById('loginPage');
    const ma = document.getElementById('mainApp');
    if (lp) lp.style.display = 'flex';
    if (ma) ma.style.display = 'none';
    // Ensure any prior idle timers are stopped when on login screen
    this.clearIdleDetection();
    this._lastActivity = Date.now();
  }

  setupIdleDetection() {
    if (this._idleSetup) return;
    this._idleSetup = true;
    const bump = () => {
      this._lastActivity = Date.now();
      // throttle touch
      if (Date.now() - this._lastTouchSent >= 30000) { this.sendHeartbeat(); }
    };
    this._activityHandler = bump;
    this._activityEvents.forEach(ev => window.addEventListener(ev, this._activityHandler, { passive: true }));
    this._idleTimerId = setInterval(() => {
      const now = Date.now();
      const idleTime = now - this._lastActivity;
      if (idleTime >= this.idleTimeoutMs) {
        this.logout();
      }
    }, 1000);
    this._heartbeatIntervalId = setInterval(() => {
      const now = Date.now();
      const timeSinceLastTouch = now - this._lastTouchSent;
      if (timeSinceLastTouch >= 30000) {
        this.sendHeartbeat();
      }
    }, 30000);
  }

  clearIdleDetection() {
    if (this._idleTimerId) clearInterval(this._idleTimerId);
    if (this._heartbeatIntervalId) clearInterval(this._heartbeatIntervalId);
    if (this._activityHandler) { this._activityEvents.forEach(ev => window.removeEventListener(ev, this._activityHandler)); this._activityHandler = null; }
    this._idleSetup = false;
  }

  sendHeartbeat() {
    this._lastTouchSent = Date.now();
    fetch('api/auth.php?action=touch', { method: 'POST', credentials: 'same-origin' })
      .then(async (res) => {
        if (res.status === 401) { throw new Error('expired'); }
        const js = await res.json().catch(()=>({}));
        if (!js?.success || (js?.data && js.data.authenticated === false)) { throw new Error('expired'); }
      })
      .catch(() => {
        // Server session expired
        this.showNotification('Session expired due to inactivity', 'warning');
        this.logout();
      });
  }

  async logout() {
    try {
      await fetch('api/auth.php?action=logout', { method: 'POST', credentials: 'same-origin' });
    } catch (_) {}
    localStorage.removeItem('superbhr_user');
    this.currentUser = null;
    this.clearIdleDetection();
    this.showLoginPage();
    this.showNotification('Logged out successfully', 'info');
  }
  showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    document.body.appendChild(notification);
    setTimeout(() => notification.classList.add('show'), 100);
    setTimeout(() => {
      notification.classList.remove('show');
      setTimeout(() => document.body.removeChild(notification), 300);
    }, 3000);
  }

  setupLoginForm() {
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
      loginForm.addEventListener('submit', (e) => {
        e.preventDefault();
        this.handleLogin();
      });
    }
  }

  async handleLogin() {
    const username = document.getElementById('username')?.value?.trim();
    const password = document.getElementById('password')?.value;
    const submitBtn = document.querySelector('#loginForm button[type="submit"]');
    if (!username || !password) { this.showNotification('Enter username and password', 'warning'); return; }
    const originalText = submitBtn ? submitBtn.innerHTML : '';
    if (submitBtn){ submitBtn.innerHTML = '<span class="spinner"></span> Signing In...'; submitBtn.disabled = true; }
    try {
      const res = await fetch('api/auth.php?action=login', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ username, password }) });
      const data = await res.json();
      if (data?.success && data.data?.user) {
        this.currentUser = data.data.user;
        localStorage.setItem('superbhr_user', JSON.stringify(this.currentUser));
        this.showMainApp();
        this.showNotification('Login successful!', 'success');
      } else {
        this.showNotification(data?.message || 'Invalid credentials', 'error');
      }
    } catch (_e) {
      this.showNotification('Login failed. Please try again.', 'error');
    } finally {
      if (submitBtn){ submitBtn.innerHTML = originalText; submitBtn.disabled = false; }
    }
  }

  async verifySession() {
    try {
      const res = await fetch('api/auth.php?action=check', { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success && data.data?.authenticated) {
        const user = data.data.user;
        if (user) {
          this.currentUser = user;
          localStorage.setItem('superbhr_user', JSON.stringify(this.currentUser));
          this.showMainApp();
        }
      } else if (data?.success && data.data && data.data.authenticated === false) {
        localStorage.removeItem('superbhr_user');
        this.currentUser = null;
        // Stop any stale watchers so they don't cause immediate logout on next login
        this.clearIdleDetection();
        this.showLoginPage();
      }
    } catch (_) { /* ignore */ }
  }

  updateUserInfo() {
    if (this.currentUser) {
      const userName = document.getElementById('userName');
      const userAvatar = document.querySelector('.user-avatar');
      if (userName) userName.textContent = `${this.currentUser.first_name || ''} ${this.currentUser.last_name || ''}`.trim();
      if (userAvatar) userAvatar.src = this.currentUser.avatar || 'assets/img/avatar-default.png';
    }
  }

  generateSidebarMenu() {
    const sidebarMenu = document.getElementById('sidebarMenu');
    if (!sidebarMenu || !this.currentUser) return;
    const roleSlug = this.currentUser.role_slug || this.currentUser.role;
    const menuItems = this.getMenuItemsForRole(roleSlug);
    sidebarMenu.innerHTML = menuItems.map(item => `
      <li>
        <a href="#/${item.page}" onclick="app.showPage('${item.page}'); return false;">
          <i class="${item.icon}"></i>
          <span>${item.label}</span>
        </a>
      </li>
    `).join('');
  }

  getMenuItemsForRole(role) {
    const roleSlug = role;
    const dashPage = ['admin','hr_head','hr_officer'].includes(roleSlug) ? 'dashboard_hr' : 'dashboard_staff';
    const allMenuItems = [
      { page: 'superadmin_dashboard', icon: 'fas fa-gauge-high', label: 'Super Admin Dashboard', roles: ['super_admin'] },
      { page: 'companies', icon: 'fas fa-building', label: 'Companies', roles: ['super_admin'] },
      { page: dashPage, icon: 'fas fa-tachometer-alt', label: 'Dashboard', roles: ['admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'employees', icon: 'fas fa-users', label: 'Employees', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'departments', icon: 'fas fa-building', label: 'Departments', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'branches', icon: 'fas fa-code-branch', label: 'Branches', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'positions', icon: 'fas fa-briefcase', label: 'Positions', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'grades', icon: 'fas fa-layer-group', label: 'Grades', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'leave', icon: 'fas fa-calendar-alt', label: 'Leave Management', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'expenses', icon: 'fas fa-wallet', label: 'Expenses', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'loans', icon: 'fas fa-hand-holding-dollar', label: 'Loans', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'training', icon: 'fas fa-graduation-cap', label: 'Training', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'learning', icon: 'fas fa-book-open', label: 'Learning', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'appraisal', icon: 'fas fa-star', label: 'Appraisals', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'appraisal_templates', icon: 'fas fa-sitemap', label: 'Appraisal Templates', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'payroll', icon: 'fas fa-money-bill-wave', label: 'Payroll', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'mypayslips', icon: 'fas fa-file-invoice-dollar', label: 'My Payslips', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'messages', icon: 'fas fa-envelope', label: 'Messages', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'documents', icon: 'fas fa-file-shield', label: 'Documents', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'planner', icon: 'fas fa-calendar-check', label: 'Planner', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'company_settings', icon: 'fas fa-building-gear', label: 'Company Settings', roles: ['super_admin', 'admin'] },
      { page: 'email_settings', icon: 'fas fa-envelope', label: 'Email Settings', roles: ['super_admin', 'admin'] },
      { page: 'recruitment', icon: 'fas fa-user-plus', label: 'Recruitment', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'attendance', icon: 'fas fa-clock', label: 'Attendance', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'onboarding', icon: 'fas fa-person-walking-arrow-right', label: 'Onboarding', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'reports', icon: 'fas fa-chart-bar', label: 'Reports', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'announcements', icon: 'fas fa-bullhorn', label: 'Announcements', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'] },
      { page: 'hr_settings', icon: 'fas fa-cog', label: 'HR Settings', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] },
      { page: 'users', icon: 'fas fa-user-cog', label: 'User Management', roles: ['super_admin', 'admin', 'hr_head', 'hr_officer'] }
    ];
    return allMenuItems.filter(item => item.roles.includes(role));
  }

  hasPermission(requiredRole) {
    if (!this.currentUser) return false;
    const roleSlug = this.currentUser.role_slug || this.currentUser.role;
    const levels = { 'super_admin':10,'admin':9,'hr_head':8,'hr_officer':6,'manager':5,'employee':1 };
    const userLevel = levels[roleSlug] || 0;
    const requiredLevel = levels[requiredRole] || 0;
    return userLevel >= requiredLevel;
  }

  canApprove(type) {
    const role = (this.currentUser?.role_slug) || (this.currentUser?.role);
    switch (type) {
      case 'leave':
      case 'expenses':
      case 'training':
        return ['super_admin', 'admin', 'hr_head', 'manager'].includes(role);
      case 'appraisal':
      case 'payroll':
      case 'exit':
        return ['super_admin', 'admin', 'hr_head'].includes(role);
      default:
        return false;
    }
  }

  canEdit(type) {
    const role = (this.currentUser?.role_slug) || (this.currentUser?.role);
    switch (type) {
      case 'employees':
      case 'departments':
      case 'positions':
      case 'payroll':
        return ['super_admin', 'admin', 'hr_head', 'hr_officer'].includes(role);
      default:
        return ['super_admin', 'admin'].includes(role);
    }
  }
}

// Global logout function
function logout() {
    if (window.auth) {
        window.auth.logout();
    }
}

// Initialize auth manager when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.auth = new AuthManager();
});
