// Attendance mobile/geofence/QR page
(function(){
  window.pages = window.pages || {};

  async function render(container){
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Attendance</h5>
          <div class="d-flex gap-2">
            <button class="btn btn-sm btn-outline-primary" id="siteManage"><i class="fas fa-map-pin"></i> Manage Sites</button>
            <button class="btn btn-sm btn-outline-secondary" id="attReload"><i class="fas fa-rotate"></i> Reload</button>
          </div>
      <div class="modal fade" id="policyModal" tabindex="-1">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header"><h5 class="modal-title">Attendance Policy</h5><button class="btn-close" data-bs-dismiss="modal"></button></div>
            <div class="modal-body">
              <div class="form-check form-switch mb-2">
                <input class="form-check-input" type="checkbox" id="polSelfie">
                <label class="form-check-label" for="polSelfie">Require selfie for clock-in/out</label>
              </div>
              <div class="form-check form-switch mb-2">
                <input class="form-check-input" type="checkbox" id="polQr">
                <label class="form-check-label" for="polQr">Require QR token</label>
              </div>
              <div class="border rounded p-2 mt-3">
                <div class="fw-bold mb-1">Company Geofence (optional)</div>
                <div class="row g-2">
                  <div class="col-md-4"><label class="form-label">Latitude</label><input id="polLat" class="form-control" placeholder="e.g. 5.6" /></div>
                  <div class="col-md-4"><label class="form-label">Longitude</label><input id="polLng" class="form-control" placeholder="e.g. -0.2" /></div>
                  <div class="col-md-4"><label class="form-label">Radius (m)</label><input id="polRadius" type="number" class="form-control" placeholder="200" /></div>
                </div>
                <div class="form-text">If set, employees must be within this radius unless a site geofence overrides it.</div>
              </div>
            </div>
            <div class="modal-footer">
              <button class="btn btn-light" data-bs-dismiss="modal">Close</button>
              <button class="btn btn-primary" id="polSave">Save Policy</button>
            </div>
          </div>
        </div>
      </div>
      <div class="modal fade" id="photoModal" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered modal-lg">
          <div class="modal-content">
            <div class="modal-body p-0"><img id="photoModalImg" src="" alt="Selfie" class="img-fluid w-100"/></div>
          </div>
        </div>
      </div>
        </div>
        <div class="card-body">
          <div class="row g-3">
            <div class="col-lg-5">
              <div class="border rounded p-3 mb-3">
                <div class="d-flex justify-content-between align-items-center">
                  <strong>Clock</strong>
                  <div class="d-flex align-items-center gap-2">
                    <button class="btn btn-sm btn-outline-secondary" id="policyBtn" style="display:none"><i class="fas fa-gear"></i></button>
                    <button class="btn btn-sm btn-success" id="btnIn"><i class="fas fa-play"></i> In</button>
                    <button class="btn btn-sm btn-danger" id="btnOut"><i class="fas fa-stop"></i> Out</button>
                  </div>
                </div>
                <div class="row g-2 mt-2">
                  <div class="col-12">
                    <label class="form-label">Site (optional)</label>
                    <select id="attSite" class="form-select"><option value="">Auto-detect nearest</option></select>
                    <div class="form-text" id="attNearest" style="display:none;"></div>
                  </div>
                  <div class="col-md-6">
                    <label class="form-label">QR Token (if required)</label>
                    <input id="attQr" class="form-control" placeholder="Scan/enter QR" />
                  </div>
                  <div class="col-md-6">
                    <label class="form-label" id="lblSelfie">Selfie (optional)</label>
                    <input type="file" accept="image/*" capture="user" id="attPhoto" class="form-control" />
                  </div>
                  <div class="col-12" id="camSection" style="display:none;">
                    <div class="border rounded p-2">
                      <div class="d-flex align-items-center justify-content-between">
                        <strong>Selfie Camera</strong>
                        <div class="btn-group btn-group-sm">
                          <button class="btn btn-outline-primary" id="camStart">Open Camera</button>
                          <button class="btn btn-outline-success" id="camSnap" disabled>Take Photo</button>
                          <button class="btn btn-outline-secondary" id="camStop" disabled>Close</button>
                        </div>
                      </div>
                      <div class="row mt-2 g-2">
                        <div class="col-md-6"><video id="camVideo" autoplay playsinline style="width:100%;max-height:220px;background:#000;border-radius:6px"></video></div>
                        <div class="col-md-6">
                          <canvas id="camCanvas" style="width:100%;max-height:220px;border:1px solid #ddd;border-radius:6px;display:none"></canvas>
                          <img id="camPreview" alt="Selfie preview" style="width:100%;max-height:220px;display:none;border-radius:6px;border:1px solid #ddd"/>
                        </div>
                      </div>
                      <div class="form-text">If required by policy, a live selfie is captured and submitted with your attendance.</div>
                    </div>
                  </div>
                </div>
                <div class="small text-muted mt-2">Location permission recommended for geofence. Your device is fingerprinted to help fraud detection.</div>
              </div>

              <div class="border rounded p-3">
                <div class="d-flex justify-content-between align-items-center">
                  <strong>Exceptions</strong>
                  <button class="btn btn-sm btn-outline-secondary" id="exReload"><i class="fas fa-rotate"></i></button>
                </div>
                <div class="table-responsive mt-2">
                  <table class="table table-sm align-middle">
                    <thead><tr><th>When</th><th>Type</th><th>Reason</th><th>Site</th><th>Selfie</th><th style="width:160px;">Actions</th></tr></thead>
                    <tbody id="exBody"><tr><td colspan="6">Loading...</td></tr></tbody>
                  </table>
                </div>
              </div>
            </div>
            <div class="col-lg-7">
              <div class="border rounded p-3 mb-3">
                <div class="d-flex justify-content-between align-items-center">
                  <strong>Map</strong>
                  <div class="d-flex align-items-center gap-2">
                    <div class="form-check" style="display: none" id="teamWrap"><input class="form-check-input" type="checkbox" id="mapTeam"><label class="form-check-label" for="mapTeam">Team</label></div>
                    <input type="date" id="mapStart" class="form-control form-control-sm" />
                    <input type="date" id="mapEnd" class="form-control form-control-sm" />
                    <button class="btn btn-sm btn-outline-secondary" id="mapReload"><i class="fas fa-rotate"></i></button>
                  </div>
                </div>
                <div id="attMap" style="height:400px;" class="mt-2 rounded border"></div>
              </div>

              <div class="border rounded p-3">
                <div class="d-flex justify-content-between align-items-center">
                  <strong>History (recent)</strong>
                  <button class="btn btn-sm btn-outline-secondary" id="histReload"><i class="fas fa-rotate"></i></button>
                </div>
                <div class="table-responsive mt-2">
                  <table class="table table-sm align-middle">
                    <thead><tr><th>Date</th><th>Status</th><th>Address</th><th>Notes</th><th>Selfie</th></tr></thead>
                    <tbody id="attHist"><tr><td colspan="5">Loading...</td></tr></tbody>
                  </table>
                </div>
              </div>
              <div class="border rounded p-3 mt-3" id="allHistWrap" style="display:none;">
                <div class="d-flex justify-content-between align-items-center">
                  <strong>All Employees History</strong>
                  <div class="d-flex align-items-center gap-2">
                    <input type="date" id="allStart" class="form-control form-control-sm" />
                    <input type="date" id="allEnd" class="form-control form-control-sm" />
                    <input type="text" id="allSearch" class="form-control form-control-sm" placeholder="Search name/number/address/type" style="min-width:220px" />
                    <button class="btn btn-sm btn-outline-secondary" id="allReload"><i class="fas fa-rotate"></i></button>
                  </div>
                </div>
                <div class="table-responsive mt-2">
                  <table class="table table-sm align-middle">
                    <thead><tr><th>Employee</th><th>Number</th><th>Time</th><th>Type</th><th>Site</th><th>Address</th><th>Selfie</th><th>Status</th><th>Approval</th></tr></thead>
                    <tbody id="attAllHist"><tr><td colspan="9">Loading...</td></tr></tbody>
                  </table>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="modal fade" id="siteModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
          <div class="modal-content">
            <div class="modal-header"><h5 class="modal-title">Sites</h5><button class="btn-close" data-bs-dismiss="modal"></button></div>
            <div class="modal-body">
              <div class="mb-2"><button class="btn btn-sm btn-primary" id="siteAdd"><i class="fas fa-plus"></i> New Site</button></div>
              <div class="table-responsive"><table class="table table-sm align-middle"><thead><tr><th>Name</th><th>Code</th><th>Lat</th><th>Lng</th><th>Radius(m)</th><th>Status</th><th style="width:180px;">Actions</th></tr></thead><tbody id="siteBody"><tr><td colspan="7">Loading...</td></tr></tbody></table></div>
            </div>
          </div>
        </div>
      </div>
    `;

    const btnIn = container.querySelector('#btnIn');
    const btnOut = container.querySelector('#btnOut');
    const qrInput = container.querySelector('#attQr');
    const fileInput = container.querySelector('#attPhoto');
    const siteSel = container.querySelector('#attSite');
    // Policy and camera state
    let policy = { photo_required: false, qr_required: false };
    let camStream = null, livePhotoData = null;
    const camSection = container.querySelector('#camSection');
    const camStart = container.querySelector('#camStart');
    const camSnap = container.querySelector('#camSnap');
    const camStop = container.querySelector('#camStop');
    const camVideo = container.querySelector('#camVideo');
    const camCanvas = container.querySelector('#camCanvas');
    const camPreview = container.querySelector('#camPreview');

    const teamWrap = container.querySelector('#teamWrap');
    if (isManagerOrHr()) teamWrap.style.display = '';

    btnIn.addEventListener('click', ()=> doClock('in'));
    btnOut.addEventListener('click', ()=> doClock('out'));
    container.querySelector('#attReload').addEventListener('click', ()=>{ loadSites(); loadHistory(); loadMap(); loadExceptions(); });
    container.querySelector('#histReload').addEventListener('click', loadHistory);
    container.querySelector('#mapReload').addEventListener('click', loadMap);
    container.querySelector('#exReload').addEventListener('click', loadExceptions);
    // HR-only all history
    const allWrap = container.querySelector('#allHistWrap');
    if (isHr()){
      allWrap.style.display = '';
      container.querySelector('#allReload').addEventListener('click', loadAllHistory);
      container.querySelector('#allStart').addEventListener('change', loadAllHistory);
      container.querySelector('#allEnd').addEventListener('change', loadAllHistory);
      container.querySelector('#allSearch').addEventListener('input', loadAllHistory);
    }
    const manageBtn = container.querySelector('#siteManage');
    if (!canManageSites()) manageBtn.style.display = 'none';
    manageBtn?.addEventListener('click', openSites);

    // Load attendance policy (selfie/qr/geofence)
    await loadPolicy();

    // Policy button (Admin/HR Head only)
    const polBtn = container.querySelector('#policyBtn');
    if (canEditPolicy()) { polBtn.style.display = ''; polBtn.addEventListener('click', openPolicyModal); }

    // Camera controls
    camStart?.addEventListener('click', openCamera);
    camSnap?.addEventListener('click', takeSnapshot);
    camStop?.addEventListener('click', closeCamera);

    // Leaflet map
    let map = L.map('attMap');
    let markersLayer = L.layerGroup().addTo(map);
    let sitesLayer = L.layerGroup().addTo(map);
    try {
      const coords = await getCoords();
      map.setView([coords.lat||0, coords.lng||0], coords.lat? 14 : 2);
    } catch(_) { map.setView([0,0], 2); }
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', { maxZoom: 19, attribution: '&copy; OpenStreetMap' }).addTo(map);
    document.getElementById('mapTeam')?.addEventListener('change', loadMap);
    document.getElementById('mapStart')?.addEventListener('change', loadMap);
    document.getElementById('mapEnd')?.addEventListener('change', loadMap);

    async function doClock(type){
      const coords = await getCoords().catch(()=>({lat:null,lng:null}));
      let photoData = livePhotoData || (await readPhoto(fileInput.files?.[0]).catch(()=>null));
      if (policy.photo_required && !photoData){ note('Selfie is required by company policy. Please take a selfie or upload one.', 'warning'); return; }
      const device_fp = await fingerprint();
      const payload = { type, lat: coords.lat, lng: coords.lng, qr: qrInput.value.trim(), photo_data: photoData, device_fp, site_id: siteSel.value || '' };
      const res = await fetch('api/attendance.php?action=clock', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (res?.success){
        if (res.data?.allowed === false){ note('Clock recorded but flagged: '+(res.data?.reason||'blocked'), 'warning'); }
        else { note('Clock recorded at '+(res.data?.address||'location'), 'success'); }
        const nearest = container.querySelector('#attNearest');
        if (res.data?.site){ nearest.style.display='block'; nearest.textContent = `Nearest site: ${res.data.site.name} (${Math.round(res.data.site.distance_m||0)}m)`; }
        await Promise.all([loadSites(), loadHistory(), loadMap(), loadExceptions()]);
      } else {
        note(res?.message||'Clock failed', 'error');
      }
    }

    async function loadPolicy(){
      try {
        const r = await fetch('api/attendance.php?action=policy', { credentials: 'same-origin' });
        const j = await r.json();
        if (j?.success){
          policy = Object.assign({ photo_required:false, qr_required:false }, j.data||{});
          const lbl = container.querySelector('#lblSelfie');
          if (lbl) lbl.textContent = `Selfie (${policy.photo_required? 'required':'optional'})`;
          if (camSection) camSection.style.display = policy.photo_required ? '' : 'none';
        }
      } catch(_) {}
    }

    function canEditPolicy(){
      const r = (window.auth?.currentUser?.role_slug) || (window.auth?.currentUser?.role);
      return ['super_admin','admin','hr_head'].includes(r);
    }

    async function openPolicyModal(){
      try{
        // Ensure latest policy loaded
        await loadPolicy();
        const m = bootstrap.Modal.getOrCreateInstance(document.getElementById('policyModal'));
        const selfCb = document.getElementById('polSelfie');
        const qrCb = document.getElementById('polQr');
        const latI = document.getElementById('polLat');
        const lngI = document.getElementById('polLng');
        const radI = document.getElementById('polRadius');
        if (selfCb) selfCb.checked = !!policy.photo_required;
        if (qrCb) qrCb.checked = !!policy.qr_required;
        if (latI) latI.value = policy.geo?.lat ?? '';
        if (lngI) lngI.value = policy.geo?.lng ?? '';
        if (radI) radI.value = policy.geo?.radius_m ?? '';
        const saveBtn = document.getElementById('polSave');
        if (saveBtn && !saveBtn._bound){
          saveBtn._bound = true;
          saveBtn.addEventListener('click', async ()=>{
            const payload = {
              photo_required: !!document.getElementById('polSelfie')?.checked,
              qr_required: !!document.getElementById('polQr')?.checked,
              geo: {
                lat: (function(){ const v = document.getElementById('polLat')?.value.trim(); return v===''? null : Number(v); })(),
                lng: (function(){ const v = document.getElementById('polLng')?.value.trim(); return v===''? null : Number(v); })(),
                radius_m: (function(){ const v = document.getElementById('polRadius')?.value.trim(); return v===''? null : Number(v); })(),
              }
            };
            try{
              const r = await fetch('api/attendance.php?action=policy_save', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
              const j = await r.json();
              if (j?.success){ note('Policy saved','success'); await loadPolicy(); m.hide(); }
              else { note(j?.message || 'Failed to save policy','error'); }
            }catch(_){ note('Failed to save policy','error'); }
          });
        }
        m.show();
      }catch(_){ note('Unable to open policy','error'); }
    }

    async function openCamera(){
      try{
        if (!navigator.mediaDevices?.getUserMedia){ note('Camera not available in this browser. You can upload a selfie file.', 'warning'); return; }
        camStream = await navigator.mediaDevices.getUserMedia({ video: { facingMode: 'user', width: { ideal: 640 }, height: { ideal: 480 } } });
        if (camVideo){ camVideo.srcObject = camStream; }
        if (camSnap) camSnap.disabled = false;
        if (camStop) camStop.disabled = false;
        if (camStart) camStart.disabled = true;
        if (camPreview){ camPreview.style.display = 'none'; camPreview.src = ''; }
        livePhotoData = null;
      } catch(e){ note('Unable to access camera. Please allow permission or upload from file.', 'warning'); }
    }

    function closeCamera(){
      try { if (camStream){ camStream.getTracks().forEach(t=>t.stop()); } } catch(_){ }
      camStream = null;
      if (camVideo) camVideo.srcObject = null;
      if (camSnap) camSnap.disabled = true;
      if (camStop) camStop.disabled = true;
      if (camStart) camStart.disabled = false;
    }

    function takeSnapshot(){
      try{
        if (!camVideo) return;
        const w = camVideo.videoWidth || 640; const h = camVideo.videoHeight || 480;
        if (camCanvas){ camCanvas.width = w; camCanvas.height = h; const ctx = camCanvas.getContext('2d'); ctx.drawImage(camVideo, 0, 0, w, h); }
        livePhotoData = camCanvas ? (camCanvas.toDataURL('image/jpeg', 0.85) || camCanvas.toDataURL('image/png')) : null;
        if (camPreview && livePhotoData){ camPreview.src = livePhotoData; camPreview.style.display = ''; }
      }catch(_){ livePhotoData = null; }
    }

    async function loadAllHistory(){
      const body = container.querySelector('#attAllHist'); if (!body) return;
      body.innerHTML = '<tr><td colspan="9">Loading...</td></tr>';
      const start = document.getElementById('allStart')?.value || '';
      const end = document.getElementById('allEnd')?.value || '';
      const qs = `${start? 'start='+encodeURIComponent(start)+'&':''}${end? 'end='+encodeURIComponent(end):''}`;
      const res = await fetch('api/attendance.php?action=events'+(qs? '&'+qs:''), { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
      if (!res?.success){ body.innerHTML = '<tr><td colspan="8" class="text-danger">Failed to load</td></tr>'; return; }
      let rows = res.data || [];
      const q = (document.getElementById('allSearch')?.value || '').toLowerCase();
      if (q){
        rows = rows.filter(ev=>{
          const s = [ev.employee_name, ev.employee_number, ev.address, ev.type, ev.site_name].map(x=> String(x||'').toLowerCase()).join(' ');
          return s.includes(q);
        });
      }
      body.innerHTML = rows.map(ev=> `
        <tr>
          <td>${esc(ev.employee_name||'')}</td>
          <td>${esc(ev.employee_number||'')}</td>
          <td>${esc(ev.event_time||'')}</td>
          <td>${esc(ev.type||'')}</td>
          <td>${esc(ev.site_name||'-')}</td>
          <td>${esc(ev.address||'')}</td>
          <td>${ev.photo_path? `<a href="#" data-photo="${esc(ev.photo_path)}" title="View"><img src="${esc(ev.photo_path)}" alt="Selfie" style="height:48px;border-radius:6px;border:1px solid #ddd"/></a>` : ''}</td>
          <td>${ev.allowed? '<span class="badge bg-success">OK</span>' : '<span class="badge bg-warning text-dark">Exception</span>'} ${!ev.allowed? esc('('+ (ev.reason||'') +')'):''}</td>
          <td>${esc(ev.approved_status||'')}</td>
        </tr>
      `).join('') || '<tr><td colspan="9" class="text-muted">No results</td></tr>';
      body.querySelectorAll('[data-photo]')?.forEach(a=>{ if (a._bound) return; a._bound=true; a.addEventListener('click', (e)=>{ e.preventDefault(); openPhotoModal(a.getAttribute('data-photo')); }); });
    }

    async function loadSites(){
      const res = await fetch('api/attendance.php?action=sites', { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
      const list = res?.success ? (res.data||[]) : [];
      siteSel.innerHTML = ['<option value="">Auto-detect nearest</option>']
        .concat(list.map(s=> `<option value="${s.id}">${esc(s.name)}${s.code? ' ['+esc(s.code)+']':''}</option>`)).join('');
    }

    async function loadHistory(){
      const body = container.querySelector('#attHist');
      body.innerHTML = '<tr><td colspan="5">Loading...</td></tr>';
      try{
        const res = await fetch('api/attendance.php?action=history', { credentials:'same-origin' });
        const js = await res.json();
        if (!js?.success){ body.innerHTML = '<tr><td colspan="5" class="text-danger">Failed to load</td></tr>'; return; }
        const rows = js.data || [];
        body.innerHTML = rows.map(r=> `
          <tr>
            <td>${esc(r.date||'')}</td>
            <td>${esc(r.status||'')}</td>
            <td>${esc(r.address||'')}</td>
            <td>${esc(r.reason||'')}</td>
            <td>${r.photo_path? `<a href="#" data-photo="${esc(r.photo_path)}" title="View"><img src="${esc(r.photo_path)}" alt="Selfie" style="height:48px;border-radius:6px;border:1px solid #ddd"/></a>` : ''}</td>
          </tr>`).join('') || '<tr><td colspan="5" class="text-muted">No records</td></tr>';
        body.querySelectorAll('[data-photo]')?.forEach(a=>{ if (a._bound) return; a._bound=true; a.addEventListener('click', (e)=>{ e.preventDefault(); openPhotoModal(a.getAttribute('data-photo')); }); });
      }catch(_){ body.innerHTML = '<tr><td colspan="5" class="text-danger">Failed to load</td></tr>'; }
    }

    async function loadMap(){
      const team = document.getElementById('mapTeam')?.checked ? 1 : 0;
      const start = document.getElementById('mapStart')?.value || '';
      const end = document.getElementById('mapEnd')?.value || '';
      const qs = `team=${team}${start? '&start='+encodeURIComponent(start):''}${end? '&end='+encodeURIComponent(end):''}`;
      const res = await fetch('api/attendance.php?action=events&'+qs, { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
      markersLayer.clearLayers();
      sitesLayer.clearLayers();
      if (!res?.success) return;
      // draw active site geofences
      try {
        const sites = await fetch('api/attendance.php?action=sites', { credentials:'same-origin' }).then(r=>r.json());
        (sites?.data||[]).forEach(s=>{
          if (s.center_lat!=null && s.center_lng!=null && s.radius_m){
            L.circle([s.center_lat, s.center_lng], { radius: Number(s.radius_m||0), color:'#1976d2', weight:1, fillOpacity:0.05 }).addTo(sitesLayer).bindTooltip(esc(s.name||''));
          }
        });
      } catch(_){ }
      (res.data||[]).forEach(ev=>{
        if (ev.lat==null || ev.lng==null) return;
        const color = ev.allowed? '#2e7d32' : (ev.approved_status==='pending'? '#f57c00' : (ev.approved_status==='rejected'? '#c62828' : '#2e7d32'));
        const m = L.circleMarker([ev.lat, ev.lng], { radius: 7, color, fillColor: color, fillOpacity: 0.8 }).addTo(markersLayer);
        const canDecide = isManagerOrHr();
        const decisionBtns = (!ev.allowed && ev.approved_status==='pending' && canDecide) ? `
          <div class="mt-2">
            <button class="btn btn-sm btn-success" data-evt-act="approve" data-id="${ev.id}">Approve</button>
            <button class="btn btn-sm btn-danger ms-2" data-evt-act="reject" data-id="${ev.id}">Reject</button>
          </div>` : '';
        const photoHtml = ev.photo_path ? `<div class="mt-2"><img src="${esc(ev.photo_path)}" alt="Selfie" style="max-width:100%;max-height:180px;border-radius:6px;border:1px solid #ddd"/></div>` : '';
        m.bindPopup(`
          <div><strong>${esc(ev.employee_name||'')}</strong> <span class="text-muted">${esc(ev.employee_number||'')}</span></div>
          <div>${esc(ev.type||'')} • ${esc(ev.event_time||'')}</div>
          <div>${esc(ev.address||'')}</div>
          <div>Site: ${esc(ev.site_name||'-')}</div>
          <div>Status: ${ev.allowed? 'OK' : 'Exception'} ${!ev.allowed? '('+esc(ev.reason||'')+')':''} • ${esc(ev.approved_status||'')}</div>
          ${photoHtml}
          ${decisionBtns}
        `);
        m.on('popupopen', ()=>{
          document.querySelectorAll('[data-evt-act]')?.forEach(btn=>{
            btn.addEventListener('click', async ()=>{
              const id = btn.getAttribute('data-id');
              const decision = btn.getAttribute('data-evt-act')==='approve' ? 'approve' : 'reject';
              const note = prompt('Optional note') || '';
              const out = await fetch('api/attendance.php?action=exception_decide', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ event_id: Number(id), decision, note }) }).then(r=>r.json()).catch(()=>null);
              if (out?.success){ note('Updated','success'); loadMap(); loadExceptions(); }
              else { note(out?.message||'Update failed','error'); }
            });
          });
        });
      });
    }

    async function loadExceptions(){
      const body = container.querySelector('#exBody');
      body.innerHTML = '<tr><td colspan="6">Loading...</td></tr>';
      const res = await fetch('api/attendance.php?action=exceptions', { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
      if (!res?.success){ body.innerHTML = '<tr><td colspan="6" class="text-danger">Failed to load</td></tr>'; return; }
      const canDecide = isManagerOrHr();
      body.innerHTML = (res.data||[]).map(ev=> `
        <tr>
          <td>${esc(ev.event_time||'')}</td>
          <td>${esc(ev.type||'')}</td>
          <td>${esc(ev.reason||'')}</td>
          <td>${esc(ev.site_name||'-')}</td>
          <td>${ev.photo_path? `<a href="#" data-photo="${esc(ev.photo_path)}" title="View"><img src="${esc(ev.photo_path)}" alt="Selfie" style="height:48px;border-radius:6px;border:1px solid #ddd"/></a>` : ''}</td>
          <td>${canDecide? `<div class="btn-group btn-group-sm"><button class="btn btn-success" data-ex="approve" data-id="${ev.id}">Approve</button><button class="btn btn-danger" data-ex="reject" data-id="${ev.id}">Reject</button></div>` : '-'}</td>
        </tr>
      `).join('') || '<tr><td colspan="6" class="text-muted">No pending exceptions</td></tr>';
      body.querySelectorAll('[data-photo]')?.forEach(a=>{ if (a._bound) return; a._bound=true; a.addEventListener('click', (e)=>{ e.preventDefault(); openPhotoModal(a.getAttribute('data-photo')); }); });
      body.querySelectorAll('[data-ex]')?.forEach(btn=>{
        btn.addEventListener('click', async ()=>{
          const id = btn.getAttribute('data-id');
          const decision = btn.getAttribute('data-ex')==='approve' ? 'approve' : 'reject';
          const note = prompt('Optional note') || '';
          const out = await fetch('api/attendance.php?action=exception_decide', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ event_id: Number(id), decision, note }) }).then(r=>r.json()).catch(()=>null);
          if (out?.success){ note('Updated','success'); loadExceptions(); loadMap(); }
          else { note(out?.message||'Update failed','error'); }
        });
      });
    }

    async function openSites(){
      const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById('siteModal'));
      const body = document.getElementById('siteBody');
      const refresh = async ()=>{
        const res = await fetch('api/attendance.php?action=sites&include_inactive=1', { credentials:'same-origin' }).then(r=>r.json()).catch(()=>null);
        body.innerHTML = (res?.data||[]).map(s=> `
          <tr>
            <td>${esc(s.name||'')}</td>
            <td>${esc(s.code||'')}</td>
            <td>${esc(s.center_lat??'')}</td>
            <td>${esc(s.center_lng??'')}</td>
            <td>${esc(s.radius_m??'')}</td>
            <td><span class="badge ${s.status==='active'?'bg-success':'bg-secondary'}">${esc(s.status||'')}</span></td>
            <td>
              <div class="btn-group btn-group-sm">
                <button class="btn btn-outline-primary" data-s="edit" data-id="${s.id}"><i class="fas fa-pen"></i></button>
                <button class="btn btn-outline-warning" data-s="toggle" data-id="${s.id}"><i class="fas fa-power-off"></i></button>
              </div>
            </td>
          </tr>
        `).join('') || '<tr><td colspan="7" class="text-muted">No sites</td></tr>';
      };
      const newSite = ()=>{
        editSite({ id:0, name:'', code:'', center_lat:'', center_lng:'', radius_m:200, qr_token:'' }, async (val)=>{
          const out = await fetch('api/attendance.php?action=site_save', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(val) }).then(r=>r.json()).catch(()=>null);
          if (out?.success){ note('Created','success'); refresh(); } else { note(out?.message||'Save failed','error'); }
        });
      };
      document.getElementById('siteAdd').onclick = newSite;
      document.getElementById('siteBody').onclick = async (e)=>{
        const btn = e.target.closest('[data-s]'); if (!btn) return;
        const id = Number(btn.getAttribute('data-id'));
        const act = btn.getAttribute('data-s');
        if (act==='toggle'){
          const out = await fetch('api/attendance.php?action=site_toggle', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ id }) }).then(r=>r.json()).catch(()=>null);
          if (out?.success){ note('Toggled','success'); refresh(); } else { note(out?.message||'Toggle failed','error'); }
        } else if (act==='edit'){
          // get info from row
          const tr = btn.closest('tr');
          const cells = tr.querySelectorAll('td');
          const val = { id, name: cells[0].innerText, code: cells[1].innerText, center_lat: cells[2].innerText, center_lng: cells[3].innerText, radius_m: cells[4].innerText };
          editSite(val, async (v)=>{
            v.id = id;
            const out = await fetch('api/attendance.php?action=site_save', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(v) }).then(r=>r.json()).catch(()=>null);
            if (out?.success){ note('Updated','success'); refresh(); } else { note(out?.message||'Save failed','error'); }
          });
        }
      };
      await refresh();
      modal.show();
    }

    function editSite(val, onSave){
      const wrap = document.createElement('div');
      wrap.innerHTML = `
        <div class="row g-2">
          <div class="col-md-4"><label class="form-label">Name</label><input class="form-control" data-k="name" value="${esc(val.name||'')}"/></div>
          <div class="col-md-3"><label class="form-label">Code</label><input class="form-control" data-k="code" value="${esc(val.code||'')}"/></div>
          <div class="col-md-2"><label class="form-label">Lat</label><input class="form-control" data-k="center_lat" value="${esc(val.center_lat||'')}"/></div>
          <div class="col-md-2"><label class="form-label">Lng</label><input class="form-control" data-k="center_lng" value="${esc(val.center_lng||'')}"/></div>
          <div class="col-md-1"><label class="form-label">r(m)</label><input class="form-control" data-k="radius_m" value="${esc(val.radius_m||'200')}"/></div>
          <div class="col-md-12"><label class="form-label">QR Token (optional)</label><input class="form-control" data-k="qr_token" value="${esc(val.qr_token||'')}"/></div>
        </div>`;
      const dlg = bootstrap.Modal.getOrCreateInstance(document.createElement('div'));
      const modalEl = document.createElement('div');
      modalEl.className = 'modal fade show';
      modalEl.style.display='block';
      modalEl.innerHTML = `<div class="modal-dialog"><div class="modal-content"><div class="modal-header"><h5 class="modal-title">${val.id? 'Edit':'New'} Site</h5><button class="btn-close" data-bs-dismiss="modal"></button></div><div class="modal-body"></div><div class="modal-footer"><button class="btn btn-light" data-bs-dismiss="modal">Close</button><button class="btn btn-primary">Save</button></div></div></div>`;
      modalEl.querySelector('.modal-body').appendChild(wrap);
      document.body.appendChild(modalEl);
      const bs = new bootstrap.Modal(modalEl);
      modalEl.querySelector('.btn-primary').onclick = ()=>{
        const payload = {};
        wrap.querySelectorAll('[data-k]').forEach(inp=>{ payload[inp.getAttribute('data-k')] = inp.value; });
        payload.center_lat = payload.center_lat!==''? Number(payload.center_lat): null;
        payload.center_lng = payload.center_lng!==''? Number(payload.center_lng): null;
        payload.radius_m = Number(payload.radius_m||200);
        onSave?.(payload);
        bs.hide();
        setTimeout(()=> modalEl.remove(), 300);
      };
      modalEl.querySelector('[data-bs-dismiss="modal"]').onclick = ()=>{ bs.hide(); setTimeout(()=> modalEl.remove(), 300); };
      bs.show();
    }

    function isManagerOrHr(){
      const r = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role; return ['super_admin','admin','hr_head','hr_officer','manager'].includes(r);
    }
    function isHr(){
      const r = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role; return ['super_admin','admin','hr_head','hr_officer'].includes(r);
    }
    function canManageSites(){
      const r = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role; return ['super_admin','admin','hr_head','hr_officer'].includes(r);
    }
    async function fingerprint(){
      // very light client fingerprint (UA + screen + timezone)
      try {
        const ua = navigator.userAgent || '';
        const scr = (screen?.width||'')+'x'+(screen?.height||'')+'x'+(screen?.colorDepth||'');
        const tz = Intl.DateTimeFormat().resolvedOptions().timeZone || '';
        const lang = navigator.language || '';
        const raw = ua+'|'+scr+'|'+tz+'|'+lang;
        // simple hash
        let h=0; for (let i=0;i<raw.length;i++){ h = (h<<5)-h + raw.charCodeAt(i); h |= 0; }
        return 'fp-'+Math.abs(h);
      } catch(_) { return null; }
    }

    async function getCoords(){
      return new Promise((resolve)=>{
        if (!navigator.geolocation) return resolve({lat:null,lng:null});
        navigator.geolocation.getCurrentPosition(
          pos=> resolve({ lat: pos.coords.latitude, lng: pos.coords.longitude }),
          _=> resolve({ lat: null, lng: null }),
          { enableHighAccuracy: true, timeout: 8000 }
        );
      });
    }

    async function readPhoto(file){
      if (!file) return null;
      return new Promise((resolve)=>{
        const reader = new FileReader();
        reader.onload = ()=> resolve(reader.result);
        reader.onerror = ()=> resolve(null);
        reader.readAsDataURL(file);
      });
    }

    function note(msg, type){ window.auth?.showNotification?.(msg, type||'info'); }
    function esc(s){ return String(s ?? '').replace(/[&<>"']/g, c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c])); }
    function openPhotoModal(url){ try{ const img = document.getElementById('photoModalImg'); if (img) img.src = url || ''; const m = bootstrap.Modal.getOrCreateInstance(document.getElementById('photoModal')); m.show(); }catch(_){} }

    await Promise.all([loadSites(), loadHistory(), loadMap(), loadExceptions()]);
    if (isHr()) await loadAllHistory();
  }

  window.pages.attendance = { render };
})();
