// Core App Controller for SuperbHR
(function () {
  class AppController {
    constructor() {
      this.currentPage = 'dashboard';
      this.container = document.getElementById('pageContent');
      this.sidebar = document.querySelector('.sidebar');
      this.overlay = document.getElementById('sidebarOverlay');
      this.setupSidebarToggle();
      this.setupRouting();
    }

    setupSidebarToggle() {
      const toggleBtn = document.getElementById('sidebarToggle');
      if (toggleBtn && this.sidebar) {
        toggleBtn.addEventListener('click', () => {
          const isMobile = window.matchMedia('(max-width: 991.98px)').matches;
          if (isMobile) {
            const open = this.sidebar.classList.toggle('sidebar-open');
            document.body.classList.toggle('sidebar-open', open);
            this.toggleOverlay(open);
          } else {
            this.sidebar.classList.toggle('collapsed');
            const main = document.querySelector('.main-content');
            if (main) {
              if (this.sidebar.classList.contains('collapsed')) {
                main.classList.add('sidebar-collapsed');
              } else {
                main.classList.remove('sidebar-collapsed');
              }
            }
          }
        });
      }

      if (this.overlay) {
        this.overlay.addEventListener('click', () => {
          this.closeMobileSidebar();
        });
      }

      window.addEventListener('resize', () => {
        if (!window.matchMedia('(max-width: 991.98px)').matches) {
          this.closeMobileSidebar();
          document.body.classList.remove('sidebar-open');
          this.toggleOverlay(false);
        }
      });
    }

    toggleOverlay(show) {
      if (!this.overlay) return;
      if (show) this.overlay.classList.add('show');
      else this.overlay.classList.remove('show');
    }

    closeMobileSidebar() {
      if (!this.sidebar) return;
      this.sidebar.classList.remove('sidebar-open');
      document.body.classList.remove('sidebar-open');
      this.toggleOverlay(false);
    }

    setupRouting() {
      window.addEventListener('hashchange', () => this.handleRoute());
      // Load route on initial page load
      this.handleRoute();
    }

    handleRoute() {
      const hash = window.location.hash || '';
      const match = hash.match(/^#\/(\w[\w-]*)/);
      const page = match ? match[1] : 'dashboard';
      if (page && page !== this.currentPage) {
        // Avoid infinite loop by passing fromHash flag
        this.showPage(page, true);
      }
    }

    async showPage(page, fromHash = false) {
      this.currentPage = page;
      if (!this.container) return;

      // Simple loading state
      this.container.innerHTML = `
        <div class="d-flex justify-content-center align-items-center" style="min-height: 200px;">
          <div class="spinner"></div>
          <span class="ms-2">Loading ${page}...</span>
        </div>
      `;

      try {
        if (window.logEvent) window.logEvent('info', 'page_navigate', { page, from_hash: !!fromHash });
        const module = window.pages?.[page];
        if (module && typeof module.render === 'function') {
          await module.render(this.container);
          this.setActiveSidebar(page);
          this.closeMobileSidebar();
          // Sync URL hash for routing if not coming from hashchange
          const newHash = `#/${page}`;
          if (!fromHash && window.location.hash !== newHash) {
            window.location.hash = newHash;
          }
        } else {
          this.container.innerHTML = `
            <div class="card">
              <div class="card-header">
                <h5 class="card-title mb-0 text-capitalize">${page}</h5>
              </div>
              <div class="card-body">
                <p>This page is not implemented yet.</p>
              </div>
            </div>
          `;
          this.setActiveSidebar(page);
          if (!fromHash) {
            const newHash = `#/${page}`;
            if (window.location.hash !== newHash) {
              window.location.hash = newHash;
            }
          }
        }
      } catch (e) {
        console.error('Failed to load page', e);
        if (window.logEvent) window.logEvent('error', 'page_render_error', { page, error: String(e?.message || e) });
        this.container.innerHTML = `
          <div class="alert alert-danger">Failed to load ${page}. Please try again.</div>
        `;
      }
    }

    setActiveSidebar(page) {
      // Highlight active menu item
      const links = document.querySelectorAll('#sidebarMenu a');
      links.forEach((a) => {
        if (a.getAttribute('onclick')?.includes(`'${page}'`)) {
          a.classList.add('active');
        } else {
          a.classList.remove('active');
        }
      });
    }
  }

  window.pages = window.pages || {};
  window.app = new AppController();
})();
