<?php
/**
 * User Management API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$database = new Database();
$db = $database->getConnection();

function ensureCanManageUsers($user){
  $allowed = in_array(($user['role_slug'] ?? ''), ['super_admin','admin','hr_head','hr_officer'], true);
  if (!$allowed) ApiResponse::forbidden('Insufficient permissions');
}

switch ($method) {
  case 'GET':
    if ($action === 'roles') getRoles($db);
    elseif ($action === 'unlinked_employees') getUnlinkedEmployees($db);
    else getUsers($db);
    break;
  case 'POST':
    if ($action === 'create') createUser($db);
    elseif ($action === 'assign_roles') assignRoles($db);
    elseif ($action === 'reset_password') resetPassword($db);
    elseif ($action === 'toggle_status') toggleStatus($db);
    else ApiResponse::error('Unknown action');
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function getUsers(PDO $db){
  $u = getCurrentUser(); ensureCanManageUsers($u);
  try {
    $sql = "SELECT 
              us.id, us.username, us.email, us.status,
              us.first_name, us.last_name, us.phone,
              e.employee_number, CONCAT(e.first_name,' ',e.last_name) AS employee_name
            FROM users us
            LEFT JOIN employees e ON e.id = us.employee_id
            WHERE us.company_id = :cid
            ORDER BY us.created_at DESC";
    $st = $db->prepare($sql); $st->bindValue(':cid', $u['company_id'], PDO::PARAM_INT); $st->execute();
    $users = $st->fetchAll();

    // Roles map per user
    $r = $db->prepare("SELECT ur.user_id, r.id AS role_id, r.name, r.slug
                       FROM user_roles ur
                       JOIN users us ON us.id = ur.user_id
                       JOIN roles r ON r.id = ur.role_id
                       WHERE us.company_id = :cid");
    $r->bindValue(':cid', $u['company_id'], PDO::PARAM_INT); $r->execute();
    $roleRows = $r->fetchAll();
    $rolesByUser = [];
    foreach ($roleRows as $rr){
      $uid = (int)$rr['user_id']; if (!isset($rolesByUser[$uid])) $rolesByUser[$uid] = [];
      $rolesByUser[$uid][] = [ 'id'=>(int)$rr['role_id'], 'name'=>$rr['name'], 'slug'=>$rr['slug'] ];
    }
    foreach ($users as &$urow){
      $uid = (int)$urow['id']; $urow['roles'] = $rolesByUser[$uid] ?? [];
    }
    ApiResponse::success($users);
  } catch (Throwable $e) { ApiResponse::error('Failed to load users: '.$e->getMessage(), 500); }
}

function getRoles(PDO $db){
  $u = getCurrentUser(); ensureCanManageUsers($u);
  try {
    $st = $db->prepare("SELECT id, name, slug, level FROM roles WHERE company_id = :cid OR company_id = 0 OR company_id IS NULL ORDER BY level DESC, name ASC");
    $st->bindValue(':cid', $u['company_id'], PDO::PARAM_INT); $st->execute();
    ApiResponse::success($st->fetchAll());
  } catch (Throwable $e) { ApiResponse::error('Failed to load roles: '.$e->getMessage(), 500); }
}

function getUnlinkedEmployees(PDO $db){
  $u = getCurrentUser(); ensureCanManageUsers($u);
  try {
    $st = $db->prepare("SELECT id, employee_number, first_name, last_name, email, phone FROM employees WHERE company_id = :cid AND (user_id IS NULL OR user_id = 0) ORDER BY first_name, last_name");
    $st->bindValue(':cid', $u['company_id'], PDO::PARAM_INT); $st->execute();
    $rows = $st->fetchAll();
    foreach ($rows as &$r){ $r['full_name'] = trim(($r['first_name'] ?? '').' '.($r['last_name'] ?? '')); }
    ApiResponse::success($rows);
  } catch (Throwable $e) { ApiResponse::error('Failed to load employees: '.$e->getMessage(), 500); }
}

function createUser(PDO $db){
  $u = getCurrentUser(); ensureCanManageUsers($u);
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $employeeId = isset($in['employee_id']) ? (int)$in['employee_id'] : 0;
  $username = trim((string)($in['username'] ?? ''));
  $email = trim((string)($in['email'] ?? ''));
  $phone = trim((string)($in['phone'] ?? ''));
  $roleIds = isset($in['role_ids']) && is_array($in['role_ids']) ? array_values(array_unique(array_map('intval', $in['role_ids']))) : [];
  if ($employeeId <= 0 || $username === '') ApiResponse::error('Employee and username are required');

  try {
    $db->beginTransaction();

    // Validate employee in company and no existing user
    $ch = $db->prepare("SELECT id, first_name, last_name, email, phone FROM employees WHERE id = :id AND company_id = :cid AND (user_id IS NULL OR user_id = 0) LIMIT 1");
    $ch->execute([':id'=>$employeeId, ':cid'=>$u['company_id']]);
    if ($ch->rowCount() === 0) ApiResponse::error('Invalid employee or already linked');

    // Check unique username/email
    $uChk = $db->prepare("SELECT 1 FROM users WHERE username = :un OR email = :em LIMIT 1");
    $uChk->execute([':un'=>$username, ':em'=>$email]);
    if ($uChk->rowCount() > 0) ApiResponse::error('Username or email already in use');

    // Insert user
    $emp = $ch->fetch();
    $first = $emp['first_name'] ?? '';
    $last = $emp['last_name'] ?? '';
    $emailFinal = $email !== '' ? $email : ($emp['email'] ?? null);
    $phoneFinal = $phone !== '' ? $phone : ($emp['phone'] ?? null);

    $sql = "INSERT INTO users (company_id, employee_id, username, email, password, first_name, last_name, phone, status, created_by, created_at)
            VALUES (:cid, :eid, :un, :em, :pw, :fn, :ln, :ph, 'active', :cb, NOW())";
    $st = $db->prepare($sql);
    $hash = password_hash('password123', PASSWORD_DEFAULT);
    $st->execute([
      ':cid'=>$u['company_id'], ':eid'=>$employeeId, ':un'=>$username, ':em'=>$emailFinal, ':pw'=>$hash,
      ':fn'=>$first, ':ln'=>$last, ':ph'=>$phoneFinal, ':cb'=>$u['id']
    ]);
    $newUserId = (int)$db->lastInsertId();

    // Link employee
    $db->prepare("UPDATE employees SET user_id = :uid WHERE id = :eid")->execute([':uid'=>$newUserId, ':eid'=>$employeeId]);

    // Assign roles
    if (!$roleIds) {
      // Try to find 'employee' role for company
      $rid = null;
      try { $rr = $db->prepare("SELECT id FROM roles WHERE (company_id = :cid OR company_id = 0 OR company_id IS NULL) AND slug = 'employee' LIMIT 1"); $rr->execute([':cid'=>$u['company_id']]); $rid = $rr->fetchColumn(); } catch (Throwable $e) {}
      if ($rid) $roleIds = [(int)$rid];
    }
    if ($roleIds) {
      $ins = $db->prepare("INSERT IGNORE INTO user_roles (user_id, role_id, assigned_by, assigned_at) VALUES (:uid,:rid,:by,NOW())");
      foreach ($roleIds as $rid) { $ins->execute([':uid'=>$newUserId, ':rid'=>$rid, ':by'=>$u['id']]); }
    }

    $db->commit();
    ApiResponse::success(['id'=>$newUserId], 'User created');
  } catch (Throwable $e) { $db->rollBack(); ApiResponse::error('Failed to create user: '.$e->getMessage(), 500); }
}

function assignRoles(PDO $db){
  $u = getCurrentUser(); ensureCanManageUsers($u);
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $userId = isset($in['user_id']) ? (int)$in['user_id'] : 0;
  $roleIds = isset($in['role_ids']) && is_array($in['role_ids']) ? array_values(array_unique(array_map('intval', $in['role_ids']))) : [];
  if ($userId <= 0) ApiResponse::error('Invalid user');
  try {
    // Ensure user belongs to same company
    $ch = $db->prepare("SELECT id FROM users WHERE id = :id AND company_id = :cid");
    $ch->execute([':id'=>$userId, ':cid'=>getCurrentUser()['company_id']]);
    if ($ch->rowCount() === 0) ApiResponse::forbidden('Cannot manage this user');

    $db->beginTransaction();
    // Fetch existing roles
    $cur = $db->prepare("SELECT role_id FROM user_roles WHERE user_id = :uid");
    $cur->execute([':uid'=>$userId]);
    $existing = array_map('intval', array_column($cur->fetchAll(), 'role_id'));

    // Delete removed
    if ($existing) {
      $place = implode(',', array_fill(0, count($existing), '?'));
      if ($roleIds) {
        // keep only those not in new set
        $del = $db->prepare("DELETE FROM user_roles WHERE user_id = ? AND role_id NOT IN (".$place.")");
        $del->execute(array_merge([$userId], $roleIds));
      } else {
        $del = $db->prepare("DELETE FROM user_roles WHERE user_id = ?");
        $del->execute([$userId]);
      }
    }

    // Insert new
    if ($roleIds) {
      $ins = $db->prepare("INSERT IGNORE INTO user_roles (user_id, role_id, assigned_by, assigned_at) VALUES (:uid,:rid,:by,NOW())");
      foreach ($roleIds as $rid) { $ins->execute([':uid'=>$userId, ':rid'=>$rid, ':by'=>$u['id']]); }
    }

    $db->commit();
    ApiResponse::success(null, 'Roles updated');
  } catch (Throwable $e) { $db->rollBack(); ApiResponse::error('Failed to update roles: '.$e->getMessage(), 500); }
}

function resetPassword(PDO $db){
  $u = getCurrentUser(); ensureCanManageUsers($u);
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $userId = isset($in['user_id']) ? (int)$in['user_id'] : 0;
  $new = isset($in['new_password']) && $in['new_password'] !== '' ? (string)$in['new_password'] : null;
  if ($userId <= 0) ApiResponse::error('Invalid user');
  try {
    $hash = password_hash($new ?? 'password123', PASSWORD_DEFAULT);
    $st = $db->prepare("UPDATE users SET password = :pw, password_changed_at = NULL, must_change_password = 1 WHERE id = :id AND company_id = :cid");
    $st->execute([':pw'=>$hash, ':id'=>$userId, ':cid'=>$u['company_id']]);
    ApiResponse::success(null, 'Password reset');
  } catch (Throwable $e) { ApiResponse::error('Failed to reset: '.$e->getMessage(), 500); }
}

function toggleStatus(PDO $db){
  $u = getCurrentUser(); ensureCanManageUsers($u);
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $userId = isset($in['user_id']) ? (int)$in['user_id'] : 0;
  $status = ($in['status'] ?? '') === 'active' ? 'active' : 'inactive';
  if ($userId <= 0) ApiResponse::error('Invalid user');
  try {
    $st = $db->prepare("UPDATE users SET status = :s WHERE id = :id AND company_id = :cid");
    $st->execute([':s'=>$status, ':id'=>$userId, ':cid'=>$u['company_id']]);
    ApiResponse::success(null, 'Status updated');
  } catch (Throwable $e) { ApiResponse::error('Failed to update: '.$e->getMessage(), 500); }
}
