<?php
/**
 * Super Admin API: cross-company management and analytics
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  ApiResponse::success(null, 'OK');
}

requireAuth();
$user = getCurrentUser();
if (!$user || ($user['role_slug'] ?? '') !== 'super_admin') {
  ApiResponse::forbidden('Super Admin only');
}

$database = new Database();
$db = $database->getConnection();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

function tableHasColumn(PDO $db, $table, $column){
  try{
    $q = $db->prepare("SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t AND COLUMN_NAME = :c");
    $q->execute([':t'=>$table, ':c'=>$column]);
    return $q->rowCount() > 0;
  }catch(Throwable $e){ return false; }
}

function ensureRole(PDO $db, int $companyId, string $slug, string $name, int $level = 0){
  $hasCompanyCol = tableHasColumn($db,'roles','company_id');
  $hasLevelCol   = tableHasColumn($db,'roles','level');
  // Try select
  try{
    if ($hasCompanyCol){
      $s = $db->prepare('SELECT id FROM roles WHERE slug = :s AND company_id = :cid LIMIT 1');
      $s->execute([':s'=>$slug, ':cid'=>$companyId]);
    } else {
      $s = $db->prepare('SELECT id FROM roles WHERE slug = :s LIMIT 1');
      $s->execute([':s'=>$slug]);
    }
    $rid = $s->fetchColumn();
    if ($rid) return (int)$rid;
  }catch(Throwable $e){ /* continue to insert path */ }

  // Insert role
  $cols = ['slug','name','created_at'];
  $vals = [':slug'=> $slug, ':name'=>$name];
  $sqlCols = '`slug`,`name`,`created_at`';
  $sqlVals = ':slug,:name,NOW()';
  if ($hasCompanyCol){ $sqlCols .= ',`company_id`'; $sqlVals .= ',:cid'; $vals[':cid'] = $companyId; }
  if ($hasLevelCol){ $sqlCols .= ',`level`'; $sqlVals .= ',:lvl'; $vals[':lvl'] = $level; }

  $sql = "INSERT INTO roles ($sqlCols) VALUES ($sqlVals)";
  $ins = $db->prepare($sql);
  $ins->execute($vals);
  return (int)$db->lastInsertId();
}

function listCompanies(PDO $db){
  try{
    // Base fields
    $baseCols = ['id','name','email','phone','website','created_at'];
    $cols = [];
    foreach ($baseCols as $c){ if (tableHasColumn($db,'companies',$c)) $cols[] = $c; }
    if (!$cols) ApiResponse::error('Companies table not available');
    // Choose order field safely
    $orderField = in_array('created_at', $cols, true) ? 'created_at' : 'id';
    $sql = 'SELECT '.implode(',', $cols).' FROM companies ORDER BY '.$orderField.' DESC';
    $st = $db->query($sql);
    $companies = $st->fetchAll() ?: [];

    // Users per company
    $uc = $db->query('SELECT company_id, COUNT(*) AS cnt FROM users GROUP BY company_id');
    $byCompany = [];
    foreach (($uc->fetchAll() ?: []) as $r){ $byCompany[(int)$r['company_id']] = (int)$r['cnt']; }

    // Role distribution per company (if roles tables exist)
    $roleDist = [];
    try{
      $q = $db->query('SELECT u.company_id AS cid, r.slug, COUNT(*) AS cnt
                       FROM user_roles ur
                       JOIN users u ON u.id = ur.user_id
                       JOIN roles r ON r.id = ur.role_id
                       GROUP BY u.company_id, r.slug');
      foreach (($q->fetchAll() ?: []) as $row){
        $cid = (int)$row['cid'];
        if (!isset($roleDist[$cid])) $roleDist[$cid] = [];
        $roleDist[$cid][$row['slug']] = (int)$row['cnt'];
      }
    }catch(Throwable $e){ /* roles tables may not exist */ }

    foreach ($companies as &$c){
      $cid = (int)($c['id'] ?? 0);
      $c['users_count'] = $byCompany[$cid] ?? 0;
      $c['roles'] = $roleDist[$cid] ?? [];
    }
    ApiResponse::success($companies);
  }catch(Throwable $e){ ApiResponse::error('Failed to list companies: '.$e->getMessage(), 500); }
}

function createCompany(PDO $db){
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $name = trim((string)($in['name'] ?? ''));
  if ($name === '') ApiResponse::error('Company name is required');
  $fields = ['name','email','phone','address','website'];
  $has = [];
  foreach ($fields as $f){ if (tableHasColumn($db,'companies',$f) && array_key_exists($f,$in)) $has[$f] = $in[$f]; }
  try{
    $cols = array_keys($has); $params = [];
    $sqlCols = '`name`'; $sqlVals = ':name'; $params[':name'] = $name;
    foreach ($cols as $f){ if ($f==='name') continue; $sqlCols .= ',`'.$f.'`'; $sqlVals .= ',:'.$f; $params[':'.$f] = $has[$f]; }
    $sql = 'INSERT INTO companies ('.$sqlCols.', created_at) VALUES ('.$sqlVals.', NOW())';
    $st = $db->prepare($sql); $st->execute($params);
    $cid = (int)$db->lastInsertId();

    // Ensure default roles for the company
    ensureRole($db,$cid,'admin','Admin',9);
    ensureRole($db,$cid,'hr_head','HR Head/Lead',8);
    ensureRole($db,$cid,'hr_officer','HR Officer',6);
    ensureRole($db,$cid,'manager','Manager',5);
    ensureRole($db,$cid,'employee','Employee',1);

    ApiResponse::success(['id'=>$cid], 'Company created');
  }catch(Throwable $e){ ApiResponse::error('Failed to create company: '.$e->getMessage(), 500); }
}

function listUsersByCompany(PDO $db){
  $cid = isset($_GET['company_id']) ? (int)$_GET['company_id'] : 0;
  if ($cid <= 0) ApiResponse::error('Invalid company');
  try{
    $st = $db->prepare('SELECT id, username, email, first_name, last_name, status, created_at FROM users WHERE company_id = :cid ORDER BY created_at DESC');
    $st->execute([':cid'=>$cid]);
    $users = $st->fetchAll() ?: [];
    // roles
    try{
      $r = $db->prepare('SELECT ur.user_id, r.slug FROM user_roles ur JOIN roles r ON r.id = ur.role_id');
      $r->execute(); $rows = $r->fetchAll() ?: [];
      $map=[]; foreach ($rows as $row){ $map[(int)$row['user_id']][] = $row['slug']; }
      foreach ($users as &$u){ $u['roles'] = $map[(int)$u['id']] ?? []; }
    }catch(Throwable $e){ /* ignore */ }
    ApiResponse::success($users);
  }catch(Throwable $e){ ApiResponse::error('Failed to list users: '.$e->getMessage(), 500); }
}

function createCompanyUser(PDO $db){
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $cid = (int)($in['company_id'] ?? 0);
  $username = trim((string)($in['username'] ?? ''));
  $password = (string)($in['password'] ?? 'password123');
  $email    = trim((string)($in['email'] ?? ''));
  $first    = trim((string)($in['first_name'] ?? ''));
  $last     = trim((string)($in['last_name'] ?? ''));
  $roleSlug = trim((string)($in['role_slug'] ?? ''));
  if ($cid<=0 || $username==='') ApiResponse::error('company_id and username required');
  if ($roleSlug==='') ApiResponse::error('role_slug required');

  try{
    $db->beginTransaction();

    // Ensure unique username/email
    $uq = $db->prepare('SELECT 1 FROM users WHERE username = :un OR email = :em LIMIT 1');
    $uq->execute([':un'=>$username, ':em'=>$email]);
    if ($uq->fetchColumn()) { ApiResponse::error('Username or email already exists'); }

    // Insert user
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $ins = $db->prepare('INSERT INTO users (company_id, username, email, password, first_name, last_name, status, created_at) VALUES (:cid,:un,:em,:pw,:fn,:ln,\'active\', NOW())');
    $ins->execute([':cid'=>$cid, ':un'=>$username, ':em'=>$email ?: null, ':pw'=>$hash, ':fn'=>$first, ':ln'=>$last]);
    $uid = (int)$db->lastInsertId();

    // Assign role
    $rid = ensureRole($db,$cid,$roleSlug, ucfirst(str_replace('_',' ',$roleSlug)));
    $ur = $db->prepare('INSERT INTO user_roles (user_id, role_id, assigned_by, assigned_at) VALUES (:uid,:rid,:by,NOW())');
    $ur->execute([':uid'=>$uid, ':rid'=>$rid, ':by'=>getCurrentUser()['id']]);

    $db->commit();
    ApiResponse::success(['id'=>$uid],'User created');
  }catch(Throwable $e){ $db->rollBack(); ApiResponse::error('Failed to create user: '.$e->getMessage(), 500); }
}

function overallStats(PDO $db){
  try{
    $totCompanies = (int)($db->query('SELECT COUNT(*) FROM companies')->fetchColumn() ?: 0);
    $totUsers     = (int)($db->query('SELECT COUNT(*) FROM users')->fetchColumn() ?: 0);

    // role distribution
    $roles = [];
    try{
      $q = $db->query('SELECT r.slug, COUNT(*) AS cnt FROM user_roles ur JOIN roles r ON r.id = ur.role_id GROUP BY r.slug');
      foreach (($q->fetchAll() ?: []) as $row){ $roles[$row['slug']] = (int)$row['cnt']; }
    }catch(Throwable $e){ /* ignore */ }

    // usage from activity_logs if available
    $events7 = 0; $active7 = 0;
    try{
      $q1 = $db->query("SELECT COUNT(*) FROM activity_logs WHERE created_at >= (NOW() - INTERVAL 7 DAY)");
      $events7 = (int)($q1->fetchColumn() ?: 0);
      $q2 = $db->query("SELECT COUNT(DISTINCT user_id) FROM activity_logs WHERE user_id IS NOT NULL AND created_at >= (NOW() - INTERVAL 7 DAY)");
      $active7 = (int)($q2->fetchColumn() ?: 0);
    }catch(Throwable $e){ /* ignore */ }

    ApiResponse::success([
      'total_companies' => $totCompanies,
      'total_users' => $totUsers,
      'roles' => $roles,
      'events_7d' => $events7,
      'active_users_7d' => $active7
    ]);
  }catch(Throwable $e){ ApiResponse::error('Failed to compute stats: '.$e->getMessage(), 500); }
}

switch ($method) {
  case 'GET':
    if ($action === 'list_companies') listCompanies($db);
    elseif ($action === 'list_users') listUsersByCompany($db);
    elseif ($action === 'overall_stats') overallStats($db);
    else ApiResponse::error('Unknown action', 400);
    break;
  case 'POST':
    if ($action === 'create_company') createCompany($db);
    elseif ($action === 'create_company_user') createCompanyUser($db);
    else ApiResponse::error('Unknown action', 400);
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}
