<?php
/**
 * Admin Settings API: Company profile and Email (SMTP) settings
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';
require_once __DIR__ . '/_sms.inc.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

$database = new Database();
$db = $database->getConnection();

function mustBeAdmin($user){
  if (!in_array($user['role_slug'], ['super_admin','admin'])) {
    ApiResponse::forbidden('Insufficient permissions');
  }

// For super_admin, allow overriding target company via ?company_id= on GET
function targetCompanyIdFromGet(array $user): int {
  $cid = (int)($user['company_id'] ?? 0);
  if (($user['role_slug'] ?? '') === 'super_admin') {
    $q = isset($_GET['company_id']) ? (int)$_GET['company_id'] : 0;
    if ($q > 0) $cid = $q;
  }
  return $cid;
}
}

function tableHasColumn(PDO $db, $table, $column){
  static $cache = [];
  $key = "$table::$column";
  if (isset($cache[$key])) return $cache[$key];
  try {
    $q = $db->prepare("SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t AND COLUMN_NAME = :c");
    $q->bindValue(':t', $table);
    $q->bindValue(':c', $column);
    $q->execute();
    return $cache[$key] = ($q->rowCount() > 0);
  } catch (Throwable $e) {
    return $cache[$key] = false;
  }
}

function storagePath($rel){
  $base = realpath(__DIR__ . '/../storage');
  if (!$base) { @mkdir(__DIR__ . '/../storage', 0777, true); $base = realpath(__DIR__ . '/../storage'); }
  $cfgDir = $base . '/config';
  if (!is_dir($cfgDir)) @mkdir($cfgDir, 0777, true);
  return $cfgDir . '/' . ltrim($rel, '/\\');
}

switch ($method) {
  case 'GET':
    if ($action === 'company') getCompany($db);
    elseif ($action === 'email') getEmail($db);
    elseif ($action === 'sms') getSms($db);
    else ApiResponse::error('Unknown action', 400);
    break;
  case 'POST':
    if ($action === 'company') saveCompany($db);
    elseif ($action === 'company_logo') uploadCompanyLogo($db);
    elseif ($action === 'company_logo_clear') clearCompanyLogo($db);
    elseif ($action === 'email') saveEmail($db);
    elseif ($action === 'email_test') sendTestEmail($db);
    elseif ($action === 'sms') saveSms($db);
    elseif ($action === 'sms_test') sendTestSms($db);
    else ApiResponse::error('Unknown action', 400);
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function getCompany(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = targetCompanyIdFromGet($user);
  // Build field list based on existing columns
  $fields = [];
  foreach (['id','name','email','phone','address','website','logo'] as $f){ if (tableHasColumn($db,'companies',$f)) $fields[] = $f; }
  if (!$fields) ApiResponse::error('Companies table not available');
  $sql = 'SELECT '.implode(',', $fields).' FROM companies WHERE id = :id LIMIT 1';
  $st = $db->prepare($sql); $st->bindValue(':id',$cid, PDO::PARAM_INT); $st->execute();
  if ($st->rowCount() === 0) ApiResponse::notFound('Company not found');
  ApiResponse::success($st->fetch());
}

function saveCompany(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = targetCompanyIdFromGet($user);
  $input = json_decode(file_get_contents('php://input'), true) ?? [];
  if (($user['role_slug'] ?? '') === 'super_admin') {
    $b = (int)($input['company_id'] ?? 0); if ($b>0) $cid = $b;
  }
  $allowed = [];
  foreach (['name','email','phone','address','website','logo'] as $f){ if (tableHasColumn($db,'companies',$f)) $allowed[] = $f; }
  if (!$allowed) ApiResponse::error('No editable company fields available');
  $set=[]; $params=[':id'=>$cid];
  foreach ($allowed as $f){ if (array_key_exists($f,$input)) { $set[] = "$f = :$f"; $params[":$f"] = $input[$f]; } }
  if (!$set) ApiResponse::error('Nothing to update');
  $sql = 'UPDATE companies SET '.implode(',',$set).', updated_at = NOW() WHERE id = :id';
  $st = $db->prepare($sql); foreach ($params as $k=>$v) $st->bindValue($k,$v); $st->execute();
  ApiResponse::success(null,'Company updated');
}

function getEmail(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = targetCompanyIdFromGet($user);
  $cols = ['smtp_host','smtp_port','smtp_secure','smtp_user','smtp_pass','email_from_name','email_from_email'];
  $exists = array_filter($cols, fn($c)=> tableHasColumn($db,'companies',$c));
  if ($exists){
    $sql = 'SELECT '.implode(',', $exists).' FROM companies WHERE id = :id LIMIT 1';
    $st = $db->prepare($sql); $st->bindValue(':id',$cid,PDO::PARAM_INT); $st->execute();
    $row = $st->fetch() ?: [];
  } else {
    $f = storagePath('email-'.$cid.'.json');
    $row = is_file($f) ? (json_decode(file_get_contents($f), true) ?: []) : [];
  }
  if (isset($row['smtp_pass'])) { $row['smtp_pass'] = '******'; $row['smtp_pass_masked'] = true; }
  ApiResponse::success($row);
}

function saveEmail(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = targetCompanyIdFromGet($user);
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  if (($user['role_slug'] ?? '') === 'super_admin') {
    $b = (int)($in['company_id'] ?? 0); if ($b>0) $cid = $b;
  }
  $fields = ['smtp_host','smtp_port','smtp_secure','smtp_user','smtp_pass','email_from_name','email_from_email'];
  $exists = array_filter($fields, fn($c)=> tableHasColumn($db,'companies',$c));
  if ($exists){
    $set=[]; $params=[':id'=>$cid];
    foreach ($fields as $f){
      if (array_key_exists($f,$in)){
        if ($f==='smtp_pass' && ($in[$f]==='' || $in[$f]===null)) continue; // skip blank password to preserve existing
        $set[] = "$f = :$f"; $params[":$f"] = $in[$f];
      }
    }
    if (!$set) ApiResponse::success(null,'No changes');
    $sql = 'UPDATE companies SET '.implode(',',$set).', updated_at = NOW() WHERE id = :id';
    $st = $db->prepare($sql); foreach ($params as $k=>$v) $st->bindValue($k,$v); $st->execute();
    ApiResponse::success(null,'Email settings saved');
  } else {
    $f = storagePath('email-'.$cid.'.json');
    $curr = is_file($f) ? (json_decode(file_get_contents($f), true) ?: []) : [];
    foreach ($fields as $fkey){
      if ($fkey==='smtp_pass'){
        if (array_key_exists('smtp_pass',$in) && $in['smtp_pass']!=='') $curr['smtp_pass'] = $in['smtp_pass'];
      } else if (array_key_exists($fkey,$in)) { $curr[$fkey] = $in[$fkey]; }
    }
    file_put_contents($f, json_encode($curr, JSON_PRETTY_PRINT));
    ApiResponse::success(null,'Email settings saved');
  }
}

function sendTestEmail(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $to = trim((string)($in['to'] ?? ''));
  if ($to==='') ApiResponse::error('Recipient required');
  // Load email settings (file or db) to determine From
  $fromName = 'SmartQuantumHR'; $fromEmail = 'no-reply@localhost';
  $cid = targetCompanyIdFromGet($user);
  if (($user['role_slug'] ?? '') === 'super_admin') { $b=(int)($in['company_id']??0); if($b>0) $cid=$b; }
  $cols = ['email_from_name','email_from_email'];
  $exists = array_filter($cols, fn($c)=> tableHasColumn($db,'companies',$c));
  if ($exists){
    $st = $db->prepare('SELECT '.implode(',',$exists).' FROM companies WHERE id = :id');
    $st->bindValue(':id', $cid, PDO::PARAM_INT);
    $st->execute(); $row = $st->fetch() ?: [];
    if (!empty($row['email_from_name'])) $fromName = $row['email_from_name'];
    if (!empty($row['email_from_email'])) $fromEmail = $row['email_from_email'];
  } else {
    $f = storagePath('email-'.$cid.'.json');
    if (is_file($f)){
      $cfg = json_decode(file_get_contents($f), true) ?: [];
      if (!empty($cfg['email_from_name'])) $fromName = $cfg['email_from_name'];
      if (!empty($cfg['email_from_email'])) $fromEmail = $cfg['email_from_email'];
    }
  }
  $subject = 'SmartQuantumHR Test Email';
  $headers = "MIME-Version: 1.0\r\n";
  $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
  $headers .= "From: ".$fromName." <".$fromEmail.">\r\n";
  $ok = @mail($to, $subject, "This is a test email from SmartQuantumHR.", $headers);
  if ($ok) ApiResponse::success(null,'Test email sent (via PHP mail).');
  ApiResponse::error('Test email failed to send');
}

// ================= SMS (mNotify) Settings =================
function getSms(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = targetCompanyIdFromGet($user);
  $cfg = sms_load_config($cid);
  $out = [
    'enabled' => (int)($cfg['enabled'] ?? 0),
    'provider' => $cfg['provider'] ?? 'mnotify',
    'sender_id' => $cfg['sender_id'] ?? '',
    'api_key_masked' => !empty($cfg['api_key'])
  ];
  ApiResponse::success($out);
}

function saveSms(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = targetCompanyIdFromGet($user);
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  if (($user['role_slug'] ?? '') === 'super_admin') { $b=(int)($in['company_id']??0); if($b>0) $cid=$b; }
  $new = [];
  if (array_key_exists('enabled',$in)) $new['enabled'] = (int)!!$in['enabled'];
  if (!empty($in['provider'])) $new['provider'] = strtolower(trim((string)$in['provider']));
  if (array_key_exists('sender_id',$in)) $new['sender_id'] = trim((string)$in['sender_id']);
  if (array_key_exists('api_key',$in)) {
    $ak = trim((string)$in['api_key']);
    if ($ak !== '') $new['api_key'] = $ak; // keep existing if blank
  }
  sms_save_config($cid, $new);
  ApiResponse::success(null,'SMS settings saved');
}

function sendTestSms(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = targetCompanyIdFromGet($user);
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  if (($user['role_slug'] ?? '') === 'super_admin') { $b=(int)($in['company_id']??0); if($b>0) $cid=$b; }
  $to = trim((string)($in['to'] ?? ''));
  if ($to === '') ApiResponse::error('Recipient required');
  $ok = sms_send($cid, $to, 'This is a test SMS from SmartQuantumHR.');
  if ($ok) ApiResponse::success(null,'Test SMS sent');
  ApiResponse::error('Test SMS failed to send');
}

// Handle company logo upload (multipart/form-data)
function uploadCompanyLogo(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = (int)$user['company_id'];
  if (!tableHasColumn($db,'companies','logo')) ApiResponse::error('Logo field not available');
  if (!isset($_FILES['logo'])) ApiResponse::error('No file uploaded');
  $f = $_FILES['logo'];
  if (!is_array($f) || ($f['error'] ?? UPLOAD_ERR_OK) !== UPLOAD_ERR_OK) ApiResponse::error('Upload failed');
  $size = (int)($f['size'] ?? 0);
  if ($size <= 0 || $size > 5*1024*1024) ApiResponse::error('File too large (max 5MB)');
  $tmp = $f['tmp_name'];
  $name = (string)($f['name'] ?? 'logo');
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  $finfo = function_exists('finfo_open') ? finfo_open(FILEINFO_MIME_TYPE) : null;
  $mime = $finfo ? finfo_file($finfo, $tmp) : null; if ($finfo) finfo_close($finfo);
  $allowedExt = ['png','jpg','jpeg','gif','webp','svg'];
  $allowedMime = ['image/png','image/jpeg','image/gif','image/webp','image/svg+xml'];
  if (!in_array($ext, $allowedExt) && $mime && !in_array($mime, $allowedMime)) ApiResponse::error('Unsupported file type');
  // Ensure uploads directory exists
  $dir = realpath(__DIR__ . '/../storage/uploads/company_logos');
  if ($dir === false) { @mkdir(__DIR__ . '/../storage/uploads/company_logos', 0777, true); $dir = realpath(__DIR__ . '/../storage/uploads/company_logos'); }
  if ($dir === false) ApiResponse::error('Failed to prepare upload directory');
  $fname = 'company-'.$cid.'-'.date('YmdHis').'-'.bin2hex(random_bytes(4)).'.'.($ext ?: 'img');
  $dest = rtrim($dir, '/\\') . DIRECTORY_SEPARATOR . $fname;
  if (!@move_uploaded_file($tmp, $dest)) ApiResponse::error('Failed to save file');
  // Build app-relative path for storage (web accessible)
  $rel = 'storage/uploads/company_logos/' . $fname;
  try {
    $st = $db->prepare('UPDATE companies SET logo = :logo, updated_at = NOW() WHERE id = :id');
    $st->bindValue(':logo', $rel);
    $st->bindValue(':id', $cid, PDO::PARAM_INT);
    $st->execute();
  } catch (Throwable $e) { ApiResponse::error('Failed to update company: '.$e->getMessage(), 500); }
  ApiResponse::success(['logo' => $rel], 'Logo updated');
}

// Clear company logo (sets companies.logo to NULL)
function clearCompanyLogo(PDO $db){
  $user = getCurrentUser(); mustBeAdmin($user);
  $cid = (int)$user['company_id'];
  if (!tableHasColumn($db,'companies','logo')) ApiResponse::error('Logo field not available');
  try {
    $st = $db->prepare('UPDATE companies SET logo = NULL, updated_at = NOW() WHERE id = :id');
    $st->bindValue(':id', $cid, PDO::PARAM_INT);
    $st->execute();
  } catch (Throwable $e) { ApiResponse::error('Failed to clear logo: '.$e->getMessage(), 500); }
  ApiResponse::success(null, 'Logo removed');
}
