<?php
/**
 * Organization Management API (Departments, Branches, Positions, Sub Companies)
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';
requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$entity = $_GET['entity'] ?? '';
$id = isset($_GET['id']) ? intval($_GET['id']) : null;

 $database = new Database();
 $db = $database->getConnection();

  function ensureBranchesTable(PDO $db) {
    try {
      $db->exec("CREATE TABLE IF NOT EXISTS branches (
        id INT AUTO_INCREMENT PRIMARY KEY,
        company_id INT NOT NULL,
        sub_company_id INT NULL,
        department_id INT NULL,
        name VARCHAR(255) NOT NULL,
        code VARCHAR(100) NULL,
        type VARCHAR(50) DEFAULT 'branch',
        address VARCHAR(255) NULL,
        phone VARCHAR(50) NULL,
        manager_id INT NULL,
        status ENUM('active','inactive') DEFAULT 'active',
        created_at DATETIME NULL,
        updated_at DATETIME NULL,
        INDEX idx_company (company_id),
        INDEX idx_manager (manager_id)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    } catch (Throwable $e) { /* ignore */ }
  }

  function canManageOrg($user) {
  // HR Officer can create/edit; Admin and HR Head full control
  return in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer']);
}

  // Deactivation (soft delete) policy: tighten for sensitive entities
  function canDeactivateOrg($user, $entity) {
    $role = $user['role_slug'] ?? null;
    if ($entity === 'positions') {
      // HR Officer cannot deactivate positions; requires HR Head/Admin/Super Admin
      return in_array($role, ['super_admin','admin','hr_head']);
    }
    // Default: use broader manage rights
    return in_array($role, ['super_admin','admin','hr_head','hr_officer']);
  }

if (!in_array($entity, ['departments','branches','positions','sub_companies'])) {
    ApiResponse::error('Invalid or missing entity');
}

switch ($method) {
    case 'GET':
        if ($id) {
            getOne($db, $entity, $id);
        } else {
            listAll($db, $entity);
        }
        break;
    case 'POST':
        createOne($db, $entity);
        break;
    case 'PUT':
        updateOne($db, $entity, $id);
        break;
    case 'DELETE':
        deleteOne($db, $entity, $id);
        break;
    default:
        ApiResponse::error('Method not allowed', 405);
}

function listAll($db, $entity) {
    $user = getCurrentUser();
    $companyId = $user['company_id'];

    switch ($entity) {
        case 'departments':
            $sql = "SELECT d.*, CONCAT(m.first_name,' ',m.last_name) AS head_name
                    FROM departments d
                    LEFT JOIN employees m ON d.head_id = m.id
                    WHERE d.company_id = :cid
                    ORDER BY d.name";
            break;
        case 'branches':
            ensureBranchesTable($db);
            $sql = "SELECT b.*, CONCAT(m.first_name,' ',m.last_name) AS manager_name
                    FROM branches b
                    LEFT JOIN employees m ON b.manager_id = m.id
                    WHERE b.company_id = :cid
                    ORDER BY b.name";
            break;
        case 'positions':
            $sql = "SELECT p.*, d.name AS department_name
                    FROM positions p
                    LEFT JOIN departments d ON p.department_id = d.id
                    WHERE p.company_id = :cid
                    ORDER BY p.title";
            break;
        case 'sub_companies':
            $sql = "SELECT sc.*, CONCAT(m.first_name,' ',m.last_name) AS manager_name
                    FROM sub_companies sc
                    LEFT JOIN employees m ON sc.manager_id = m.id
                    WHERE sc.parent_company_id = :cid
                    ORDER BY sc.name";
            break;
    }

    $stmt = $db->prepare($sql);
    $stmt->bindValue(':cid', $companyId, PDO::PARAM_INT);
    $stmt->execute();
    ApiResponse::success($stmt->fetchAll());
}

function getOne($db, $entity, $id) {
    $user = getCurrentUser();
    $companyId = $user['company_id'];

    switch ($entity) {
        case 'departments':
            $sql = "SELECT * FROM departments WHERE id = :id AND company_id = :cid";
            break;
        case 'branches':
            ensureBranchesTable($db);
            $sql = "SELECT * FROM branches WHERE id = :id AND company_id = :cid";
            break;
        case 'positions':
            $sql = "SELECT * FROM positions WHERE id = :id AND company_id = :cid";
            break;
        case 'sub_companies':
            $sql = "SELECT * FROM sub_companies WHERE id = :id AND parent_company_id = :cid";
            break;
    }

    $stmt = $db->prepare($sql);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->bindValue(':cid', $companyId, PDO::PARAM_INT);
    $stmt->execute();

    if ($stmt->rowCount() === 0) ApiResponse::notFound('Record not found');
    ApiResponse::success($stmt->fetch());
}

function createOne($db, $entity) {
    $user = getCurrentUser();
    if (!canManageOrg($user)) ApiResponse::forbidden('Insufficient permissions');

    $input = json_decode(file_get_contents('php://input'), true) ?? [];
    $cid = $user['company_id'];

    switch ($entity) {
        case 'departments':
            required($input, ['name','code']);
            $sql = "INSERT INTO departments (company_id, sub_company_id, name, code, description, head_id, parent_department_id, budget, status, created_at)
                    VALUES (:cid, :sub_company_id, :name, :code, :description, :head_id, :parent_id, :budget, :status, NOW())";
            $stmt = $db->prepare($sql);
            bind($stmt, [
                ':cid' => $cid,
                ':sub_company_id' => $input['sub_company_id'] ?? null,
                ':name' => $input['name'],
                ':code' => $input['code'],
                ':description' => $input['description'] ?? null,
                ':head_id' => $input['head_id'] ?? null,
                ':parent_id' => $input['parent_department_id'] ?? null,
                ':budget' => $input['budget'] ?? null,
                ':status' => $input['status'] ?? 'active',
            ]);
            break;
        case 'branches':
            ensureBranchesTable($db);
            required($input, ['name','code']);
            $sql = "INSERT INTO branches (company_id, sub_company_id, department_id, name, code, type, address, phone, manager_id, status, created_at)
                    VALUES (:cid, :sub_company_id, :department_id, :name, :code, :type, :address, :phone, :manager_id, :status, NOW())";
            $stmt = $db->prepare($sql);
            bind($stmt, [
                ':cid' => $cid,
                ':sub_company_id' => $input['sub_company_id'] ?? null,
                ':department_id' => $input['department_id'] ?? null,
                ':name' => $input['name'],
                ':code' => $input['code'],
                ':type' => $input['type'] ?? 'branch',
                ':address' => $input['address'] ?? null,
                ':phone' => $input['phone'] ?? null,
                ':manager_id' => $input['manager_id'] ?? null,
                ':status' => $input['status'] ?? 'active',
            ]);
            break;
        case 'positions':
            required($input, ['title','code']);
            $sql = "INSERT INTO positions (company_id, department_id, title, code, description, requirements, responsibilities, min_salary, max_salary, level, status, created_at)
                    VALUES (:cid, :department_id, :title, :code, :description, :requirements, :responsibilities, :min_salary, :max_salary, :level, :status, NOW())";
            $stmt = $db->prepare($sql);
            // Enforce unique code per company for positions
            try {
                $du = $db->prepare("SELECT id FROM positions WHERE company_id = :cid AND code = :code LIMIT 1");
                $du->bindValue(':cid', $cid, PDO::PARAM_INT);
                $du->bindValue(':code', $input['code']);
                $du->execute();
                if ($du->fetchColumn()) { ApiResponse::error('Position code already exists'); }
            } catch (Throwable $e) { /* best-effort; continue */ }
            bind($stmt, [
                ':cid' => $cid,
                ':department_id' => $input['department_id'] ?? null,
                ':title' => $input['title'],
                ':code' => $input['code'],
                ':description' => $input['description'] ?? null,
                ':requirements' => $input['requirements'] ?? null,
                ':responsibilities' => $input['responsibilities'] ?? null,
                ':min_salary' => $input['min_salary'] ?? null,
                ':max_salary' => $input['max_salary'] ?? null,
                ':level' => $input['level'] ?? 'entry',
                ':status' => $input['status'] ?? 'active',
            ]);
            break;
        case 'sub_companies':
            required($input, ['name','code']);
            $sql = "INSERT INTO sub_companies (parent_company_id, name, code, email, phone, address, manager_id, status, settings, created_at)
                    VALUES (:cid, :name, :code, :email, :phone, :address, :manager_id, :status, :settings, NOW())";
            $stmt = $db->prepare($sql);
            bind($stmt, [
                ':cid' => $cid,
                ':name' => $input['name'],
                ':code' => $input['code'],
                ':email' => $input['email'] ?? null,
                ':phone' => $input['phone'] ?? null,
                ':address' => $input['address'] ?? null,
                ':manager_id' => $input['manager_id'] ?? null,
                ':status' => $input['status'] ?? 'active',
                ':settings' => isset($input['settings']) ? json_encode($input['settings']) : null,
            ]);
            break;
    }

    $stmt->execute();
    $id = $db->lastInsertId();
    // Log create action
    try { app_log('info', 'org_create', ['entity'=>$entity, 'id'=>$id, 'user_id'=>getCurrentUser()['id'] ?? null]); } catch (Throwable $e) {}
    try { recordActivity($db, (int)$user['company_id'], (int)$user['id'], 'create_'.$entity, $entity, (int)$id, null, ['payload'=>$input]); } catch (Throwable $e) {}
    ApiResponse::success(['id' => $id], 'Created successfully');
}

function updateOne($db, $entity, $id) {
    $user = getCurrentUser();
    if (!canManageOrg($user)) ApiResponse::forbidden('Insufficient permissions');
    if (!$id) ApiResponse::error('ID required');
    $input = json_decode(file_get_contents('php://input'), true) ?? [];
    switch ($entity) {
        case 'departments':
            $allowed = ['name','code','description','head_id','parent_department_id','budget','status'];
            $table = 'departments';
            break;
        case 'branches':
            $allowed = ['name','code','type','address','phone','manager_id','department_id','sub_company_id','status'];
            $table = 'branches';
            break;
        case 'positions':
            $allowed = ['title','code','description','requirements','responsibilities','min_salary','max_salary','level','status','department_id'];
            $table = 'positions';
            break;
        case 'sub_companies':
            $allowed = ['name','code','email','phone','address','manager_id','status','settings'];
            $table = 'sub_companies';
            break;
    }

    $set = [];
    $params = [':id' => $id, ':cid' => ($entity==='sub_companies' ? getCurrentUser()['company_id'] : getCurrentUser()['company_id'])];

    foreach ($allowed as $f) {
        if (array_key_exists($f, $input)) {
            $set[] = "$f = :$f";
            $params[":$f"] = $f === 'settings' ? json_encode($input[$f]) : $input[$f];
        }
    }
    if (!$set) ApiResponse::error('No valid fields to update');

    if ($entity === 'sub_companies') {
        $sql = "UPDATE $table SET ".implode(', ', $set).", updated_at = NOW() WHERE id = :id AND parent_company_id = :cid";
    } else {
        $sql = "UPDATE $table SET ".implode(', ', $set).", updated_at = NOW() WHERE id = :id AND company_id = :cid";
    }

    $stmt = $db->prepare($sql);
    foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
    $stmt->execute();
    // Log update action
    try { app_log('info', 'org_update', ['entity'=>$entity, 'id'=>$id, 'user_id'=>getCurrentUser()['id'] ?? null]); } catch (Throwable $e) {}
    try { recordActivity($db, (int)getCurrentUser()['company_id'], (int)getCurrentUser()['id'], 'update_'.$entity, $entity, (int)$id, null, ['payload'=>$input]); } catch (Throwable $e) {}
    ApiResponse::success(null, 'Updated successfully');
}

function deleteOne($db, $entity, $id) {
    $user = getCurrentUser();
    if (!canDeactivateOrg($user, $entity)) ApiResponse::forbidden('Insufficient permissions');
    if (!$id) ApiResponse::error('ID required');

    switch ($entity) {
        case 'departments':
            $sql = "UPDATE departments SET status='inactive', updated_at = NOW() WHERE id = :id AND company_id = :cid";
            break;
        case 'branches':
            $sql = "UPDATE branches SET status='inactive', updated_at = NOW() WHERE id = :id AND company_id = :cid";
            break;
        case 'positions':
            // Safety: prevent deactivation if there are active employees on this position
            try {
                $chk = $db->prepare("SELECT COUNT(*) FROM employees WHERE company_id = :cid AND position_id = :id AND (employment_status IS NULL OR employment_status = 'active')");
                $chk->bindValue(':cid', $user['company_id'], PDO::PARAM_INT);
                $chk->bindValue(':id', $id, PDO::PARAM_INT);
                $chk->execute();
                $cnt = (int)$chk->fetchColumn();
                if ($cnt > 0) {
                    ApiResponse::error("Cannot deactivate position: $cnt active employee(s) assigned. Reassign before deactivating.");
                }
            } catch (Throwable $e) { /* best-effort */ }
            $sql = "UPDATE positions SET status='inactive', updated_at = NOW() WHERE id = :id AND company_id = :cid";
            break;
        case 'sub_companies':
            $sql = "UPDATE sub_companies SET status='inactive', updated_at = NOW() WHERE id = :id AND parent_company_id = :cid";
            break;
    }

    $stmt = $db->prepare($sql);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->bindValue(':cid', $user['company_id'], PDO::PARAM_INT);
    $stmt->execute();
    // Log delete (deactivate) action
    try { app_log('info', 'org_deactivate', ['entity'=>$entity, 'id'=>$id, 'user_id'=>$user['id'] ?? null]); } catch (Throwable $e) {}
    try { recordActivity($db, (int)$user['company_id'], (int)$user['id'], 'deactivate_'.$entity, $entity, (int)$id, null, null); } catch (Throwable $e) {}
    ApiResponse::success(null, 'Deleted successfully');
}

function required($input, $fields) {
    foreach ($fields as $f) {
        if (!isset($input[$f]) || $input[$f] === '') {
            ApiResponse::error("Field '$f' is required");
        }
    }
}

function bind($stmt, $map) {
    foreach ($map as $k=>$v) {
        $stmt->bindValue($k, $v);
    }
}
