<?php
/**
 * Notifications API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? null;

$database = new Database();
$db = $database->getConnection();

switch ($method) {
    case 'GET':
        if ($action === 'unread_count') {
            getUnreadCount($db);
        } elseif ($action === 'detail') {
            notificationDetail($db);
        } else {
            listNotifications($db);
        }
        break;

    case 'POST':
        if ($action === 'read') {
            markAsRead($db, $id);
        } elseif ($action === 'read_all') {
            markAllAsRead($db);
        } else {
            ApiResponse::error('Invalid action');
        }
        break;

    default:
        ApiResponse::error('Method not allowed', 405);
}

function listNotifications($db) {
    $user = getCurrentUser();
    $limit = max(1, intval($_GET['limit'] ?? 20));
    $offset = max(0, intval($_GET['offset'] ?? 0));
    $status = strtolower(trim((string)($_GET['status'] ?? 'all')));

    $conds = ['user_id = :user_id'];
    if ($status === 'unread') { $conds[] = 'read_at IS NULL'; }

    $query = "SELECT id, type, title, content, data, read_at, created_at
              FROM notifications
              WHERE ".implode(' AND ', $conds)."
              ORDER BY COALESCE(read_at, created_at) DESC, id DESC
              LIMIT :limit OFFSET :offset";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':user_id', $user['id'], PDO::PARAM_INT);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();

    $items = $stmt->fetchAll();
    ApiResponse::success($items);
}

function getUnreadCount($db) {
    $user = getCurrentUser();
    $query = "SELECT COUNT(*) AS unread FROM notifications WHERE user_id = :user_id AND read_at IS NULL";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':user_id', $user['id'], PDO::PARAM_INT);
    $stmt->execute();
    $row = $stmt->fetch();
    ApiResponse::success(['unread' => intval($row['unread'] ?? 0)]);
}

function markAsRead($db, $id) {
    $user = getCurrentUser();
    if (!$id) ApiResponse::error('Notification id required');

    $query = "UPDATE notifications SET read_at = NOW() WHERE id = :id AND user_id = :user_id";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->bindValue(':user_id', $user['id'], PDO::PARAM_INT);
    $stmt->execute();

    ApiResponse::success(null, 'Notification marked as read');
}

function markAllAsRead($db) {
    $user = getCurrentUser();
    $query = "UPDATE notifications SET read_at = NOW() WHERE user_id = :user_id AND read_at IS NULL";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':user_id', $user['id'], PDO::PARAM_INT);
    $stmt->execute();

    ApiResponse::success(null, 'All notifications marked as read');
}

function notificationDetail(PDO $db){
    $user = getCurrentUser();
    $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
    if ($id <= 0) ApiResponse::error('Invalid id');
    $st = $db->prepare("SELECT id, type, title, content, data, read_at, created_at FROM notifications WHERE id = :id AND user_id = :uid LIMIT 1");
    $st->bindValue(':id', $id, PDO::PARAM_INT);
    $st->bindValue(':uid', $user['id'], PDO::PARAM_INT);
    $st->execute();
    if ($st->rowCount() === 0) ApiResponse::notFound('Not found');
    ApiResponse::success($st->fetch());
}
