<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

$database = new Database();
$db = $database->getConnection();

// Basic role helpers
function learning_is_hr(array $u): bool { $r = strtolower($u['role_slug'] ?? ''); return in_array($r, ['super_admin','admin','hr_head','hr_officer'], true); }

// Ensure minimal schema; detailed tables will be added as we implement
function learning_ensure_min_schema(PDO $db) {
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS lessons (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      title VARCHAR(255) NOT NULL,
      description TEXT NULL,
      content_type ENUM('text','video','link') NOT NULL DEFAULT 'text',
      content_url TEXT NULL,
      content_text LONGTEXT NULL,
      status ENUM('draft','active','archived') NOT NULL DEFAULT 'active',
      created_by INT NOT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      KEY idx_company (company_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  // Assessments schema
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS assessments (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      title VARCHAR(255) NOT NULL,
      description TEXT NULL,
      type ENUM('quiz','assignment','test') NOT NULL DEFAULT 'quiz',
      delivery ENUM('online','offline') NOT NULL DEFAULT 'online',
      linked_type ENUM('lesson','course') NULL,
      linked_id INT NULL,
      total_points INT NOT NULL DEFAULT 0,
      duration_minutes INT NULL,
      grading_scheme LONGTEXT NULL,
      status ENUM('draft','active','archived') NOT NULL DEFAULT 'active',
      created_by INT NOT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      KEY idx_company (company_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  // Backfill: add duration_minutes if table already existed
  try { $db->exec("ALTER TABLE assessments ADD COLUMN duration_minutes INT NULL AFTER total_points"); } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS assessment_questions (
      id INT AUTO_INCREMENT PRIMARY KEY,
      assessment_id INT NOT NULL,
      qtype ENUM('mcq','truefalse','text') NOT NULL DEFAULT 'mcq',
      question_text TEXT NOT NULL,
      options_json LONGTEXT NULL,
      correct_json LONGTEXT NULL,
      points INT NOT NULL DEFAULT 1,
      sort_order INT NOT NULL DEFAULT 0,
      KEY idx_assessment (assessment_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS assessment_submissions (
      id INT AUTO_INCREMENT PRIMARY KEY,
      assessment_id INT NOT NULL,
      employee_id INT NOT NULL,
      attempt_number INT NOT NULL DEFAULT 1,
      submitted_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
      status ENUM('submitted','graded') NOT NULL DEFAULT 'submitted',
      score INT NULL,
      grade VARCHAR(20) NULL,
      feedback TEXT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      KEY idx_assessment_emp (assessment_id, employee_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS submission_answers (
      id INT AUTO_INCREMENT PRIMARY KEY,
      submission_id INT NOT NULL,
      question_id INT NOT NULL,
      answer_text LONGTEXT NULL,
      answer_json LONGTEXT NULL,
      KEY idx_sub (submission_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
}

// Extended schema for courses, plans, enrollments, progress
function learning_ensure_extended_schema(PDO $db) {
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS courses (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      title VARCHAR(255) NOT NULL,
      description TEXT NULL,
      level VARCHAR(50) NULL,
      category VARCHAR(100) NULL,
      status ENUM('draft','active','archived') NOT NULL DEFAULT 'active',
      created_by INT NOT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      KEY idx_company (company_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS course_lessons (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      course_id INT NOT NULL,
      lesson_id INT NOT NULL,
      sort_order INT NOT NULL DEFAULT 0,
      UNIQUE KEY uniq_course_lesson (course_id, lesson_id),
      KEY idx_company (company_id),
      KEY idx_course (course_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS learning_plans (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      owner_user_id INT NULL,
      title VARCHAR(255) NOT NULL,
      purpose VARCHAR(255) NULL,
      outcome TEXT NULL,
      status ENUM('draft','active','archived') NOT NULL DEFAULT 'active',
      created_by INT NOT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      KEY idx_company (company_id),
      KEY idx_owner (owner_user_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS plan_items (
      id INT AUTO_INCREMENT PRIMARY KEY,
      plan_id INT NOT NULL,
      item_type ENUM('lesson','course') NOT NULL,
      item_id INT NOT NULL,
      sort_order INT NOT NULL DEFAULT 0,
      UNIQUE KEY uniq_item (plan_id, item_type, item_id),
      KEY idx_plan (plan_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS plan_enrollments (
      id INT AUTO_INCREMENT PRIMARY KEY,
      plan_id INT NOT NULL,
      employee_id INT NOT NULL,
      status ENUM('enrolled','in_progress','completed') NOT NULL DEFAULT 'enrolled',
      progress_percent TINYINT NOT NULL DEFAULT 0,
      started_at TIMESTAMP NULL DEFAULT NULL,
      completed_at TIMESTAMP NULL DEFAULT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      UNIQUE KEY uniq_enroll (plan_id, employee_id),
      KEY idx_plan (plan_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS lesson_progress (
      id INT AUTO_INCREMENT PRIMARY KEY,
      plan_id INT NOT NULL,
      employee_id INT NOT NULL,
      lesson_id INT NOT NULL,
      status ENUM('not_started','in_progress','completed') NOT NULL DEFAULT 'in_progress',
      progress TINYINT NOT NULL DEFAULT 0,
      last_accessed TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP,
      completed_at TIMESTAMP NULL DEFAULT NULL,
      UNIQUE KEY uniq_prog (plan_id, employee_id, lesson_id),
      KEY idx_plan_emp (plan_id, employee_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  // Applicability rules for plans
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS plan_rules (
      id INT AUTO_INCREMENT PRIMARY KEY,
      plan_id INT NOT NULL,
      rule_type ENUM('enrollment','outcome') NOT NULL,
      condition_json LONGTEXT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      KEY idx_plan (plan_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
  // Course ratings for analytics
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS course_ratings (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      course_id INT NOT NULL,
      user_id INT NOT NULL,
      rating TINYINT NOT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      UNIQUE KEY uniq_course_user (course_id, user_id),
      KEY idx_company (company_id),
      KEY idx_course (course_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  } catch (Throwable $e) {}
}

learning_ensure_min_schema($db);
learning_ensure_extended_schema($db);
$user = getCurrentUser();

switch ($method) {
  case 'GET':
    if ($action === 'lessons') {
      $st = $db->prepare('SELECT * FROM lessons WHERE company_id = :cid AND status <> "archived" ORDER BY created_at DESC');
      $st->bindValue(':cid', (int)$user['company_id'], PDO::PARAM_INT);
      $st->execute();
      ApiResponse::success($st->fetchAll());
    } elseif ($action === 'courses') {
      $st = $db->prepare('SELECT c.*, (SELECT COUNT(*) FROM course_lessons cl WHERE cl.course_id=c.id) AS lesson_count FROM courses c WHERE c.company_id=:cid AND c.status<>"archived" ORDER BY c.created_at DESC');
      $st->bindValue(':cid', (int)$user['company_id'], PDO::PARAM_INT);
      $st->execute();
      ApiResponse::success($st->fetchAll());
    } elseif ($action === 'course' && $id) {
      $q=$db->prepare('SELECT * FROM courses WHERE id=:id AND company_id=:cid');
      $q->bindValue(':id',$id,PDO::PARAM_INT); $q->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT); $q->execute();
      if ($q->rowCount()===0) ApiResponse::notFound('Course not found');
      $course=$q->fetch();
      $cl=$db->prepare('SELECT l.* FROM course_lessons cl JOIN lessons l ON l.id = cl.lesson_id WHERE cl.course_id = :cid2 ORDER BY cl.sort_order');
      $cl->bindValue(':cid2',$id,PDO::PARAM_INT); $cl->execute(); $course['lessons']=$cl->fetchAll();
      ApiResponse::success($course);
    } elseif ($action === 'catalog') {
      $cid=(int)$user['company_id'];
      $ls=$db->prepare('SELECT id, title, description, content_type, content_url, status FROM lessons WHERE company_id=:c AND status="active" ORDER BY created_at DESC');
      $ls->bindValue(':c',$cid,PDO::PARAM_INT); $ls->execute(); $lessons=$ls->fetchAll();
      $cs=$db->prepare('SELECT id, title, description, status, (SELECT COUNT(*) FROM course_lessons cl WHERE cl.course_id=c.id) AS lesson_count FROM courses c WHERE company_id=:c AND status="active" ORDER BY created_at DESC');
      $cs->bindValue(':c',$cid,PDO::PARAM_INT); $cs->execute(); $courses=$cs->fetchAll();
      ApiResponse::success(['lessons'=>$lessons,'courses'=>$courses]);
    } elseif ($action === 'plans') {
      $cid=(int)$user['company_id'];
      $role=strtolower($user['role_slug'] ?? '');
      if (in_array($role,['super_admin','admin','hr_head','hr_officer'],true)){
        $st=$db->prepare('SELECT * FROM learning_plans WHERE company_id=:c AND status<>"archived" ORDER BY created_at DESC');
        $st->bindValue(':c',$cid,PDO::PARAM_INT); $st->execute(); ApiResponse::success($st->fetchAll());
      } else {
        $st=$db->prepare('SELECT DISTINCT p.* FROM learning_plans p LEFT JOIN plan_enrollments pe ON pe.plan_id = p.id LEFT JOIN employees e ON e.id = pe.employee_id WHERE p.company_id=:c AND p.status<>"archived" AND (p.owner_user_id=:u OR e.user_id=:u) ORDER BY p.created_at DESC');
        $st->bindValue(':c',$cid,PDO::PARAM_INT); $st->bindValue(':u',(int)$user['id'],PDO::PARAM_INT); $st->execute(); ApiResponse::success($st->fetchAll());
      }
    } elseif ($action === 'plan' && $id) {
      $q=$db->prepare('SELECT * FROM learning_plans WHERE id=:id AND company_id=:cid');
      $q->bindValue(':id',$id,PDO::PARAM_INT); $q->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT); $q->execute();
      if ($q->rowCount()===0) ApiResponse::notFound('Plan not found');
      $plan=$q->fetch();
      $it=$db->prepare('SELECT * FROM plan_items WHERE plan_id=:p ORDER BY sort_order'); $it->bindValue(':p',$id,PDO::PARAM_INT); $it->execute(); $plan['items']=$it->fetchAll();
      ApiResponse::success($plan);
    } elseif ($action === 'assessments') {
      $st=$db->prepare('SELECT * FROM assessments WHERE company_id=:cid AND status<>"archived" ORDER BY created_at DESC');
      $st->bindValue(':cid', (int)$user['company_id'], PDO::PARAM_INT); $st->execute();
      ApiResponse::success($st->fetchAll());
    } elseif ($action === 'assessment' && $id) {
      $q=$db->prepare('SELECT * FROM assessments WHERE id=:id AND company_id=:cid'); $q->bindValue(':id',$id,PDO::PARAM_INT); $q->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT); $q->execute(); if ($q->rowCount()===0) ApiResponse::notFound('Assessment not found');
      $ass=$q->fetch();
      $qq=$db->prepare('SELECT * FROM assessment_questions WHERE assessment_id=:a ORDER BY sort_order'); $qq->bindValue(':a',$id,PDO::PARAM_INT); $qq->execute(); $ass['questions']=$qq->fetchAll();
      ApiResponse::success($ass);
    } elseif ($action === 'plan_rules' && $id) {
      // List applicability rules for a plan
      $st=$db->prepare('SELECT * FROM plan_rules WHERE plan_id=:p ORDER BY id DESC');
      $st->bindValue(':p',$id,PDO::PARAM_INT); $st->execute();
      ApiResponse::success($st->fetchAll());
    } elseif ($action === 'submissions') {
      // List assessment submissions (HR/Admin only)
      $role = strtolower($user['role_slug'] ?? '');
      if (!in_array($role, ['super_admin','admin','hr_head','hr_officer'], true)) ApiResponse::forbidden('Only HR/Admin can view submissions');
      $aid = isset($_GET['assessment_id']) ? (int)$_GET['assessment_id'] : ($id ?? 0);
      $sql = 'SELECT s.*, e.id AS employee_id, u.first_name, u.last_name FROM assessment_submissions s LEFT JOIN employees e ON e.id = s.employee_id LEFT JOIN users u ON u.id = e.user_id';
      if ($aid) { $sql .= ' WHERE s.assessment_id = :aid'; }
      $sql .= ' ORDER BY s.created_at DESC';
      $st=$db->prepare($sql);
      if ($aid) $st->bindValue(':aid',$aid,PDO::PARAM_INT);
      $st->execute();
      ApiResponse::success($st->fetchAll());
    } elseif ($action === 'reports_course_ratings') {
      // Aggregate ratings per course for org analytics
      $cid=(int)$user['company_id'];
      $sql = 'SELECT c.id, c.title, r.rating, COUNT(*) as cnt FROM courses c LEFT JOIN course_ratings r ON r.course_id = c.id AND r.company_id = :cid WHERE c.company_id = :cid2 GROUP BY c.id, r.rating ORDER BY c.title';
      $st=$db->prepare($sql); $st->bindValue(':cid',$cid,PDO::PARAM_INT); $st->bindValue(':cid2',$cid,PDO::PARAM_INT); $st->execute();
      $rows=$st->fetchAll();
      // reshape
      $out = [];
      foreach ($rows as $r){ $id=(int)$r['id']; if(!isset($out[$id])) $out[$id] = ['course_id'=>$id,'title'=>$r['title'],'ratings'=>[1=>0,2=>0,3=>0,4=>0,5=>0]]; $rate=(int)($r['rating'] ?? 0); if($rate>=1 && $rate<=5){ $out[$id]['ratings'][$rate] = (int)$r['cnt']; } }
      ApiResponse::success(array_values($out));
    } else {
      ApiResponse::success(['message' => 'Learning API ready']);
    }
    break;

  case 'POST':
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    if ($action === 'create_lesson') {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can create lessons');
      if (empty($in['title'])) ApiResponse::error("Field 'title' is required");
      $q = $db->prepare('INSERT INTO lessons (company_id,title,description,content_type,content_url,content_text,status,created_by,created_at) VALUES (:cid,:t,:d,:ct,:cu,:cx,:s,:u,NOW())');
      $q->bindValue(':cid', (int)$user['company_id'], PDO::PARAM_INT);
      $q->bindValue(':t', $in['title']);
      $q->bindValue(':d', $in['description'] ?? null);
      $q->bindValue(':ct', $in['content_type'] ?? 'text');
      $q->bindValue(':cu', $in['content_url'] ?? null);
      $q->bindValue(':cx', $in['content_text'] ?? null);
      $q->bindValue(':s', $in['status'] ?? 'active');
      $q->bindValue(':u', (int)$user['id'], PDO::PARAM_INT);
      $q->execute();
      ApiResponse::success(['id'=>$db->lastInsertId()], 'Lesson created');
    } elseif ($action === 'create_course') {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can create courses');
      if (empty($in['title'])) ApiResponse::error("Field 'title' is required");
      $st=$db->prepare('INSERT INTO courses (company_id,title,description,level,category,status,created_by,created_at) VALUES (:cid,:t,:d,:lvl,:cat,:s,:u,NOW())');
      $st->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT);
      $st->bindValue(':t',$in['title']);
      $st->bindValue(':d',$in['description'] ?? null);
      $st->bindValue(':lvl',$in['level'] ?? null);
      $st->bindValue(':cat',$in['category'] ?? null);
      $st->bindValue(':s',$in['status'] ?? 'active');
      $st->bindValue(':u',(int)$user['id'],PDO::PARAM_INT);
      $st->execute();
      ApiResponse::success(['id'=>$db->lastInsertId()], 'Course created');
    } elseif ($action === 'update_course' && $id) {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can update courses');
      $st=$db->prepare('UPDATE courses SET title = COALESCE(:t, title), description = COALESCE(:d, description), level = COALESCE(:lvl, level), category = COALESCE(:cat, category), status = COALESCE(:s, status), updated_at = NOW() WHERE id = :id AND company_id = :cid');
      $st->bindValue(':t', $in['title'] ?? null);
      $st->bindValue(':d', $in['description'] ?? null);
      $st->bindValue(':lvl', $in['level'] ?? null);
      $st->bindValue(':cat', $in['category'] ?? null);
      $st->bindValue(':s', $in['status'] ?? null);
      $st->bindValue(':id',$id,PDO::PARAM_INT);
      $st->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT);
      $st->execute();
      ApiResponse::success(null,'Course updated');
    } elseif ($action === 'publish_course' && $id) {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can publish');
      $st=$db->prepare('UPDATE courses SET status = "active", updated_at = NOW() WHERE id = :id AND company_id = :cid');
      $st->bindValue(':id',$id,PDO::PARAM_INT); $st->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT); $st->execute();
      ApiResponse::success(null,'Course published');
    } elseif ($action === 'set_course_lessons' && $id) {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can set course lessons');
      $ids = isset($in['lesson_ids']) && is_array($in['lesson_ids']) ? $in['lesson_ids'] : [];
      $db->beginTransaction();
      try {
        $del=$db->prepare('DELETE FROM course_lessons WHERE course_id = :cid'); $del->bindValue(':cid',$id,PDO::PARAM_INT); $del->execute();
        $ins=$db->prepare('INSERT INTO course_lessons (company_id, course_id, lesson_id, sort_order) VALUES (:co,:c,:l,:s)');
        $order=0; foreach ($ids as $lid){ $order++; $ins->bindValue(':co',(int)$user['company_id'],PDO::PARAM_INT); $ins->bindValue(':c',$id,PDO::PARAM_INT); $ins->bindValue(':l',(int)$lid,PDO::PARAM_INT); $ins->bindValue(':s',$order,PDO::PARAM_INT); $ins->execute(); }
        $db->commit();
      } catch (Throwable $e){ $db->rollBack(); ApiResponse::error('Failed to set lessons: '.$e->getMessage(),500); }
      ApiResponse::success(null,'Course lessons updated');
    } elseif ($action === 'add_course_rating' && $id) {
      $rating = isset($in['rating']) ? (int)$in['rating'] : 0; if ($rating < 1 || $rating > 5) ApiResponse::error('Rating must be 1-5');
      $st=$db->prepare('INSERT INTO course_ratings (company_id, course_id, user_id, rating, created_at) VALUES (:c,:cid,:u,:r, NOW()) ON DUPLICATE KEY UPDATE rating = VALUES(rating), updated_at = NOW()');
      $st->bindValue(':c',(int)$user['company_id'],PDO::PARAM_INT); $st->bindValue(':cid',$id,PDO::PARAM_INT); $st->bindValue(':u',(int)$user['id'],PDO::PARAM_INT); $st->bindValue(':r',$rating,PDO::PARAM_INT); $st->execute();
      ApiResponse::success(null,'Rating saved');
    } elseif ($action === 'create_plan') {
      $title=trim((string)($in['title'] ?? ''));
      if ($title==='') ApiResponse::error("Field 'title' is required");
      $owner = $user['id'];
      $q=$db->prepare('INSERT INTO learning_plans (company_id, owner_user_id, title, purpose, outcome, status, created_by, created_at) VALUES (:c,:o,:t,:p,:o2,:s,:u,NOW())');
      $q->bindValue(':c',(int)$user['company_id'],PDO::PARAM_INT); $q->bindValue(':o',(int)$owner,PDO::PARAM_INT); $q->bindValue(':t',$title); $q->bindValue(':p',$in['purpose'] ?? null); $q->bindValue(':o2',$in['outcome'] ?? null); $q->bindValue(':s',$in['status'] ?? 'active'); $q->bindValue(':u',(int)$user['id'],PDO::PARAM_INT); $q->execute();
      ApiResponse::success(['id'=>$db->lastInsertId()],'Plan created');
    } elseif ($action === 'add_plan_rule' && $id) {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can add rules');
      $ruleType = $in['rule_type'] ?? 'enrollment';
      $cond = isset($in['condition']) ? json_encode($in['condition'], JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE) : null;
      $st=$db->prepare('INSERT INTO plan_rules (plan_id, rule_type, condition_json, created_at) VALUES (:p,:t,:j,NOW())');
      $st->bindValue(':p',$id,PDO::PARAM_INT); $st->bindValue(':t',$ruleType); $st->bindValue(':j',$cond); $st->execute();
      ApiResponse::success(['id'=>$db->lastInsertId()],'Rule added');
    } elseif ($action === 'delete_plan_rule' && $id) {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can delete rules');
      $st=$db->prepare('DELETE FROM plan_rules WHERE id = :id'); $st->bindValue(':id',$id,PDO::PARAM_INT); $st->execute();
      ApiResponse::success(null,'Rule deleted');
    } elseif ($action === 'add_plan_items' && $id) {
      $items = isset($in['items']) && is_array($in['items']) ? $in['items'] : [];
      $ins=$db->prepare('INSERT IGNORE INTO plan_items (plan_id, item_type, item_id, sort_order) VALUES (:p,:t,:i,:s)');
      $orderBase = (int)($in['start_order'] ?? 0); $i=0;
      foreach ($items as $it){ $i++; $ins->bindValue(':p',$id,PDO::PARAM_INT); $ins->bindValue(':t',$it['type']); $ins->bindValue(':i',(int)$it['id'],PDO::PARAM_INT); $ins->bindValue(':s',$orderBase + $i,PDO::PARAM_INT); $ins->execute(); }
      ApiResponse::success(null,'Items added');
    } elseif ($action === 'enroll_plan' && $id) {
      // get current employee id
      $emp = null; try { $q=$db->prepare('SELECT id FROM employees WHERE user_id=:u'); $q->bindValue(':u',(int)$user['id'],PDO::PARAM_INT); $q->execute(); $r=$q->fetch(); if ($r) $emp=(int)$r['id']; } catch(Throwable $e){}
      if (!$emp) ApiResponse::error('Employee record not found');
      $chk=$db->prepare('SELECT id FROM learning_plans WHERE id=:id AND company_id=:cid AND status<>"archived"'); $chk->bindValue(':id',$id,PDO::PARAM_INT); $chk->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT); $chk->execute(); if($chk->rowCount()===0) ApiResponse::notFound('Plan not found');
      $ins=$db->prepare('INSERT INTO plan_enrollments (plan_id, employee_id, status, started_at, created_at) VALUES (:p,:e,"in_progress", NOW(), NOW()) ON DUPLICATE KEY UPDATE status = VALUES(status), started_at = NOW(), updated_at = NOW()');
      $ins->bindValue(':p',$id,PDO::PARAM_INT); $ins->bindValue(':e',$emp,PDO::PARAM_INT); $ins->execute();
      ApiResponse::success(null,'Enrolled to plan');
    } elseif ($action === 'mark_lesson') {
      $planId = (int)($in['plan_id'] ?? 0); $lessonId = (int)($in['lesson_id'] ?? 0); $status = ($in['status'] ?? 'completed');
      $emp = null; try { $q=$db->prepare('SELECT id FROM employees WHERE user_id=:u'); $q->bindValue(':u',(int)$user['id'],PDO::PARAM_INT); $q->execute(); $r=$q->fetch(); if ($r) $emp=(int)$r['id']; } catch(Throwable $e){}
      if (!$emp) ApiResponse::error('Employee record not found');
      if ($planId<=0 || $lessonId<=0) ApiResponse::error('Missing plan_id or lesson_id');
      $en=$db->prepare('SELECT id FROM plan_enrollments WHERE plan_id=:p AND employee_id=:e'); $en->bindValue(':p',$planId,PDO::PARAM_INT); $en->bindValue(':e',$emp,PDO::PARAM_INT); $en->execute(); if ($en->rowCount()===0) ApiResponse::forbidden('Not enrolled in plan');
      $st=$db->prepare('INSERT INTO lesson_progress (plan_id, employee_id, lesson_id, status, progress, last_accessed, completed_at) VALUES (:p,:e,:l,:s,:pr,NOW(), CASE WHEN :s2 = "completed" THEN NOW() ELSE NULL END) ON DUPLICATE KEY UPDATE status = VALUES(status), progress = VALUES(progress), last_accessed = NOW(), completed_at = CASE WHEN VALUES(status) = "completed" THEN NOW() ELSE completed_at END');
      $st->bindValue(':p',$planId,PDO::PARAM_INT); $st->bindValue(':e',$emp,PDO::PARAM_INT); $st->bindValue(':l',$lessonId,PDO::PARAM_INT); $st->bindValue(':s',$status); $st->bindValue(':pr', (int)($in['progress'] ?? ($status==='completed'?100:0)), PDO::PARAM_INT); $st->bindValue(':s2',$status);
      $st->execute();
      ApiResponse::success(null,'Progress saved');
    } elseif ($action === 'create_assessment') {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can create assessments');
      if (empty($in['title'])) ApiResponse::error("Field 'title' is required");
      $st=$db->prepare('INSERT INTO assessments (company_id,title,description,type,delivery,linked_type,linked_id,total_points,duration_minutes,grading_scheme,status,created_by,created_at) VALUES (:cid,:t,:d,:ty,:del,:lt,:lid,:tp,:dm,:gs,:st,:uid,NOW())');
      $st->bindValue(':cid',(int)$user['company_id'],PDO::PARAM_INT);
      $st->bindValue(':t',$in['title']);
      $st->bindValue(':d',$in['description'] ?? null);
      $st->bindValue(':ty',$in['type'] ?? 'quiz');
      $st->bindValue(':del',$in['delivery'] ?? 'online');
      $st->bindValue(':lt',$in['linked_type'] ?? null);
      if (isset($in['linked_id'])) { $st->bindValue(':lid',(int)$in['linked_id'],PDO::PARAM_INT); } else { $st->bindValue(':lid', null, PDO::PARAM_NULL); }
      $st->bindValue(':tp', isset($in['total_points']) ? (int)$in['total_points'] : 0, PDO::PARAM_INT);
      $st->bindValue(':dm', isset($in['duration_minutes']) ? (int)$in['duration_minutes'] : null);
      $st->bindValue(':gs', isset($in['grading_scheme']) ? json_encode($in['grading_scheme']) : null);
      $st->bindValue(':st',$in['status'] ?? 'active');
      $st->bindValue(':uid',(int)$user['id'],PDO::PARAM_INT);
      $st->execute();
      ApiResponse::success(['id'=>$db->lastInsertId()], 'Assessment created');
    } elseif ($action === 'add_question' && $id) {
      if (!learning_is_hr($user)) ApiResponse::forbidden('Only HR/Admin can add questions');
      if (empty($in['question_text'])) ApiResponse::error("Field 'question_text' is required");
      $qtype = $in['qtype'] ?? 'mcq';
      $opt = isset($in['options']) ? json_encode($in['options']) : null;
      $corr = isset($in['correct']) ? json_encode($in['correct']) : null;
      $pts = isset($in['points']) ? (int)$in['points'] : 1;
      $order = isset($in['sort_order']) ? (int)$in['sort_order'] : 0;
      $st=$db->prepare('INSERT INTO assessment_questions (assessment_id,qtype,question_text,options_json,correct_json,points,sort_order) VALUES (:a,:qt,:tx,:op,:cr,:pt,:so)');
      $st->bindValue(':a',$id,PDO::PARAM_INT); $st->bindValue(':qt',$qtype); $st->bindValue(':tx',$in['question_text']); $st->bindValue(':op',$opt); $st->bindValue(':cr',$corr); $st->bindValue(':pt',$pts,PDO::PARAM_INT); $st->bindValue(':so',$order,PDO::PARAM_INT);
      $st->execute();
      ApiResponse::success(['id'=>$db->lastInsertId()], 'Question added');
    } elseif ($action === 'submit_assessment' && $id) {
      // employee submits attempt (basic autoscore for mcq/truefalse)
      $emp = null; try { $q=$db->prepare('SELECT id FROM employees WHERE user_id=:u'); $q->bindValue(':u',(int)$user['id'],PDO::PARAM_INT); $q->execute(); $r=$q->fetch(); if ($r) $emp=(int)$r['id']; } catch(Throwable $e){}
      if (!$emp) ApiResponse::error('Employee record not found');
      $answers = is_array($in['answers'] ?? null) ? $in['answers'] : [];
      $qa=$db->prepare('SELECT * FROM assessment_questions WHERE assessment_id=:a'); $qa->bindValue(':a',$id,PDO::PARAM_INT); $qa->execute(); $questions=$qa->fetchAll();
      $map=[]; $total=0; foreach($questions as $qr){ $map[(int)$qr['id']]=$qr; $total += (int)($qr['points'] ?? 0); }
      $ins=$db->prepare('INSERT INTO assessment_submissions (assessment_id, employee_id, status, created_at) VALUES (:a,:e,"submitted", NOW())');
      $ins->bindValue(':a',$id,PDO::PARAM_INT); $ins->bindValue(':e',$emp,PDO::PARAM_INT); $ins->execute(); $subId=(int)$db->lastInsertId();
      $ansIns=$db->prepare('INSERT INTO submission_answers (submission_id, question_id, answer_text, answer_json) VALUES (:s,:q,:t,:j)');
      $score=0;
      foreach ($answers as $ans){
        $qid = (int)($ans['question_id'] ?? 0); if ($qid<=0) continue;
        $aText = $ans['answer_text'] ?? null; $aJson = isset($ans['answer']) ? json_encode($ans['answer']) : null;
        $ansIns->bindValue(':s',$subId,PDO::PARAM_INT); $ansIns->bindValue(':q',$qid,PDO::PARAM_INT); $ansIns->bindValue(':t',$aText); $ansIns->bindValue(':j',$aJson); $ansIns->execute();
        if (isset($map[$qid])){
          $qrow=$map[$qid]; $pts=(int)($qrow['points'] ?? 0); $qt=$qrow['qtype'];
          if (in_array($qt,['mcq','truefalse'],true)){
            $correct = json_decode($qrow['correct_json'], true);
            $given = isset($ans['answer']) ? $ans['answer'] : (isset($ans['answer_text'])?$ans['answer_text']:null);
            if ($correct !== null && $given !== null && json_encode($correct) === json_encode($given)) { $score += $pts; }
          }
        }
      }
      // update submission score if any auto-points
      $up=$db->prepare('UPDATE assessment_submissions SET score = :sc, updated_at = NOW() WHERE id = :id');
      $up->bindValue(':sc',$score,PDO::PARAM_INT); $up->bindValue(':id',$subId,PDO::PARAM_INT); $up->execute();
      ApiResponse::success(['submission_id'=>$subId,'score'=>$score,'total_points'=>$total],'Submission received');
    } elseif ($action === 'grade_submission' && $id) {
      $role = strtolower($user['role_slug'] ?? '');
      if (!in_array($role, ['super_admin','admin','hr_head'], true)) ApiResponse::forbidden('Only Admin/HR Head can grade');
      $score = isset($in['score']) ? (int)$in['score'] : null; $grade = $in['grade'] ?? null; $feedback = $in['feedback'] ?? null;
      $st=$db->prepare('UPDATE assessment_submissions SET status = "graded", score = :sc, grade = :gr, feedback = :fb, updated_at = NOW() WHERE id = :id');
      if ($score===null){ $st->bindValue(':sc', null, PDO::PARAM_NULL); } else { $st->bindValue(':sc',$score,PDO::PARAM_INT); }
      $st->bindValue(':gr',$grade); $st->bindValue(':fb',$feedback); $st->bindValue(':id',$id,PDO::PARAM_INT); $st->execute();
      ApiResponse::success(null,'Submission graded');
    } else {
      ApiResponse::error('Unknown action', 400);
    }
    break;

  case 'PUT':
  case 'DELETE':
    ApiResponse::error('Not implemented yet', 400);
    break;

  default:
    ApiResponse::error('Method not allowed', 405);
}
