<?php
/**
 * Grades Management API
 * - Manage grades (name, code, base_salary)
 * - Manage grade default allowances and deductions
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ApiResponse::success(null, 'OK');
}

function bulkMapEmployeesToGrades(PDO $db, $user) {
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $mappings = isset($in['mappings']) && is_array($in['mappings']) ? $in['mappings'] : [];
    if (empty($mappings)) ApiResponse::error('mappings array is required');
    // Normalize and dedupe by employee_id
    $clean = [];
    foreach ($mappings as $m) {
        $eid = isset($m['employee_id']) && ctype_digit(strval($m['employee_id'])) ? (int)$m['employee_id'] : 0;
        $gid = (isset($m['grade_id']) && $m['grade_id'] !== '' && $m['grade_id'] !== null && ctype_digit(strval($m['grade_id']))) ? (int)$m['grade_id'] : null; // allow null to clear
        if ($eid <= 0) continue;
        $clean[$eid] = $gid; // last write wins
    }
    if (empty($clean)) ApiResponse::error('No valid mappings');

    // Validate grades belong to company (exclude NULL)
    $gidSet = array_values(array_unique(array_filter(array_values($clean), fn($v)=>$v !== null)));
    $validGrades = [];
    if (!empty($gidSet)) {
        $place = implode(',', array_fill(0, count($gidSet), '?'));
        $st = $db->prepare("SELECT id FROM grades WHERE company_id = ? AND id IN ($place)");
        $st->bindValue(1, $user['company_id'], PDO::PARAM_INT);
        foreach ($gidSet as $i=>$v) { $st->bindValue($i+2, (int)$v, PDO::PARAM_INT); }
        $st->execute();
        foreach ($st->fetchAll() as $r) { $validGrades[(int)$r['id']] = true; }
    }

    $updated = 0; $skipped = [];
    $db->beginTransaction();
    try {
        // Scope employees to company
        $empIds = array_keys($clean);
        $placeE = implode(',', array_fill(0, count($empIds), '?'));
        $se = $db->prepare("SELECT id FROM employees WHERE company_id = ? AND id IN ($placeE)");
        $se->bindValue(1, $user['company_id'], PDO::PARAM_INT);
        foreach ($empIds as $i=>$v) { $se->bindValue($i+2, (int)$v, PDO::PARAM_INT); }
        $se->execute();
        $validEmp = []; foreach ($se->fetchAll() as $r) { $validEmp[(int)$r['id']] = true; }

        $u = $db->prepare("UPDATE employees SET grade_id = :gid, updated_at = NOW() WHERE id = :id AND company_id = :cid");
        foreach ($clean as $eid=>$gid) {
            if (empty($validEmp[$eid])) { $skipped[] = ['employee_id'=>$eid, 'reason'=>'not_in_company']; continue; }
            if ($gid !== null && empty($validGrades[$gid])) { $skipped[] = ['employee_id'=>$eid, 'reason'=>'invalid_grade']; continue; }
            $u->bindValue(':gid', $gid, $gid === null ? PDO::PARAM_NULL : PDO::PARAM_INT);
            $u->bindValue(':id', $eid, PDO::PARAM_INT);
            $u->bindValue(':cid', $user['company_id'], PDO::PARAM_INT);
            $u->execute();
            $updated += ($u->rowCount() > 0) ? 1 : 0;
        }
        $db->commit();
    } catch (Throwable $e) {
        $db->rollBack();
        ApiResponse::error('Bulk map failed: ' . $e->getMessage(), 500);
    }
    ApiResponse::success(['updated'=>$updated, 'skipped'=>$skipped], 'Bulk mapping completed');
}

function saveGradeDeduction(PDO $db, $user) {
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($in['id']) ? (int)$in['id'] : 0;
    $gradeId = (int)($in['grade_id'] ?? 0);
    $typeId = (int)($in['deduction_type_id'] ?? 0);
    $calcMode = in_array(($in['calc_mode'] ?? 'fixed'), ['fixed','percent_basic','percent_gross'], true) ? $in['calc_mode'] : 'fixed';
    $rate = isset($in['rate']) && $in['rate'] !== '' ? (float)$in['rate'] : null;
    $amount = (float)($in['amount'] ?? 0);
    if (!$gradeId || !$typeId) ApiResponse::error('grade_id and deduction_type_id are required');
    $g = $db->prepare("SELECT id FROM grades WHERE id = :id AND company_id = :cid");
    $g->execute([':id'=>$gradeId, ':cid'=>$user['company_id']]);
    if ($g->rowCount() === 0) ApiResponse::error('Invalid grade');

    if ($id > 0) {
        $q = $db->prepare("UPDATE grade_deductions SET deduction_type_id = :t, amount = :a, calc_mode = :m, rate = :r WHERE id = :id AND company_id = :cid");
        $q->execute([':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate, ':id'=>$id, ':cid'=>$user['company_id']]);
    } else {
        $q = $db->prepare("INSERT INTO grade_deductions (company_id, grade_id, deduction_type_id, amount, calc_mode, rate, created_at) VALUES (:cid, :gid, :t, :a, :m, :r, NOW())");
        $q->execute([':cid'=>$user['company_id'], ':gid'=>$gradeId, ':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate]);
        $id = (int)$db->lastInsertId();
    }
    ApiResponse::success(['id'=>$id], 'Saved');
}

requireAuth();
$user = getCurrentUser();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$database = new Database();
$db = $database->getConnection();

// Ensure grade component tables have calc_mode/rate columns (idempotent)
function ensureGradeComponentCalcSchema(PDO $db){
    try { $db->exec("ALTER TABLE grade_allowances ADD COLUMN calc_mode ENUM('fixed','percent_basic','percent_gross') NOT NULL DEFAULT 'fixed' AFTER amount"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE grade_allowances ADD COLUMN rate DECIMAL(5,2) NULL AFTER calc_mode"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE grade_deductions ADD COLUMN calc_mode ENUM('fixed','percent_basic','percent_gross') NOT NULL DEFAULT 'fixed' AFTER amount"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE grade_deductions ADD COLUMN rate DECIMAL(5,2) NULL AFTER calc_mode"); } catch (Throwable $e) {}
}
ensureGradeComponentCalcSchema($db);

// Ensure employees table has grade_id (idempotent)
function ensureEmployeeGradeSchema(PDO $db) {
    try { $db->exec("ALTER TABLE employees ADD COLUMN grade_id INT NULL AFTER position_id"); } catch (Throwable $e) {}
}
ensureEmployeeGradeSchema($db);

// Permissions: editing grades allowed for admin, hr_head, hr_officer; approvals not applicable here
function ensureCanEditGrades($user) {
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
}

switch ($method) {
    case 'GET':
        if ($action === 'detail' && $id) {
            getGradeDetail($db, $user, $id);
        } else {
            listGrades($db, $user);
        }
        break;
    case 'POST':
        ensureCanEditGrades($user);
        switch ($action) {
            case 'save':
                saveGrade($db, $user);
                break;
            case 'delete':
                deleteGrade($db, $user, $id);
                break;
            case 'save_allowance':
                saveGradeAllowance($db, $user);
                break;
            case 'delete_allowance':
                deleteGradeAllowance($db, $user, $id);
                break;
            case 'save_deduction':
                saveGradeDeduction($db, $user);
                break;
            case 'delete_deduction':
                deleteGradeDeduction($db, $user, $id);
                break;
            case 'bulk_map':
                bulkMapEmployeesToGrades($db, $user);
                break;
            default:
                ApiResponse::error('Invalid action');
        }
        break;
    default:
        ApiResponse::error('Method not allowed', 405);
}

function listGrades(PDO $db, $user) {
    $q = $db->prepare("SELECT id, name, code, base_salary, created_at FROM grades WHERE company_id = :cid ORDER BY name");
    $q->execute([':cid'=>$user['company_id']]);
    ApiResponse::success($q->fetchAll());
}

function getGradeDetail(PDO $db, $user, $id) {
    // Basic grade info
    $g = $db->prepare("SELECT id, name, code, base_salary FROM grades WHERE id = :id AND company_id = :cid");
    $g->execute([':id'=>$id, ':cid'=>$user['company_id']]);
    if ($g->rowCount() === 0) ApiResponse::notFound('Grade not found');
    $grade = $g->fetch();

    // Allowances
    $a = $db->prepare("SELECT ga.id, ga.allowance_type_id, at.name, at.code, ga.amount, ga.calc_mode, ga.rate
                        FROM grade_allowances ga
                        JOIN allowance_types at ON ga.allowance_type_id = at.id
                        WHERE ga.grade_id = :gid AND ga.company_id = :cid
                        ORDER BY at.name");
    $a->execute([':gid'=>$id, ':cid'=>$user['company_id']]);
    $grade['allowances'] = $a->fetchAll();

    // Deductions
    $d = $db->prepare("SELECT gd.id, gd.deduction_type_id, dt.name, dt.code, gd.amount, gd.calc_mode, gd.rate
                        FROM grade_deductions gd
                        JOIN deduction_types dt ON gd.deduction_type_id = dt.id
                        WHERE gd.grade_id = :gid AND gd.company_id = :cid
                        ORDER BY dt.name");
    $d->execute([':gid'=>$id, ':cid'=>$user['company_id']]);
    $grade['deductions'] = $d->fetchAll();

    ApiResponse::success($grade);
}

function saveGrade(PDO $db, $user) {
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($in['id']) ? (int)$in['id'] : 0;
    $name = trim($in['name'] ?? '');
    $code = trim($in['code'] ?? '');
    $baseSalary = isset($in['base_salary']) && $in['base_salary'] !== '' ? (float)$in['base_salary'] : null;
    if ($name === '') ApiResponse::error('Name is required');

    if ($id > 0) {
        $q = $db->prepare("UPDATE grades SET name = :n, code = :c, base_salary = :b WHERE id = :id AND company_id = :cid");
        $q->execute([':n'=>$name, ':c'=>$code, ':b'=>$baseSalary, ':id'=>$id, ':cid'=>$user['company_id']]);
    } else {
        $q = $db->prepare("INSERT INTO grades (company_id, name, code, base_salary, created_by, created_at) VALUES (:cid, :n, :c, :b, :uid, NOW())");
        $q->execute([':cid'=>$user['company_id'], ':n'=>$name, ':c'=>$code, ':b'=>$baseSalary, ':uid'=>$user['id']]);
        $id = (int)$db->lastInsertId();
    }
    ApiResponse::success(['id'=>$id], 'Saved');
}

function deleteGrade(PDO $db, $user, $id) {
    if (!$id) ApiResponse::error('ID required');
    // Prevent delete if employees reference this grade
    $c = $db->prepare("SELECT COUNT(*) FROM employees WHERE grade_id = :id");
    $c->execute([':id'=>$id]);
    if ((int)$c->fetchColumn() > 0) ApiResponse::error('Cannot delete: grade is assigned to employees');

    // Delete child rows then grade
    $db->beginTransaction();
    try {
        $db->prepare("DELETE FROM grade_allowances WHERE grade_id = :id AND company_id = :cid")->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        $db->prepare("DELETE FROM grade_deductions WHERE grade_id = :id AND company_id = :cid")->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        $db->prepare("DELETE FROM grades WHERE id = :id AND company_id = :cid")->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        $db->commit();
    } catch (Throwable $e) {
        $db->rollBack();
        ApiResponse::error('Failed to delete grade: ' . $e->getMessage(), 500);
    }
    ApiResponse::success(null, 'Deleted');
}

function saveGradeAllowance(PDO $db, $user) {
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($in['id']) ? (int)$in['id'] : 0;
    $gradeId = (int)($in['grade_id'] ?? 0);
    $typeId = (int)($in['allowance_type_id'] ?? 0);
    $calcMode = in_array(($in['calc_mode'] ?? 'fixed'), ['fixed','percent_basic','percent_gross'], true) ? $in['calc_mode'] : 'fixed';
    $rate = isset($in['rate']) && $in['rate'] !== '' ? (float)$in['rate'] : null;
    $amount = (float)($in['amount'] ?? 0);
    if (!$gradeId || !$typeId) ApiResponse::error('grade_id and allowance_type_id are required');
    $g = $db->prepare("SELECT id FROM grades WHERE id = :id AND company_id = :cid");
    $g->execute([':id'=>$gradeId, ':cid'=>$user['company_id']]);
    if ($g->rowCount() === 0) ApiResponse::error('Invalid grade');

    if ($id > 0) {
        $q = $db->prepare("UPDATE grade_allowances SET allowance_type_id = :t, amount = :a, calc_mode = :m, rate = :r WHERE id = :id AND company_id = :cid");
        $q->execute([':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate, ':id'=>$id, ':cid'=>$user['company_id']]);
    } else {
        $q = $db->prepare("INSERT INTO grade_allowances (company_id, grade_id, allowance_type_id, amount, calc_mode, rate, created_at) VALUES (:cid, :gid, :t, :a, :m, :r, NOW())");
        $q->execute([':cid'=>$user['company_id'], ':gid'=>$gradeId, ':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate]);
    }
    ApiResponse::success(['id'=>$id], 'Saved');
}

function deleteGradeAllowance(PDO $db, $user, $id) {
    if (!$id) ApiResponse::error('ID required');
    $q = $db->prepare("DELETE FROM grade_allowances WHERE id = :id AND company_id = :cid");
    $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
    ApiResponse::success(null, 'Deleted');
}

function deleteGradeDeduction(PDO $db, $user, $id) {
    if (!$id) ApiResponse::error('ID required');
    $q = $db->prepare("DELETE FROM grade_deductions WHERE id = :id AND company_id = :cid");
    $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
    ApiResponse::success(null, 'Deleted');
}
