<?php
/**
 * Expenses/Imprest/Per-diem/Overtime Requests API with multi-stage approvals
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$database = new Database();
$db = $database->getConnection();

// ===== Schema helpers =====
function ensureExpensesTables(PDO $db){
  try{
    $db->exec("CREATE TABLE IF NOT EXISTS expenses (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      employee_id INT NOT NULL,
      type ENUM('imprest','reimbursement','overtime','per_diem','other') NOT NULL,
      request_date DATE NOT NULL,
      amount DECIMAL(12,2) NULL,
      hours DECIMAL(8,2) NULL,
      currency VARCHAR(8) NULL,
      description TEXT NULL,
      status ENUM('pending','approved','rejected','cancelled') NOT NULL DEFAULT 'pending',
      approved_by INT NULL,
      approved_at TIMESTAMP NULL,
      rejection_reason TEXT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
      KEY idx_company (company_id),
      KEY idx_employee (employee_id),
      KEY idx_status (status),
      KEY idx_type (type),
      KEY idx_date (request_date)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  }catch(Throwable $e){ /* ignore */ }
  try{
    $db->exec("CREATE TABLE IF NOT EXISTS expense_files (
      id INT AUTO_INCREMENT PRIMARY KEY,
      expense_id INT NOT NULL,
      file_path VARCHAR(1024) NOT NULL,
      uploaded_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      KEY idx_exp (expense_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  }catch(Throwable $e){ /* ignore */ }
  try{
    $db->exec("CREATE TABLE IF NOT EXISTS expense_approval_logs (
      id INT AUTO_INCREMENT PRIMARY KEY,
      expense_id INT NOT NULL,
      stage TINYINT NOT NULL,
      action ENUM('approved','rejected') NOT NULL,
      approver_user_id INT NOT NULL,
      notes TEXT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      KEY idx_exp_stage (expense_id, stage)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
  }catch(Throwable $e){ /* ignore */ }
  // Notifications table may be used
  if (!function_exists('ensureNotificationsTable')){
    // fallback definition
    try{
      $db->exec("CREATE TABLE IF NOT EXISTS notifications (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        type VARCHAR(64) NOT NULL,
        title VARCHAR(255) NOT NULL,
        content TEXT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
        KEY idx_user_isread (user_id, is_read),
        KEY idx_created_at (created_at)
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
    }catch(Throwable $e){ /* ignore */ }
  }
}

function saveUploadedFile($subdir){
  if (!isset($_FILES['file']) || !is_array($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) return null;
  $fn = $_FILES['file']['name'] ?? 'file';
  $ext = strtolower(pathinfo($fn, PATHINFO_EXTENSION));
  $allowed = ['pdf','doc','docx','png','jpg','jpeg','txt','xls','xlsx'];
  if (!in_array($ext, $allowed, true)) return null;
  if (($_FILES['file']['size'] ?? 0) > 15*1024*1024) return null; // 15MB
  $base = dirname(__DIR__);
  $dir = rtrim($base, DIRECTORY_SEPARATOR).DIRECTORY_SEPARATOR.'uploads'.DIRECTORY_SEPARATOR.$subdir;
  if (!is_dir($dir)) @mkdir($dir, 0775, true);
  $safe = preg_replace('/[^a-zA-Z0-9._-]/','_', basename($fn));
  $name = date('Ymd_His').'_'.$safe;
  $path = $dir.DIRECTORY_SEPARATOR.$name;
  if (!@move_uploaded_file($_FILES['file']['tmp_name'], $path)) return null;
  return 'uploads/'.trim($subdir,'/').'/'.$name;
}

// ===== Workflow helpers =====
function getExpenseContext(PDO $db, int $expenseId): ?array {
  ensureExpensesTables($db);
  $q = "SELECT x.id, x.employee_id, e.company_id, e.user_id AS employee_user_id, e.manager_id,
               mgr.user_id AS manager_user_id,
               urole.slug AS submitter_role
        FROM expenses x
        JOIN employees e ON e.id = x.employee_id
        LEFT JOIN employees mgr ON mgr.id = e.manager_id
        LEFT JOIN users u ON u.id = e.user_id
        LEFT JOIN user_roles ur ON ur.user_id = u.id
        LEFT JOIN roles urole ON urole.id = ur.role_id
        WHERE x.id = :id LIMIT 1";
  $st = $db->prepare($q);
  $st->bindValue(':id', $expenseId, PDO::PARAM_INT);
  $st->execute();
  if ($st->rowCount() === 0) return null;
  return $st->fetch();
}

function getExpenseApprovedStageCount(PDO $db, int $expenseId): int {
  $st = $db->prepare("SELECT COUNT(*) FROM expense_approval_logs WHERE expense_id = :id AND action = 'approved'");
  $st->bindValue(':id', $expenseId, PDO::PARAM_INT);
  $st->execute();
  return (int)$st->fetchColumn();
}

function getExpenseNextStageRole(PDO $db, int $expenseId): ?string {
  $ctx = getExpenseContext($db, $expenseId);
  if (!$ctx) return null;
  $stages = [];
  // If raised by HR Officer, skip manager and go to HR Head
  if (($ctx['submitter_role'] ?? '') === 'hr_officer') {
    $stages[] = 'hr_head';
  } else {
    if (!empty($ctx['manager_user_id'])) $stages[] = 'manager';
    $stages[] = 'hr_head';
  }
  $approved = getExpenseApprovedStageCount($db, $expenseId);
  if ($approved >= count($stages)) return null; // done
  return $stages[$approved];
}

function canUserApproveExpenseStage(array $user, string $stageRole, ?int $managerUserId): bool {
  $role = $user['role_slug'] ?? '';
  $uid = (int)($user['id'] ?? 0);
  switch ($stageRole) {
    case 'manager':
      return ($role === 'manager') && $managerUserId !== null && $uid === (int)$managerUserId;
    case 'hr_head':
      return in_array($role, ['hr_head','admin','super_admin'], true);
    default:
      return false;
  }
}

function notifyExpenseNextStage(PDO $db, int $expenseId, string $nextStageRole): void {
  // Ensure notifications table exists
  ensureExpensesTables($db);
  $ctx = getExpenseContext($db, $expenseId);
  if (!$ctx) return;
  // Load basic info
  $st = $db->prepare("SELECT x.id, x.type, x.amount, x.request_date,
                             CONCAT(e.first_name,' ',e.last_name) AS employee_name
                      FROM expenses x JOIN employees e ON e.id = x.employee_id WHERE x.id = :id");
  $st->execute([':id'=>$expenseId]);
  $info = $st->fetch();
  if (!$info) return;

  $title = 'Expense Approval Required';
  $content = sprintf('%s submitted %s on %s.', $info['employee_name'] ?? 'Employee', (string)($info['type'] ?? 'expense'), (string)($info['request_date'] ?? ''));

  $recipients = [];
  if ($nextStageRole === 'hr_head') {
    // all HR Heads in company
    $q = $db->prepare("SELECT u.id AS user_id FROM users u
                       JOIN user_roles ur ON ur.user_id = u.id
                       JOIN roles r ON r.id = ur.role_id
                       JOIN employees e ON e.user_id = u.id
                       WHERE e.company_id = :cid AND r.slug = 'hr_head'");
    $q->execute([':cid'=>$ctx['company_id']]);
    foreach ($q->fetchAll() as $r) { $recipients[] = (int)$r['user_id']; }
  } elseif ($nextStageRole === 'manager') {
    if (!empty($ctx['manager_user_id'])) $recipients[] = (int)$ctx['manager_user_id'];
  }
  if (!$recipients) return;
  foreach ($recipients as $uid) {
    $ins = $db->prepare("INSERT INTO notifications (user_id, type, title, content, created_at) VALUES (:uid, 'expense_needs_approval', :t, :c, NOW())");
    $ins->execute([':uid'=>$uid, ':t'=>$title, ':c'=>$content]);
  }
}

// ===== Routing =====
switch ($method) {
  case 'GET':
    if ($action === 'files') listFiles($db);
    else listExpenses($db);
    break;
  case 'POST':
    if ($action === 'create') createExpense($db);
    elseif ($action === 'upload_file') uploadExpenseFile($db);
    elseif ($action === 'approve') approveExpense($db, $id);
    elseif ($action === 'reject') rejectExpense($db, $id);
    else ApiResponse::error('Invalid action');
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

// ===== Handlers =====
function listExpenses(PDO $db){
  ensureExpensesTables($db);
  $u = getCurrentUser();
  $role = $u['role_slug'] ?? '';
  $cid = (int)$u['company_id'];
  $status = isset($_GET['status']) ? trim((string)$_GET['status']) : '';
  $type = isset($_GET['type']) ? trim((string)$_GET['type']) : '';
  $forApproval = isset($_GET['for_approval']) ? (int)$_GET['for_approval'] : 0;
  $where = ['e.company_id = :cid'];
  $params = [':cid'=>$cid];

  if ($role === 'employee') {
    $meEmp = $db->prepare('SELECT id FROM employees WHERE user_id = :uid LIMIT 1');
    $meEmp->execute([':uid'=>$u['id']]);
    $empId = (int)($meEmp->fetchColumn() ?: 0);
    if ($empId<=0) { ApiResponse::success([]); return; }
    $where[] = 'e.employee_id = :me'; $params[':me'] = $empId;
  } elseif ($role === 'manager') {
    // show own & direct reports
    $meEmp = $db->prepare('SELECT id FROM employees WHERE user_id = :uid LIMIT 1');
    $meEmp->execute([':uid'=>$u['id']]);
    $managerEmpId = (int)($meEmp->fetchColumn() ?: 0);
    $where[] = '(emp.manager_id = :mgr OR e.employee_id = :mgr)';
    $params[':mgr'] = $managerEmpId;
  } else {
    // HR/Admin see all
  }

  if ($status !== '') { $where[] = 'e.status = :st'; $params[':st'] = $status; }
  if ($type !== '') { $where[] = 'e.type = :tp'; $params[':tp'] = $type; }

  $sql = "SELECT e.*, CONCAT(emp.first_name,' ',emp.last_name) AS employee_name, d.name AS department_name
          FROM expenses e
          JOIN employees emp ON emp.id = e.employee_id
          LEFT JOIN departments d ON d.id = emp.department_id
          WHERE ".implode(' AND ',$where)." ORDER BY e.created_at DESC LIMIT 500";
  $st = $db->prepare($sql);
  foreach($params as $k=>$v) $st->bindValue($k, $v);
  $st->execute();
  $rows = $st->fetchAll() ?: [];

  // Attach files grouped
  if ($rows){
    $ids = array_map(function($r){ return (int)$r['id']; }, $rows);
    $place = implode(',', array_fill(0, count($ids), '?'));
    $fs = $db->prepare("SELECT expense_id, file_path FROM expense_files WHERE expense_id IN ($place)");
    foreach ($ids as $i=>$val) $fs->bindValue($i+1, $val, PDO::PARAM_INT);
    $fs->execute();
    $files = $fs->fetchAll() ?: [];
    $map = [];
    foreach($files as $f){ $eid=(int)$f['expense_id']; if(!isset($map[$eid])) $map[$eid]=[]; $map[$eid][] = $f['file_path']; }
    foreach($rows as &$r){ $r['attachments'] = $map[(int)$r['id']] ?? []; }
  }

  // Approvals view: only show items awaiting current user's stage when status=pending and for_approval flag set
  if ($forApproval && $status === 'pending'){
    $filtered = [];
    foreach ($rows as $r){
      $stageRole = getExpenseNextStageRole($db, (int)$r['id']);
      if ($stageRole === null) continue;
      $ctx = getExpenseContext($db, (int)$r['id']);
      $mgrUserId = $ctx['manager_user_id'] ?? null;
      if (canUserApproveExpenseStage($u, $stageRole, $mgrUserId)) $filtered[] = $r;
    }
    $rows = $filtered;
  }

  ApiResponse::success($rows);
}

function listFiles(PDO $db){
  $eid = isset($_GET['expense_id']) ? (int)$_GET['expense_id'] : 0; if ($eid<=0) ApiResponse::error('Invalid expense');
  $st = $db->prepare('SELECT id, file_path, uploaded_at FROM expense_files WHERE expense_id = :id ORDER BY id');
  $st->execute([':id'=>$eid]);
  ApiResponse::success($st->fetchAll());
}

function createExpense(PDO $db){
  ensureExpensesTables($db);
  $u = getCurrentUser();
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $type = in_array(($in['type'] ?? ''), ['imprest','reimbursement','overtime','per_diem','other'], true) ? $in['type'] : 'other';
  $requestDate = $in['request_date'] ?? date('Y-m-d');
  $desc = $in['description'] ?? null;
  $amount = isset($in['amount']) ? (float)$in['amount'] : null;
  $hours = isset($in['hours']) ? (float)$in['hours'] : null;
  $currency = $in['currency'] ?? null;

  // Resolve employee
  $me = $db->prepare('SELECT id FROM employees WHERE user_id = :uid AND company_id = :cid');
  $me->execute([':uid'=>$u['id'], ':cid'=>$u['company_id']]);
  $empId = (int)$me->fetchColumn();
  if ($empId<=0) ApiResponse::error('Employee record not found');

  try{
    $st = $db->prepare("INSERT INTO expenses (company_id, employee_id, type, request_date, amount, hours, currency, description, status, created_at) VALUES (:cid, :eid, :tp, :dt, :amt, :hrs, :cur, :ds, 'pending', NOW())");
    $st->execute([':cid'=>$u['company_id'], ':eid'=>$empId, ':tp'=>$type, ':dt'=>$requestDate, ':amt'=>$amount, ':hrs'=>$hours, ':cur'=>$currency, ':ds'=>$desc]);
    $id = (int)$db->lastInsertId();
    // Notify first-stage approvers
    $firstStage = getExpenseNextStageRole($db, $id);
    if ($firstStage !== null) notifyExpenseNextStage($db, $id, $firstStage);
    ApiResponse::success(['id'=>$id], 'Expense submitted');
  }catch(Throwable $e){ ApiResponse::error('Create failed: '.$e->getMessage(), 500); }
}

function uploadExpenseFile(PDO $db){
  ensureExpensesTables($db);
  $u = getCurrentUser();
  $eid = isset($_POST['expense_id']) ? (int)$_POST['expense_id'] : 0; if ($eid<=0) ApiResponse::error('Invalid expense');
  // Ensure ownership company
  $chk = $db->prepare('SELECT e.company_id, e.employee_id, emp.user_id AS owner_user_id FROM expenses e JOIN employees emp ON emp.id = e.employee_id WHERE e.id = :id');
  $chk->execute([':id'=>$eid]);
  $row = $chk->fetch(); if(!$row || (int)$row['company_id'] !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
  // Allow owner or approver roles to upload
  $role = $u['role_slug'] ?? '';
  if ((int)$row['owner_user_id'] !== (int)$u['id'] && !in_array($role, ['manager','hr_head','admin','super_admin'], true)) ApiResponse::forbidden('Not allowed');

  $rel = saveUploadedFile('expenses'); if (!$rel) ApiResponse::error('Upload failed');
  try{
    $ins = $db->prepare('INSERT INTO expense_files (expense_id, file_path, uploaded_at) VALUES (:id, :p, NOW())');
    $ins->execute([':id'=>$eid, ':p'=>$rel]);
    ApiResponse::success(['path'=>$rel], 'File uploaded');
  }catch(Throwable $e){ ApiResponse::error('Failed to save file: '.$e->getMessage(), 500); }
}

function approveExpense(PDO $db, int $id){
  ensureExpensesTables($db);
  $u = getCurrentUser(); if ($id<=0) ApiResponse::error('Invalid expense');
  $input = json_decode(file_get_contents('php://input'), true) ?: [];

  // Verify exists and pending in same company
  $check = $db->prepare('SELECT e.*, emp.user_id AS employee_user_id, emp.manager_id, emp.company_id 
                         FROM expenses e 
                         JOIN employees emp ON emp.id = e.employee_id 
                         WHERE e.id = :id AND e.status = "pending" AND emp.company_id = :cid');
  $check->execute([':id'=>$id, ':cid'=>$u['company_id']]); if ($check->rowCount()===0) ApiResponse::error('Expense not found or already processed');

  $stageRole = getExpenseNextStageRole($db, $id);
  if ($stageRole === null) ApiResponse::error('Already finalized');
  $ctx = getExpenseContext($db, $id); $mgrUserId = $ctx['manager_user_id'] ?? null;
  if (!canUserApproveExpenseStage($u, $stageRole, $mgrUserId)) ApiResponse::forbidden('Not authorized for this stage');

  // Write stage approval log
  $stageIndex = getExpenseApprovedStageCount($db, $id) + 1;
  $log = $db->prepare("INSERT INTO expense_approval_logs (expense_id, stage, action, approver_user_id, notes, created_at) VALUES (:id,:st,'approved',:uid,:notes,NOW())");
  $log->execute([':id'=>$id, ':st'=>$stageIndex, ':uid'=>$u['id'], ':notes'=>$input['notes'] ?? null]);

  // Move to next or finalize
  $nextStage = getExpenseNextStageRole($db, $id);
  if ($nextStage === null){
    $up = $db->prepare("UPDATE expenses SET status='approved', approved_by=:uid, approved_at=NOW(), updated_at=NOW() WHERE id = :id");
    $up->execute([':uid'=>$u['id'], ':id'=>$id]);
    // Notify employee (optional)
    ApiResponse::success(null, 'Expense approved');
  }else{
    notifyExpenseNextStage($db, $id, $nextStage);
    ApiResponse::success(null, 'Stage approved');
  }
}

function rejectExpense(PDO $db, int $id){
  ensureExpensesTables($db);
  $u = getCurrentUser(); if ($id<=0) ApiResponse::error('Invalid expense');
  $input = json_decode(file_get_contents('php://input'), true) ?: [];
  if (empty($input['rejection_reason'])) ApiResponse::error('Rejection reason required');

  $check = $db->prepare('SELECT e.*, emp.user_id AS employee_user_id, emp.company_id 
                         FROM expenses e 
                         JOIN employees emp ON emp.id = e.employee_id 
                         WHERE e.id = :id AND e.status = "pending" AND emp.company_id = :cid');
  $check->execute([':id'=>$id, ':cid'=>$u['company_id']]); if ($check->rowCount()===0) ApiResponse::error('Expense not found or already processed');

  $stageRole = getExpenseNextStageRole($db, $id);
  if ($stageRole === null) ApiResponse::error('Already finalized');
  $ctx = getExpenseContext($db, $id); $mgrUserId = $ctx['manager_user_id'] ?? null;
  if (!canUserApproveExpenseStage($u, $stageRole, $mgrUserId)) ApiResponse::forbidden('Not authorized for this stage');

  // Log rejection
  $stageIndex = getExpenseApprovedStageCount($db, $id) + 1;
  $log = $db->prepare("INSERT INTO expense_approval_logs (expense_id, stage, action, approver_user_id, notes, created_at) VALUES (:id,:st,'rejected',:uid,:notes,NOW())");
  $log->execute([':id'=>$id, ':st'=>$stageIndex, ':uid'=>$u['id'], ':notes'=>$input['rejection_reason']]);

  // Update request
  $up = $db->prepare("UPDATE expenses SET status='rejected', approved_by=:uid, approved_at=NOW(), rejection_reason=:rsn, updated_at=NOW() WHERE id = :id");
  $up->execute([':uid'=>$u['id'], ':rsn'=>$input['rejection_reason'], ':id'=>$id]);

  ApiResponse::success(null, 'Expense rejected');
}
