<?php
/**
 * Employee Management API Endpoints
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

// Require authentication for all endpoints in this file
requireAuth();

// Return existing column names for a DB table (cached per request)
function getTableColumns(PDO $db, $table) {
    static $cache = [];
    if (isset($cache[$table])) return $cache[$table];
    try {
        $q = $db->prepare("SELECT COLUMN_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t");
        $q->bindValue(':t', $table);
        $q->execute();
        $cols = [];
        foreach ($q->fetchAll() as $r) { $cols[] = $r['COLUMN_NAME']; }
        return $cache[$table] = $cols;
    } catch (Throwable $e) {
        return $cache[$table] = [];
    }
}
// Ensure employee allowances/deductions have calc fields (idempotent)
function ensureEmployeeComponentCalcSchema(PDO $db) {
    try { $db->exec("ALTER TABLE employee_allowances ADD COLUMN calc_mode ENUM('fixed','percent_basic','percent_gross') NOT NULL DEFAULT 'fixed' AFTER amount"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employee_allowances ADD COLUMN rate DECIMAL(5,2) NULL AFTER calc_mode"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employee_deductions ADD COLUMN calc_mode ENUM('fixed','percent_basic','percent_gross') NOT NULL DEFAULT 'fixed' AFTER amount"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employee_deductions ADD COLUMN rate DECIMAL(5,2) NULL AFTER calc_mode"); } catch (Throwable $e) {}
}

// Ensure employee payment default columns (bank/momo) exist (idempotent)
function ensureEmployeePaymentSchema(PDO $db) {
    try { $db->exec("ALTER TABLE employees ADD COLUMN bank_code VARCHAR(32) NULL AFTER phone"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employees ADD COLUMN bank_account VARCHAR(64) NULL AFTER bank_code"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employees ADD COLUMN momo_provider VARCHAR(64) NULL AFTER bank_account"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employees ADD COLUMN momo_number VARCHAR(64) NULL AFTER momo_provider"); } catch (Throwable $e) {}
}

// Ensure employees table has a grade_id column to map to grades
function ensureEmployeeGradeSchema(PDO $db) {
    try { $db->exec("ALTER TABLE employees ADD COLUMN grade_id INT NULL AFTER position_id"); } catch (Throwable $e) {}
}

function getEmployeeComponents(PDO $db, $employeeId) {
    $user = getCurrentUser();
    if (!$employeeId) ApiResponse::error('Employee ID required');
    ensureEmployeeComponentCalcSchema($db);
    try {
        // Permissions: HR roles can view any employee; employees can only view their own
        $isHr = in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'], true);
        if (!$isHr) {
            $self = $db->prepare("SELECT id FROM employees WHERE id = :id AND user_id = :uid");
            $self->execute([':id'=>(int)$employeeId, ':uid'=>$user['id'] ?? 0]);
            if ($self->rowCount() === 0) ApiResponse::forbidden('Insufficient permissions');
        }
        $empCheck = $db->prepare("SELECT id FROM employees WHERE id = :id AND company_id = :cid");
        $empCheck->execute([':id'=>(int)$employeeId, ':cid'=>$user['company_id']]);
        if ($empCheck->rowCount() === 0) ApiResponse::forbidden('Access denied');
        $als = $db->prepare("SELECT ea.id, ea.allowance_type_id, at.name, at.code, ea.amount, ea.calc_mode, ea.rate, ea.status, ea.start_date, ea.end_date
                              FROM employee_allowances ea
                              LEFT JOIN allowance_types at ON at.id = ea.allowance_type_id
                              WHERE ea.employee_id = :eid ORDER BY at.name");
        $als->execute([':eid'=>(int)$employeeId]);
        $dds = $db->prepare("SELECT ed.id, ed.deduction_type_id, dt.name, dt.code, ed.amount, ed.calc_mode, ed.rate, ed.status, ed.start_date, ed.end_date
                              FROM employee_deductions ed
                              LEFT JOIN deduction_types dt ON dt.id = ed.deduction_type_id
                              WHERE ed.employee_id = :eid ORDER BY dt.name");
        $dds->execute([':eid'=>(int)$employeeId]);
        ApiResponse::success(['allowances'=>$als->fetchAll(), 'deductions'=>$dds->fetchAll()]);
    } catch (Throwable $e) {
        ApiResponse::error('Failed to load components: '.$e->getMessage(), 500);
    }
}

function saveEmployeeAllowance(PDO $db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
    ensureEmployeeComponentCalcSchema($db);
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($in['id']) ? (int)$in['id'] : 0;
    $employeeId = (int)($in['employee_id'] ?? 0);
    $typeId = (int)($in['allowance_type_id'] ?? 0);
    $amount = (float)($in['amount'] ?? 0);
    $calcMode = in_array(($in['calc_mode'] ?? 'fixed'), ['fixed','percent_basic','percent_gross'], true) ? $in['calc_mode'] : 'fixed';
    $rate = isset($in['rate']) && $in['rate'] !== '' ? (float)$in['rate'] : null;
    if (!$employeeId || !$typeId) ApiResponse::error('employee_id and allowance_type_id are required');
    // Company scope check
    $empOk = $db->prepare('SELECT id FROM employees WHERE id = :id AND company_id = :cid');
    $empOk->execute([':id'=>$employeeId, ':cid'=>$user['company_id']]);
    if ($empOk->rowCount() === 0) ApiResponse::forbidden('Access denied');
    try {
        if ($id > 0) {
            $q = $db->prepare("UPDATE employee_allowances SET allowance_type_id=:t, amount=:a, calc_mode=:m, rate=:r, updated_at=NOW() WHERE id=:id AND employee_id=:eid");
            $q->execute([':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate, ':id'=>$id, ':eid'=>$employeeId]);
        } else {
            $q = $db->prepare("INSERT INTO employee_allowances (employee_id, allowance_type_id, amount, calc_mode, rate, status, start_date, created_at) VALUES (:eid, :t, :a, :m, :r, 'active', CURDATE(), NOW())");
            $q->execute([':eid'=>$employeeId, ':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate]);
            $id = (int)$db->lastInsertId();
        }
        ApiResponse::success(['id'=>$id], 'Saved');
    } catch (Throwable $e) {
        ApiResponse::error('Failed to save: '.$e->getMessage(), 500);
    }
}

function deleteEmployeeAllowance(PDO $db, int $id) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
    if (!$id) ApiResponse::error('ID required');
    try {
        // Soft delete by status if column exists else delete
        $cols = getTableColumns($db, 'employee_allowances');
        if (in_array('status', $cols, true)) {
            $q = $db->prepare("UPDATE employee_allowances ea JOIN employees e ON e.id = ea.employee_id SET ea.status='inactive', ea.end_date = CURDATE() WHERE ea.id = :id AND e.company_id = :cid");
            $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        } else {
            $q = $db->prepare("DELETE ea FROM employee_allowances ea JOIN employees e ON e.id = ea.employee_id WHERE ea.id = :id AND e.company_id = :cid");
            $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        }
        ApiResponse::success(null, 'Deleted');
    } catch (Throwable $e) { ApiResponse::error('Failed to delete: '.$e->getMessage(), 500); }
}

function saveEmployeeDeduction(PDO $db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
    ensureEmployeeComponentCalcSchema($db);
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($in['id']) ? (int)$in['id'] : 0;
    $employeeId = (int)($in['employee_id'] ?? 0);
    $typeId = (int)($in['deduction_type_id'] ?? 0);
    $amount = (float)($in['amount'] ?? 0);
    $calcMode = in_array(($in['calc_mode'] ?? 'fixed'), ['fixed','percent_basic','percent_gross'], true) ? $in['calc_mode'] : 'fixed';
    $rate = isset($in['rate']) && $in['rate'] !== '' ? (float)$in['rate'] : null;
    if (!$employeeId || !$typeId) ApiResponse::error('employee_id and deduction_type_id are required');
    $empOk = $db->prepare('SELECT id FROM employees WHERE id = :id AND company_id = :cid');
    $empOk->execute([':id'=>$employeeId, ':cid'=>$user['company_id']]);
    if ($empOk->rowCount() === 0) ApiResponse::forbidden('Access denied');
    try {
        if ($id > 0) {
            $q = $db->prepare("UPDATE employee_deductions SET deduction_type_id=:t, amount=:a, calc_mode=:m, rate=:r, updated_at=NOW() WHERE id=:id AND employee_id=:eid");
            $q->execute([':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate, ':id'=>$id, ':eid'=>$employeeId]);
        } else {
            $q = $db->prepare("INSERT INTO employee_deductions (employee_id, deduction_type_id, amount, calc_mode, rate, status, start_date, created_at) VALUES (:eid, :t, :a, :m, :r, 'active', CURDATE(), NOW())");
            $q->execute([':eid'=>$employeeId, ':t'=>$typeId, ':a'=>$amount, ':m'=>$calcMode, ':r'=>$rate]);
            $id = (int)$db->lastInsertId();
        }
        ApiResponse::success(['id'=>$id], 'Saved');
    } catch (Throwable $e) {
        ApiResponse::error('Failed to save: '.$e->getMessage(), 500);
    }
}

function deleteEmployeeDeduction(PDO $db, int $id) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
    if (!$id) ApiResponse::error('ID required');
    try {
        $cols = getTableColumns($db, 'employee_deductions');
        if (in_array('status', $cols, true)) {
            $q = $db->prepare("UPDATE employee_deductions ed JOIN employees e ON e.id = ed.employee_id SET ed.status='inactive', ed.end_date = CURDATE() WHERE ed.id = :id AND e.company_id = :cid");
            $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        } else {
            $q = $db->prepare("DELETE ed FROM employee_deductions ed JOIN employees e ON e.id = ed.employee_id WHERE ed.id = :id AND e.company_id = :cid");
            $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        }
        ApiResponse::success(null, 'Deleted');
    } catch (Throwable $e) { ApiResponse::error('Failed to delete: '.$e->getMessage(), 500); }
}

function bulkCreateEmployees(PDO $db) {
    $user = getCurrentUser();
    // Permissions similar to createEmployee
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }

    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $rows = isset($input['rows']) && is_array($input['rows']) ? $input['rows'] : null;
    if (!$rows) ApiResponse::error('No rows provided');

    $cid = (int)$user['company_id'];

    // Log start of bulk import
    try { app_log('info', 'bulk_employees_import_start', ['company_id' => $cid, 'user_id' => $user['id'] ?? null, 'total_rows' => count($rows)]); } catch (Throwable $e) {}

    // Preload maps for fast lookups
    $deptByName = $deptByCode = [];
    $posByTitle = $posByCode = [];
    $branchByName = $branchByCode = [];
    $mgrByEmpNo = [];
    try {
        $q = $db->prepare("SELECT id, name, code FROM departments WHERE company_id = :cid");
        $q->execute([':cid'=>$cid]);
        foreach ($q->fetchAll() as $r){ if (!empty($r['name'])) $deptByName[strtolower($r['name'])]=(int)$r['id']; if (!empty($r['code'])) $deptByCode[strtolower($r['code'])]=(int)$r['id']; }
    } catch (Throwable $e) {}
    try {
        $q = $db->prepare("SELECT id, title, code FROM positions WHERE company_id = :cid");
        $q->execute([':cid'=>$cid]);
        foreach ($q->fetchAll() as $r){ if (!empty($r['title'])) $posByTitle[strtolower($r['title'])]=(int)$r['id']; if (!empty($r['code'])) $posByCode[strtolower($r['code'])]=(int)$r['id']; }
    } catch (Throwable $e) {}
    try {
        $q = $db->prepare("SELECT id, name, code FROM branches WHERE company_id = :cid");
        $q->execute([':cid'=>$cid]);
        foreach ($q->fetchAll() as $r){ if (!empty($r['name'])) $branchByName[strtolower($r['name'])]=(int)$r['id']; if (!empty($r['code'])) $branchByCode[strtolower($r['code'])]=(int)$r['id']; }
    } catch (Throwable $e) {}
    try {
        $q = $db->prepare("SELECT id, employee_number FROM employees WHERE company_id = :cid");
        $q->execute([':cid'=>$cid]);
        foreach ($q->fetchAll() as $r){ if (!empty($r['employee_number'])) $mgrByEmpNo[strtoupper(trim($r['employee_number']))]=(int)$r['id']; }
    } catch (Throwable $e) {}

    $summary = ['total'=>count($rows), 'success_count'=>0, 'fail_count'=>0, 'rows'=>[]];
    // Track duplicates within the same file (case-insensitive)
    $seenEmpNos = [];

    foreach ($rows as $idx => $row) {
        $empno = normalizeEmpNumber($row['employee_number'] ?? '');
        $first = trim((string)($row['first_name'] ?? ''));
        $last = trim((string)($row['last_name'] ?? ''));
        $hireRaw = trim((string)($row['hire_date'] ?? ''));

        $errors = [];
        $warnings = [];

        // Required fields check (collect all missing)
        $missing = [];
        if ($empno === '') $missing[] = 'employee_number';
        if ($first === '') $missing[] = 'first_name';
        if ($last === '') $missing[] = 'last_name';
        if ($hireRaw === '') $missing[] = 'hire_date';
        if (!empty($missing)) {
            $errors[] = 'Missing required field(s): ' . implode(', ', $missing);
        }

        // Normalize dates with validation
        $hire_date = $hireRaw !== '' ? normalizeBulkDate($hireRaw) : null;
        if ($hireRaw !== '' && $hire_date === null) {
            $errors[] = "Invalid hire_date format: '" . $hireRaw . "' (expected YYYY-MM-DD or dd/mm/yyyy)";
        }
        $dob = null;
        if (isset($row['date_of_birth']) && trim((string)$row['date_of_birth']) !== '') {
            $dob = normalizeBulkDate($row['date_of_birth']);
            if ($dob === null) {
                $warnings[] = "Invalid date_of_birth format: '" . (string)$row['date_of_birth'] . "' (skipped)";
            }
        }

        // Basic email format validation if provided
        $emailVal = isset($row['email']) ? trim((string)$row['email']) : '';
        if ($emailVal !== '' && !filter_var($emailVal, FILTER_VALIDATE_EMAIL)) {
            $errors[] = "Invalid email: '" . $emailVal . "'";
        }

        // In-file duplicate check
        if ($empno !== '') {
            $key = strtoupper($empno);
            if (isset($seenEmpNos[$key])) {
                $errors[] = 'Duplicate employee_number in file';
            } else {
                $seenEmpNos[$key] = true;
            }
        }

        // Duplicate check against database
        try {
            if ($empno !== '') {
                $du = $db->prepare('SELECT id FROM employees WHERE company_id = :cid AND UPPER(employee_number) = UPPER(:no) LIMIT 1');
                $du->execute([':cid'=>$cid, ':no'=>$empno]);
                if ($du->fetchColumn()) {
                    $errors[] = 'Duplicate employee_number';
                }
            }
        } catch (Throwable $e) {
            // ignore duplicate check errors and continue validation
        }

        // Map helpers
        $department_id = null; $position_id = null; $branch_id = null; $manager_id = null;
        $depIn = $row['department_id'] ?? ($row['department'] ?? ($row['dept'] ?? null));
        if ($depIn !== null && $depIn !== '') {
            if (is_numeric($depIn)) { $department_id = (int)$depIn; }
            else {
                $key = strtolower(trim((string)$depIn));
                if (isset($deptByName[$key])) $department_id = $deptByName[$key];
                elseif (isset($deptByCode[$key])) $department_id = $deptByCode[$key];
                if ($department_id === null) { $warnings[] = "Department not found: '" . (string)$depIn . "'"; }
            }
        }
        $posIn = $row['position_id'] ?? ($row['position'] ?? ($row['job_title'] ?? null));
        if ($posIn !== null && $posIn !== '') {
            if (is_numeric($posIn)) { $position_id = (int)$posIn; }
            else {
                $key = strtolower(trim((string)$posIn));
                if (isset($posByTitle[$key])) $position_id = $posByTitle[$key];
                elseif (isset($posByCode[$key])) $position_id = $posByCode[$key];
                if ($position_id === null) { $warnings[] = "Position not found: '" . (string)$posIn . "'"; }
            }
        }
        $brIn = $row['branch_id'] ?? ($row['branch'] ?? null);
        if ($brIn !== null && $brIn !== '') {
            if (is_numeric($brIn)) { $branch_id = (int)$brIn; }
            else {
                $key = strtolower(trim((string)$brIn));
                if (isset($branchByName[$key])) $branch_id = $branchByName[$key];
                elseif (isset($branchByCode[$key])) $branch_id = $branchByCode[$key];
                if ($branch_id === null) { $warnings[] = "Branch not found: '" . (string)$brIn . "'"; }
            }
        }
        $mgrNo = trim((string)($row['manager_employee_number'] ?? ''));
        if ($mgrNo !== '') {
            if (isset($mgrByEmpNo[strtoupper($mgrNo)])) {
                $manager_id = $mgrByEmpNo[strtoupper($mgrNo)];
            } else {
                $warnings[] = "Manager not found: '" . $mgrNo . "'";
            }
        }

        // If we have any errors, log and continue to next row
        if (!empty($errors)) {
            $msg = 'Errors: ' . implode('; ', $errors);
            if (!empty($warnings)) { $msg .= ' | Warnings: ' . implode('; ', $warnings); }
            $summary['rows'][] = ['employee_number'=>$empno, 'success'=>false, 'message'=>$msg];
            $summary['fail_count']++;
            try { app_log('warning', 'bulk_employee_row_failed', ['row_index'=>$idx+1, 'employee_number'=>$empno, 'errors'=>$errors, 'warnings'=>$warnings, 'row'=>$row]); } catch (Throwable $e) {}
            continue;
        }

        // Build payload
        $payload = [
            'company_id' => $cid,
            'employee_number' => $empno,
            'first_name' => $first,
            'middle_name' => nullIfEmpty($row['middle_name'] ?? null),
            'last_name' => $last,
            'hire_date' => $hire_date,
            'email' => nullIfEmpty($row['email'] ?? null),
            'phone' => nullIfEmpty($row['phone'] ?? null),
            'gender' => nullIfEmpty($row['gender'] ?? null),
            'date_of_birth' => $dob,
            'department_id' => $department_id,
            'position_id' => $position_id,
            'branch_id' => $branch_id,
            'manager_id' => $manager_id,
            'employment_status' => $row['employment_status'] ?? 'active',
            'created_by' => $user['id'],
            'created_at' => date('Y-m-d H:i:s'),
        ];
        try {
            $ok = insertDynamic($db, 'employees', $payload);
            if ($ok) {
                $newId = (int)$db->lastInsertId();
                // Extend manager map so subsequent rows can reference this new employee
                $mgrByEmpNo[strtoupper($empno)] = $newId;
                $msg = 'created';
                if (!empty($warnings)) { $msg .= ' | Warnings: ' . implode('; ', $warnings); }
                $summary['rows'][] = ['employee_number'=>$empno, 'success'=>true, 'message'=>$msg];
                $summary['success_count']++;
            } else {
                $summary['rows'][] = ['employee_number'=>$empno, 'success'=>false, 'message'=>'insert failed'];
                $summary['fail_count']++;
                try { app_log('error', 'bulk_employee_row_insert_failed', ['row_index'=>$idx+1, 'employee_number'=>$empno]); } catch (Throwable $e) {}
            }
        } catch (Throwable $e) {
            // Friendly duplicate message on SQL unique constraint
            $msg = $e->getMessage();
            $code = (string)$e->getCode();
            if ($code === '23000' || stripos($msg, 'duplicate') !== false) {
                $msg = 'Duplicate employee_number';
            }
            $summary['rows'][] = ['employee_number'=>$empno, 'success'=>false, 'message'=>$msg];
            $summary['fail_count']++;
            try { app_log('error', 'bulk_employee_row_exception', ['row_index'=>$idx+1, 'employee_number'=>$empno, 'error'=>$e->getMessage()]); } catch (Throwable $e2) {}
        }
    }

    // Log end of bulk import and record activity (compact)
    try { app_log('info', 'bulk_employees_import_end', ['total'=>$summary['total'], 'success'=>$summary['success_count'], 'fail'=>$summary['fail_count']]); } catch (Throwable $e) {}
    try { recordActivity($db, $cid, (int)$user['id'], 'bulk_import_employees', 'employee', null, 'Bulk create processed', ['total'=>$summary['total'], 'success'=>$summary['success_count'], 'fail'=>$summary['fail_count']]); } catch (Throwable $e) {}

    ApiResponse::success($summary, 'Bulk create processed');
}

// Parse dd/mm/yyyy to yyyy-mm-dd or passthrough ISO
function normalizeBulkDate($v) {
    if ($v === null) return null;
    $t = trim((string)$v);
    if ($t === '') return null;
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $t)) return $t;
    if (preg_match('/^(\d{1,2})\/(\d{1,2})\/(\d{4})$/', $t, $m)) {
        // Assume dd/mm/yyyy as used in UI
        $d = (int)$m[1]; $mo = (int)$m[2]; $y = (int)$m[3];
        return sprintf('%04d-%02d-%02d', $y, $mo, $d);
    }
    // Fallback: try strtotime
    $ts = strtotime($t);
    if ($ts) return date('Y-m-d', $ts);
    return null;
}

// Insert a row building the statement from the intersection of provided data keys and existing columns
function insertDynamic(PDO $db, $table, array $data) {
    $cols = getTableColumns($db, $table);
    if (!$cols) return false;
    $keys = array_values(array_intersect(array_keys($data), $cols));
    if (empty($keys)) return false;
    $fields = implode(', ', array_map(function($c){ return "`$c`"; }, $keys));
    $placeholders = implode(', ', array_map(function($c){ return ":$c"; }, $keys));
    $sql = "INSERT INTO `$table` ($fields) VALUES ($placeholders)";
    $stmt = $db->prepare($sql);
    foreach ($keys as $k) {
        $stmt->bindValue(':'.$k, $data[$k]);
    }
    return $stmt->execute();
}

// Normalize empty string values to NULL (useful for optional text/date inputs)
function nullIfEmpty($v) {
    if ($v === null) return null;
    if (is_string($v)) {
        return (trim($v) === '') ? null : $v;
    }
    return $v;
}

// Normalize optional foreign keys: treat '', null, 0, '0' as NULL
function normalizeFk($v) {
    if ($v === null) return null;
    if (is_string($v)) {
        $t = trim($v);
        if ($t === '' || $t === '0') return null;
        if (ctype_digit($t)) return (int)$t;
        return null; // any non-numeric string becomes NULL to avoid FK violation
    }
    if ($v === 0) return null;
    return is_int($v) ? $v : (int)$v;
}

// Normalize employee number for consistent duplicate checks
function normalizeEmpNumber($v) {
    $t = (string)$v;
    // Replace non-breaking spaces with regular spaces, then trim
    $t = str_replace("\xC2\xA0", ' ', $t);
    $t = trim($t);
    return $t;
}

// Resolve a grade id from input payload. Accepts multiple keys:
// - numeric: grade_id, gradeId
// - string labels: grade_code, gradeCode, grade_name, gradeName, grade, current_grade, starting_grade, currentGrade, startingGrade
// Returns int|null
function resolveGradeIdFromInput(PDO $db, int $companyId, array $input) {
    // Prefer explicit numeric IDs first
    foreach (['grade_id','gradeId'] as $k) {
        if (isset($input[$k]) && $input[$k] !== '' && $input[$k] !== null) {
            $val = $input[$k];
            if (is_int($val) || ctype_digit((string)$val)) {
                return (int)$val;
            }
        }
    }
    // Fallback to textual labels
    $labelKeys = ['grade_code','gradeCode','grade_name','gradeName','grade','current_grade','starting_grade','currentGrade','startingGrade'];
    $labels = [];
    foreach ($labelKeys as $lk) {
        if (!empty($input[$lk]) && is_string($input[$lk])) {
            $labels[] = trim($input[$lk]);
        }
    }
    foreach ($labels as $g) {
        try {
            $q = $db->prepare("SELECT id FROM grades WHERE company_id = :cid AND (LOWER(code) = LOWER(:g) OR LOWER(name) = LOWER(:g)) LIMIT 1");
            $q->execute([':cid'=>$companyId, ':g'=>$g]);
            $gid = $q->fetchColumn();
            if ($gid) return (int)$gid;
        } catch (Throwable $e) { /* ignore and continue */ }
    }
    return null;
}

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? null;

$database = new Database();
$db = $database->getConnection();

switch ($method) {
    case 'GET':
        if ($action === 'lists') {
            getOnboardingLists($db);
        } elseif ($action === 'components') {
            getEmployeeComponents($db, $id);
        } elseif ($id) {
            getEmployee($db, $id);
        } else {
            getEmployees($db);
        }
        break;
    
    case 'POST':
        if ($action === 'bulk_create') {
            bulkCreateEmployees($db);
        } elseif ($action === 'save_allowance') {
            saveEmployeeAllowance($db);
        } elseif ($action === 'delete_allowance') {
            deleteEmployeeAllowance($db, isset($_GET['id']) ? (int)$_GET['id'] : 0);
        } elseif ($action === 'save_deduction') {
            saveEmployeeDeduction($db);
        } elseif ($action === 'delete_deduction') {
            deleteEmployeeDeduction($db, isset($_GET['id']) ? (int)$_GET['id'] : 0);
        } else {
            createEmployee($db);
        }
        break;
    
    case 'PUT':
        updateEmployee($db, $id);
        break;
    
    case 'DELETE':
        deleteEmployee($db, $id);
        break;
    
    default:
        ApiResponse::error('Method not allowed', 405);
}

function getEmployees($db) {
    $user = getCurrentUser();
    $filters = [];
    $params = [];
    
    // Build query based on filters
    $query = "SELECT e.*, d.name as department_name, p.title as position_title,
                     b.name AS branch_name,
                     CONCAT(manager.first_name, ' ', manager.last_name) AS manager_full_name,
                     u.username, u.email as user_email, u.status as user_status,
                     CONCAT(e.first_name, ' ', e.last_name) as full_name
              FROM employees e
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN positions p ON e.position_id = p.id
              LEFT JOIN branches b ON e.branch_id = b.id
              LEFT JOIN employees manager ON e.manager_id = manager.id
              LEFT JOIN users u ON e.user_id = u.id
              WHERE e.company_id = :company_id";
    
    $params[':company_id'] = $user['company_id'];
    
    // Apply filters
    if (isset($_GET['department']) && !empty($_GET['department'])) {
        $query .= " AND d.name = :department";
        $params[':department'] = $_GET['department'];
    }
    
    if (isset($_GET['status']) && !empty($_GET['status'])) {
        $query .= " AND e.employment_status = :status";
        $params[':status'] = $_GET['status'];
    }
    
    if (isset($_GET['search']) && !empty($_GET['search'])) {
        $query .= " AND (CONCAT(e.first_name, ' ', e.last_name) LIKE :search 
                    OR e.employee_number LIKE :search 
                    OR e.email LIKE :search)";
        $params[':search'] = '%' . $_GET['search'] . '%';
    }
    
    $query .= " ORDER BY e.created_at DESC";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();
    
    $employees = $stmt->fetchAll();
    
    ApiResponse::success($employees);
}

function getOnboardingLists($db) {
    $user = getCurrentUser();
    try {
        // Skills
        $skillsStmt = $db->prepare("SELECT id, name FROM skills WHERE company_id = :cid ORDER BY name");
        $skillsStmt->bindValue(':cid', $user['company_id']);
        $skillsStmt->execute();
        $skills = $skillsStmt->fetchAll();

        // Certificates
        $certStmt = $db->prepare("SELECT id, name FROM certificates WHERE company_id = :cid ORDER BY name");
        $certStmt->bindValue(':cid', $user['company_id']);
        $certStmt->execute();
        $certificates = $certStmt->fetchAll();

        // Static enums
        $document_types = ['contract','id_copy','passport','certificate','resume','other'];
        $proficiency_levels = ['beginner','intermediate','advanced','expert'];
        $certificate_status = ['active','expired','revoked'];
        $regions = ['Greater Accra','Ashanti','Western','Western North','Central','Eastern','Volta','Oti','Northern','North East','Savannah','Upper East','Upper West','Bono','Bono East','Ahafo'];
        $religions = ['Christian','Muslim','Traditional','Others'];

        // Grades (for payroll grade setup)
        $gradesStmt = $db->prepare("SELECT id, name, code, base_salary FROM grades WHERE company_id = :cid ORDER BY name");
        $gradesStmt->bindValue(':cid', $user['company_id']);
        $gradesStmt->execute();
        $grades = $gradesStmt->fetchAll();

        ApiResponse::success([
            'skills' => $skills,
            'certificates' => $certificates,
            'document_types' => $document_types,
            'proficiency_levels' => $proficiency_levels,
            'certificate_status' => $certificate_status,
            'regions' => $regions,
            'religions' => $religions,
            'grades' => $grades,
        ]);
    } catch (Throwable $e) {
        ApiResponse::error('Failed to load lists: ' . $e->getMessage(), 500);
    }
}

// Safely delete a file inside storage/uploads only
function safeDeleteUploadedFile($relPath) {
    try {
        if (!$relPath) return;
        $base = realpath(__DIR__ . '/../storage/uploads');
        $full = realpath(__DIR__ . '/../' . ltrim($relPath, '/\\'));
        if ($full && $base && strpos($full, $base) === 0 && is_file($full)) {
            @unlink($full);
        }
    } catch (Throwable $e) {
        // ignore cleanup failures
    }
}

function getEmployee($db, $id) {
    $user = getCurrentUser();
    
    $query = "SELECT e.*, d.name as department_name, p.title as position_title,
                     b.name as branch_name, sc.name as sub_company_name,
                     u.username, u.email as user_email, u.status as user_status,
                     CONCAT(e.first_name, ' ', e.last_name) as full_name,
                     manager.first_name as manager_first_name,
                     manager.last_name as manager_last_name
              FROM employees e
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN positions p ON e.position_id = p.id
              LEFT JOIN branches b ON e.branch_id = b.id
              LEFT JOIN sub_companies sc ON e.sub_company_id = sc.id
              LEFT JOIN users u ON e.user_id = u.id
              LEFT JOIN employees manager ON e.manager_id = manager.id
              WHERE e.id = :id AND e.company_id = :company_id";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':id', $id);
    $stmt->bindParam(':company_id', $user['company_id']);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $employee = $stmt->fetch();

        // Load related arrays for profile view
        // Documents
        $dStmt = $db->prepare("SELECT id, document_type, title, description, file_path, file_size, mime_type, is_confidential, uploaded_by, created_at
                               FROM employee_documents WHERE employee_id = :eid ORDER BY created_at DESC");
        $dStmt->bindValue(':eid', $id, PDO::PARAM_INT);
        $dStmt->execute();
        $employee['documents'] = $dStmt->fetchAll();

        // Skills with skill name
        $sStmt = $db->prepare("SELECT es.*, s.name AS skill_name
                               FROM employee_skills es
                               LEFT JOIN skills s ON es.skill_id = s.id
                               WHERE es.employee_id = :eid ORDER BY es.created_at DESC");
        $sStmt->bindValue(':eid', $id, PDO::PARAM_INT);
        $sStmt->execute();
        $employee['skills'] = $sStmt->fetchAll();

        // Employment history
        $hStmt = $db->prepare("SELECT * FROM employment_history WHERE employee_id = :eid ORDER BY start_date DESC, id DESC");
        $hStmt->bindValue(':eid', $id, PDO::PARAM_INT);
        $hStmt->execute();
        $employee['employment_history'] = $hStmt->fetchAll();

        // Certificates with certificate name
        $cStmt = $db->prepare("SELECT ec.*, c.name AS certificate_name
                               FROM employee_certificates ec
                               LEFT JOIN certificates c ON ec.certificate_id = c.id
                               WHERE ec.employee_id = :eid ORDER BY ec.issue_date DESC, ec.id DESC");
        $cStmt->bindValue(':eid', $id, PDO::PARAM_INT);
        $cStmt->execute();
        $employee['certificates'] = $cStmt->fetchAll();

        ApiResponse::success($employee);
    } else {
        ApiResponse::notFound('Employee not found');
    }
}

function createEmployee($db) {
    $user = getCurrentUser();
    
    // Check permissions
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    // Normalize key inputs
    if (isset($input['employee_number'])) {
        $input['employee_number'] = normalizeEmpNumber($input['employee_number']);
    }
    
    // Validate required fields
    $required = ['first_name', 'last_name', 'employee_number', 'hire_date'];
    foreach ($required as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            ApiResponse::error("Field '$field' is required");
        }
    }
    
    try {
        ensureEmployeePaymentSchema($db);
        ensureEmployeeGradeSchema($db);
        $db->beginTransaction();
        
        // Check if employee number already exists (case-insensitive, company-scoped)
        $checkQuery = "SELECT id FROM employees WHERE company_id = :company_id AND UPPER(employee_number) = UPPER(:emp_num)";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':emp_num', $input['employee_number']);
        $checkStmt->bindParam(':company_id', $user['company_id']);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() > 0) {
            ApiResponse::error('Employee number already exists for this company');
        }
        
        // Resolve grade_id from various possible inputs if provided (id, code, name)
        $resolvedGradeId = resolveGradeIdFromInput($db, $user['company_id'], $input);

        // Insert employee (schema-aware)
        $empPayload = [
            'company_id' => $user['company_id'],
            'sub_company_id' => normalizeFk($input['sub_company_id'] ?? null),
            'department_id' => normalizeFk($input['department_id'] ?? null),
            'branch_id' => normalizeFk($input['branch_id'] ?? null),
            'position_id' => normalizeFk($input['position_id'] ?? null),
            'grade_id' => $resolvedGradeId !== null ? $resolvedGradeId : normalizeFk($input['grade_id'] ?? null),
            'employee_number' => $input['employee_number'],
            'first_name' => $input['first_name'],
            'last_name' => $input['last_name'],
            'middle_name' => nullIfEmpty($input['middle_name'] ?? null),
            'email' => nullIfEmpty($input['email'] ?? null),
            'official_email' => nullIfEmpty($input['official_email'] ?? null),
            'phone' => nullIfEmpty($input['phone'] ?? null),
            'mobile' => nullIfEmpty($input['mobile'] ?? null),
            // Bank/MoMo defaults
            'bank_code' => nullIfEmpty($input['bank_code'] ?? null),
            'bank_account' => nullIfEmpty($input['bank_account'] ?? null),
            'momo_provider' => nullIfEmpty($input['momo_provider'] ?? null),
            'momo_number' => nullIfEmpty($input['momo_number'] ?? null),
            'date_of_birth' => nullIfEmpty($input['date_of_birth'] ?? null),
            'gender' => nullIfEmpty($input['gender'] ?? null),
            'marital_status' => nullIfEmpty($input['marital_status'] ?? null),
            'nationality' => nullIfEmpty($input['nationality'] ?? null),
            'address' => nullIfEmpty($input['address'] ?? null),
            // Contact & Address Extras
            'digital_address' => nullIfEmpty($input['digital_address'] ?? null),
            'house_number' => nullIfEmpty($input['house_number'] ?? null),
            'land_mark' => nullIfEmpty($input['land_mark'] ?? null),
            'postal_address' => nullIfEmpty($input['postal_address'] ?? null),
            'region' => nullIfEmpty($input['region'] ?? null),
            'religion' => nullIfEmpty($input['religion'] ?? null),
            'residential_ownership' => nullIfEmpty($input['residential_ownership'] ?? null),
            'staff_location' => nullIfEmpty($input['staff_location'] ?? null),
            'location' => nullIfEmpty($input['location'] ?? null),
            // Identity & titles
            'salutation' => nullIfEmpty($input['salutation'] ?? null),
            'title' => nullIfEmpty($input['title'] ?? null),
            'id_type' => nullIfEmpty($input['id_type'] ?? null),
            'id_number' => nullIfEmpty($input['id_number'] ?? null),
            'emergency_contact_name' => nullIfEmpty($input['emergency_contact_name'] ?? null),
            'emergency_contact_phone' => nullIfEmpty($input['emergency_contact_phone'] ?? null),
            'emergency_contact_relationship' => nullIfEmpty($input['emergency_contact_relationship'] ?? null),
            'hire_date' => $input['hire_date'],
            'probation_end_date' => nullIfEmpty($input['probation_end_date'] ?? null),
            'contract_type' => $input['contract_type'] ?? 'permanent',
            'employment_status' => $input['employment_status'] ?? 'active',
            'salary' => nullIfEmpty($input['salary'] ?? null),
            'currency' => $input['currency'] ?? 'USD',
            'manager_id' => normalizeFk($input['manager_id'] ?? null),
            'notes' => nullIfEmpty($input['notes'] ?? null),
            // Education
            'certificate_type' => nullIfEmpty($input['certificate_type'] ?? null),
            'first_degree_program' => nullIfEmpty($input['first_degree_program'] ?? null),
            'first_degree_class' => nullIfEmpty($input['first_degree_class'] ?? null),
            'first_degree_institution' => nullIfEmpty($input['first_degree_institution'] ?? null),
            'second_degree' => nullIfEmpty($input['second_degree'] ?? null),
            'second_degree_program' => nullIfEmpty($input['second_degree_program'] ?? null),
            'second_degree_class' => nullIfEmpty($input['second_degree_class'] ?? null),
            'second_degree_institution' => nullIfEmpty($input['second_degree_institution'] ?? null),
            'professional_certificate' => nullIfEmpty($input['professional_certificate'] ?? null),
            'professional_institution' => nullIfEmpty($input['professional_institution'] ?? null),
            'professional_results' => nullIfEmpty($input['professional_results'] ?? null),
            'academic_program' => nullIfEmpty($input['academic_program'] ?? null),
            // Next of kin
            'nok_name' => nullIfEmpty($input['nok_name'] ?? null),
            'nok_contact' => nullIfEmpty($input['nok_contact'] ?? null),
            'nok_postal_address' => nullIfEmpty($input['nok_postal_address'] ?? null),
            'nok_date' => nullIfEmpty($input['nok_date'] ?? null),
            'relationship' => nullIfEmpty($input['relationship'] ?? null),
            'nok_home_address' => nullIfEmpty($input['nok_home_address'] ?? null),
            // Job extras
            'job_title' => nullIfEmpty($input['job_title'] ?? null),
            'designation' => nullIfEmpty($input['designation'] ?? null),
            'job_description' => nullIfEmpty($input['job_description'] ?? null),
            'department_group' => nullIfEmpty($input['department_group'] ?? null),
            'department_unit' => nullIfEmpty($input['department_unit'] ?? null),
            'branch_type' => nullIfEmpty($input['branch_type'] ?? null),
            'starting_grade' => nullIfEmpty($input['starting_grade'] ?? null),
            'current_grade' => nullIfEmpty($input['current_grade'] ?? null),
            'joined_department_date' => nullIfEmpty($input['joined_department_date'] ?? null),
            'last_promotion_date' => nullIfEmpty($input['last_promotion_date'] ?? null),
            'confirmation' => nullIfEmpty($input['confirmation'] ?? null),
            'service_length' => nullIfEmpty($input['service_length'] ?? null),
            'staff_type' => nullIfEmpty($input['staff_type'] ?? null),
            'employee_type' => nullIfEmpty($input['employee_type'] ?? null),
            'person_type' => nullIfEmpty($input['person_type'] ?? null),
            'staff_status' => nullIfEmpty($input['staff_status'] ?? null),
            // Verification & misc
            'assigned_role' => nullIfEmpty($input['assigned_role'] ?? null),
            'verified' => isset($input['verified']) ? (int)!!$input['verified'] : null,
            'verified_by' => nullIfEmpty($input['verified_by'] ?? null),
            'verification_date' => nullIfEmpty($input['verification_date'] ?? null),
            'staff_exited_date' => nullIfEmpty($input['staff_exited_date'] ?? null),
            'created_by' => $user['id'],
            'created_at' => date('Y-m-d H:i:s'),
        ];
        insertDynamic($db, 'employees', $empPayload);
        $employeeId = $db->lastInsertId();

        // Minimal, schema-aware follow-up update: map front-end "profile_pic" to DB column "photo" if present
        $empCols = getTableColumns($db, 'employees');
        $updates = [];
        $params = [':id' => $employeeId];
        if (isset($input['profile_pic']) && in_array('photo', $empCols, true)) {
            $updates[] = 'photo = :photo';
            $params[':photo'] = nullIfEmpty($input['profile_pic']);
        }
        if (!empty($updates)) {
            $updates[] = 'updated_at = NOW()';
            $q = 'UPDATE employees SET ' . implode(', ', $updates) . ' WHERE id = :id';
            $uStmt = $db->prepare($q);
            foreach ($params as $k => $v) { $uStmt->bindValue($k, $v); }
            $uStmt->execute();
        }
        
        // Insert onboarding child records if provided
        // Documents (schema-aware)
        if (isset($input['documents']) && is_array($input['documents'])) {
            foreach ($input['documents'] as $doc) {
                if (!isset($doc['document_type']) || !isset($doc['title'])) continue;
                if (empty($doc['file_path'])) continue; // file_path is NOT NULL in schema
                $payload = [
                    'employee_id' => (int)$employeeId,
                    'document_type' => $doc['document_type'],
                    'title' => $doc['title'],
                    'description' => $doc['description'] ?? null,
                    'file_path' => $doc['file_path'],
                    'file_size' => isset($doc['file_size']) ? (int)$doc['file_size'] : null,
                    'mime_type' => $doc['mime_type'] ?? null,
                    'is_confidential' => (int)!!($doc['is_confidential'] ?? 0),
                    'uploaded_by' => (int)$user['id'],
                    'created_at' => date('Y-m-d H:i:s'),
                ];
                insertDynamic($db, 'employee_documents', $payload);
            }
        }

        // Skills
        if (isset($input['skills']) && is_array($input['skills'])) {
            $skillSql = "INSERT INTO employee_skills (employee_id, skill_id, proficiency_level, years_of_experience, certified, certification_name, certification_date, notes, created_at)
                         VALUES (:eid, :skill_id, :proficiency_level, :years_of_experience, :certified, :certification_name, :certification_date, :notes, NOW())";
            $skillStmt = $db->prepare($skillSql);
            foreach ($input['skills'] as $sk) {
                if (empty($sk['skill_id'])) continue;
                $skillStmt->bindValue(':eid', $employeeId, PDO::PARAM_INT);
                $skillStmt->bindValue(':skill_id', $sk['skill_id'], PDO::PARAM_INT);
                $skillStmt->bindValue(':proficiency_level', $sk['proficiency_level'] ?? 'beginner');
                $skillStmt->bindValue(':years_of_experience', isset($sk['years_of_experience']) ? (int)$sk['years_of_experience'] : 0, PDO::PARAM_INT);
                $skillStmt->bindValue(':certified', (int)!!($sk['certified'] ?? 0), PDO::PARAM_INT);
                $skillStmt->bindValue(':certification_name', $sk['certification_name'] ?? null);
                $skillStmt->bindValue(':certification_date', $sk['certification_date'] ?? null);
                $skillStmt->bindValue(':notes', $sk['notes'] ?? null);
                $skillStmt->execute();
            }
        }

        // Employment History
        if (isset($input['employment_history']) && is_array($input['employment_history'])) {
            $histSql = "INSERT INTO employment_history (employee_id, position_id, department_id, branch_id, start_date, end_date, salary, reason_for_change, notes, created_by, created_at)
                        VALUES (:eid, :position_id, :department_id, :branch_id, :start_date, :end_date, :salary, :reason_for_change, :notes, :created_by, NOW())";
            $histStmt = $db->prepare($histSql);
            foreach ($input['employment_history'] as $h) {
                if (empty($h['start_date'])) continue;
                $histStmt->bindValue(':eid', $employeeId, PDO::PARAM_INT);
                $histStmt->bindValue(':position_id', $h['position_id'] ?? null);
                $histStmt->bindValue(':department_id', $h['department_id'] ?? null);
                $histStmt->bindValue(':branch_id', $h['branch_id'] ?? null);
                $histStmt->bindValue(':start_date', $h['start_date']);
                $histStmt->bindValue(':end_date', $h['end_date'] ?? null);
                $histStmt->bindValue(':salary', $h['salary'] ?? null);
                $histStmt->bindValue(':reason_for_change', $h['reason_for_change'] ?? null);
                $histStmt->bindValue(':notes', $h['notes'] ?? null);
                $histStmt->bindValue(':created_by', $user['id'], PDO::PARAM_INT);
                $histStmt->execute();
            }
        }

        // Certificates (schema-aware)
        if (isset($input['certificates']) && is_array($input['certificates'])) {
            foreach ($input['certificates'] as $c) {
                if (empty($c['certificate_id']) || empty($c['issue_date'])) continue;
                $payload = [
                    'employee_id' => (int)$employeeId,
                    'certificate_id' => (int)$c['certificate_id'],
                    'issue_date' => $c['issue_date'],
                    'expiry_date' => $c['expiry_date'] ?? null,
                    'certificate_number' => $c['certificate_number'] ?? null,
                    'file_path' => $c['file_path'] ?? null,
                    'status' => $c['status'] ?? 'active',
                    'created_at' => date('Y-m-d H:i:s'),
                ];
                insertDynamic($db, 'employee_certificates', $payload);
            }
        }
        
        // Create user account if requested
        if (isset($input['create_user']) && $input['create_user'] && !empty($input['username'])) {
            $userQuery = "INSERT INTO users (
                            company_id, employee_id, username, email, password,
                            first_name, last_name, phone, status, created_by, created_at
                          ) VALUES (
                            :company_id, :employee_id, :username, :email, :password,
                            :first_name, :last_name, :phone, 'active', :created_by, NOW()
                          )";
            
            $userStmt = $db->prepare($userQuery);
            $defaultPassword = password_hash('password123', PASSWORD_DEFAULT);
            
            $userStmt->bindParam(':company_id', $user['company_id']);
            $userStmt->bindParam(':employee_id', $employeeId);
            $userStmt->bindParam(':username', $input['username']);
            $userStmt->bindParam(':email', $input['email']);
            $userStmt->bindParam(':password', $defaultPassword);
            $userStmt->bindParam(':first_name', $input['first_name']);
            $userStmt->bindParam(':last_name', $input['last_name']);
            $userStmt->bindParam(':phone', $input['phone']);
            $userStmt->bindParam(':created_by', $user['id']);
            
            $userStmt->execute();
            $userId = $db->lastInsertId();
            
            // Assign default role
            $roleQuery = "INSERT INTO user_roles (user_id, role_id, assigned_by, assigned_at) 
                          VALUES (:user_id, :role_id, :assigned_by, NOW())";
            $roleStmt = $db->prepare($roleQuery);
            $defaultRoleId = 6; // Employee role
            $roleStmt->bindParam(':user_id', $userId);
            $roleStmt->bindParam(':role_id', $defaultRoleId);
            $roleStmt->bindParam(':assigned_by', $user['id']);
            $roleStmt->execute();
            
            // Update employee with user_id
            $updateEmpQuery = "UPDATE employees SET user_id = :user_id WHERE id = :id";
            $updateEmpStmt = $db->prepare($updateEmpQuery);
            $updateEmpStmt->bindParam(':user_id', $userId);
            $updateEmpStmt->bindParam(':id', $employeeId);
            $updateEmpStmt->execute();
        }
        
        $db->commit();
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'employee_created', ['employee_id' => $employeeId]);
        
        ApiResponse::success(['id' => $employeeId], 'Employee created successfully');
        
    } catch (Exception $e) {
        $db->rollback();
        $msg = $e->getMessage();
        $code = (string)$e->getCode();
        if ($code === '23000' || stripos($msg, 'duplicate') !== false) {
            ApiResponse::error('Employee number already exists for this company');
        }
        ApiResponse::error('Failed to create employee: ' . $msg);
    }
}

function updateEmployee($db, $id) {
    $user = getCurrentUser();
    
    // Check permissions
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    if (!$id) {
        ApiResponse::error('Employee ID required');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    try {
        $db->beginTransaction();
        // Check if employee exists
        $checkQuery = "SELECT id FROM employees WHERE id = :id AND company_id = :company_id";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id);
        $checkStmt->bindParam(':company_id', $user['company_id']);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::notFound('Employee not found');
        }

        // Remember previous profile picture for cleanup if replaced
        $prevPic = null;
        try {
            $prevStmt = $db->prepare("SELECT photo FROM employees WHERE id = :id");
            $prevStmt->bindValue(':id', $id, PDO::PARAM_INT);
            $prevStmt->execute();
            $prevPic = $prevStmt->fetchColumn();
        } catch (Throwable $e) { /* ignore */ }

        // Attempt to resolve grade_id from various inputs and fold back into request payload
        $maybeGradeId = resolveGradeIdFromInput($db, $user['company_id'], $input);
        if ($maybeGradeId !== null) {
            $input['grade_id'] = $maybeGradeId;
        }

        // Build update set from actual schema columns only (schema-aware)
        $empCols = getTableColumns($db, 'employees');
        $updateFields = [];
        $params = [':id' => $id];
        $fieldMap = [
            // Core
            'first_name' => 'first_name',
            'last_name' => 'last_name',
            'middle_name' => 'middle_name',
            'email' => 'email',
            'official_email' => 'official_email',
            'phone' => 'phone',
            'mobile' => 'mobile',
            // Bank/MoMo defaults
            'bank_code' => 'bank_code',
            'bank_account' => 'bank_account',
            'momo_provider' => 'momo_provider',
            'momo_number' => 'momo_number',
            'date_of_birth' => 'date_of_birth',
            'gender' => 'gender',
            'marital_status' => 'marital_status',
            'nationality' => 'nationality',
            'address' => 'address',
            // Contact & Address extras
            'digital_address' => 'digital_address',
            'house_number' => 'house_number',
            'land_mark' => 'land_mark',
            'postal_address' => 'postal_address',
            'region' => 'region',
            'religion' => 'religion',
            'residential_ownership' => 'residential_ownership',
            'staff_location' => 'staff_location',
            'location' => 'location',
            // Identity & titles
            'salutation' => 'salutation',
            'title' => 'title',
            'id_type' => 'id_type',
            'id_number' => 'id_number',
            'emergency_contact_name' => 'emergency_contact_name',
            'emergency_contact_phone' => 'emergency_contact_phone',
            'emergency_contact_relationship' => 'emergency_contact_relationship',
            // Placement / FKs
            'department_id' => 'department_id',
            'position_id' => 'position_id',
            'branch_id' => 'branch_id',
            'manager_id' => 'manager_id',
            'sub_company_id' => 'sub_company_id',
            'grade_id' => 'grade_id',
            // Employment
            'salary' => 'salary',
            'currency' => 'currency',
            'contract_type' => 'contract_type',
            'employment_status' => 'employment_status',
            'probation_end_date' => 'probation_end_date',
            'hire_date' => 'hire_date',
            // Media mapping
            'profile_pic' => 'photo',
            // Misc
            'notes' => 'notes',
            // Education
            'certificate_type' => 'certificate_type',
            'first_degree_program' => 'first_degree_program',
            'first_degree_class' => 'first_degree_class',
            'first_degree_institution' => 'first_degree_institution',
            'second_degree' => 'second_degree',
            'second_degree_program' => 'second_degree_program',
            'second_degree_class' => 'second_degree_class',
            'second_degree_institution' => 'second_degree_institution',
            'professional_certificate' => 'professional_certificate',
            'professional_institution' => 'professional_institution',
            'professional_results' => 'professional_results',
            'academic_program' => 'academic_program',
            // Next of kin
            'nok_name' => 'nok_name',
            'nok_contact' => 'nok_contact',
            'nok_postal_address' => 'nok_postal_address',
            'nok_date' => 'nok_date',
            'relationship' => 'relationship',
            'nok_home_address' => 'nok_home_address',
            // Job extras
            'job_title' => 'job_title',
            'designation' => 'designation',
            'job_description' => 'job_description',
            'department_group' => 'department_group',
            'department_unit' => 'department_unit',
            'branch_type' => 'branch_type',
            'starting_grade' => 'starting_grade',
            'current_grade' => 'current_grade',
            'joined_department_date' => 'joined_department_date',
            'last_promotion_date' => 'last_promotion_date',
            'confirmation' => 'confirmation',
            'service_length' => 'service_length',
            'staff_type' => 'staff_type',
            'employee_type' => 'employee_type',
            'person_type' => 'person_type',
            'staff_status' => 'staff_status',
            // Verification & misc
            'assigned_role' => 'assigned_role',
            'verified' => 'verified',
            'verified_by' => 'verified_by',
            'verification_date' => 'verification_date',
            'staff_exited_date' => 'staff_exited_date',
        ];

        foreach ($fieldMap as $inField => $dbField) {
            if (!array_key_exists($inField, $input)) continue;
            if (!in_array($dbField, $empCols, true)) continue;
            $raw = $input[$inField];
            $val = $raw;
            if (in_array($dbField, ['department_id','position_id','branch_id','manager_id','sub_company_id','grade_id'], true)) {
                $val = normalizeFk($val);
            } else {
                $val = nullIfEmpty($val);
            }
            // Special case: avoid clearing existing photo if client sends empty profile_pic
            if ($dbField === 'photo' && $val === null) {
                continue;
            }
            $updateFields[] = "$dbField = :$dbField";
            $params[":$dbField"] = $val;
        }

        if (empty($updateFields)) {
            ApiResponse::error('No valid fields to update');
        }

        $updateFields[] = "updated_at = NOW()";
        $query = "UPDATE employees SET " . implode(', ', $updateFields) . " WHERE id = :id";
        $stmt = $db->prepare($query);
        foreach ($params as $key => $value) { $stmt->bindValue($key, $value); }
        $stmt->execute();

        // Prepare arrays to cleanup files after successful commit
        $cleanupDocPaths = [];
        $cleanupCertPaths = [];

        // Replace onboarding child arrays if provided
        // Documents (gather old file paths to cleanup after reinsertion)
        if (array_key_exists('documents', $input) && is_array($input['documents'])) {
            $oldDocPaths = [];
            try {
                $oldD = $db->prepare("SELECT file_path FROM employee_documents WHERE employee_id = :eid AND file_path IS NOT NULL");
                $oldD->execute([':eid' => $id]);
                foreach ($oldD->fetchAll() as $r) { if (!empty($r['file_path'])) $oldDocPaths[] = $r['file_path']; }
            } catch (Throwable $e) { /* ignore */ }
            $del = $db->prepare("DELETE FROM employee_documents WHERE employee_id = :eid");
            $del->execute([':eid' => $id]);
            $docSql = "INSERT INTO employee_documents (employee_id, document_type, title, description, file_path, file_size, mime_type, is_confidential, uploaded_by, created_at)
                       VALUES (:eid, :document_type, :title, :description, :file_path, :file_size, :mime_type, :is_confidential, :uploaded_by, NOW())";
            $docStmt = $db->prepare($docSql);
            foreach ($input['documents'] as $doc) {
                if (!isset($doc['document_type']) || !isset($doc['title'])) continue;
                if (empty($doc['file_path'])) continue; // file_path is NOT NULL in schema
                $docStmt->bindValue(':eid', $id, PDO::PARAM_INT);
                $docStmt->bindValue(':document_type', $doc['document_type']);
                $docStmt->bindValue(':title', $doc['title']);
                $docStmt->bindValue(':description', $doc['description'] ?? null);
                $docStmt->bindValue(':file_path', $doc['file_path']);
                $docStmt->bindValue(':file_size', isset($doc['file_size']) ? (int)$doc['file_size'] : null, isset($doc['file_size']) ? PDO::PARAM_INT : PDO::PARAM_NULL);
                $docStmt->bindValue(':mime_type', $doc['mime_type'] ?? null);
                $docStmt->bindValue(':is_confidential', (int)!!($doc['is_confidential'] ?? 0), PDO::PARAM_INT);
                $docStmt->bindValue(':uploaded_by', $user['id'], PDO::PARAM_INT);
                $docStmt->execute();
            }
            // Mark old files that are no longer referenced for cleanup after commit
            try {
                $newDocPaths = [];
                foreach ($input['documents'] as $doc) { if (!empty($doc['file_path'])) $newDocPaths[] = $doc['file_path']; }
                $cleanupDocPaths = array_values(array_diff($oldDocPaths, $newDocPaths));
            } catch (Throwable $e) { /* ignore */ }
        }

        // Skills
        if (array_key_exists('skills', $input) && is_array($input['skills'])) {
            $db->prepare("DELETE FROM employee_skills WHERE employee_id = :eid")->execute([':eid' => $id]);
            $skillSql = "INSERT INTO employee_skills (employee_id, skill_id, proficiency_level, years_of_experience, certified, certification_name, certification_date, notes, created_at)
                         VALUES (:eid, :skill_id, :proficiency_level, :years_of_experience, :certified, :certification_name, :certification_date, :notes, NOW())";
            $skillStmt = $db->prepare($skillSql);
            $seen = [];
            foreach ($input['skills'] as $sk) {
                if (empty($sk['skill_id'])) continue;
                if (isset($seen[$sk['skill_id']])) continue;
                $seen[$sk['skill_id']] = true;
                $skillStmt->bindValue(':eid', $id, PDO::PARAM_INT);
                $skillStmt->bindValue(':skill_id', $sk['skill_id'], PDO::PARAM_INT);
                $skillStmt->bindValue(':proficiency_level', $sk['proficiency_level'] ?? 'beginner');
                $skillStmt->bindValue(':years_of_experience', isset($sk['years_of_experience']) ? (int)$sk['years_of_experience'] : 0, PDO::PARAM_INT);
                $skillStmt->bindValue(':certified', (int)!!($sk['certified'] ?? 0), PDO::PARAM_INT);
                $skillStmt->bindValue(':certification_name', $sk['certification_name'] ?? null);
                $skillStmt->bindValue(':certification_date', $sk['certification_date'] ?? null);
                $skillStmt->bindValue(':notes', $sk['notes'] ?? null);
                $skillStmt->execute();
            }
        }

        // Employment history
        if (array_key_exists('employment_history', $input) && is_array($input['employment_history'])) {
            $db->prepare("DELETE FROM employment_history WHERE employee_id = :eid")->execute([':eid' => $id]);
            $histSql = "INSERT INTO employment_history (employee_id, position_id, department_id, branch_id, start_date, end_date, salary, reason_for_change, notes, created_by, created_at)
                        VALUES (:eid, :position_id, :department_id, :branch_id, :start_date, :end_date, :salary, :reason_for_change, :notes, :created_by, NOW())";
            $histStmt = $db->prepare($histSql);
            foreach ($input['employment_history'] as $h) {
                if (empty($h['start_date'])) continue;
                $histStmt->bindValue(':eid', $id, PDO::PARAM_INT);
                $histStmt->bindValue(':position_id', $h['position_id'] ?? null);
                $histStmt->bindValue(':department_id', $h['department_id'] ?? null);
                $histStmt->bindValue(':branch_id', $h['branch_id'] ?? null);
                $histStmt->bindValue(':start_date', $h['start_date']);
                $histStmt->bindValue(':end_date', $h['end_date'] ?? null);
                $histStmt->bindValue(':salary', $h['salary'] ?? null);
                $histStmt->bindValue(':reason_for_change', $h['reason_for_change'] ?? null);
                $histStmt->bindValue(':notes', $h['notes'] ?? null);
                $histStmt->bindValue(':created_by', $user['id'], PDO::PARAM_INT);
                $histStmt->execute();
            }
        }

        // Certificates (gather old file paths to cleanup after reinsertion)
        if (array_key_exists('certificates', $input) && is_array($input['certificates'])) {
            $oldCertPaths = [];
            try {
                $oldC = $db->prepare("SELECT file_path FROM employee_certificates WHERE employee_id = :eid AND file_path IS NOT NULL");
                $oldC->execute([':eid' => $id]);
                foreach ($oldC->fetchAll() as $r) { if (!empty($r['file_path'])) $oldCertPaths[] = $r['file_path']; }
            } catch (Throwable $e) { /* ignore */ }
            $db->prepare("DELETE FROM employee_certificates WHERE employee_id = :eid")->execute([':eid' => $id]);
            $certSql = "INSERT INTO employee_certificates (employee_id, certificate_id, issue_date, expiry_date, certificate_number, file_path, status, created_at)
                        VALUES (:eid, :certificate_id, :issue_date, :expiry_date, :certificate_number, :file_path, :status, NOW())";
            $certStmt = $db->prepare($certSql);
            foreach ($input['certificates'] as $c) {
                if (empty($c['certificate_id']) || empty($c['issue_date'])) continue;
                $certStmt->bindValue(':eid', $id, PDO::PARAM_INT);
                $certStmt->bindValue(':certificate_id', $c['certificate_id'], PDO::PARAM_INT);
                $certStmt->bindValue(':issue_date', $c['issue_date']);
                $certStmt->bindValue(':expiry_date', $c['expiry_date'] ?? null);
                $certStmt->bindValue(':certificate_number', $c['certificate_number'] ?? null);
                $certStmt->bindValue(':file_path', $c['file_path'] ?? null);
                $certStmt->bindValue(':status', $c['status'] ?? 'active');
                $certStmt->execute();
            }
            // Mark old certificate files that are no longer referenced for cleanup after commit
            try {
                $newCertPaths = [];
                foreach ($input['certificates'] as $c) { if (!empty($c['file_path'])) $newCertPaths[] = $c['file_path']; }
                $cleanupCertPaths = array_values(array_diff($oldCertPaths, $newCertPaths));
            } catch (Throwable $e) { /* ignore */ }
        }

        $db->commit();

        // After commit, perform filesystem cleanup (non-transactional)
        try {
            if (isset($input['profile_pic']) && $input['profile_pic'] && $prevPic && $prevPic !== $input['profile_pic']) {
                safeDeleteUploadedFile($prevPic);
            }
            foreach ($cleanupDocPaths as $fp) { safeDeleteUploadedFile($fp); }
            foreach ($cleanupCertPaths as $fp) { safeDeleteUploadedFile($fp); }
        } catch (Throwable $e) { /* ignore */ }

        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'employee_updated', ['employee_id' => $id, 'changes' => $input]);
        
        ApiResponse::success(null, 'Employee updated successfully');
        
    } catch (Exception $e) {
        if ($db->inTransaction()) $db->rollback();
        ApiResponse::error('Failed to update employee: ' . $e->getMessage());
    }
}

function deleteEmployee($db, $id) {
    $user = getCurrentUser();
    
    // Check permissions - only super admin and admin can delete
    if (!in_array($user['role_slug'], ['super_admin', 'admin'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    if (!$id) {
        ApiResponse::error('Employee ID required');
    }
    
    try {
        $db->beginTransaction();
        
        // Check if employee exists
        $checkQuery = "SELECT id, user_id FROM employees WHERE id = :id AND company_id = :company_id";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id);
        $checkStmt->bindParam(':company_id', $user['company_id']);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::notFound('Employee not found');
        }
        
        $employee = $checkStmt->fetch();
        
        // Soft delete - update status instead of actual deletion
        $query = "UPDATE employees SET employment_status = 'terminated', updated_at = NOW() WHERE id = :id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        // Deactivate user account if exists
        if ($employee['user_id']) {
            $userQuery = "UPDATE users SET status = 'inactive', updated_at = NOW() WHERE id = :user_id";
            $userStmt = $db->prepare($userQuery);
            $userStmt->bindParam(':user_id', $employee['user_id']);
            $userStmt->execute();
        }
        
        $db->commit();
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'employee_deleted', ['employee_id' => $id]);
        
        ApiResponse::success(null, 'Employee deleted successfully');
        
    } catch (Exception $e) {
        $db->rollback();
        ApiResponse::error('Failed to delete employee: ' . $e->getMessage());
    }
}
?>
