<?php
/**
 * Public Contact/Lead API (no auth required)
 * Accepts JSON POST from intro.html forms and sends an email + stores a lead file.
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
  ApiResponse::success(null, 'OK');
}

$method = $_SERVER['REQUEST_METHOD'];
if ($method !== 'POST') {
  ApiResponse::error('Method not allowed', 405);
}

function loadContactConfig(){
  $file = __DIR__ . '/../storage/config/contact.json';
  if (is_file($file)) {
    $cfg = json_decode(@file_get_contents($file), true);
    if (is_array($cfg)) return $cfg;
  }
  return [
    'to' => 'sales@smartquantumhr.com',
    'from_name' => 'SmartQuantumHR Website',
    'from_email' => 'no-reply@localhost',
    'bcc' => null
  ];
}

function storageEnsure($relDir){
  $base = realpath(__DIR__ . '/../storage');
  if ($base === false) { @mkdir(__DIR__ . '/../storage', 0777, true); $base = realpath(__DIR__ . '/../storage'); }
  $dir = rtrim($base, '/\\') . '/' . ltrim($relDir, '/\\');
  if (!is_dir($dir)) @mkdir($dir, 0777, true);
  return $dir;
}

function tooManyRequests(){
  // Simple rate limit by IP: max 5 per hour
  $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
  $dir = storageEnsure('tmp/ratelimit');
  $key = $dir . '/' . preg_replace('/[^a-zA-Z0-9_.-]/','_', $ip) . '.json';
  $now = time();
  $window = 3600; $limit = 5;
  $state = ['t' => $now, 'hits' => 0];
  if (is_file($key)){
    $state = json_decode(@file_get_contents($key), true) ?: $state;
    if (($now - ($state['t'] ?? 0)) > $window){ $state = ['t'=>$now,'hits'=>0]; }
  }
  if (($state['hits'] ?? 0) >= $limit) return true;
  $state['hits'] = ($state['hits'] ?? 0) + 1; $state['t'] = $now;
  @file_put_contents($key, json_encode($state));
  return false;
}

try {
  if (tooManyRequests()) ApiResponse::error('Too many requests, please try again later', 429);

  $input = json_decode(file_get_contents('php://input'), true) ?: [];
  $company   = trim((string)($input['company'] ?? ''));
  $name      = trim((string)($input['name'] ?? ''));
  $email     = trim((string)($input['email'] ?? ''));
  $phone     = trim((string)($input['phone'] ?? ''));
  $employees = trim((string)($input['employees'] ?? ''));
  $message   = trim((string)($input['message'] ?? ''));
  $source    = trim((string)($input['source'] ?? 'contact'));

  if ($company === '' || $name === '' || $email === '' || $phone === '') {
    ApiResponse::error('Please provide company, name, email and phone');
  }
  if (!filter_var($email, FILTER_VALIDATE_EMAIL)) ApiResponse::error('Invalid email');

  $cfg = loadContactConfig();

  // Persist lead to file
  $lead = [
    'company' => $company,
    'name' => $name,
    'email' => $email,
    'phone' => $phone,
    'employees' => $employees,
    'message' => $message,
    'source' => $source,
    'ip' => $_SERVER['REMOTE_ADDR'] ?? null,
    'ua' => $_SERVER['HTTP_USER_AGENT'] ?? null,
    'created_at' => date('c')
  ];
  $leadsDir = storageEnsure('leads/' . date('Y/m'));
  $id = 'lead-' . date('Ymd-His') . '-' . bin2hex(random_bytes(3));
  @file_put_contents($leadsDir . '/' . $id . '.json', json_encode($lead, JSON_PRETTY_PRINT));

  // Compose email
  $subject = '[SmartQuantumHR] New Website Inquiry (' . strtoupper($source) . ')';
  $body = "New inquiry from SmartQuantumHR website\n\n"
        . "Company: $company\n"
        . "Name: $name\n"
        . "Email: $email\n"
        . "Phone: $phone\n"
        . "Employees: $employees\n"
        . "Source: $source\n"
        . "IP: ".($_SERVER['REMOTE_ADDR'] ?? '-') ."\n"
        . "User-Agent: ".($_SERVER['HTTP_USER_AGENT'] ?? '-') ."\n\n"
        . "Message:\n$message\n";

  $headers = "MIME-Version: 1.0\r\n";
  $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
  $headers .= 'X-Request-ID: '. get_request_id() ."\r\n";
  $headers .= 'From: ' . ($cfg['from_name'] ?? 'SmartQuantumHR Website') . ' <' . ($cfg['from_email'] ?? 'no-reply@localhost') . ">\r\n";
  $headers .= 'Reply-To: ' . $name . ' <' . $email . ">\r\n";
  if (!empty($cfg['bcc'])) $headers .= 'Bcc: ' . $cfg['bcc'] . "\r\n";

  $ok = @mail($cfg['to'], $subject, $body, $headers);
  if (!$ok) {
    app_log('error', 'contact_mail_failed', ['to'=>$cfg['to']]);
    // Still succeed to the user but inform failure message
    ApiResponse::success(['id'=>$id, 'queued'=>true], 'Thanks! We have received your message.');
  }

  ApiResponse::success(['id'=>$id], 'Thanks! We have received your message.');

} catch (Throwable $e) {
  app_log('error', 'contact_error', ['error'=>$e->getMessage()]);
  ApiResponse::error('Failed to submit, please try again later', 500);
}
