<?php
/**
 * Attendance API: geofenced/QR clock in/out and history
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$database = new Database();
$db = $database->getConnection();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$user = getCurrentUser();
// ensure tables for sites and events exist
ensureAttendanceTables($db);

switch ($method) {
  case 'GET':
    if ($action === 'history') listHistory($db, $user);
    elseif ($action === 'events') listEvents($db, $user);
    elseif ($action === 'sites') listSites($db, $user);
    elseif ($action === 'exceptions') listExceptions($db, $user);
    elseif ($action === 'policy') policy($db, $user);
    elseif ($action === 'stats_year') statsYear($db, $user);
    else ApiResponse::error('Unknown action', 400);
    break;
  case 'POST':
    if ($action === 'clock') clock($db, $user);
    elseif ($action === 'site_save') siteSave($db, $user);
    elseif ($action === 'site_toggle') siteToggle($db, $user);
    elseif ($action === 'exception_decide') exceptionDecide($db, $user);
    elseif ($action === 'policy_save') policySave($db, $user);
    else ApiResponse::error('Unknown action', 400);
    break;
  default: ApiResponse::error('Method not allowed', 405);
}

function policy(PDO $db, array $user){
  // Read company-level attendance policy from companies.settings JSON
  $settings = [];
  try {
    $st = $db->prepare('SELECT settings FROM companies WHERE id = :id');
    $st->execute([':id' => $user['company_id']]);
    $row = $st->fetch();
    $settings = $row && !empty($row['settings']) ? (json_decode($row['settings'], true) ?: []) : [];
  } catch (Throwable $e) { $settings = []; }
  $out = [
    'photo_required' => !empty($settings['photo_required']),
    'qr_required' => !empty($settings['qr_required']),
    'geo' => $settings['geo'] ?? null,
  ];
  ApiResponse::success($out);
}

function policySave(PDO $db, array $user){
  $role = $user['role_slug'] ?? '';
  if (!in_array($role, ['super_admin','admin','hr_head'])) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $cid = (int)$user['company_id'];
  try {
    $st = $db->prepare('SELECT settings FROM companies WHERE id = :id');
    $st->execute([':id'=>$cid]);
    $row = $st->fetch();
    $settings = $row && !empty($row['settings']) ? (json_decode($row['settings'], true) ?: []) : [];
  } catch (Throwable $e) { $settings = []; }
  // Merge incoming flags
  if (array_key_exists('photo_required',$in)) $settings['photo_required'] = !!$in['photo_required'];
  if (array_key_exists('qr_required',$in)) $settings['qr_required'] = !!$in['qr_required'];
  if (array_key_exists('geo',$in) && is_array($in['geo'])){
    $g = $in['geo'];
    $settings['geo'] = [
      'lat' => isset($g['lat']) && is_numeric($g['lat']) ? (float)$g['lat'] : null,
      'lng' => isset($g['lng']) && is_numeric($g['lng']) ? (float)$g['lng'] : null,
      'radius_m' => isset($g['radius_m']) && is_numeric($g['radius_m']) ? (int)$g['radius_m'] : 200,
    ];
  }
  try {
    $u = $db->prepare('UPDATE companies SET settings = :s, updated_at = NOW() WHERE id = :id');
    $u->execute([':s'=> json_encode($settings, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE), ':id'=>$cid]);
  } catch (Throwable $e) { ApiResponse::error('Failed to save policy: '.$e->getMessage(), 500); }
  ApiResponse::success(null, 'Attendance policy saved');
}

function statsYear(PDO $db, array $user){
  $year = isset($_GET['year']) && is_numeric($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
  $role = $user['role_slug'] ?? '';
  // HR/Admin: overall stats
  if (in_array($role, ['super_admin','admin','hr_head','hr_officer'])){
    $st = $db->prepare("SELECT 
        SUM(status='clock_in') AS clock_in_total,
        SUM(status='clock_out') AS clock_out_total,
        COUNT(*) AS records
      FROM attendance a JOIN employees e ON a.employee_id=e.id
      WHERE e.company_id=:cid AND YEAR(a.date)=:yr");
    $st->execute([':cid'=>$user['company_id'], ':yr'=>$year]);
    $row = $st->fetch() ?: ['clock_in_total'=>0,'clock_out_total'=>0,'records'=>0];
    ApiResponse::success(['scope'=>'company','year'=>$year] + array_map('intval',$row));
  }
  // Manager: team stats
  if ($role==='manager'){
    $mid = getEmpId($db, $user['id']); if (!$mid) ApiResponse::success(['scope'=>'team','year'=>$year,'clock_in_total'=>0,'clock_out_total'=>0,'records'=>0,'team_members'=>0]);
    $teamCount = (int)singleValuePDO($db, "SELECT COUNT(*) FROM employees WHERE manager_id=:mid", [':mid'=>$mid]);
    $st = $db->prepare("SELECT 
        SUM(a.status='clock_in') AS clock_in_total,
        SUM(a.status='clock_out') AS clock_out_total,
        COUNT(*) AS records
      FROM attendance a JOIN employees e ON a.employee_id=e.id
      WHERE e.manager_id=:mid AND YEAR(a.date)=:yr");
    $st->execute([':mid'=>$mid, ':yr'=>$year]);
    $row = $st->fetch() ?: ['clock_in_total'=>0,'clock_out_total'=>0,'records'=>0];
    ApiResponse::success(['scope'=>'team','year'=>$year,'team_members'=>$teamCount] + array_map('intval',$row));
  }
  // Employee: self stats
  $eid = getEmpId($db, $user['id']); if (!$eid) ApiResponse::success(['scope'=>'self','year'=>$year,'clock_in'=>0,'clock_out'=>0,'records'=>0]);
  $st = $db->prepare("SELECT 
      SUM(status='clock_in') AS clock_in,
      SUM(status='clock_out') AS clock_out,
      COUNT(*) AS records
    FROM attendance WHERE employee_id=:eid AND YEAR(date)=:yr");
  $st->execute([':eid'=>$eid, ':yr'=>$year]);
  $row = $st->fetch() ?: ['clock_in'=>0,'clock_out'=>0,'records'=>0];
  ApiResponse::success(['scope'=>'self','year'=>$year] + array_map('intval',$row));
}

function singleValuePDO(PDO $db, string $sql, array $params){
  try { $st=$db->prepare($sql); foreach($params as $k=>$v){ $st->bindValue($k,$v); } $st->execute(); $row=$st->fetch(); if (!$row) return 0; $vals=array_values($row); return (int)($vals[0]??0);} catch (Throwable $e){ return 0; }
}

function listHistory(PDO $db, array $user){
  $empId = getEmpId($db, $user['id']); if (!$empId) ApiResponse::success([]);
  $st = $db->prepare("SELECT ev.id, ev.event_time AS date, ev.type AS status, ev.address, ev.reason, ev.allowed, ev.approved_status, ev.photo_path
                      FROM attendance_events ev
                      WHERE ev.company_id = :cid AND ev.employee_id = :eid
                      ORDER BY ev.event_time DESC LIMIT 200");
  $st->execute([':cid'=>$user['company_id'], ':eid'=>$empId]);
  ApiResponse::success($st->fetchAll());
}

function clock(PDO $db, array $user){
  $empId = getEmpId($db, $user['id']); if (!$empId) ApiResponse::error('Employee profile not found');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $type = ($in['type'] ?? 'in') === 'out' ? 'out' : 'in';
  $lat = is_numeric($in['lat'] ?? null) ? (float)$in['lat'] : null;
  $lng = is_numeric($in['lng'] ?? null) ? (float)$in['lng'] : null;
  $qr = trim((string)($in['qr'] ?? ''));
  $photoData = $in['photo_data'] ?? null; // data URL or base64
  $siteId = isset($in['site_id']) && $in['site_id'] !== '' ? (int)$in['site_id'] : null;
  $deviceFp = isset($in['device_fp']) ? substr((string)$in['device_fp'], 0, 255) : null;

  // Determine policy and site
  $allowed = true; $reason = 'ok'; $resolvedSite = null; $siteDist = null; $settings = [];
  try {
    $st = $db->prepare('SELECT settings FROM companies WHERE id = :id');
    $st->execute([':id'=>$user['company_id']]); $row = $st->fetch();
    $settings = $row && !empty($row['settings']) ? (json_decode($row['settings'], true) ?: []) : [];
  } catch (Throwable $e) { $settings = []; }

  // Locate nearest active site if not provided
  if ($siteId === null && $lat !== null && $lng !== null){
    $resolvedSite = findNearestSite($db, (int)$user['company_id'], $lat, $lng);
    if ($resolvedSite) { $siteId = (int)$resolvedSite['id']; $siteDist = $resolvedSite['distance_m']; }
  } else if ($siteId !== null) {
    $resolvedSite = getSiteById($db, (int)$user['company_id'], $siteId);
  }

  // Geofence policy: use site geofence if available, else company geo
  if ($lat === null || $lng === null){
    if (!empty($settings['geo']) || $resolvedSite) { $allowed = false; $reason = 'no_location'; }
  } else {
    if ($resolvedSite) {
      $dist = haversine((float)$resolvedSite['center_lat'], (float)$resolvedSite['center_lng'], $lat, $lng);
      $siteDist = $dist;
      if ($dist > (float)$resolvedSite['radius_m']) { $allowed = false; $reason = 'outside_geofence'; }
    } elseif (!empty($settings['geo'])) {
      $g = $settings['geo']; $centerLat = (float)($g['lat'] ?? 0); $centerLng = (float)($g['lng'] ?? 0); $radius = (float)($g['radius_m'] ?? 200);
      if ($centerLat && $centerLng){
        $dist = haversine($centerLat, $centerLng, $lat, $lng);
        if ($dist > $radius) { $allowed = false; $reason = 'outside_geofence'; }
      }
    }
  }

  // QR policy: if site has token, require; else company-wide QR policy
  if ($resolvedSite && !empty($resolvedSite['qr_token'])){
    if ($qr === '' || $qr !== (string)$resolvedSite['qr_token']) { $allowed = false; $reason = 'invalid_qr'; }
  } else if (!empty($settings['qr_required']) && $settings['qr_required']){
    $token = (string)($settings['qr_token'] ?? '');
    if ($token !== '' && $qr !== $token) { $allowed = false; $reason = 'invalid_qr'; }
    if ($token === '' && $qr === '') { $allowed = false; $reason = 'qr_required'; }
  }

  // Photo policy
  if (!empty($settings['photo_required']) && $settings['photo_required'] && empty($photoData)){
    $allowed = false; $reason = 'no_photo';
  }

  // Store photo if provided (data URL)
  $photoPath = null;
  if ($photoData && is_string($photoData)){
    $photoPath = savePhoto($photoData, $user['id']);
  }

  // Upsert by (employee_id, date) to keep one row per day
  $st = $db->prepare('INSERT INTO attendance (employee_id, date, status, location, notes, created_at) VALUES (:eid, CURDATE(), :status, :loc, :notes, NOW())
                      ON DUPLICATE KEY UPDATE status = VALUES(status), location = VALUES(location), notes = VALUES(notes)');
  $status = $type==='in' ? 'clock_in' : 'clock_out';
  $loc = ($lat!==null && $lng!==null)? ($lat.','.$lng) : null;
  $notes = $allowed? null : $reason;
  $st->execute([':eid'=>$empId, ':status'=>$status, ':loc'=>$loc, ':notes'=>$notes]);

  // Reverse geocode to address
  $address = ($lat!==null && $lng!==null) ? reverseGeocode($lat, $lng, $settings) : null;
  $ua = $_SERVER['HTTP_USER_AGENT'] ?? null;
  // Log event
  $ev = $db->prepare('INSERT INTO attendance_events (company_id, employee_id, event_time, type, lat, lng, address, site_id, qr_used, photo_path, device_fp, user_agent, allowed, reason, approved_status, created_at)
                      VALUES (:cid, :eid, NOW(), :type, :lat, :lng, :addr, :sid, :qru, :photo, :fp, :ua, :alw, :rs, :appr, NOW())');
  $ev->execute([
    ':cid'=>$user['company_id'], ':eid'=>$empId, ':type'=>$status,
    ':lat'=>$lat, ':lng'=>$lng, ':addr'=>$address, ':sid'=>$siteId,
    ':qru'=> $qr!=='' ? 1 : 0, ':photo'=>$photoPath, ':fp'=>$deviceFp, ':ua'=>$ua,
    ':alw'=> $allowed ? 1 : 0, ':rs'=>$reason, ':appr'=> $allowed ? 'auto' : 'pending'
  ]);

  ApiResponse::success([
    'allowed'=>$allowed,
    'reason'=>$reason,
    'photo'=>$photoPath,
    'address'=>$address,
    'site'=> $resolvedSite ? ['id'=>$resolvedSite['id'], 'name'=>$resolvedSite['name'], 'distance_m'=>$siteDist] : null
  ], 'Clock recorded');
}

function getEmpId(PDO $db, int $userId){
  try { $q=$db->prepare('SELECT id FROM employees WHERE user_id = :u'); $q->execute([':u'=>$userId]); return (int)($q->fetchColumn() ?: 0); } catch (Throwable $e){ return 0; }
}

function haversine($lat1,$lon1,$lat2,$lon2){
  $R=6371000; // meters
  $phi1=deg2rad($lat1); $phi2=deg2rad($lat2);
  $dphi=deg2rad($lat2-$lat1); $dl=deg2rad($lon2-$lon1);
  $a=sin($dphi/2)*sin($dphi/2)+cos($phi1)*cos($phi2)*sin($dl/2)*sin($dl/2);
  $c=2*atan2(sqrt($a),sqrt(1-$a));
  return $R*$c;
}

function savePhoto($data, $userId){
  // expects data URL or base64 string
  if (strpos($data, 'data:') === 0){ $parts = explode(',', $data, 2); $data = $parts[1] ?? ''; }
  $bin = base64_decode($data, true);
  if (!$bin) return null;
  $dir = realpath(__DIR__ . '/../storage/uploads/attendance_photos');
  if ($dir === false){ @mkdir(__DIR__ . '/../storage/uploads/attendance_photos', 0777, true); $dir = realpath(__DIR__ . '/../storage/uploads/attendance_photos'); }
  if ($dir === false) return null;
  $name = 'att-'.date('Ymd-His').'-u'.$userId.'-'.bin2hex(random_bytes(3)).'.jpg';
  $path = rtrim($dir,'/\\').DIRECTORY_SEPARATOR.$name;
  @file_put_contents($path, $bin);
  return 'storage/uploads/attendance_photos/'.$name;
}

// ===== New helpers & endpoints =====
function ensureAttendanceTables(PDO $db){
  $db->exec("CREATE TABLE IF NOT EXISTS attendance (
    id INT AUTO_INCREMENT PRIMARY KEY,
    employee_id INT NOT NULL,
    date DATE NOT NULL,
    status ENUM('clock_in','clock_out') NOT NULL,
    location VARCHAR(64) NULL,
    notes VARCHAR(255) NULL,
    created_at DATETIME NOT NULL,
    UNIQUE KEY uniq_emp_day (employee_id, date),
    INDEX(employee_id), INDEX(date)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  $db->exec("CREATE TABLE IF NOT EXISTS attendance_sites (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    name VARCHAR(255) NOT NULL,
    code VARCHAR(64) NULL,
    center_lat DECIMAL(10,7) NULL,
    center_lng DECIMAL(10,7) NULL,
    radius_m INT NOT NULL DEFAULT 200,
    qr_token VARCHAR(255) NULL,
    status ENUM('active','inactive') NOT NULL DEFAULT 'active',
    created_by INT NULL,
    created_at DATETIME NOT NULL,
    INDEX(company_id), INDEX(status)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  $db->exec("CREATE TABLE IF NOT EXISTS attendance_events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    employee_id INT NOT NULL,
    event_time DATETIME NOT NULL,
    type ENUM('clock_in','clock_out') NOT NULL,
    lat DECIMAL(10,7) NULL,
    lng DECIMAL(10,7) NULL,
    address VARCHAR(255) NULL,
    site_id INT NULL,
    qr_used TINYINT(1) NOT NULL DEFAULT 0,
    photo_path VARCHAR(255) NULL,
    device_fp VARCHAR(255) NULL,
    user_agent VARCHAR(255) NULL,
    allowed TINYINT(1) NOT NULL DEFAULT 1,
    reason VARCHAR(64) NULL,
    approved_status ENUM('auto','pending','approved','rejected') NOT NULL DEFAULT 'auto',
    approver_id INT NULL,
    decision_at DATETIME NULL,
    notes VARCHAR(255) NULL,
    created_at DATETIME NOT NULL,
    INDEX(company_id), INDEX(employee_id), INDEX(event_time), INDEX(site_id), INDEX(approved_status)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}

function listSites(PDO $db, array $user){
  $params = [':cid'=>$user['company_id']];
  $where = 'company_id = :cid';
  if (isset($_GET['include_inactive']) && $_GET['include_inactive']=='1'){} else { $where .= " AND status='active'"; }
  $st = $db->prepare("SELECT id, name, code, center_lat, center_lng, radius_m, status FROM attendance_sites WHERE $where ORDER BY name");
  $st->execute($params);
  ApiResponse::success($st->fetchAll());
}

function siteSave(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0);
  $name = trim((string)($in['name'] ?? '')); if ($name==='') ApiResponse::error('name required');
  $code = trim((string)($in['code'] ?? '')) ?: null;
  $lat = isset($in['center_lat']) && is_numeric($in['center_lat']) ? (float)$in['center_lat'] : null;
  $lng = isset($in['center_lng']) && is_numeric($in['center_lng']) ? (float)$in['center_lng'] : null;
  $radius = isset($in['radius_m']) && is_numeric($in['radius_m']) ? (int)$in['radius_m'] : 200;
  $qr = isset($in['qr_token']) && $in['qr_token']!=='' ? substr((string)$in['qr_token'],0,255) : null;
  if ($id>0){
    $st = $db->prepare("UPDATE attendance_sites SET name=:n, code=:c, center_lat=:lat, center_lng=:lng, radius_m=:r, qr_token=:qr WHERE id=:id AND company_id=:cid");
    $st->execute([':n'=>$name, ':c'=>$code, ':lat'=>$lat, ':lng'=>$lng, ':r'=>$radius, ':qr'=>$qr, ':id'=>$id, ':cid'=>$user['company_id']]);
    ApiResponse::success(null, 'Updated');
  } else {
    $st = $db->prepare("INSERT INTO attendance_sites (company_id, name, code, center_lat, center_lng, radius_m, qr_token, status, created_by, created_at)
                        VALUES (:cid,:n,:c,:lat,:lng,:r,:qr,'active',:uid,NOW())");
    $st->execute([':cid'=>$user['company_id'], ':n'=>$name, ':c'=>$code, ':lat'=>$lat, ':lng'=>$lng, ':r'=>$radius, ':qr'=>$qr, ':uid'=>$user['id']]);
    ApiResponse::success(['id'=>$db->lastInsertId()], 'Created');
  }
}

function siteToggle(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  $st = $db->prepare("UPDATE attendance_sites SET status = IF(status='active','inactive','active') WHERE id=:id AND company_id=:cid");
  $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
  ApiResponse::success(null, 'Toggled');
}

function listEvents(PDO $db, array $user){
  $companyId = (int)$user['company_id'];
  $team = isset($_GET['team']) && $_GET['team']=='1';
  $limit = isset($_GET['limit']) && is_numeric($_GET['limit']) ? max(1, min((int)$_GET['limit'], 500)) : 200;
  $start = $_GET['start'] ?? null; $end = $_GET['end'] ?? null;
  $params = [':cid'=>$companyId];
  $where = 'ev.company_id = :cid';
  $role = $user['role_slug'] ?? '';
  if (in_array($role, ['super_admin','admin','hr_head','hr_officer'])){
    // see all
  } else {
    $empId = getEmpId($db, $user['id']); if (!$empId) ApiResponse::success([]);
    if ($team && $role==='manager'){
      $where .= ' AND (emp.manager_id = :mid OR ev.employee_id = :eid)'; $params[':mid']=$empId; $params[':eid']=$empId;
    } else {
      $where .= ' AND ev.employee_id = :eid'; $params[':eid']=$empId;
    }
  }
  if ($start){ $where .= ' AND ev.event_time >= :start'; $params[':start']=$start; }
  if ($end){ $where .= ' AND ev.event_time <= :end'; $params[':end']=$end; }
  $sql = "SELECT ev.*, CONCAT(emp.first_name,' ',emp.last_name) AS employee_name, emp.employee_number, s.name AS site_name
          FROM attendance_events ev
          JOIN employees emp ON ev.employee_id = emp.id
          LEFT JOIN attendance_sites s ON ev.site_id = s.id
          WHERE $where
          ORDER BY ev.event_time DESC
          LIMIT $limit";
  $st = $db->prepare($sql); foreach($params as $k=>$v){ $st->bindValue($k,$v); } $st->execute();
  ApiResponse::success($st->fetchAll());
}

function listExceptions(PDO $db, array $user){
  $companyId = (int)$user['company_id'];
  $params = [':cid'=>$companyId];
  $where = "ev.company_id = :cid AND ev.allowed = 0 AND ev.approved_status = 'pending'";
  $role = $user['role_slug'] ?? '';
  if (in_array($role, ['super_admin','admin','hr_head','hr_officer'])){
    // see all pending exceptions
  } elseif ($role === 'manager'){
    $empId = getEmpId($db, $user['id']); if (!$empId) ApiResponse::success([]);
    $where .= ' AND e.manager_id = :mid'; $params[':mid']=$empId;
  } else {
    // employee sees own exceptions
    $empId = getEmpId($db, $user['id']); if (!$empId) ApiResponse::success([]);
    $where .= ' AND ev.employee_id = :eid'; $params[':eid']=$empId;
  }
  $sql = "SELECT ev.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name, e.employee_number, s.name AS site_name
          FROM attendance_events ev
          JOIN employees e ON ev.employee_id = e.id
          LEFT JOIN attendance_sites s ON ev.site_id = s.id
          WHERE $where ORDER BY ev.event_time DESC LIMIT 300";
  $st = $db->prepare($sql); foreach($params as $k=>$v){ $st->bindValue($k,$v); } $st->execute();
  ApiResponse::success($st->fetchAll());
}

function exceptionDecide(PDO $db, array $user){
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['event_id'] ?? 0); if ($id<=0) ApiResponse::error('event_id required');
  $decision = ($in['decision'] ?? '') === 'approve' ? 'approved' : 'rejected';
  $note = isset($in['note']) ? substr((string)$in['note'],0,255) : null;
  // Load event and check permission
  $st = $db->prepare('SELECT ev.*, e.manager_id FROM attendance_events ev JOIN employees e ON ev.employee_id = e.id WHERE ev.id = :id AND ev.approved_status = \''.'pending'.'\'' );
  $st->execute([':id'=>$id]);
  if ($st->rowCount()===0) ApiResponse::notFound('Event not found or already decided');
  $ev = $st->fetch();
  $role = $user['role_slug'] ?? '';
  $empId = getEmpId($db, $user['id']);
  $can = in_array($role, ['super_admin','admin','hr_head']) || ($role==='manager' && $empId && (int)$ev['manager_id'] === (int)$empId);
  if (!$can) ApiResponse::forbidden('Not allowed');
  $u = $db->prepare("UPDATE attendance_events SET approved_status = :st, approver_id = :uid, decision_at = NOW(), notes = :note WHERE id = :id");
  $u->execute([':st'=>$decision, ':uid'=>$user['id'], ':note'=>$note, ':id'=>$id]);
  ApiResponse::success(null, 'Updated');
}

function getSiteById(PDO $db, int $companyId, int $siteId){
  $st = $db->prepare("SELECT * FROM attendance_sites WHERE id = :id AND company_id = :cid AND status='active'");
  $st->execute([':id'=>$siteId, ':cid'=>$companyId]);
  return $st->fetch() ?: null;
}

function findNearestSite(PDO $db, int $companyId, float $lat, float $lng){
  $st = $db->prepare("SELECT id, name, center_lat, center_lng, radius_m FROM attendance_sites WHERE company_id = :cid AND status='active'");
  $st->execute([':cid'=>$companyId]);
  $best = null; $bestDist = null;
  while ($row = $st->fetch()){
    if ($row['center_lat']===null || $row['center_lng']===null) continue;
    $d = haversine((float)$row['center_lat'], (float)$row['center_lng'], $lat, $lng);
    if ($best===null || $d < $bestDist){ $best = $row; $bestDist = $d; }
  }
  if ($best){ $best['distance_m'] = $bestDist; }
  return $best;
}

function reverseGeocode(float $lat, float $lng, array $settings){
  // Prefer Google if API key provided
  $key = $settings['google_maps_api_key'] ?? null;
  if ($key){
    $url = 'https://maps.googleapis.com/maps/api/geocode/json?latlng='.urlencode($lat.','.$lng).'&key='.urlencode($key);
    try { $json = @file_get_contents($url); if ($json){ $obj = json_decode($json,true); $addr = $obj['results'][0]['formatted_address'] ?? null; if ($addr) return $addr; } } catch (Throwable $e) {}
  }
  // Fallback to Nominatim (OpenStreetMap)
  $url = 'https://nominatim.openstreetmap.org/reverse?format=json&lat='.urlencode((string)$lat).'&lon='.urlencode((string)$lng).'&zoom=18&addressdetails=1';
  $ctx = stream_context_create(['http'=>['header'=>"User-Agent: SmartQuantumHR/1.0\r\n"]]);
  try { $json = @file_get_contents($url, false, $ctx); if ($json){ $obj = json_decode($json,true); $addr = $obj['display_name'] ?? null; if ($addr) return $addr; } } catch (Throwable $e) {}
  return $lat.','.$lng;
}
