<?php
/**
 * Appraisal System API with Line Manager Workflow
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? null;

$database = new Database();
$db = $database->getConnection();

switch ($method) {
    case 'GET':
        // Ensure tables exist before any reads
        ensureAppraisalTables($db);
        if ($action === 'templates') {
            ensureAppraisalTables($db);
            getAppraisalTemplates($db);
        } elseif ($action === 'kpis' && $id) {
            ensureAppraisalTables($db);
            getAppraisalKpis($db, $id);
        } elseif ($id) {
            getAppraisal($db, $id);
        } else {
            getAppraisals($db);
        }
        break;
    
    case 'POST':
        if ($action === 'submit') {
            ensureAppraisalTables($db);
            submitAppraisal($db, $id);
        } elseif ($action === 'review') {
            ensureAppraisalTables($db);
            reviewAppraisal($db, $id);
        } elseif ($action === 'approve') {
            approveAppraisal($db, $id);
        } elseif ($action === 'template_create') {
            ensureAppraisalTables($db); templateCreate($db);
        } elseif ($action === 'template_update') {
            ensureAppraisalTables($db); templateUpdate($db);
        } elseif ($action === 'template_delete') {
            ensureAppraisalTables($db); templateDelete($db);
        } elseif ($action === 'kpi_create') {
            ensureAppraisalTables($db); kpiCreate($db);
        } elseif ($action === 'kpi_update') {
            ensureAppraisalTables($db); kpiUpdate($db);
        } elseif ($action === 'kpi_delete') {
            ensureAppraisalTables($db); kpiDelete($db);
        } else {
            ensureAppraisalTables($db); createAppraisal($db);
        }
        break;
    
    case 'PUT':
        updateAppraisal($db, $id);
        break;
    
    case 'DELETE':
        deleteAppraisal($db, $id);
        break;
    
    default:
        ApiResponse::error('Method not allowed', 405);
}
function getAppraisals($db) {
    $user = getCurrentUser();
    $role = $user['role_slug'];
    
    $query = "SELECT 
                a.*,
                CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                e.employee_number,
                d.name as department_name,
                p.title as position_title,
                CONCAT(reviewer.first_name, ' ', reviewer.last_name) as reviewer_name,
                t.name AS template_name
              FROM appraisals a
              JOIN employees e ON a.employee_id = e.id
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN positions p ON e.position_id = p.id
              LEFT JOIN employees reviewer ON a.reviewer_id = reviewer.id
              LEFT JOIN appraisal_meta am ON am.appraisal_id = a.id AND am.meta_key = 'template_id'
              LEFT JOIN appraisal_templates t ON t.id = am.meta_value
              WHERE e.company_id = :company_id";
    
    $params = [':company_id' => $user['company_id']];
    
    // Apply role-based filtering
    if ($role === 'employee') {
        // Employees can see their own appraisals
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        
        if ($empStmt->rowCount() > 0) {
            $emp = $empStmt->fetch();
            $query .= " AND a.employee_id = :employee_id";
            $params[':employee_id'] = $emp['id'];
        } else {
            ApiResponse::success([]);
        }
    } elseif ($role === 'manager') {
        // Managers can see appraisals for their direct reports and their own
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        
        if ($empStmt->rowCount() > 0) {
            $manager = $empStmt->fetch();
            $query .= " AND (e.manager_id = :manager_id OR a.employee_id = :employee_id OR a.reviewer_id = :reviewer_id)";
            $params[':manager_id'] = $manager['id'];
            $params[':employee_id'] = $manager['id'];
            $params[':reviewer_id'] = $manager['id'];
        }
    }
    // HR roles can see all appraisals (no additional filtering)
    
    // Apply additional filters
    if (isset($_GET['status']) && !empty($_GET['status'])) {
        $query .= " AND a.status = :status";
        $params[':status'] = $_GET['status'];
    }
    
    if (isset($_GET['period_year']) && !empty($_GET['period_year'])) {
        $query .= " AND YEAR(a.period_start) = :period_year";
        $params[':period_year'] = $_GET['period_year'];
    }
    
    $query .= " ORDER BY a.created_at DESC";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();
    
    $appraisals = $stmt->fetchAll();
    ApiResponse::success($appraisals);
}

// Ensure core appraisals table and related template tables exist
function ensureAppraisalTables(PDO $db){
    try {
        // Core appraisals table
        $db->exec("CREATE TABLE IF NOT EXISTS appraisals (
            id INT AUTO_INCREMENT PRIMARY KEY,
            employee_id INT NOT NULL,
            reviewer_id INT NOT NULL,
            period_start DATE NOT NULL,
            period_end DATE NOT NULL,
            goals_achievement TEXT NULL,
            strengths TEXT NULL,
            areas_for_improvement TEXT NULL,
            development_plan TEXT NULL,
            comments TEXT NULL,
            overall_rating VARCHAR(50) NULL,
            status ENUM('draft','submitted','reviewed','approved','rejected') NOT NULL DEFAULT 'draft',
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            submitted_at TIMESTAMP NULL DEFAULT NULL,
            reviewed_at TIMESTAMP NULL DEFAULT NULL,
            approved_at TIMESTAMP NULL DEFAULT NULL,
            KEY idx_employee (employee_id),
            KEY idx_reviewer (reviewer_id),
            KEY idx_period (period_start, period_end),
            KEY idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
    } catch (Throwable $e) { /* ignore */ }
    // ensure dependent tables
    ensureTemplateTables($db);
}

function createAppraisal($db) {
    $user = getCurrentUser();
    
    $role = $user['role_slug'];
    $input = json_decode(file_get_contents('php://input'), true);
    
    try {
        ensureTemplateTables($db);
        $employeeId = null;
        $reviewerId = null;
        $periodStart = $input['period_start'] ?? null;
        $periodEnd = $input['period_end'] ?? null;
        $templateId = isset($input['template_id']) ? (int)$input['template_id'] : null;
        
        if (in_array($role, ['employee', 'manager'])) {
            // Self-setup: resolve employee by current user
            $empQ = $db->prepare("SELECT id, manager_id, department_id FROM employees WHERE user_id = :uid");
            $empQ->execute([':uid' => $user['id']]);
            if ($empQ->rowCount() === 0) {
                ApiResponse::error('Employee profile not found for current user');
            }
            $emp = $empQ->fetch();
            $employeeId = (int)$emp['id'];
            // Determine reviewer: prefer manager_id; fallback to department head
            if (!empty($emp['manager_id'])) {
                $reviewerId = (int)$emp['manager_id'];
            } else if (!empty($emp['department_id'])) {
                $dh = $db->prepare("SELECT head_id FROM departments WHERE id = :did");
                $dh->execute([':did' => $emp['department_id']]);
                $row = $dh->fetch();
                if ($row && !empty($row['head_id'])) $reviewerId = (int)$row['head_id'];
            }
            if (!$reviewerId) {
                ApiResponse::error('No reviewer configured (manager/department head). Please contact HR.');
            }
            if (!$periodStart || !$periodEnd) {
                ApiResponse::error("Fields 'period_start' and 'period_end' are required");
            }
        } else {
            // HR roles: require all fields as before
            if (!in_array($role, ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
                ApiResponse::forbidden('Insufficient permissions');
            }
            $req = ['employee_id','reviewer_id','period_start','period_end'];
            foreach ($req as $f) {
                if (empty($input[$f])) ApiResponse::error("Field '$f' is required");
            }
            $employeeId = (int)$input['employee_id'];
            $reviewerId = (int)$input['reviewer_id'];
            $periodStart = $input['period_start'];
            $periodEnd = $input['period_end'];
        }
        
        // Check duplicate period for employee
        $checkQuery = "SELECT id FROM appraisals 
                       WHERE employee_id = :employee_id 
                       AND period_start = :period_start 
                       AND period_end = :period_end";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':employee_id', $employeeId);
        $checkStmt->bindParam(':period_start', $periodStart);
        $checkStmt->bindParam(':period_end', $periodEnd);
        $checkStmt->execute();
        if ($checkStmt->rowCount() > 0) {
            ApiResponse::error('Appraisal already exists for this period');
        }
        
        // Create appraisal
        $query = "INSERT INTO appraisals (
                    employee_id, reviewer_id, period_start, period_end,
                    goals_achievement, strengths, areas_for_improvement,
                    development_plan, comments, status, created_at
                  ) VALUES (
                    :employee_id, :reviewer_id, :period_start, :period_end,
                    :goals_achievement, :strengths, :areas_for_improvement,
                    :development_plan, :comments, 'draft', NOW()
                  )";
        $stmt = $db->prepare($query);
        $stmt->bindValue(':employee_id', $employeeId, PDO::PARAM_INT);
        $stmt->bindValue(':reviewer_id', $reviewerId, PDO::PARAM_INT);
        $stmt->bindValue(':period_start', $periodStart);
        $stmt->bindValue(':period_end', $periodEnd);
        $stmt->bindValue(':goals_achievement', $input['goals_achievement'] ?? null);
        $stmt->bindValue(':strengths', $input['strengths'] ?? null);
        $stmt->bindValue(':areas_for_improvement', $input['areas_for_improvement'] ?? null);
        $stmt->bindValue(':development_plan', $input['development_plan'] ?? null);
        $stmt->bindValue(':comments', $input['comments'] ?? null);
        $stmt->execute();
        $appraisalId = $db->lastInsertId();

        // If a template was selected, record it and pre-create KPI score rows
        if ($templateId) {
            // store meta
            $db->prepare("INSERT INTO appraisal_meta (appraisal_id, meta_key, meta_value) VALUES (:aid,'template_id',:tid)")
               ->execute([':aid'=>$appraisalId, ':tid'=>$templateId]);
            // copy template KPIs
            $kp = $db->prepare("SELECT id FROM appraisal_template_kpis WHERE template_id = :tid");
            $kp->execute([':tid'=>$templateId]);
            while ($row = $kp->fetch()) {
                $db->prepare("INSERT IGNORE INTO appraisal_kpi_scores (appraisal_id, kpi_id) VALUES (:aid, :kid)")
                   ->execute([':aid'=>$appraisalId, ':kid'=>$row['id']]);
            }
        }
        
        createAppraisalNotification($db, $appraisalId, 'appraisal_created');
        $auth = new Auth();
        $auth->logActivity($user['id'], 'appraisal_created', ['appraisal_id' => $appraisalId]);
        ApiResponse::success(['id' => $appraisalId], 'Appraisal created successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to create appraisal: ' . $e->getMessage());
    }
}

function submitAppraisal($db, $id) {
    $user = getCurrentUser();
    
    if (!$id) {
        ApiResponse::error('Appraisal ID required');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    try {
        ensureTemplateTables($db);
        // Check if user can submit this appraisal (employee or reviewer)
        $checkQuery = "SELECT a.*, e.user_id as employee_user_id, r.user_id as reviewer_user_id
                       FROM appraisals a
                       JOIN employees e ON a.employee_id = e.id
                       JOIN employees r ON a.reviewer_id = r.id
                       WHERE a.id = :id AND a.status = 'draft'";
        
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Appraisal not found or already submitted');
        }
        
        $appraisal = $checkStmt->fetch();
        
        // Check if user is authorized to submit
        if ($user['id'] !== $appraisal['employee_user_id'] && $user['id'] !== $appraisal['reviewer_user_id']) {
            ApiResponse::forbidden('Not authorized to submit this appraisal');
        }
        
        // Update appraisal
        $updateFields = [];
        $params = [':id' => $id];
        
        $allowedFields = [
            'goals_achievement', 'strengths', 'areas_for_improvement',
            'development_plan', 'comments', 'overall_rating'
        ];
        
        foreach ($allowedFields as $field) {
            if (isset($input[$field])) {
                $updateFields[] = "$field = :$field";
                $params[":$field"] = $input[$field];
            }
        }

        // Persist KPI scores if provided
        if (!empty($input['kpi_scores']) && is_array($input['kpi_scores'])) {
            foreach ($input['kpi_scores'] as $kpiId => $score) {
                $kpiId = (int)$kpiId; $score = is_numeric($score) ? $score : null;
                if ($kpiId && $score !== null) {
                    $st = $db->prepare("INSERT INTO appraisal_kpi_scores (appraisal_id, kpi_id, employee_score) VALUES (:aid,:kid,:sc)
                                         ON DUPLICATE KEY UPDATE employee_score = VALUES(employee_score)");
                    $st->execute([':aid'=>$id, ':kid'=>$kpiId, ':sc'=>$score]);
                }
            }
        }
        
        $updateFields[] = "status = 'submitted'";
        $updateFields[] = "submitted_at = NOW()";
        $updateFields[] = "updated_at = NOW()";
        
        $query = "UPDATE appraisals SET " . implode(', ', $updateFields) . " WHERE id = :id";
        $stmt = $db->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        
        // Create notification for line manager
        createAppraisalNotification($db, $id, 'appraisal_submitted');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'appraisal_submitted', ['appraisal_id' => $id]);
        
        ApiResponse::success(null, 'Appraisal submitted successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to submit appraisal: ' . $e->getMessage());
    }
}

function reviewAppraisal($db, $id) {
    $user = getCurrentUser();
    
    if (!$id) {
        ApiResponse::error('Appraisal ID required');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    try {
        // Check if user can review this appraisal (line manager or department head or assigned reviewer)
        $checkQuery = "SELECT a.*, e.manager_id, e.department_id, d.head_id AS dept_head_id, r.user_id as reviewer_user_id
                       FROM appraisals a
                       JOIN employees e ON a.employee_id = e.id
                       LEFT JOIN departments d ON e.department_id = d.id
                       LEFT JOIN employees r ON a.reviewer_id = r.id
                       WHERE a.id = :id AND a.status = 'submitted'";
        
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Appraisal not found or not ready for review');
        }
        
        $appraisal = $checkStmt->fetch();
        
        // Resolve current user's employee id
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        $currentEmpId = $empStmt->fetchColumn();
        
        $canReview = false;
        if ($currentEmpId) {
            $canReview = ($currentEmpId == $appraisal['manager_id']) ||
                         ($currentEmpId == $appraisal['dept_head_id']);
        }
        // Also allow if reviewer user matches or HR approvers
        if (!$canReview) {
            $canReview = ($user['id'] == ($appraisal['reviewer_user_id'] ?? 0)) || in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head']);
        }
        
        if (!$canReview) {
            ApiResponse::forbidden('Not authorized to review this appraisal');
        }
        
        // Persist manager KPI scores if provided
        if (!empty($input['manager_scores']) && is_array($input['manager_scores'])) {
            foreach ($input['manager_scores'] as $kpiId => $score) {
                $kpiId = (int)$kpiId; $score = is_numeric($score) ? $score : null;
                if ($kpiId && $score !== null) {
                    $st = $db->prepare("INSERT INTO appraisal_kpi_scores (appraisal_id, kpi_id, manager_score) VALUES (:aid,:kid,:sc)
                                         ON DUPLICATE KEY UPDATE manager_score = VALUES(manager_score)");
                    $st->execute([':aid'=>$id, ':kid'=>$kpiId, ':sc'=>$score]);
                }
            }
        }

        // Update appraisal with manager's review
        $query = "UPDATE appraisals SET 
                    status = 'reviewed',
                    reviewed_at = NOW(),
                    updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        // Create notification for HR
        createAppraisalNotification($db, $id, 'appraisal_reviewed');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'appraisal_reviewed', ['appraisal_id' => $id]);
        
        ApiResponse::success(null, 'Appraisal reviewed successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to review appraisal: ' . $e->getMessage());
    }
}

function approveAppraisal($db, $id) {
    $user = getCurrentUser();
    
    // Check if user can approve appraisals (HR only)
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head'])) {
        ApiResponse::forbidden('Insufficient permissions to approve appraisals');
    }
    
    if (!$id) {
        ApiResponse::error('Appraisal ID required');
    }
    
    try {
        // Check if appraisal exists and is reviewed
        $checkQuery = "SELECT a.*, e.company_id 
                       FROM appraisals a 
                       JOIN employees e ON a.employee_id = e.id 
                       WHERE a.id = :id AND a.status = 'reviewed' AND e.company_id = :company_id";
        
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id);
        $checkStmt->bindParam(':company_id', $user['company_id']);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Appraisal not found or not ready for approval');
        }
        
        // Update appraisal status
        $query = "UPDATE appraisals SET 
                    status = 'approved',
                    approved_at = NOW(),
                    updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        // Create announcement for completion
        createAppraisalAnnouncement($db, $id);
        
        // Create notification for employee
        createAppraisalNotification($db, $id, 'appraisal_approved');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'appraisal_approved', ['appraisal_id' => $id]);
        
        ApiResponse::success(null, 'Appraisal approved successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to approve appraisal: ' . $e->getMessage());
    }
}

function createAppraisalNotification($db, $appraisalId, $type) {
    // Get appraisal details
    $query = "SELECT a.*, 
                     CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                     e.user_id as employee_user_id, e.manager_id,
                     CONCAT(r.first_name, ' ', r.last_name) as reviewer_name,
                     r.user_id as reviewer_user_id
              FROM appraisals a
              JOIN employees e ON a.employee_id = e.id
              JOIN employees r ON a.reviewer_id = r.id
              WHERE a.id = :appraisal_id";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':appraisal_id', $appraisalId);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $appraisal = $stmt->fetch();
        
        $notifications = [];
        
        switch ($type) {
            case 'appraisal_created':
                // Notify employee and reviewer
                $notifications[] = [
                    'user_id' => $appraisal['employee_user_id'],
                    'title' => 'New Appraisal Created',
                    'content' => "Your performance appraisal for the period has been created."
                ];
                
                if ($appraisal['reviewer_user_id'] !== $appraisal['employee_user_id']) {
                    $notifications[] = [
                        'user_id' => $appraisal['reviewer_user_id'],
                        'title' => 'Appraisal Assignment',
                        'content' => "You have been assigned to review {$appraisal['employee_name']}'s performance appraisal."
                    ];
                }
                break;
                
            case 'appraisal_submitted':
                // Notify manager and HR
                if ($appraisal['manager_id']) {
                    $managerQuery = "SELECT user_id FROM employees WHERE id = :manager_id";
                    $managerStmt = $db->prepare($managerQuery);
                    $managerStmt->bindParam(':manager_id', $appraisal['manager_id']);
                    $managerStmt->execute();
                    
                    if ($managerStmt->rowCount() > 0) {
                        $manager = $managerStmt->fetch();
                        $notifications[] = [
                            'user_id' => $manager['user_id'],
                            'title' => 'Appraisal Ready for Review',
                            'content' => "{$appraisal['employee_name']}'s appraisal is ready for your review."
                        ];
                    }
                }
                break;
                
            case 'appraisal_reviewed':
                // Notify HR
                $hrQuery = "SELECT u.id FROM users u 
                           JOIN user_roles ur ON u.id = ur.user_id 
                           JOIN roles r ON ur.role_id = r.id 
                           WHERE r.slug IN ('hr_head', 'admin', 'super_admin') 
                           AND u.company_id = :company_id";
                $hrStmt = $db->prepare($hrQuery);
                $hrStmt->bindParam(':company_id', $appraisal['company_id'] ?? 1);
                $hrStmt->execute();
                
                while ($hr = $hrStmt->fetch()) {
                    $notifications[] = [
                        'user_id' => $hr['id'],
                        'title' => 'Appraisal Ready for Approval',
                        'content' => "{$appraisal['employee_name']}'s appraisal has been reviewed and is ready for final approval."
                    ];
                }
                break;
                
            case 'appraisal_approved':
                // Notify employee
                $notifications[] = [
                    'user_id' => $appraisal['employee_user_id'],
                    'title' => 'Appraisal Completed',
                    'content' => "Your performance appraisal has been completed and approved."
                ];
                break;
        }
        
        // Insert notifications
        foreach ($notifications as $notification) {
            $notifQuery = "INSERT INTO notifications (user_id, type, title, content, created_at) 
                           VALUES (:user_id, :type, :title, :content, NOW())";
            $notifStmt = $db->prepare($notifQuery);
            $notifStmt->bindParam(':user_id', $notification['user_id']);
            $notifStmt->bindParam(':type', $type);
            $notifStmt->bindParam(':title', $notification['title']);
            $notifStmt->bindParam(':content', $notification['content']);
            $notifStmt->execute();
        }
    }
}

function createAppraisalAnnouncement($db, $appraisalId) {
    // Get appraisal details
    $query = "SELECT a.*, 
                     CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                     e.company_id,
                     d.name as department_name
              FROM appraisals a
              JOIN employees e ON a.employee_id = e.id
              LEFT JOIN departments d ON e.department_id = d.id
              WHERE a.id = :appraisal_id";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':appraisal_id', $appraisalId);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $appraisal = $stmt->fetch();
        
        // Create announcement
        $announcementQuery = "INSERT INTO announcements (
                                company_id, title, content, priority, status,
                                published_at, author_id, created_at
                              ) VALUES (
                                :company_id, :title, :content, 'medium', 'published',
                                NOW(), :author_id, NOW()
                              )";
        
        $announcementStmt = $db->prepare($announcementQuery);
        $title = "Performance Appraisal Completed";
        $content = "Performance appraisal for {$appraisal['employee_name']} ({$appraisal['department_name']}) has been successfully completed for the period {$appraisal['period_start']} to {$appraisal['period_end']}.";
        
        $announcementStmt->bindParam(':company_id', $appraisal['company_id']);
        $announcementStmt->bindParam(':title', $title);
        $announcementStmt->bindParam(':content', $content);
        $announcementStmt->bindParam(':author_id', 1); // System user
        $announcementStmt->execute();
    }
}

// Ensure schema for templates/KPIs/scores/meta exists
function ensureTemplateTables(PDO $db){
    try {
        $db->exec("CREATE TABLE IF NOT EXISTS appraisal_templates (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            name VARCHAR(255) NOT NULL,
            department_id INT NULL,
            created_by INT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            KEY idx_company (company_id),
            KEY idx_dept (department_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
        $db->exec("CREATE TABLE IF NOT EXISTS appraisal_template_kpis (
            id INT AUTO_INCREMENT PRIMARY KEY,
            template_id INT NOT NULL,
            title VARCHAR(255) NOT NULL,
            weight DECIMAL(6,2) NOT NULL DEFAULT 0,
            KEY idx_tpl (template_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
        $db->exec("CREATE TABLE IF NOT EXISTS appraisal_kpi_scores (
            appraisal_id INT NOT NULL,
            kpi_id INT NOT NULL,
            employee_score DECIMAL(6,2) NULL,
            manager_score DECIMAL(6,2) NULL,
            PRIMARY KEY (appraisal_id, kpi_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
        $db->exec("CREATE TABLE IF NOT EXISTS appraisal_meta (
            id INT AUTO_INCREMENT PRIMARY KEY,
            appraisal_id INT NOT NULL,
            meta_key VARCHAR(64) NOT NULL,
            meta_value VARCHAR(255) NULL,
            KEY idx_app_meta (appraisal_id, meta_key)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
    } catch (Throwable $e) { /* ignore */ }
}

// Single appraisal fetch (with names)
function getAppraisal(PDO $db, $id){
    $user = getCurrentUser();
    $q = "SELECT a.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name,
                 CONCAT(r.first_name,' ',r.last_name) AS reviewer_name
          FROM appraisals a
          JOIN employees e ON e.id = a.employee_id
          JOIN employees r ON r.id = a.reviewer_id
          WHERE a.id = :id AND e.company_id = :cid";
    $st = $db->prepare($q);
    $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
    $row = $st->fetch();
    if (!$row) ApiResponse::notFound('Appraisal not found');
    ApiResponse::success($row);
}

// Templates with nested KPIs for current company
function getAppraisalTemplates(PDO $db){
    $user = getCurrentUser();
    $tpl = $db->prepare("SELECT t.*, d.name AS department_name
                         FROM appraisal_templates t
                         LEFT JOIN departments d ON d.id = t.department_id
                         WHERE t.company_id = :cid
                         ORDER BY t.created_at DESC");
    $tpl->execute([':cid'=>$user['company_id']]);
    $templates = $tpl->fetchAll();
    if (!$templates) { ApiResponse::success([]); return; }
    $ids = array_map(fn($r)=> (int)$r['id'], $templates);
    $in = implode(',', array_fill(0, count($ids), '?'));
    $kpisByTpl = [];
    if ($ids){
        $kp = $db->prepare("SELECT id, template_id, title, weight FROM appraisal_template_kpis WHERE template_id IN ($in) ORDER BY id ASC");
        $kp->execute($ids);
        while ($r = $kp->fetch()) { $kpisByTpl[(int)$r['template_id']][] = $r; }
    }
    foreach ($templates as &$t){ $t['kpis'] = $kpisByTpl[(int)$t['id']] ?? []; }
    ApiResponse::success($templates);
}

// KPIs for an appraisal, including current scores
function getAppraisalKpis(PDO $db, $appraisalId){
    // find template assigned to this appraisal
    $tplId = $db->prepare("SELECT meta_value FROM appraisal_meta WHERE appraisal_id = :aid AND meta_key = 'template_id' LIMIT 1");
    $tplId->execute([':aid'=>$appraisalId]);
    $tid = (int)($tplId->fetchColumn() ?: 0);
    if ($tid <= 0){ ApiResponse::success([]); return; }
    $q = $db->prepare("SELECT k.id AS kpi_id, k.title, k.weight,
                              s.employee_score, s.manager_score
                       FROM appraisal_template_kpis k
                       LEFT JOIN appraisal_kpi_scores s ON s.kpi_id = k.id AND s.appraisal_id = :aid
                       WHERE k.template_id = :tid
                       ORDER BY k.id ASC");
    $q->execute([':aid'=>$appraisalId, ':tid'=>$tid]);
    ApiResponse::success($q->fetchAll());
}

// ================= Template/KPI Management (CRUD) =================
function ensureHrRole(array $user){
    if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
}

function templateCreate(PDO $db){
    $user = getCurrentUser(); ensureHrRole($user); ensureTemplateTables($db);
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $name = trim((string)($in['name'] ?? ''));
    $dept = isset($in['department_id']) && $in['department_id'] !== '' ? (int)$in['department_id'] : null;
    if ($name === '') ApiResponse::error('Name is required');
    $st = $db->prepare('INSERT INTO appraisal_templates (company_id, name, department_id, created_by, created_at) VALUES (:cid,:name,:dept,:uid,NOW())');
    $st->bindValue(':cid', $user['company_id'], PDO::PARAM_INT);
    $st->bindValue(':name', $name);
    $st->bindValue(':dept', $dept);
    $st->bindValue(':uid', $user['id'], PDO::PARAM_INT);
    $st->execute();
    ApiResponse::success(['id'=>$db->lastInsertId()], 'Template created');
}

function templateUpdate(PDO $db){
    $user = getCurrentUser(); ensureHrRole($user); ensureTemplateTables($db);
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
    $name = isset($in['name']) ? trim((string)$in['name']) : null;
    $dept = array_key_exists('department_id',$in) ? ($in['department_id'] !== '' ? (int)$in['department_id'] : null) : null;
    $set=[]; $params=[':id'=>$id, ':cid'=>$user['company_id']];
    if ($name !== null){ $set[]='name = :name'; $params[':name']=$name; }
    if ($dept !== null){ $set[]='department_id = :dept'; $params[':dept']=$dept; }
    if (!$set) ApiResponse::success(null,'No changes');
    $sql = 'UPDATE appraisal_templates SET '.implode(', ',$set).' WHERE id = :id AND company_id = :cid';
    $st = $db->prepare($sql); foreach($params as $k=>$v){ $st->bindValue($k,$v); } $st->execute();
    ApiResponse::success(null,'Updated');
}

function templateDelete(PDO $db){
    $user = getCurrentUser(); ensureHrRole($user); ensureTemplateTables($db);
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
    // Check if any KPI is referenced by existing appraisal scores
    $check = $db->prepare('SELECT COUNT(*) AS cnt FROM appraisal_kpi_scores WHERE kpi_id IN (SELECT id FROM appraisal_template_kpis WHERE template_id = :tid)');
    $check->execute([':tid'=>$id]);
    $row = $check->fetch();
    if (($row['cnt'] ?? 0) > 0) ApiResponse::error('Cannot delete: KPIs from this template are used in appraisals');
    // Delete KPIs then template
    $db->prepare('DELETE FROM appraisal_template_kpis WHERE template_id = :tid')->execute([':tid'=>$id]);
    $st = $db->prepare('DELETE FROM appraisal_templates WHERE id = :id AND company_id = :cid');
    $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
    ApiResponse::success(null,'Deleted');
}

function kpiCreate(PDO $db){
    $user = getCurrentUser(); ensureHrRole($user); ensureTemplateTables($db);
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $templateId = (int)($in['template_id'] ?? 0); if ($templateId<=0) ApiResponse::error('template_id required');
    $title = trim((string)($in['title'] ?? '')); if ($title==='') ApiResponse::error('title required');
    $weight = is_numeric($in['weight'] ?? null) ? (float)$in['weight'] : 0;
    $st = $db->prepare('INSERT INTO appraisal_template_kpis (template_id, title, weight) VALUES (:tid,:title,:weight)');
    $st->execute([':tid'=>$templateId, ':title'=>$title, ':weight'=>$weight]);
    ApiResponse::success(['id'=>$db->lastInsertId()], 'KPI created');
}

function kpiUpdate(PDO $db){
    $user = getCurrentUser(); ensureHrRole($user); ensureTemplateTables($db);
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
    $title = isset($in['title']) ? trim((string)$in['title']) : null;
    $weight = array_key_exists('weight',$in) ? (is_numeric($in['weight']) ? (float)$in['weight'] : 0) : null;
    $set=[]; $params=[':id'=>$id];
    if ($title !== null){ $set[]='title = :title'; $params[':title']=$title; }
    if ($weight !== null){ $set[]='weight = :weight'; $params[':weight']=$weight; }
    if (!$set) ApiResponse::success(null,'No changes');
    $sql = 'UPDATE appraisal_template_kpis SET '.implode(', ',$set).' WHERE id = :id';
    $st = $db->prepare($sql); foreach($params as $k=>$v){ $st->bindValue($k,$v); } $st->execute();
    ApiResponse::success(null,'Updated');
}

function kpiDelete(PDO $db){
    $user = getCurrentUser(); ensureHrRole($user); ensureTemplateTables($db);
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
    $check = $db->prepare('SELECT COUNT(*) AS cnt FROM appraisal_kpi_scores WHERE kpi_id = :id');
    $check->execute([':id'=>$id]);
    $row = $check->fetch();
    if (($row['cnt'] ?? 0) > 0) ApiResponse::error('Cannot delete: KPI is used in appraisals');
    $st = $db->prepare('DELETE FROM appraisal_template_kpis WHERE id = :id');
    $st->execute([':id'=>$id]);
    ApiResponse::success(null,'Deleted');
}
