<?php
/**
 * Analytics API - returns metrics and insights for dashboard
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';
// bring in leave allocation utilities for grade-based allowed days calculations
require_once __DIR__ . '/_leave_allocations.inc.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

$database = new Database();
$db = $database->getConnection();

if ($method !== 'GET') {
    ApiResponse::error('Method not allowed', 405);
}

function hrDashboardStats(PDO $db){
    $user = getCurrentUser();
    $cid = (int)$user['company_id'];
    $year = isset($_GET['year']) && is_numeric($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');

    // Ensure leave schema presence
    if (function_exists('ensureLeaveTypesTable')) { ensureLeaveTypesTable($db); }
    if (function_exists('ensureAllocationsTable')) { ensureAllocationsTable($db); }
    if (function_exists('ensureEmployeesCurrentGradeColumn')) { ensureEmployeesCurrentGradeColumn($db); }

    // Headcount
    $head = singleRow($db, "SELECT 
        COUNT(*) AS total,
        SUM(CASE WHEN employment_status='active' THEN 1 ELSE 0 END) AS active
      FROM employees WHERE company_id=:cid", [':cid'=>$cid]);

    // Gender distribution (active only)
    $gender = [];
    try{
        $st = $db->prepare("SELECT LOWER(COALESCE(NULLIF(TRIM(gender),''),'unspecified')) AS g, COUNT(*) AS c
                             FROM employees WHERE company_id=:cid AND employment_status='active'
                             GROUP BY g ORDER BY c DESC");
        $st->execute([':cid'=>$cid]);
        $gender = $st->fetchAll();
    }catch(Throwable $e){ $gender=[]; }

    // Department distribution (active only)
    $dept = [];
    try{
        $st = $db->prepare("SELECT d.id, d.name, COUNT(e.id) AS count
                             FROM departments d
                             LEFT JOIN employees e ON e.department_id=d.id AND e.employment_status='active'
                             WHERE d.company_id=:cid
                             GROUP BY d.id, d.name
                             ORDER BY count DESC, d.name ASC");
        $st->execute([':cid'=>$cid]);
        $dept = $st->fetchAll();
    }catch(Throwable $e){ $dept=[]; }

    // Vacancies
    $vac = [
        'open_current' => 0,
        'opened_this_year' => 0,
        'filled_this_year' => 0,
    ];
    try{
        $vac['open_current'] = (int)singleValue($db, "SELECT COUNT(*) FROM job_postings WHERE company_id=:cid AND status='open'", [':cid'=>$cid]);
        $vac['opened_this_year'] = (int)singleValue($db, "SELECT COUNT(*) FROM job_postings WHERE company_id=:cid AND YEAR(created_at)=:yr AND status IN ('open','closed')", [':cid'=>$cid, ':yr'=>$year]);
        $vac['filled_this_year'] = (int)singleValue($db, "SELECT COUNT(*) FROM job_applications a JOIN job_postings jp ON jp.id=a.job_posting_id WHERE jp.company_id=:cid AND a.status='hired' AND YEAR(a.application_date)=:yr", [':cid'=>$cid, ':yr'=>$year]);
    }catch(Throwable $e){ /* defaults */ }

    // Leave allocations vs used per type for the year
    $leave = [];
    try{
        // Allowed total days across active employees per type
        $allowedSql = "SELECT lt.id AS leave_type_id, lt.name, lt.code,
                               COALESCE(SUM(COALESCE(la.days_per_year, lt.days_per_year)),0) AS allowed_total_days
                        FROM leave_types lt
                        JOIN employees e ON e.company_id = lt.company_id AND e.employment_status='active'
                        LEFT JOIN grades g ON g.id = e.grade_id
                        LEFT JOIN leave_allocations la 
                          ON la.company_id = lt.company_id 
                         AND la.leave_type_id = lt.id 
                         AND la.grade = COALESCE(NULLIF(TRIM(g.code),''), g.name, e.current_grade)
                        WHERE lt.company_id = :cid
                        GROUP BY lt.id, lt.name, lt.code
                        ORDER BY lt.name";
        $a = $db->prepare($allowedSql); $a->execute([':cid'=>$cid]); $allowedRows = $a->fetchAll();
        $map = [];
        foreach ($allowedRows as $r){ $map[(int)$r['leave_type_id']] = [
            'leave_type_id'=>(int)$r['leave_type_id'],
            'name'=>$r['name'],
            'code'=>$r['code'],
            'allowed_total_days'=>(int)$r['allowed_total_days'],
            'used_days'=>0,
            'pending_days'=>0,
        ]; }

        // Used and pending days in the year
        $usedSql = "SELECT lr.leave_type_id,
                           SUM(CASE WHEN lr.status='approved' THEN lr.days_requested ELSE 0 END) AS used_days,
                           SUM(CASE WHEN lr.status='pending' THEN lr.days_requested ELSE 0 END) AS pending_days
                    FROM leave_requests lr
                    JOIN employees e ON e.id = lr.employee_id
                    WHERE e.company_id=:cid AND YEAR(lr.start_date)=:yr
                    GROUP BY lr.leave_type_id";
        $u = $db->prepare($usedSql); $u->execute([':cid'=>$cid, ':yr'=>$year]);
        foreach ($u->fetchAll() as $r){ $id = (int)($r['leave_type_id'] ?? 0); if ($id<=0) continue; if (!isset($map[$id])) $map[$id] = ['leave_type_id'=>$id,'name'=>'','code'=>'','allowed_total_days'=>0,'used_days'=>0,'pending_days'=>0]; $map[$id]['used_days'] = (int)($r['used_days'] ?? 0); $map[$id]['pending_days'] = (int)($r['pending_days'] ?? 0); }

        $leave = array_values($map);
    }catch(Throwable $e){ $leave = []; }

    // Retention (last 12 months): compute month start/end, headcount at start, terminations in month, retention_rate
    $retention = [ 'labels'=>[], 'headcount_start'=>[], 'terminations'=>[], 'retention_rate'=>[] ];
    try{
        $months = [];
        for ($i=11; $i>=0; $i--) {
            $ts = strtotime(date('Y-m-01', strtotime("-$i months")));
            $start = date('Y-m-01', $ts);
            $end = date('Y-m-t', $ts);
            $label = date('Y-m', $ts);
            $months[] = ['start'=>$start,'end'=>$end,'label'=>$label];
        }
        foreach ($months as $m) {
            $hcStart = (int)singleValue($db, "SELECT COUNT(*) FROM employees WHERE company_id=:cid AND hire_date <= :start AND (staff_exited_date IS NULL OR staff_exited_date > :start)", [':cid'=>$cid, ':start'=>$m['start']]);
            $terms = (int)singleValue($db, "SELECT COUNT(*) FROM employees WHERE company_id=:cid AND staff_exited_date IS NOT NULL AND staff_exited_date BETWEEN :s AND :e", [':cid'=>$cid, ':s'=>$m['start'], ':e'=>$m['end']]);
            $rate = $hcStart > 0 ? max(0.0, min(100.0, 100.0 * (1.0 - ($terms / $hcStart)))) : 100.0;
            $retention['labels'][] = $m['label'];
            $retention['headcount_start'][] = $hcStart;
            $retention['terminations'][] = $terms;
            $retention['retention_rate'][] = round($rate, 1);
        }
    }catch(Throwable $e){ $retention = [ 'labels'=>[], 'headcount_start'=>[], 'terminations'=>[], 'retention_rate'=>[] ]; }

    // Pending exit requests (if table exists)
    $exitsPending = 0;
    try { $exitsPending = (int)singleValue($db, "SELECT COUNT(*) FROM exit_requests er WHERE er.company_id=:cid AND er.status='pending'", [':cid'=>$cid]); } catch (Throwable $e) { $exitsPending = 0; }

    ApiResponse::success([
        'year' => $year,
        'employees' => [
            'total' => (int)($head['total'] ?? 0),
            'active' => (int)($head['active'] ?? 0),
        ],
        'gender_distribution' => $gender,
        'department_distribution' => $dept,
        'vacancies' => $vac,
        'leave_types' => $leave,
        'retention' => $retention,
        'exits_pending' => $exitsPending,
    ]);
}

function peopleWindow(PDO $db){
    $user = getCurrentUser();
    $companyId = (int)$user['company_id'];
    $days = isset($_GET['days']) && is_numeric($_GET['days']) ? max(1, min(31, (int)$_GET['days'])) : 3;
    $end = date('Y-m-d', time() + $days*86400);

    // Upcoming birthdays (next occurrence within N days)
    $birthdays = [];
    try {
        $occExpr = "CASE 
            WHEN DATE_FORMAT(e.date_of_birth,'%m-%d') >= DATE_FORMAT(CURDATE(),'%m-%d') 
              THEN STR_TO_DATE(CONCAT(YEAR(CURDATE()), '-', DATE_FORMAT(e.date_of_birth,'%m-%d')), '%Y-%m-%d') 
            ELSE DATE_ADD(STR_TO_DATE(CONCAT(YEAR(CURDATE()), '-', DATE_FORMAT(e.date_of_birth,'%m-%d')), '%Y-%m-%d'), INTERVAL 1 YEAR) 
          END";
        $sql = "SELECT e.id AS employee_id, e.first_name, e.last_name, e.date_of_birth,
                       d.name AS department_name, u.id AS user_id,
                       $occExpr AS occurs_on
                FROM employees e
                LEFT JOIN departments d ON d.id = e.department_id
                LEFT JOIN users u ON u.id = e.user_id
                WHERE e.company_id = :cid AND e.employment_status='active' AND e.date_of_birth IS NOT NULL
                  AND ($occExpr) <= :end
                ORDER BY occurs_on ASC, e.first_name, e.last_name
                LIMIT 100";
        $st = $db->prepare($sql);
        $st->bindValue(':cid', $companyId, PDO::PARAM_INT);
        $st->bindValue(':end', $end);
        $st->execute();
        $birthdays = $st->fetchAll();
    } catch (Throwable $e){ $birthdays = []; }

    // Upcoming anniversaries (>=1 year)
    $anniv = [];
    try {
        $occExpr = "CASE 
            WHEN DATE_FORMAT(e.hire_date,'%m-%d') >= DATE_FORMAT(CURDATE(),'%m-%d') 
              THEN STR_TO_DATE(CONCAT(YEAR(CURDATE()), '-', DATE_FORMAT(e.hire_date,'%m-%d')), '%Y-%m-%d') 
            ELSE DATE_ADD(STR_TO_DATE(CONCAT(YEAR(CURDATE()), '-', DATE_FORMAT(e.hire_date,'%m-%d')), '%Y-%m-%d'), INTERVAL 1 YEAR) 
          END";
        $sql = "SELECT e.id AS employee_id, e.first_name, e.last_name, e.hire_date,
                       d.name AS department_name, u.id AS user_id,
                       $occExpr AS occurs_on,
                       TIMESTAMPDIFF(YEAR, e.hire_date, $occExpr) AS years
                FROM employees e
                LEFT JOIN departments d ON d.id = e.department_id
                LEFT JOIN users u ON u.id = e.user_id
                WHERE e.company_id = :cid AND e.employment_status='active' AND e.hire_date IS NOT NULL
                  AND ($occExpr) <= :end
                  AND TIMESTAMPDIFF(YEAR, e.hire_date, $occExpr) >= 1
                ORDER BY occurs_on ASC, e.first_name, e.last_name
                LIMIT 100";
        $st = $db->prepare($sql);
        $st->bindValue(':cid', $companyId, PDO::PARAM_INT);
        $st->bindValue(':end', $end);
        $st->execute();
        $anniv = $st->fetchAll();
    } catch (Throwable $e){ $anniv = []; }

    ApiResponse::success([
        'birthdays' => $birthdays,
        'anniversaries' => $anniv,
        'end' => $end,
        'days' => $days,
    ]);
}

function people(PDO $db) {
    $user = getCurrentUser();
    $companyId = (int)$user['company_id'];
    $userId = (int)$user['id'];

    // Profile summary for current user
    $profile = [];
    try {
        $q = $db->prepare("SELECT u.id AS user_id, u.username, u.first_name, u.last_name, u.email,
                                   e.id AS employee_id, e.department_id, d.name AS department_name
                            FROM users u
                            LEFT JOIN employees e ON e.id = u.employee_id
                            LEFT JOIN departments d ON d.id = e.department_id
                            WHERE u.id = :uid LIMIT 1");
        $q->bindValue(':uid', $userId, PDO::PARAM_INT); $q->execute();
        $profile = $q->fetch() ?: [];

        // Roles for current user
        $r = $db->prepare("SELECT r.name, r.slug, r.level FROM user_roles ur JOIN roles r ON r.id = ur.role_id WHERE ur.user_id = :uid ORDER BY r.level DESC, r.name");
        $r->bindValue(':uid', $userId, PDO::PARAM_INT); $r->execute();
        $profile['roles'] = $r->fetchAll();
    } catch (Throwable $e) { $profile = []; }

    // Today's birthdays
    $birthdays = [];
    try {
        $b = $db->prepare("SELECT e.id AS employee_id, e.first_name, e.last_name, e.date_of_birth,
                                   d.name AS department_name, u.id AS user_id,
                                   TIMESTAMPDIFF(YEAR, e.date_of_birth, CURDATE()) AS turning
                            FROM employees e
                            LEFT JOIN departments d ON d.id = e.department_id
                            LEFT JOIN users u ON u.id = e.user_id
                            WHERE e.company_id = :cid AND e.employment_status = 'active'
                              AND e.date_of_birth IS NOT NULL
                              AND DATE_FORMAT(e.date_of_birth, '%m-%d') = DATE_FORMAT(CURDATE(), '%m-%d')");
        $b->bindValue(':cid', $companyId, PDO::PARAM_INT); $b->execute();
        $birthdays = $b->fetchAll();
    } catch (Throwable $e) { $birthdays = []; }

    // Today's work anniversaries (>= 1 year)
    $anniv = [];
    try {
        $a = $db->prepare("SELECT e.id AS employee_id, e.first_name, e.last_name, e.hire_date,
                                   d.name AS department_name, u.id AS user_id,
                                   TIMESTAMPDIFF(YEAR, e.hire_date, CURDATE()) AS years
                            FROM employees e
                            LEFT JOIN departments d ON d.id = e.department_id
                            LEFT JOIN users u ON u.id = e.user_id
                            WHERE e.company_id = :cid AND e.employment_status = 'active'
                              AND e.hire_date IS NOT NULL
                              AND e.hire_date <= CURDATE()
                              AND DATE_FORMAT(e.hire_date, '%m-%d') = DATE_FORMAT(CURDATE(), '%m-%d')
                              AND TIMESTAMPDIFF(YEAR, e.hire_date, CURDATE()) >= 1");
        $a->bindValue(':cid', $companyId, PDO::PARAM_INT); $a->execute();
        $anniv = $a->fetchAll();
    } catch (Throwable $e) { $anniv = []; }

    ApiResponse::success([
        'profile' => $profile,
        'birthdays_today' => $birthdays,
        'anniversaries_today' => $anniv,
    ]);
}

switch ($action) {
    case 'insights':
        insights($db);
        break;
    case 'people':
        people($db);
        break;
    case 'people_window':
        peopleWindow($db);
        break;
    case 'hr_dashboard_stats':
        hrDashboardStats($db);
        break;
    default:
        metrics($db);
}

function metrics($db) {
    $user = getCurrentUser();
    $companyId = $user['company_id'];

    // Headcount metrics
    $totals = singleRow($db, "SELECT 
        COUNT(*) AS employees_total,
        SUM(CASE WHEN employment_status = 'active' THEN 1 ELSE 0 END) AS employees_active
      FROM employees WHERE company_id = :cid", [':cid' => $companyId]);

    $newHires30 = singleValue($db, "SELECT COUNT(*) FROM employees WHERE company_id = :cid AND hire_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)", [':cid' => $companyId]);

    $terminations30 = singleValue($db, "SELECT COUNT(*) FROM employees WHERE company_id = :cid AND employment_status = 'terminated' AND updated_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)", [':cid' => $companyId]);

    // Workflow counts
    $leavePending = singleValue($db, "SELECT COUNT(*) FROM leave_requests lr JOIN employees e ON lr.employee_id = e.id WHERE e.company_id = :cid AND lr.status = 'pending'", [':cid' => $companyId]);

    $trainingActive = singleValue($db, "SELECT COUNT(*) FROM trainings WHERE company_id = :cid AND status = 'active'", [':cid' => $companyId]);

    $appraisalsInProgress = singleValue($db, "SELECT COUNT(*) FROM appraisals a JOIN employees e ON a.employee_id = e.id WHERE e.company_id = :cid AND a.status IN ('submitted','reviewed')", [':cid' => $companyId]);

    $payrollPending = singleValue($db, "SELECT COUNT(*) FROM payrolls WHERE company_id = :cid AND status IN ('calculated')", [':cid' => $companyId]);

    // Department distribution
    $deptDist = listRows($db, "SELECT d.id, d.name, COUNT(e.id) AS count
                               FROM departments d
                               LEFT JOIN employees e ON e.department_id = d.id
                               WHERE d.company_id = :cid
                               GROUP BY d.id, d.name
                               ORDER BY count DESC, d.name ASC", [':cid' => $companyId]);

    // New hires by month (last 6 months)
    $hiresTrend = listRows($db, "SELECT DATE_FORMAT(hire_date, '%Y-%m') AS ym, COUNT(*) AS hires
                                  FROM employees
                                  WHERE company_id = :cid AND hire_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
                                  GROUP BY ym ORDER BY ym ASC", [':cid' => $companyId]);

    ApiResponse::success([
        'headcount' => [
            'total' => intval($totals['employees_total'] ?? 0),
            'active' => intval($totals['employees_active'] ?? 0),
            'new_hires_30d' => intval($newHires30 ?? 0),
            'terminations_30d' => intval($terminations30 ?? 0)
        ],
        'workflows' => [
            'leave_pending' => intval($leavePending ?? 0),
            'training_active' => intval($trainingActive ?? 0),
            'appraisals_in_progress' => intval($appraisalsInProgress ?? 0),
            'payroll_pending' => intval($payrollPending ?? 0)
        ],
        'department_distribution' => $deptDist,
        'hires_trend' => $hiresTrend
    ]);
}

function insights($db) {
    $user = getCurrentUser();
    $companyId = $user['company_id'];

    $metrics = [
        'active' => singleValue($db, "SELECT SUM(CASE WHEN employment_status='active' THEN 1 ELSE 0 END) FROM employees WHERE company_id=:cid", [':cid' => $companyId]),
        'pendingLeave' => singleValue($db, "SELECT COUNT(*) FROM leave_requests lr JOIN employees e ON lr.employee_id = e.id WHERE e.company_id=:cid AND lr.status='pending'", [':cid' => $companyId]),
        'inProgressAppraisal' => singleValue($db, "SELECT COUNT(*) FROM appraisals a JOIN employees e ON a.employee_id=e.id WHERE e.company_id=:cid AND a.status IN ('submitted','reviewed')", [':cid' => $companyId]),
        'activeTraining' => singleValue($db, "SELECT COUNT(*) FROM trainings WHERE company_id=:cid AND status='active'", [':cid' => $companyId])
    ];

    $topDept = singleRow($db, "SELECT d.name, COUNT(e.id) AS c FROM departments d LEFT JOIN employees e ON e.department_id=d.id WHERE d.company_id=:cid GROUP BY d.id ORDER BY c DESC LIMIT 1", [':cid' => $companyId]);

    $messages = [];
    $active = intval($metrics['active'] ?? 0);
    $pl = intval($metrics['pendingLeave'] ?? 0);
    $ap = intval($metrics['inProgressAppraisal'] ?? 0);
    $tr = intval($metrics['activeTraining'] ?? 0);

    $messages[] = "Active headcount is $active. " . ($topDept ? ("Largest department: ".$topDept['name']." (".intval($topDept['c'])." employees).") : '');
    if ($pl > 0) $messages[] = "There are $pl pending leave requests requiring manager/HR attention.";
    if ($ap > 0) $messages[] = "$ap appraisals are awaiting reviews or approvals.";
    if ($tr > 0) $messages[] = "$tr trainings are currently active; consider targeting high-turnover departments for upskilling.";

    ApiResponse::success(['insights' => $messages]);
}

function singleValue($db, $sql, $params) {
    $stmt = $db->prepare($sql);
    foreach ($params as $k => $v) $stmt->bindValue($k, $v);
    $stmt->execute();
    $row = $stmt->fetch();
    if ($row) {
        $first = array_values($row)[0];
        return is_null($first) ? 0 : $first;
    }
    return 0;
}

function singleRow($db, $sql, $params) {
    $stmt = $db->prepare($sql);
    foreach ($params as $k => $v) $stmt->bindValue($k, $v);
    $stmt->execute();
    return $stmt->fetch() ?: [];
}

function listRows($db, $sql, $params) {
    $stmt = $db->prepare($sql);
    foreach ($params as $k => $v) $stmt->bindValue($k, $v);
    $stmt->execute();
    return $stmt->fetchAll();
}
