<?php
/**
 * Admin Migration Tools (DEV ONLY)
 * Adds missing employee fields inspired by legacy `staff` table.
 *
 * Security: Requires authenticated user with role super_admin or admin.
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ApiResponse::success(null, 'OK');
}

requireAuth();
$user = getCurrentUser();
if (!in_array($user['role_slug'], ['super_admin', 'admin'])) {
    ApiResponse::forbidden('Admin only');
}

$database = new Database();
$db = $database->getConnection();
$action = $_GET['action'] ?? '';

switch ($action) {
    case 'extend_employees':
        extendEmployees($db);
        break;
    case 'extend_payroll_ghana':
        extendPayrollGhana($db);
        break;
    default:
        ApiResponse::error('Invalid action');
}

function ensureColumn(PDO $db, $table, $column, $definition) {
    $q = $db->prepare("SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t AND COLUMN_NAME = :c");
    $q->bindParam(':t', $table);
    $q->bindParam(':c', $column);
    $q->execute();
    if ($q->rowCount() === 0) {
        $sql = "ALTER TABLE `$table` ADD `$column` $definition";
        $db->exec($sql);
        return true;
    }
    return false;
}

function extendEmployees(PDO $db) {
    $added = [];
    $table = 'employees';

    $columns = [
        // Identity & salutations
        'salutation' => "VARCHAR(50) NULL",
        'title' => "VARCHAR(255) NULL",
        'id_type' => "VARCHAR(50) NULL",
        'id_number' => "VARCHAR(100) NULL",

        // Contact & Address
        'official_email' => "VARCHAR(255) NULL",
        'digital_address' => "VARCHAR(255) NULL",
        'house_number' => "VARCHAR(255) NULL",
        'land_mark' => "VARCHAR(255) NULL",
        'postal_address' => "VARCHAR(255) NULL",
        'region' => "VARCHAR(255) NULL",
        'religion' => "VARCHAR(255) NULL",
        'residential_ownership' => "VARCHAR(255) NULL",
        'staff_location' => "VARCHAR(255) NULL",
        'location' => "VARCHAR(255) NULL",

        // Education
        'certificate_type' => "VARCHAR(255) NULL",
        'first_degree_program' => "VARCHAR(255) NULL",
        'first_degree_class' => "VARCHAR(255) NULL",
        'first_degree_institution' => "MEDIUMTEXT NULL",
        'second_degree' => "VARCHAR(255) NULL",
        'second_degree_program' => "VARCHAR(255) NULL",
        'second_degree_class' => "VARCHAR(255) NULL",
        'second_degree_institution' => "MEDIUMTEXT NULL",
        'professional_certificate' => "VARCHAR(255) NULL",
        'professional_institution' => "VARCHAR(255) NULL",
        'professional_results' => "VARCHAR(255) NULL",
        'academic_program' => "MEDIUMTEXT NULL",

        // Next of kin (distinct from emergency_contact_*)
        'nok_name' => "VARCHAR(255) NULL",
        'nok_contact' => "VARCHAR(255) NULL",
        'nok_postal_address' => "VARCHAR(255) NULL",
        'nok_date' => "DATE NULL",
        'relationship' => "VARCHAR(255) NULL",
        'nok_home_address' => "VARCHAR(255) NULL",

        // Job info & grades
        'job_title' => "VARCHAR(255) NULL",
        'designation' => "VARCHAR(255) NULL",
        'job_description' => "VARCHAR(255) NULL",
        'department_group' => "VARCHAR(255) NULL",
        'department_unit' => "VARCHAR(255) NULL",
        'branch_type' => "VARCHAR(255) NULL",
        'starting_grade' => "VARCHAR(255) NULL",
        'current_grade' => "VARCHAR(255) NULL",
        'joined_department_date' => "DATETIME NULL",
        'last_promotion_date' => "DATE NULL",
        'confirmation' => "VARCHAR(255) NULL",
        'service_length' => "VARCHAR(255) NULL",
        'staff_type' => "VARCHAR(255) NULL",
        'employee_type' => "VARCHAR(20) NULL",
        'person_type' => "VARCHAR(20) NULL",
        'staff_status' => "VARCHAR(255) NULL",

        // Verification & misc
        'assigned_role' => "VARCHAR(255) NULL",
        'verified' => "TINYINT(1) NULL DEFAULT 0",
        'verified_by' => "VARCHAR(255) NULL",
        'verification_date' => "DATETIME NULL",
        'profile_pic' => "VARCHAR(255) NULL",
        'staff_exited_date' => "DATE NULL",

        // Maintenance
        'last_modified_by' => "INT NULL",
        'is_deleted' => "TINYINT(1) NULL DEFAULT 0"
    ];

    foreach ($columns as $col => $def) {
        if (ensureColumn($db, $table, $col, $def)) {
            $added[] = $col;
        }
    }

    ApiResponse::success(['added' => $added], 'Employees table extended');
}

function ensureTable(PDO $db, $table, $ddl) {
    $q = $db->prepare("SELECT 1 FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t");
    $q->bindParam(':t', $table);
    $q->execute();
    if ($q->rowCount() === 0) {
        $db->exec($ddl);
        return true;
    }
    return false;
}

function extendPayrollGhana(PDO $db) {
    $created = [];
    $addedCols = [];

    // 1) Base types for payroll
    if (ensureTable($db, 'allowance_types',
        "CREATE TABLE allowance_types (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            name VARCHAR(255) NOT NULL,
            code VARCHAR(50) NULL,
            taxable TINYINT(1) NOT NULL DEFAULT 1,
            taxable_rate DECIMAL(5,2) NULL,
            created_at DATETIME DEFAULT NOW()
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4")) { $created[] = 'allowance_types'; }
    // Ensure new columns exist if table already present
    ensureColumn($db, 'allowance_types', 'taxable', 'TINYINT(1) NOT NULL DEFAULT 1');
    ensureColumn($db, 'allowance_types', 'taxable_rate', 'DECIMAL(5,2) NULL');

    if (ensureTable($db, 'deduction_types',
        "CREATE TABLE deduction_types (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            name VARCHAR(255) NOT NULL,
            code VARCHAR(50) NULL,
            created_at DATETIME DEFAULT NOW()
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4")) { $created[] = 'deduction_types'; }

    // 2) Employee allowance/deduction linking tables (add missing columns if needed)
    ensureTable($db, 'employee_allowances',
        "CREATE TABLE employee_allowances (
            id INT AUTO_INCREMENT PRIMARY KEY,
            employee_id INT NOT NULL,
            allowance_type_id INT NOT NULL,
            amount DECIMAL(12,2) NOT NULL DEFAULT 0,
            status VARCHAR(20) NOT NULL DEFAULT 'active',
            start_date DATE NULL,
            end_date DATE NULL,
            created_at DATETIME DEFAULT NOW()
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    ensureColumn($db, 'employee_allowances', 'allowance_type_id', 'INT NOT NULL');

    ensureTable($db, 'employee_deductions',
        "CREATE TABLE employee_deductions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            employee_id INT NOT NULL,
            deduction_type_id INT NOT NULL,
            amount DECIMAL(12,2) NOT NULL DEFAULT 0,
            status VARCHAR(20) NOT NULL DEFAULT 'active',
            start_date DATE NULL,
            end_date DATE NULL,
            created_at DATETIME DEFAULT NOW()
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    ensureColumn($db, 'employee_deductions', 'deduction_type_id', 'INT NOT NULL');

    // 3) Grades and default components
    if (ensureTable($db, 'grades',
        "CREATE TABLE grades (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            name VARCHAR(255) NOT NULL,
            code VARCHAR(50) NULL,
            base_salary DECIMAL(12,2) NULL,
            created_by INT NULL,
            created_at DATETIME DEFAULT NOW()
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4")) { $created[] = 'grades'; }

    if (ensureTable($db, 'grade_allowances',
        "CREATE TABLE grade_allowances (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            grade_id INT NOT NULL,
            allowance_type_id INT NOT NULL,
            amount DECIMAL(12,2) NOT NULL DEFAULT 0,
            created_at DATETIME DEFAULT NOW()
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4")) { $created[] = 'grade_allowances'; }

    if (ensureTable($db, 'grade_deductions',
        "CREATE TABLE grade_deductions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            grade_id INT NOT NULL,
            deduction_type_id INT NOT NULL,
            amount DECIMAL(12,2) NOT NULL DEFAULT 0,
            created_at DATETIME DEFAULT NOW()
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4")) { $created[] = 'grade_deductions'; }

    // 4) Employees: grade_id
    if (ensureColumn($db, 'employees', 'grade_id', 'INT NULL')) { $addedCols[] = 'employees.grade_id'; }

    // 5) Payrolls breakdown for Ghana
    if (ensureColumn($db, 'payrolls', 'ssnit_employee', 'DECIMAL(12,2) NULL')) { $addedCols[] = 'payrolls.ssnit_employee'; }
    if (ensureColumn($db, 'payrolls', 'ssnit_employer', 'DECIMAL(12,2) NULL')) { $addedCols[] = 'payrolls.ssnit_employer'; }
    if (ensureColumn($db, 'payrolls', 'chargeable_income', 'DECIMAL(12,2) NULL')) { $addedCols[] = 'payrolls.chargeable_income'; }
    if (ensureColumn($db, 'payrolls', 'taxable_allowances', 'DECIMAL(12,2) NULL')) { $addedCols[] = 'payrolls.taxable_allowances'; }
    if (ensureColumn($db, 'payrolls', 'exempt_allowances', 'DECIMAL(12,2) NULL')) { $addedCols[] = 'payrolls.exempt_allowances'; }
    if (ensureColumn($db, 'payrolls', 'approved_at', 'DATETIME NULL')) { $addedCols[] = 'payrolls.approved_at'; }
    if (ensureColumn($db, 'payrolls', 'paid_at', 'DATETIME NULL')) { $addedCols[] = 'payrolls.paid_at'; }
    if (ensureColumn($db, 'payrolls', 'updated_at', 'DATETIME NULL')) { $addedCols[] = 'payrolls.updated_at'; }

    ApiResponse::success([
        'created_tables' => $created,
        'added_columns' => $addedCols
    ], 'Payroll (Ghana) and Grades schema ensured');
}
