<?php
// Helper and endpoints for grade-based leave allocations

if (!function_exists('ensureAllocationsTable')) {
    function ensureAllocationsTable(PDO $db){
        $sql = "CREATE TABLE IF NOT EXISTS leave_allocations (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    company_id INT NOT NULL,
                    grade VARCHAR(100) NOT NULL,
                    leave_type_id INT NOT NULL,
                    days_per_year INT NOT NULL,
                    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    UNIQUE KEY uniq_company_grade_type (company_id, grade, leave_type_id)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        try { $db->exec($sql); } catch (Throwable $e) { /* ignore */ }
    }
}

// Ensure base leave_types table exists (used by balance and allocations)
if (!function_exists('ensureLeaveTypesTable')) {
    function ensureLeaveTypesTable(PDO $db){
        $sql = "CREATE TABLE IF NOT EXISTS leave_types (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    company_id INT NOT NULL,
                    name VARCHAR(100) NOT NULL,
                    code VARCHAR(50) NULL,
                    days_per_year INT NOT NULL DEFAULT 0,
                    color VARCHAR(20) NULL,
                    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    KEY idx_company (company_id),
                    KEY idx_code (code)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        try { $db->exec($sql); } catch (Throwable $e) { /* ignore */ }
    }
}

// Ensure employees.current_grade exists for grade-based allocations
if (!function_exists('ensureEmployeesCurrentGradeColumn')) {
    function ensureEmployeesCurrentGradeColumn(PDO $db){
        try {
            $q = $db->prepare("SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'employees' AND COLUMN_NAME = 'current_grade'");
            $q->execute();
            if ($q->rowCount() === 0) {
                $db->exec("ALTER TABLE employees ADD current_grade VARCHAR(255) NULL");
            }
        } catch (Throwable $e) { /* ignore */ }
    }
}

if (!function_exists('getLeaveAllocations')) {
    function getLeaveAllocations(PDO $db){
        $user = getCurrentUser();
        $role = $user['role_slug'] ?? '';
        $allowed = in_array($role, ['super_admin','admin','hr_head','hr_officer'], true);
        if (!$allowed) ApiResponse::forbidden('You do not have access to allocations');

        $cid = (int)$user['company_id'];

        try {
            // Ensure required schema is present
            if (function_exists('ensureLeaveTypesTable')) { ensureLeaveTypesTable($db); }
            if (function_exists('ensureEmployeesCurrentGradeColumn')) { ensureEmployeesCurrentGradeColumn($db); }
            if (function_exists('ensureAllocationsTable')) { ensureAllocationsTable($db); }

            // Load leave types
            $typesStmt = $db->prepare('SELECT id, name, code FROM leave_types WHERE company_id = :cid ORDER BY name');
            $typesStmt->bindValue(':cid', $cid, PDO::PARAM_INT);
            $typesStmt->execute();
            $types = $typesStmt->fetchAll();

            // Build grade list from grades table (prefer code, fallback to name) + any distinct employee current_grade
            $grades = [];
            $seen = [];
            // 1) From grades table
            try {
                $g1 = $db->prepare("SELECT COALESCE(NULLIF(TRIM(code),''), name) AS g FROM grades WHERE company_id = :cid ORDER BY name");
                $g1->bindValue(':cid', $cid, PDO::PARAM_INT);
                $g1->execute();
                foreach ($g1->fetchAll() as $row) {
                    $v = isset($row['g']) ? trim((string)$row['g']) : '';
                    if ($v !== '') {
                        $k = strtolower($v);
                        if (!isset($seen[$k])) { $seen[$k] = true; $grades[] = $v; }
                    }
                }
            } catch (Throwable $e) { /* ignore */ }

            // 2) From employees.current_grade
            try {
                $g2 = $db->prepare("SELECT DISTINCT current_grade AS g FROM employees WHERE company_id = :cid AND current_grade IS NOT NULL AND current_grade <> ''");
                $g2->bindValue(':cid', $cid, PDO::PARAM_INT);
                $g2->execute();
                foreach ($g2->fetchAll() as $row) {
                    $v = isset($row['g']) ? trim((string)$row['g']) : '';
                    if ($v !== '') {
                        $k = strtolower($v);
                        if (!isset($seen[$k])) { $seen[$k] = true; $grades[] = $v; }
                    }
                }
            } catch (Throwable $e) { /* ignore */ }

            // Load allocations
            $allocStmt = $db->prepare('SELECT grade, leave_type_id, days_per_year FROM leave_allocations WHERE company_id = :cid');
            $allocStmt->bindValue(':cid', $cid, PDO::PARAM_INT);
            $allocStmt->execute();
            $allocRows = $allocStmt->fetchAll();

            ApiResponse::success([
                'leave_types' => $types,
                'grades' => $grades,
                'allocations' => $allocRows
            ]);
        } catch (Throwable $e) {
            ApiResponse::error('Failed to load allocations: ' . $e->getMessage(), 500);
        }
    }
}

if (!function_exists('saveLeaveAllocations')) {
    function saveLeaveAllocations(PDO $db){
        $user = getCurrentUser();
        $role = $user['role_slug'] ?? '';
        $allowed = in_array($role, ['super_admin','admin','hr_head','hr_officer'], true);
        if (!$allowed) ApiResponse::forbidden('You do not have permission to save allocations');

        ensureAllocationsTable($db);

        $cid = (int)$user['company_id'];
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        $allocs = isset($input['allocations']) && is_array($input['allocations']) ? $input['allocations'] : [];
        if (!$allocs) ApiResponse::error('No allocations provided');

        $db->beginTransaction();
        try {
            $sql = "INSERT INTO leave_allocations (company_id, grade, leave_type_id, days_per_year)
                    VALUES (:cid, :grade, :type_id, :days)
                    ON DUPLICATE KEY UPDATE days_per_year = VALUES(days_per_year), updated_at = NOW()";
            $stmt = $db->prepare($sql);
            foreach ($allocs as $a){
                $grade = trim((string)($a['grade'] ?? ''));
                $typeId = (int)($a['leave_type_id'] ?? 0);
                $days = (int)($a['days_per_year'] ?? 0);
                if ($grade === '' || $typeId <= 0 || $days < 0) continue;
                $stmt->bindValue(':cid', $cid, PDO::PARAM_INT);
                $stmt->bindValue(':grade', $grade);
                $stmt->bindValue(':type_id', $typeId, PDO::PARAM_INT);
                $stmt->bindValue(':days', $days, PDO::PARAM_INT);
                $stmt->execute();
            }
            $db->commit();
        } catch (Throwable $e){
            $db->rollBack();
            ApiResponse::error('Failed to save allocations: '.$e->getMessage(), 500);
        }
        ApiResponse::success(null, 'Allocations saved');
    }
}

// Utility to resolve an employee's current grade
if (!function_exists('lookupEmployeeGrade')) {
    function lookupEmployeeGrade(PDO $db, int $employeeId): ?string {
        // 1) Prefer relational grade via grade_id -> grades.code/name
        try {
            $q = $db->prepare("SELECT COALESCE(NULLIF(TRIM(g.code),''), g.name) AS g
                                FROM employees e
                                JOIN grades g ON g.id = e.grade_id
                                WHERE e.id = :id LIMIT 1");
            $q->bindValue(':id', $employeeId, PDO::PARAM_INT);
            $q->execute();
            if ($q->rowCount() > 0) {
                $r = $q->fetch();
                $v = isset($r['g']) ? trim((string)$r['g']) : '';
                if ($v !== '') return $v;
            }
        } catch (Throwable $e) { /* ignore */ }

        // 2) Fallback to textual hint column current_grade
        try {
            $st = $db->prepare('SELECT current_grade FROM employees WHERE id = :id LIMIT 1');
            $st->bindValue(':id', $employeeId, PDO::PARAM_INT);
            $st->execute();
            if ($st->rowCount() === 0) return null;
            $row = $st->fetch();
            $g = isset($row['current_grade']) ? trim((string)$row['current_grade']) : '';
            return $g !== '' ? $g : null;
        } catch (Throwable $e) { return null; }
    }
}
